﻿using MCRA.Utils.ExtensionMethods;
using MCRA.Utils.Xml;
using System.Reflection;

namespace MCRA.Simulation.OutputGeneration.Helpers {

    /// <summary>
    /// Methods to automatically generate the HTML for summarizing a section-based report.
    /// </summary>
    public static class DataSectionHelper {

        /// <summary>
        /// Creates a data section and adds is to the summary section.
        /// </summary>
        /// <typeparam name="TRecord"></typeparam>
        /// <param name="name"></param>
        /// <param name="section"></param>
        /// <param name="items"></param>
        /// <param name="viewBag"></param>
        /// <param name="header"></param>
        /// <param name="hiddenProperties"></param>
        /// <returns></returns>
        public static CsvDataSummarySection CreateCsvDataSection<TRecord>(
            string name,
            SummarySection section,
            IList<TRecord> items,
            ViewParameters viewBag,
            bool header = true,
            IList<string> hiddenProperties = null
        ) {
            var id = StringExtensions.CreateFingerprint(section.SectionId + name);
            var builder = new DataSectionBuilder<TRecord>(section, header, viewBag, hiddenProperties) {
                Id = id,
                Section = section,
                ShowHeader = header,
                HiddenProperties = hiddenProperties,
            };
            return builder.Build(items, name, id);
        }

        /// <summary>
        /// Creates a data section with the csv data as obtained from
        /// the file generated by the write csv action.
        /// </summary>
        /// <param name="name"></param>
        /// <param name="section"></param>
        /// <param name="viewBag"></param>
        /// <param name="writeCsv"></param>
        /// <param name="visibleProperties"></param>
        /// <returns></returns>
        public static CsvDataSummarySection CreateCsvDataSection(
            string name,
            SummarySection section,
            ViewParameters viewBag,
            Action<string> writeCsv,
            IEnumerable<PropertyInfo> visibleProperties = null
        ) {
            var id = StringExtensions.CreateFingerprint(section.SectionId + name);
            var csvTempFile = $"{id}.csv";
            var filename = Path.Combine(viewBag.TempPath ?? Path.GetTempPath(), csvTempFile);

            var dataSection = new CsvDataSummarySection(
                name,
                filename,
                viewBag?.TitlePath,
                visibleProperties,
                viewBag?.UnitsDictionary
            );
            section.DataSections.Add(dataSection);

            writeCsv(dataSection.CsvFileName);

            return dataSection;
        }

        /// <summary>
        /// Creates a data section and adds is to the summary section.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="name"></param>
        /// <param name="section"></param>
        /// <param name="data"></param>
        /// <param name="titlePath"></param>
        /// <returns></returns>
        public static XmlDataSummarySection CreateXmlDataSection<T>(
            string name,
            SummarySection section,
            T data,
            string titlePath,
            string sectionId
        ) where T : new() {
            var xml = XmlSerialization.ToXml(data);
            var dataSection = new XmlDataSummarySection(
                name,
                xml,
                titlePath,
                section.SectionLabel
            );
            if (!string.IsNullOrEmpty(sectionId)) {
                dataSection.SectionId = sectionId;
            }
            section.XmlDataSections.Add(dataSection);
            return dataSection;
        }
    }
}
