﻿using MCRA.Data.Compiled.Objects;
using MCRA.General;
using MCRA.Simulation.Calculators.ExternalExposureCalculation;
using MCRA.Simulation.Calculators.KineticModelCalculation;
using MCRA.Simulation.Calculators.TargetExposuresCalculation.AggregateExposures;
using MCRA.Utils.ProgressReporting;
using MCRA.Utils.Statistics;

namespace MCRA.Simulation.Calculators.TargetExposuresCalculation.TargetExposuresCalculators {
    public class InternalTargetExposuresCalculator : ITargetExposuresCalculator {

        private readonly ICollection<IKineticModelCalculator> _kineticConversionCalculator;

        public InternalTargetExposuresCalculator(
            IDictionary<Compound, IKineticModelCalculator> kineticModelCalculators
        ) {
            _kineticConversionCalculator = kineticModelCalculators.Values;
        }

        public ICollection<AggregateIndividualDayExposure> ComputeAcute(
            ICollection<IExternalIndividualDayExposure> externalIndividualDayExposures,
            ICollection<Compound> substances,
            ICollection<ExposureRoute> routes,
            ExposureUnitTriple exposureUnit,
            ICollection<TargetUnit> targetUnits,
            IRandom generator,
            ProgressState progressState
        ) {
            var aggregateIndividualDayExposures = externalIndividualDayExposures
                .ToDictionary(
                    r => r.SimulatedIndividualDayId,
                    r => new AggregateIndividualDayExposure() {
                        SimulatedIndividual = r.SimulatedIndividual,
                        SimulatedIndividualDayId = r.SimulatedIndividualDayId,
                        Day = r.Day,
                        ExternalIndividualDayExposures = [r],
                    }
                );

            foreach (var calculator in _kineticConversionCalculator) {
                // Compute aggregate, internal exposures for this calculator.
                var internalIndividualDayExposures = calculator
                    .CalculateIndividualDayTargetExposures(
                        externalIndividualDayExposures,
                        routes,
                        exposureUnit,
                        targetUnits,
                        progressState,
                        generator
                    );

                // Merge the internal individual day exposures generated by the kinetic
                // conversion model with the combined individual target exposures.
                foreach (var individualTargetExposures in internalIndividualDayExposures) {
                    var aggregateIndividualDayExposure = aggregateIndividualDayExposures[individualTargetExposures.SimulatedIndividualDayId];

                    // Per internal target: merge substance target exposures with substance
                    // target exposures of the combined collection.
                    foreach (var target in individualTargetExposures.InternalTargetExposures) {
                        var exposureTarget = target.Key;
                        if (!aggregateIndividualDayExposure.InternalTargetExposures
                            .TryGetValue(exposureTarget, out var targetIndividualExposure)
                        ) {
                            // If the combined collection does not yet contain substance target exposures
                            // for the current target, then initialize the collection based on current records.
                            targetIndividualExposure = target.Value.ToDictionary(r => r.Key, r => r.Value);
                            aggregateIndividualDayExposure.InternalTargetExposures[exposureTarget] = targetIndividualExposure;
                        } else {
                            // The combined collection already contains substance target exposures for the
                            // current target; merge the substance target exposures with the existing records
                            // of the combined collection.
                            foreach (var value in target.Value) {
                                var substance = value.Key;
                                var substanceTargetExposure = value.Value;
                                if (!targetIndividualExposure.ContainsKey(substance)) {
                                    targetIndividualExposure[substance] = substanceTargetExposure;
                                } else {
                                    var msg = "Aggregation of multiple substance target exposures "
                                        + "from multiple kinetic conversion models not supported";
                                    throw new NotImplementedException(msg);
                                }
                            }
                        }
                    }
                }
            }

            var result = aggregateIndividualDayExposures.Values;
            return result;
        }

        public ICollection<AggregateIndividualExposure> ComputeChronic(
            ICollection<IExternalIndividualExposure> externalIndividualExposures,
            ICollection<Compound> substances,
            ICollection<ExposureRoute> routes,
            ExposureUnitTriple exposureUnit,
            ICollection<TargetUnit> targetUnits,
            IRandom generator,
            ProgressState progressState
        ) {
            var aggregateIndividualExposures = externalIndividualExposures
                .ToDictionary(
                    r => r.SimulatedIndividual.Id,
                    r => new AggregateIndividualExposure() {
                        SimulatedIndividual = r.SimulatedIndividual,
                        ExternalIndividualDayExposures = r.ExternalIndividualDayExposures,
                    }
                );

            foreach (var calculator in _kineticConversionCalculator) {
                // Compute aggregate, internal exposures for this calculator.
                var internalIndividualExposures = calculator
                    .CalculateIndividualTargetExposures(
                        externalIndividualExposures,
                        routes,
                        exposureUnit,
                        targetUnits,
                        progressState,
                        generator
                    );

                // Merge the internal individual day exposures generated by the kinetic
                // conversion model with the combined individual target exposures.
                foreach (var individualTargetExposures in internalIndividualExposures) {
                    var aggregateIndividualExposure = aggregateIndividualExposures[individualTargetExposures.SimulatedIndividual.Id];

                    // Per internal target: merge substance target exposures with substance
                    // target exposures of the combined collection.
                    foreach (var target in individualTargetExposures.InternalTargetExposures) {
                        var exposureTarget = target.Key;
                        if (!aggregateIndividualExposure.InternalTargetExposures
                            .TryGetValue(exposureTarget, out var targetIndividualExposure)
                        ) {
                            // If the combined collection does not yet contain substance target exposures
                            // for the current target, then initialize the collection based on current records.
                            targetIndividualExposure = target.Value.ToDictionary(r => r.Key, r => r.Value);
                            aggregateIndividualExposure.InternalTargetExposures[exposureTarget] = targetIndividualExposure;
                        } else {
                            // The combined collection already contains substance target exposures for the
                            // current target; merge the substance target exposures with the existing records
                            // of the combined collection.
                            foreach (var value in target.Value) {
                                var substance = value.Key;
                                var substanceTargetExposure = value.Value;
                                if (!targetIndividualExposure.ContainsKey(substance)) {
                                    targetIndividualExposure[substance] = substanceTargetExposure;
                                } else {
                                    var msg = "Aggregation of multiple substance target exposures "
                                        + "from multiple kinetic conversion models not supported";
                                    throw new NotImplementedException(msg);
                                }
                            }
                        }
                    }
                }
            }

            var result = aggregateIndividualExposures.Values;
            return result;
        }
    }
}
