﻿using MCRA.Data.Compiled.Wrappers;
using MCRA.Data.Management.RawDataObjectConverters;
using MCRA.Data.Management.RawDataWriters;
using MCRA.Data.Raw.Objects.RawTableGroups;
using MCRA.General;
using MCRA.General.ModuleDefinitions.Settings;

namespace MCRA.Simulation.Actions.TargetExposures {

    public sealed class TargetExposuresOutputWriter {

        /// <summary>
        /// Writes the output data generated by a target exposures action to the provided raw data writer.
        /// </summary>
        /// <param name="config"></param>
        /// <param name="data"></param>
        /// <param name="result"></param>
        /// <param name="rawDataWriter"></param>
        public void WriteOutputData(
            TargetExposuresModuleConfig config,
            ActionData data,
            TargetExposuresActionResult result,
            IRawDataWriter rawDataWriter
        ) {
            var rawDataConverter = new RawTargetExposuresDataConverter();
            var exposureStatistics = getExposureStatistics(config, data, result);
            if (exposureStatistics != null) {
                var rawData = rawDataConverter.ToRaw(exposureStatistics, config.SelectedPercentiles.ToArray());
                rawDataWriter.Set(rawData);
            }
        }

        /// <summary>
        /// Updates the target exposures data of the raw data writer. Appends
        /// another bootstrap record / uncertainty set.
        /// </summary>
        /// <param name="config"></param>
        /// <param name="rawDataWriter"></param>
        /// <param name="data"></param>
        /// <param name="result"></param>
        /// <param name="idBootstrap"></param>
        public void UpdateOutputData(
            TargetExposuresModuleConfig config,
            IRawDataWriter rawDataWriter,
            ActionData data,
            TargetExposuresActionResult result,
            int idBootstrap
        ) {
            var rawData = rawDataWriter.Get(SourceTableGroup.TargetExposures) as RawTargetExposuresData;
            var rawDataConverter = new RawTargetExposuresDataConverter();
            if (data.ActiveSubstances.Count == 1 || data.CorrectedRelativePotencyFactors != null) {
                var exposureStatistics = getExposureStatistics(config, data, result);
                if (exposureStatistics != null) {
                    rawDataConverter.AppendUncertaintyRunValues(
                        rawData,
                        idBootstrap,
                        exposureStatistics,
                        [.. config.SelectedPercentiles]
                    );
                }
            }
        }

        private ICollection<SimpleExposureStatistics> getExposureStatistics(
            TargetExposuresModuleConfig config,
            ActionData data,
            TargetExposuresActionResult result
        ) {
            var exposureStatistics = new List<SimpleExposureStatistics>();
            if (config.ExposureType == ExposureType.Acute) {
                if (data.CorrectedRelativePotencyFactors != null || data.ActiveSubstances.Count == 1) {
                    var individualExposures = result.AggregateIndividualDayExposures;
                    var statistics = new SimpleExposureStatistics() {
                        Code = config.Id,
                        Name = config.Name,
                        Description = config.Description,
                        TargetUnit = data.TargetExposureUnit,
                        Substance = data.ReferenceSubstance,
                        SamplingWeights = individualExposures.Select(c => c.SimulatedIndividual.SamplingWeight).ToList()
                    };
                    if (data.ActiveSubstances.Count > 1) {
                        statistics.Intakes = individualExposures
                            .Select(c => c.GetTotalExposureAtTarget(
                                data.TargetExposureUnit.Target,
                                data.CorrectedRelativePotencyFactors,
                                data.MembershipProbabilities
                            ))
                            .ToList();
                    } else {
                        statistics.Intakes = individualExposures
                            .Select(c => c.GetSubstanceExposure(
                                data.TargetExposureUnit.Target,
                                data.ActiveSubstances.First()
                            ))
                            .ToList();
                    }
                    exposureStatistics.Add(statistics);
                }
            } else {
                if (data.CorrectedRelativePotencyFactors != null || data.ActiveSubstances.Count == 1) {

                    //Summarize internal, OIM
                    var individualExposures = result.AggregateIndividualExposures;
                    var intakes = (data.ActiveSubstances.Count > 1)
                        ? individualExposures
                            .Select(c => c.GetTotalExposureAtTarget(
                                data.TargetExposureUnit.Target,
                                data.CorrectedRelativePotencyFactors,
                                data.MembershipProbabilities
                            ))
                            .ToList()
                        : individualExposures
                            .Select(c => c.GetSubstanceExposure(
                                data.TargetExposureUnit.Target,
                                data.ActiveSubstances.First()
                            ))
                            .ToList();
                    var weights = individualExposures
                        .Select(c => c.SimulatedIndividual.SamplingWeight)
                        .ToList();
                    var statistics = new SimpleExposureStatistics() {
                        Code = config.Id,
                        Name = config.Name,
                        Description = config.Description,
                        TargetUnit = data.TargetExposureUnit,
                        Substance = data.ReferenceSubstance,
                        Intakes = intakes,
                        SamplingWeights = weights
                    };
                    exposureStatistics.Add(statistics);
                }
            }
            return exposureStatistics;
        }
    }
}
