﻿using MCRA.Data.Compiled.Objects;
using MCRA.Data.Management.RawDataObjectConverters;
using MCRA.Data.Management.RawDataWriters;
using MCRA.Data.Raw.Objects.RawTableGroups;
using MCRA.General;
using MCRA.General.ModuleDefinitions.Settings;

namespace MCRA.Simulation.Actions.Risks {

    public sealed class RisksOutputWriter {

        /// <summary>
        /// Writes the output data generated by a dietary exposures action to the provided raw data writer.
        /// </summary>
        /// <param name="config"></param>
        /// <param name="data"></param>
        /// <param name="result"></param>
        /// <param name="rawDataWriter"></param>
        public void WriteOutputData(
            RisksModuleConfig config,
            ActionData data,
            RisksActionResult result,
            IRawDataWriter rawDataWriter
        ) {
            var rawDataConverter = new RawRisksDataConverter();
            if (result.RiskPercentiles?.Count > 0) {
                var riskModel = createRiskModel(config, data, result);
                var rawData = rawDataConverter.ToRaw([riskModel]);
                rawDataWriter.Set(rawData);
            }
        }

        /// <summary>
        /// Updates the dietary exposures data of the raw data writer. Appends
        /// another bootstrap record / uncertainty set.
        /// </summary>
        /// <param name="config"></param>
        /// <param name="rawDataWriter"></param>
        /// <param name="data"></param>
        /// <param name="result"></param>
        /// <param name="idBootstrap"></param>
        public void UpdateOutputData(
            RisksModuleConfig config,
            IRawDataWriter rawDataWriter,
            ActionData data,
            RisksActionResult result,
            int idBootstrap
        ) {
            var rawData = rawDataWriter.Get(SourceTableGroup.Risks) as RawRisksData;
            var rawDataConverter = new RawRisksDataConverter();
            if (result.RiskPercentiles?.Count > 0) {
                var riskModel = createRiskModel(config, data, result);
                rawDataConverter.AppendUncertaintyRunValues(
                    rawData,
                    idBootstrap,
                    [riskModel]
                );
            }
        }

        private static RiskModel createRiskModel(RisksModuleConfig config, ActionData data, RisksActionResult result) {
            return new RiskModel() {
                Code = config.Id,
                Name = $"{config.Name}-{config.RiskMetricType}",
                Description = config.Description,
                Compound = data.ReferenceSubstance,
                RiskPercentiles = result.RiskPercentiles
                    .Select(r => new RiskPercentile() {
                        Percentage = r.Percentage,
                        Risk = config.RiskMetricType == RiskMetricType.HazardExposureRatio
                            ? r.HazardExposureRatio : r.HazardQuotient
                    })
                    .ToDictionary(r => r.Percentage),
                RiskMetric = config.RiskMetricType
            };
        }
    }
}
