﻿using MCRA.Data.Compiled.Wrappers;
using MCRA.Data.Management.RawDataObjectConverters;
using MCRA.Data.Management.RawDataWriters;
using MCRA.Data.Raw.Objects.RawTableGroups;
using MCRA.General;
using MCRA.General.ModuleDefinitions.Settings;

namespace MCRA.Simulation.Actions.DietaryExposures {

    public sealed class DietaryExposuresOutputWriter {

        /// <summary>
        /// Writes the output data generated by a dietary exposures action to the provided raw data writer.
        /// </summary>
        /// <param name="config"></param>
        /// <param name="data"></param>
        /// <param name="result"></param>
        /// <param name="rawDataWriter"></param>
        public void WriteOutputData(
            DietaryExposuresModuleConfig config,
            ActionData data,
            DietaryExposuresActionResult result,
            IRawDataWriter rawDataWriter
        ) {
            var rawDataConverter = new RawDietaryExposuresDataConverter();
            var exposureStatistics = getExposureStatistics(config, data, result);
            if (exposureStatistics != null) {
                var rawData = rawDataConverter.ToRaw(exposureStatistics, config.SelectedPercentiles.ToArray());
                rawDataWriter.Set(rawData);
            }
        }

        /// <summary>
        /// Updates the dietary exposures data of the raw data writer. Appends
        /// another bootstrap record / uncertainty set.
        /// </summary>
        /// <param name="config"></param>
        /// <param name="rawDataWriter"></param>
        /// <param name="data"></param>
        /// <param name="result"></param>
        /// <param name="idBootstrap"></param>
        public void UpdateOutputData(
            DietaryExposuresModuleConfig config,
            IRawDataWriter rawDataWriter,
            ActionData data,
            DietaryExposuresActionResult result,
            int idBootstrap
        ) {
            var rawData = rawDataWriter.Get(SourceTableGroup.DietaryExposures) as RawDietaryExposuresData;
            var rawDataConverter = new RawDietaryExposuresDataConverter();
            var exposureStatistics = getExposureStatistics(config, data, result);
            if (exposureStatistics != null) {
                rawDataConverter.AppendUncertaintyRunValues(
                    rawData,
                    idBootstrap,
                    exposureStatistics,
                    config.SelectedPercentiles.ToArray()
                );
            }
        }

        private ICollection<SimpleExposureStatistics> getExposureStatistics(
            DietaryExposuresModuleConfig config,
            ActionData data,
            DietaryExposuresActionResult result
        ) {
            var exposureStatistics = new List<SimpleExposureStatistics>();
            if (config.ExposureType == ExposureType.Acute && result.DietaryIndividualDayIntakes != null) {
                if (data.CorrectedRelativePotencyFactors != null || data.ActiveSubstances.Count == 1) {
                    // Acute cumulative
                    var statistics = new SimpleExposureStatistics() {
                        Code = config.Id,
                        Name = config.Name,
                        Description = config.Description,
                        TargetUnit = data.DietaryExposureUnit,
                        Substance = data.ActiveSubstances.Count == 1 ? data.ActiveSubstances.First() : data.ReferenceSubstance,
                        SamplingWeights = result.DietaryIndividualDayIntakes.Select(c => c.SimulatedIndividual.SamplingWeight).ToList()
                    };
                    if (data.ActiveSubstances.Count > 1) {
                        statistics.Intakes = result.DietaryIndividualDayIntakes
                            .Select(c => c.TotalExposurePerMassUnit(data.CorrectedRelativePotencyFactors, data.MembershipProbabilities, config.IsPerPerson))
                            .ToList();
                    } else {
                        statistics.Intakes = result.DietaryIndividualDayIntakes
                            .Select(c => c.GetSubstanceTotalExposurePerMassUnit(data.ActiveSubstances.First(), config.IsPerPerson))
                            .ToList();
                    }
                    exposureStatistics.Add(statistics);
                }
            } else {
                if (config.IntakeFirstModelThenAdd && result.DietaryObservedIndividualMeans != null) {
                    // Model-then-add
                    var intakes = result.DietaryObservedIndividualMeans.Select(c => c.DietaryIntakePerMassUnit).ToList();
                    var weights = result.DietaryObservedIndividualMeans.Select(c => c.SimulatedIndividual.SamplingWeight).ToList();
                    var statistics = new SimpleExposureStatistics() {
                        Code = $"{config.Id}-Model-Assisted",
                        Name = $"{config.Name} (Model assisted)",
                        Description = config.Description,
                        TargetUnit = data.DietaryExposureUnit,
                        Substance = data.ActiveSubstances.Count == 1 ? data.ActiveSubstances.First() : data.ReferenceSubstance,
                        Intakes = intakes,
                        SamplingWeights = weights
                    };
                    exposureStatistics.Add(statistics);
                } else if (result.DietaryModelAssistedIntakes != null) {
                    // Model assisted
                    var intakes = result.DietaryModelAssistedIntakes.Select(c => c.DietaryIntakePerMassUnit).ToList();
                    var weights = result.DietaryModelAssistedIntakes.Select(c => c.SimulatedIndividual.SamplingWeight).ToList();
                    var statistics = new SimpleExposureStatistics() {
                        Code = $"{config.Id}-Model-Assisted",
                        Name = $"{config.Name} (Model assisted)",
                        Description = config.Description,
                        TargetUnit = data.DietaryExposureUnit,
                        Substance = data.ActiveSubstances.Count == 1 ? data.ActiveSubstances.First() : data.ReferenceSubstance,
                        Intakes = intakes,
                        SamplingWeights = weights
                    };
                    exposureStatistics.Add(statistics);
                } else if (result.DietaryObservedIndividualMeans != null) {
                    var intakes = result.DietaryObservedIndividualMeans.Select(c => c.DietaryIntakePerMassUnit).ToList();
                    var weights = result.DietaryObservedIndividualMeans.Select(c => c.SimulatedIndividual.SamplingWeight).ToList();
                    var statistics = new SimpleExposureStatistics() {
                        Code = $"{config.Id}-OIM",
                        Name = $"{config.Name} (OIM)",
                        Description = config.Description,
                        TargetUnit = data.DietaryExposureUnit,
                        Substance = data.ActiveSubstances.Count == 1 ? data.ActiveSubstances.First() : data.ReferenceSubstance,
                        Intakes = intakes,
                        SamplingWeights = weights
                    };
                    exposureStatistics.Add(statistics);
                }
            }
            return exposureStatistics;
        }
    }
}
