#ifndef PRIMITIVE_HPP
#define PRIMITIVE_HPP

// original taken from:
//  https://codereview.stackexchange.com/questions/142457/primitive-type-wrapper-in-c
// this version contains slight modification

#include <type_traits>
#include <iosfwd>

template <typename T, typename = std::enable_if_t< std::is_arithmetic<T>::value >>
class primitive_type_wrapper {
    T m_value;

public:
    using value_type = T;

    constexpr primitive_type_wrapper() noexcept: m_value() {}
    constexpr primitive_type_wrapper(T const& value) noexcept: m_value(value) {}

    primitive_type_wrapper(primitive_type_wrapper const&) noexcept = default;
    primitive_type_wrapper(primitive_type_wrapper &&) noexcept = default;

    primitive_type_wrapper& operator=(primitive_type_wrapper const&) noexcept = default;
    primitive_type_wrapper& operator=(primitive_type_wrapper &&) noexcept = default;

    constexpr T const& get() const noexcept { return m_value; }

    template <typename U = T, typename = std::enable_if_t<!std::is_same<U, bool>::value  >>
    constexpr primitive_type_wrapper const& operator+() const noexcept { return *this; }
    template <typename U = T, typename = std::enable_if_t<!std::is_same<U, bool>::value  >>
    constexpr primitive_type_wrapper operator-() const noexcept { return -this->get(); }

    template <typename U = T, typename = std::enable_if_t< std::is_integral<U>::value && !std::is_same<U, bool>::value >>
    constexpr primitive_type_wrapper operator~() const noexcept { return ~this->get(); }

    constexpr primitive_type_wrapper<bool> operator!() const noexcept { return !this->get(); }

    primitive_type_wrapper& operator++() noexcept {
        ++m_value;
        return *this;
    }
    primitive_type_wrapper operator++(int) noexcept {
        return m_value++;
    }

    primitive_type_wrapper& operator--() noexcept {
        --m_value;
        return *this;
    }
    primitive_type_wrapper operator--(int) noexcept {
        return m_value--;
    }

    template <typename U>
    primitive_type_wrapper& operator+=(U const& other) noexcept {
        m_value += other;
        return *this;
    }
    template <typename U>
    primitive_type_wrapper& operator+=(primitive_type_wrapper<U> const& other) noexcept {
        m_value += other.get();
        return *this;
    }

    template <typename U>
    primitive_type_wrapper& operator-=(U const& other) noexcept {
        m_value -= other;
        return *this;
    }
    template <typename U>
    primitive_type_wrapper& operator-=(primitive_type_wrapper<U> const& other) noexcept {
        m_value -= other.get();
        return *this;
    }

    template <typename U>
    primitive_type_wrapper& operator*=(U const& other) noexcept {
        m_value *= other;
        return *this;
    }
    template <typename U>
    primitive_type_wrapper& operator*=(primitive_type_wrapper<U> const& other) noexcept {
        m_value *= other.get();
        return *this;
    }

    template <typename U>
    primitive_type_wrapper& operator/=(U const& other) noexcept {
        m_value /= other;
        return *this;
    }
    template <typename U>
    primitive_type_wrapper& operator/=(primitive_type_wrapper<U> const& other) noexcept {
        m_value /= other.get();
        return *this;
    }

    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator%=(U const& other) noexcept {
        m_value %= other;
        return *this;
    }
    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator%=(primitive_type_wrapper<U> const& other) noexcept {
        m_value %= other.get();
        return *this;
    }

    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator<<=(U const& other) noexcept {
        m_value <<= other;
        return *this;
    }
    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator<<=(primitive_type_wrapper<U> const& other) noexcept {
        m_value <<= other.get();
        return *this;
    }

    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator>>=(U const& other) noexcept {
        m_value >>= other;
        return *this;
    }
    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator>>=(primitive_type_wrapper<U> const& other) noexcept {
        m_value >>= other.get();
        return *this;
    }

    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator&=(U const& other) noexcept {
        m_value &= other;
        return *this;
    }
    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator&=(primitive_type_wrapper<U> const& other) noexcept {
        m_value &= other.get();
        return *this;
    }

    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator|=(U const& other) noexcept {
        m_value |= other;
        return *this;
    }
    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator|=(primitive_type_wrapper<U> const& other) noexcept {
        m_value |= other.get();
        return *this;
    }

    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator^=(U const& other) noexcept {
        m_value ^= other;
        return *this;
    }
    template <typename U, typename = std::enable_if_t< std::is_integral<T>::value && std::is_integral<U>::value >>
    primitive_type_wrapper& operator^=(primitive_type_wrapper<U> const& other) noexcept {
        m_value ^= other.get();
        return *this;
    }

    template <typename U = T, typename = std::enable_if_t< std::is_same<U, signed char>::value >>
    constexpr operator primitive_type_wrapper<short>() const noexcept { return m_value; }

    template <typename U = T, typename = std::enable_if_t< std::is_same<U, unsigned char>::value >>
    constexpr operator primitive_type_wrapper<unsigned short>() const noexcept { return m_value; }

    template <typename U = T, typename = std::enable_if_t< std::is_same<U, signed char>::value || std::is_same<U, short>::value >>
    constexpr operator primitive_type_wrapper<int>() const noexcept { return m_value; }

    template <typename U = T, typename = std::enable_if_t< std::is_same<U, unsigned char>::value || std::is_same<U, unsigned short>::value >>
    constexpr operator primitive_type_wrapper<unsigned int>() const noexcept { return m_value; }

    template <typename U = T, typename = std::enable_if_t<
            std::is_same<U, signed char>::value
            || std::is_same<U, short>::value
            || std::is_same<U, int>::value
    >>
    constexpr operator primitive_type_wrapper<long>() const noexcept { return m_value; }

    template <typename U = T, typename = std::enable_if_t<
            std::is_same<U, unsigned char>::value
            || std::is_same<U, unsigned short>::value
            || std::is_same<U, unsigned int>::value
    >>
    constexpr operator primitive_type_wrapper<unsigned long>() const noexcept { return m_value; }

    template <typename U = T, typename = std::enable_if_t<
            std::is_same<U, signed char>::value
            || std::is_same<U, short>::value
            || std::is_same<U, int>::value
            || std::is_same<U, long>::value
    >>
    constexpr operator primitive_type_wrapper<long long>() const noexcept { return m_value; }

    template <typename U = T, typename = std::enable_if_t<
            std::is_same<U, unsigned char>::value
            || std::is_same<U, unsigned short>::value
            || std::is_same<U, unsigned int>::value
            || std::is_same<U, unsigned long>::value
    >>
    constexpr operator primitive_type_wrapper<unsigned long long>() const noexcept { return m_value; }

    template <typename U = T, typename = std::enable_if_t<
            std::is_same<U, signed char>::value
            || std::is_same<U, unsigned char>::value
            || std::is_same<U, short>::value
            || std::is_same<U, unsigned short>::value
            || std::is_same<U, int>::value
            || std::is_same<U, unsigned int>::value
            || std::is_same<U, long>::value
            || std::is_same<U, unsigned long>::value
            || std::is_same<U, float>::value
    >>
    constexpr operator primitive_type_wrapper<double>() const noexcept { return m_value; }

    template <typename U = T, typename = std::enable_if_t<
            std::is_same<U, signed char>::value
            || std::is_same<U, unsigned char>::value
            || std::is_same<U, short>::value
            || std::is_same<U, unsigned short>::value
            || std::is_same<U, int>::value
            || std::is_same<U, unsigned int>::value
            || std::is_same<U, long>::value
            || std::is_same<U, unsigned long>::value
            || std::is_same<U, float>::value
            || std::is_same<U, double>::value
    >>
    constexpr operator primitive_type_wrapper<long double>() const noexcept { return m_value; }

    template <typename U>
    constexpr explicit operator primitive_type_wrapper<U>() const noexcept { return static_cast<U>(m_value); }

    friend std::istream& operator>>(std::istream& lhs, primitive_type_wrapper<T> & rhs) { return lhs >> rhs.m_value; }
};

template <typename T>
constexpr primitive_type_wrapper<T> operator+(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() + rhs; }
template <typename T>
constexpr primitive_type_wrapper<T> operator+(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs + rhs.get(); }
template <typename T1, typename T2>
constexpr auto operator+(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept -> primitive_type_wrapper<decltype(lhs.get() + rhs.get())> {
    return lhs.get() + rhs.get();
}

template <typename T>
constexpr primitive_type_wrapper<T> operator-(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() - rhs; }
template <typename T>
constexpr primitive_type_wrapper<T> operator-(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs - rhs.get(); }
template <typename T1, typename T2>
constexpr auto operator-(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept -> primitive_type_wrapper<decltype(lhs.get() - rhs.get())> {
    return lhs.get() - rhs.get();
}

template <typename T>
constexpr primitive_type_wrapper<T> operator*(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() * rhs; }
template <typename T>
constexpr primitive_type_wrapper<T> operator*(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs * rhs.get(); }
template <typename T1, typename T2>
constexpr auto operator*(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept -> primitive_type_wrapper<decltype(lhs.get() * rhs.get())> {
    return lhs.get() * rhs.get();
}

template <typename T>
constexpr primitive_type_wrapper<T> operator/(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() / rhs; }
template <typename T>
constexpr primitive_type_wrapper<T> operator/(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs / rhs.get(); }
template <typename T1, typename T2>
constexpr auto operator/(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept -> primitive_type_wrapper<decltype(lhs.get() / rhs.get())> {
    return lhs.get() / rhs.get();
}

template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator%(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() % rhs; }
template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator%(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs % rhs.get(); }
template <typename T1, typename T2, typename = std::enable_if_t< std::is_integral<T1>::value && std::is_integral<T2>::value >>
constexpr auto operator%(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept -> primitive_type_wrapper<decltype(lhs.get() % rhs.get())> {
    return lhs.get() % rhs.get();
}

template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator&(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() & rhs; }
template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator&(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs & rhs.get(); }
template <typename T1, typename T2, typename = std::enable_if_t< std::is_integral<T1>::value && std::is_integral<T2>::value >>
constexpr auto operator&(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept -> primitive_type_wrapper<decltype(lhs.get() & rhs.get())> {
    return lhs.get() & rhs.get();
}

template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator|(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() | rhs; }
template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator|(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs | rhs.get(); }
template <typename T1, typename T2, typename = std::enable_if_t< std::is_integral<T1>::value && std::is_integral<T2>::value >>
constexpr auto operator|(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept -> primitive_type_wrapper<decltype(lhs.get() | rhs.get())> {
    return lhs.get() | rhs.get();
}

template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator^(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() ^ rhs; }
template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator^(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs ^ rhs.get(); }
template <typename T1, typename T2, typename = std::enable_if_t< std::is_integral<T1>::value && std::is_integral<T2>::value >>
constexpr auto operator^(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept -> primitive_type_wrapper<decltype(lhs.get() ^ rhs.get())>  {
    return lhs.get() ^ rhs.get();
}

template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator<<(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() << rhs; }
template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator<<(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs << rhs.get(); }
template <typename T1, typename T2, typename = std::enable_if_t< std::is_integral<T1>::value && std::is_integral<T2>::value >>
constexpr auto operator<<(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept -> primitive_type_wrapper<decltype(lhs.get() << rhs.get())> {
    return lhs.get() << rhs.get();
}

template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator>>(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() >> rhs; }
template <typename T, typename = std::enable_if_t< std::is_integral<T>::value >>
constexpr primitive_type_wrapper<T> operator>>(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs >> rhs.get(); }
template <typename T1, typename T2, typename = std::enable_if_t< std::is_integral<T1>::value && std::is_integral<T2>::value >>
constexpr auto operator>>(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept -> primitive_type_wrapper<decltype(lhs.get() >> rhs.get())> {
    return lhs.get() >> rhs.get();
}

constexpr primitive_type_wrapper<bool> operator&&(primitive_type_wrapper<bool> const& lhs, bool const& rhs) noexcept { return lhs.get() && rhs; }
constexpr primitive_type_wrapper<bool> operator&&(bool const& lhs, primitive_type_wrapper<bool> const& rhs) noexcept { return lhs && rhs.get(); }
constexpr primitive_type_wrapper<bool> operator&&(primitive_type_wrapper<bool> const& lhs, primitive_type_wrapper<bool> const& rhs) noexcept {
    return lhs.get() && rhs.get();
}

constexpr primitive_type_wrapper<bool> operator||(primitive_type_wrapper<bool> const& lhs, bool const& rhs) noexcept { return lhs.get() || rhs; }
constexpr primitive_type_wrapper<bool> operator||(bool const& lhs, primitive_type_wrapper<bool> const& rhs) noexcept { return lhs || rhs.get(); }
constexpr primitive_type_wrapper<bool> operator||(primitive_type_wrapper<bool> const& lhs, primitive_type_wrapper<bool> const& rhs) noexcept {
    return lhs.get() || rhs.get();
}

template <typename T>
constexpr bool operator==(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() == rhs; }
template <typename T>
constexpr bool operator==(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs == rhs.get(); }
template <typename T1, typename T2>
constexpr bool operator==(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept {
    return lhs.get() == rhs.get();
}

template <typename T>
constexpr bool operator!=(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() != rhs; }
template <typename T>
constexpr bool operator!=(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs != rhs.get(); }
template <typename T1, typename T2>
constexpr bool operator!=(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept {
    return lhs.get() != rhs.get();
}

template <typename T>
constexpr bool operator<(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() < rhs; }
template <typename T>
constexpr bool operator<(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs < rhs.get(); }
template <typename T1, typename T2>
constexpr bool operator<(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept {
    return lhs.get() < rhs.get();
}

template <typename T>
constexpr bool operator<=(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() <= rhs; }
template <typename T>
constexpr bool operator<=(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs <= rhs.get(); }
template <typename T1, typename T2>
constexpr bool operator<=(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept {
    return lhs.get() <= rhs.get();
}

template <typename T>
constexpr bool operator>(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() > rhs; }
template <typename T>
constexpr bool operator>(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs > rhs.get(); }
template <typename T1, typename T2>
constexpr bool operator>(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept {
    return lhs.get() > rhs.get();
}

template <typename T>
constexpr bool operator>=(primitive_type_wrapper<T> const& lhs, T const& rhs) noexcept { return lhs.get() >= rhs; }
template <typename T>
constexpr bool operator>=(T const& lhs, primitive_type_wrapper<T> const& rhs) noexcept { return lhs >= rhs.get(); }
template <typename T1, typename T2>
constexpr bool operator>=(primitive_type_wrapper<T1> const& lhs, primitive_type_wrapper<T2> const& rhs) noexcept {
    return lhs.get() >= rhs.get();
}

template <typename T>
std::ostream& operator<<(std::ostream& lhs, primitive_type_wrapper<T> const& rhs) { return lhs << rhs.get(); }

namespace std {
template<typename T>
struct hash<primitive_type_wrapper<T>> {
    std::size_t operator()(const primitive_type_wrapper <T> &twr) const noexcept {
        return twr.get();
    }
};
}

#endif