#ifndef FAST_BACKWARD_TOKEN_H
#define FAST_BACKWARD_TOKEN_H

#include <sstream>
#include <unordered_set>
#include <unordered_map>
#include <memory>
#include "magic_enum.hpp"
#include "../../utils/type_defs.h"

namespace HELP { 

namespace Lexer { //TODO: different name?, no namespace Lexer needed?
enum TOKEN_TYPE {
    // internal shenanigans
    NO_TOKEN,
    SOME_STR,

    // identifier
    VARIABLE, // starting with ? (also called parameter)
    ID, // default
    STRING_LITERAL,

    NUMBER,

    // characters
    DOT,
    COMMA,
    EQUAL_SIGN,
    LBRACKET,
    RBRACKET,
    LPAREN,
    RPAREN,

    // Datalog specific
    DL_RULE_SPECIFIER,
    DL_RULE_TYPE
};

extern std::unordered_set<char> special_id_starting_chars;

extern std::unordered_set<char> special_id_chars;

extern std::unordered_set<char> special_id_seperator;

extern std::unordered_set<char> literal_start_end;

extern std::unordered_set<std::string> rule_type_col;
//TODO: static assert that intersect with mid id is empty

extern std::unordered_map<TOKEN_TYPE, std::string> identifierable_m;

bool is_valid_number_start(char c);

// first char for id
bool valid_id_start(char c);

// mid id
bool valid_id_char(char c);

// id seperator
bool valid_id_end(char c);

// string literal
bool is_literal_start(char c);

/*
* Check if the given string has trailing whitespaces.
*/
inline bool not_padded(std::string &s) { //TODO: probably should move things like this in a util class
    if (s.empty()) {
        return true;
    }
    return !isspace(*s.begin()) && !isspace(*std::prev(s.end()));
}

struct TokenPosition {
    int x;
    int y;

    friend std::ostream &operator<<(std::ostream &outs, TokenPosition &t) {
        return outs << "[" << t.x << ", " << t.y << "]";
    }
};

class Token {
protected:
    TOKEN_TYPE type;
    TokenPosition position;
    //TODO: err with position
    //TODO: err position unit tests

public:
    Token(TOKEN_TYPE type, TokenPosition position) : type(type), position(position) {}
    virtual ~Token() = default;

    TOKEN_TYPE get_type() {
        return type;
    }

    TokenPosition &get_position() {
        return position;
    }

    virtual void dump(std::ostream &outs) {
        outs << magic_enum::enum_name(type) << position;
    }

    friend std::ostream &operator<<(std::ostream &outs, Token &t) {
        t.dump(outs);
        return outs;
    }

    virtual bool is_string_type() {
        return false;
    }

    virtual bool is_number() {
        return false;
    }

    bool is_type(TOKEN_TYPE tp) {
        return get_type() == tp;
    }

    // TODO: can be transformed into identifier
    bool identifierable() {
        return identifierable_m.contains(get_type());
    }

    const std::string &get_id_conversion_string() {
        assert(identifierable());
        return identifierable_m[get_type()];
    }
};

class StringToken : public Token {
    std::string name;

public:
    StringToken(TOKEN_TYPE type, TokenPosition position, std::string name) : Token(type, position), name(name) { //TODO: avoid copies as much as possible
        assert(!name.empty());
        assert(not_padded(name));
    }

    std::string &get_string() {
        return name;
    }

    virtual void dump(std::ostream &outs) override {
        Token::dump(outs);
        outs << "('" << name << "')";
    }

    virtual bool is_string_type() override {
        return true;
    }
};

inline StringToken to_id_token(Token &token) {
    return StringToken(ID, token.get_position(), token.get_id_conversion_string());
} //TODO: move?

class NumberToken : public Token {
    int num;

public:
    NumberToken(TokenPosition position, ll num) : Token(NUMBER, position), num(num) { }

    ll get_num() {
        return num;
    }

    virtual void dump(std::ostream &outs) override { //TODO: combine as extra_f
        Token::dump(outs);
        outs << "('" << num << "')";
    }

    bool is_number() override {
        return true;
    }
};

inline bool is_rule_type(std::string &s) {
    return rule_type_col.contains(s);
}

inline TOKEN_TYPE determine_string_type(std::string &s) {
    assert(!s.empty());

    if (*s.begin() == '?') {
        return VARIABLE;
    } else if (is_rule_type(s)) {
        return DL_RULE_TYPE;
    }

    return ID;
}

inline std::shared_ptr<Token> create_token(TOKEN_TYPE type, std::stringstream &sstream, TokenPosition &pos) { //TODO: prob. want to move the result
    assert(type != NO_TOKEN);

    switch (type) { //TODO: if stays this way, do if/else
        case STRING_LITERAL: {
            std::string s = sstream.str();
            s = s.substr(1, s.size()-2);
            return std::make_shared<StringToken>(type, pos, s);
        }
        case SOME_STR: {
            std::string s = sstream.str(); //TODO: prob do beforehand
            TOKEN_TYPE t = determine_string_type(s);
            return std::make_shared<StringToken>(t, pos, s);
        }
        case NUMBER: {
            std::string s = sstream.str();
            return std::make_shared<NumberToken>(pos, std::stoi(s));
        }
        default: {
            return std::make_shared<Token>(type, pos);
        }
    }
}

//TODO: add stirng literal like "..." and '..."

//TODO: assertions of class get token_type that is valid for class

}

}

#endif