/****************************************************************************
 *
 *   Copyright (c) 2021-2023 PX4 Development Team. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name PX4 nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/**
 * Test the yaw estimator
 * @author Mathieu Bresciani <mathieu@auterion.com>
 */

#include <gtest/gtest.h>
#include "EKF/ekf.h"
#include "sensor_simulator/sensor_simulator.h"
#include "sensor_simulator/ekf_wrapper.h"

class EKFYawEstimatorTest : public ::testing::Test
{
public:

	EKFYawEstimatorTest(): ::testing::Test(),
		_ekf{std::make_shared<Ekf>()},
		_sensor_simulator(_ekf),
		_ekf_wrapper(_ekf) {};

	std::shared_ptr<Ekf> _ekf;
	SensorSimulator _sensor_simulator;
	EkfWrapper _ekf_wrapper;

	// Setup the Ekf with mag aiding disabled
	void SetUp() override
	{
		// run briefly to init, then manually set in air and at rest (default for a real vehicle)
		_ekf->init(0);
		_sensor_simulator.runSeconds(0.1);
		_ekf->set_in_air_status(false);
		_ekf->set_vehicle_at_rest(true);

		_ekf_wrapper.setMagFuseTypeNone();
		_sensor_simulator.runSeconds(2);
		_ekf_wrapper.enableGpsFusion();
		_sensor_simulator.startGps();
		_sensor_simulator.runSeconds(11);
	}

	// Use this method to clean up any memory, network etc. after each test
	void TearDown() override
	{
	}
};

TEST_F(EKFYawEstimatorTest, inAirYawAlignment)
{
	// GIVEN: an accelerating vehicle with unknown heading
	EXPECT_EQ(1, (int) _ekf->control_status_flags().tilt_align);
	EXPECT_EQ(0, (int) _ekf->control_status_flags().yaw_align);

	// AND: a true heading far from North
	const float yaw = math::radians(-130.f);
	const Dcmf R_to_earth{Eulerf(0.f, 0.f, yaw)};
	_sensor_simulator.setOrientation(R_to_earth);

	// WHEN: the vehicle starts accelerating in the horizontal plane
	_sensor_simulator.setTrajectoryTargetVelocity(Vector3f(2.f, -2.f, -1.f));
	_ekf->set_in_air_status(true);

	_sensor_simulator.runTrajectorySeconds(3.f);

	// THEN: the heading can be estimated and then used to fuse GNSS vel and pos to the main EKF
	float yaw_est{};
	float yaw_est_var{};
	float dummy[5];
	_ekf->getDataEKFGSF(&yaw_est, &yaw_est_var, dummy, dummy, dummy, dummy);

	const float tolerance_rad = math::radians(5.f);
	EXPECT_NEAR(yaw_est, yaw, tolerance_rad);
	EXPECT_LT(yaw_est_var, tolerance_rad);

	EXPECT_LT(_ekf->aid_src_gnss_pos().test_ratio[0], 0.5f);
	EXPECT_LT(_ekf->aid_src_gnss_pos().test_ratio[1], 0.5f);

	EXPECT_LT(_ekf->aid_src_gnss_vel().test_ratio[0], 0.1f);
	EXPECT_LT(_ekf->aid_src_gnss_vel().test_ratio[1], 0.1f);
	EXPECT_LT(_ekf->aid_src_gnss_vel().test_ratio[2], 0.1f);

	EXPECT_TRUE(_ekf->isLocalHorizontalPositionValid());
	EXPECT_TRUE(_ekf->isGlobalHorizontalPositionValid());
}
