/****************************************************************************
 *
 *   Copyright (c) 2019-2023 PX4 Development Team. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name PX4 nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

#include <gtest/gtest.h>
#include <math.h>
#include <memory>
#include "EKF/ekf.h"
#include "sensor_simulator/sensor_simulator.h"
#include "sensor_simulator/ekf_wrapper.h"
#include "sensor_simulator/ekf_logger.h"

class EkfReplayTest : public ::testing::Test
{
public:
	EkfReplayTest(): ::testing::Test(),
		_ekf{std::make_shared<Ekf>()},
		_sensor_simulator(_ekf),
		_ekf_wrapper(_ekf),
		_ekf_logger(_ekf) {};

	std::shared_ptr<Ekf> _ekf;
	SensorSimulator _sensor_simulator;
	EkfWrapper _ekf_wrapper;
	EkfLogger _ekf_logger;
};

TEST_F(EkfReplayTest, irisGps)
{
	_sensor_simulator.loadSensorDataFromFile(TEST_DATA_PATH"/replay_data/iris_gps.csv");
	_ekf_logger.setFilePath(TEST_DATA_PATH"/change_indication/iris_gps.csv");

	// Start simulation and enable fusion of additional sensor types here
	// By default the IMU, Baro and Mag sensor simulators are already running
	_sensor_simulator.startGps();
	_ekf_wrapper.enableGpsFusion();

	uint8_t logging_rate_hz = 10;

	for (int i = 0; i < 35 * logging_rate_hz; ++i) {
		_sensor_simulator.runReplaySeconds(1.0f / logging_rate_hz);
		_ekf_logger.writeStateToFile();
	}
}

TEST_F(EkfReplayTest, ekfGsfReset)
{
	_sensor_simulator.loadSensorDataFromFile(TEST_DATA_PATH"/replay_data/ekf_gsf_reset.csv");
	_ekf_logger.setFilePath(TEST_DATA_PATH"/change_indication/ekf_gsf_reset.csv");

	// Start simulation and enable fusion of additional sensor types here
	// By default the IMU, Baro and Mag sensor simulators are already running
	_sensor_simulator.startGps();
	_ekf_wrapper.enableGpsFusion();
	auto params = _ekf->getParamHandle();
	params->gps_vel_innov_gate = 1.f;
	params->gps_pos_innov_gate = 1.f;

	uint8_t logging_rate_hz = 10;

	for (int i = 0; i < 39 * logging_rate_hz; ++i) {
		_sensor_simulator.runReplaySeconds(1.0f / logging_rate_hz);
		_ekf_logger.writeStateToFile();
	}
}
