/****************************************************************************
 *
 *   Copyright (c) 2019 PX4 Development Team. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name PX4 nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

#include "ManualVelocitySmoothingXY.hpp"

#include <mathlib/mathlib.h>
#include <float.h>

using namespace matrix;

void ManualVelocitySmoothingXY::reset(const Vector2f &accel, const Vector2f &vel, const Vector2f &pos)
{
	for (int i = 0; i < 2; i++) {
		_trajectory[i].reset(accel(i), vel(i), pos(i));
	}

	resetPositionLock();
}

void ManualVelocitySmoothingXY::resetPositionLock()
{
	_position_lock_active = false;
	_position_setpoint_locked(0) = NAN;
	_position_setpoint_locked(1) = NAN;
}

void ManualVelocitySmoothingXY::update(float dt, const Vector2f &velocity_target)
{
	// Update state
	updateTrajectories(dt);

	// Lock or unlock position
	// Has to be done before _updateTrajDurations()
	checkPositionLock(velocity_target);

	// Update durations and sync XY
	updateTrajDurations(velocity_target);
}

void ManualVelocitySmoothingXY::updateTrajectories(float dt)
{
	for (int i = 0; i < 2; ++i) {
		_trajectory[i].updateTraj(dt);

		_state.j(i) = _trajectory[i].getCurrentJerk();
		_state.a(i) = _trajectory[i].getCurrentAcceleration();
		_state.v(i) = _trajectory[i].getCurrentVelocity();
		_state.x(i) = _trajectory[i].getCurrentPosition();
	}
}

void ManualVelocitySmoothingXY::updateTrajDurations(const Vector2f &velocity_target)
{
	for (int i = 0; i < 2; ++i) {
		_trajectory[i].updateDurations(velocity_target(i));
	}

	VelocitySmoothing::timeSynchronization(_trajectory, 2);
}

void ManualVelocitySmoothingXY::checkPositionLock(const Vector2f &velocity_target)
{
	/**
	 * During a position lock -> position unlock transition, we have to make sure that the velocity setpoint
	 * is continuous. We know that the output of the position loop (part of the velocity setpoint)
	 * will suddenly become null
	 * and only the feedforward (generated by this flight task) will remain.
	 * This is why the previous input of the velocity controller
	 * is used to set current velocity of the trajectory.
	 */
	if (_state.v.length() < 0.1f &&
	    _state.a.length() < .2f &&
	    velocity_target.length() <= FLT_EPSILON) {
		// Lock position
		_position_lock_active = true;
		_position_setpoint_locked = _state.x;

	} else {
		// Unlock position
		if (_position_lock_active) {
			// Start the trajectory at the current velocity setpoint
			_trajectory[0].setCurrentVelocity(_velocity_setpoint_feedback(0));
			_trajectory[1].setCurrentVelocity(_velocity_setpoint_feedback(1));
			_state.v = _velocity_setpoint_feedback;
			resetPositionLock();
		}

		_trajectory[0].setCurrentPosition(_position_estimate(0));
		_trajectory[1].setCurrentPosition(_position_estimate(1));
	}
}
