/****************************************************************************
 *
 *   Copyright (c) 2020 PX4 Development Team. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name PX4 nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

#pragma once

#include <cstdint>
#include <atomic>

#include <px4_platform_common/sem.h>

/**
 * @class LockstepComponents
 * Allows to register components (threads) that need to be updated or waited for in every lockstep cycle (barrier).
 * Registered components need to ensure they poll on topics that is updated in every lockstep cycle.
 */
class LockstepComponents
{
public:
	LockstepComponents(bool no_cleanup_on_destroy = false);
	~LockstepComponents();

	/**
	 * Register a component
	 * @return a valid component ID > 0 or 0 on error (or unsupported)
	 */
	int register_component();
	void unregister_component(int component);

	/**
	 * signal an update from a component
	 * @param component component ID
	 */
	void lockstep_progress(int component);

	/**
	 * Wait for all registered components to call lockstep_progress()
	 * Note: only 1 thread can call this
	 */
	void wait_for_components();

private:
	const bool _no_cleanup_on_destroy;

	px4_sem_t _components_sem;

	std::atomic_int _components_used_bitset{0};
	std::atomic_int _components_progress_bitset{0};
};
