/****************************************************************************
 *
 *   Copyright (c) 2017-2021 PX4 Development Team. All rights reserved.
 *       Author: David Sidrane <david_s5@nscdg.com>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name PX4 nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>
#include "nuttx/arch.h"
#include "arm_internal.h"

#include "boot_config.h"

/****************************************************************************
 * Pre-processor Definitions
 ****************************************************************************/

/****************************************************************************
 * Private Types
 ****************************************************************************/

/****************************************************************************
 * Private Function Prototypes
 ****************************************************************************/

/****************************************************************************
 * Private Data
 ****************************************************************************/

/****************************************************************************
 * Public Data
 ****************************************************************************/

/****************************************************************************
 * Private Functions
 ****************************************************************************/

/****************************************************************************
 * Public Functions
 ****************************************************************************/
__EXPORT int __wrap_nxsem_wait(void *sem);
__EXPORT int __wrap_nxsem_post(void *sem);

int main(int argc, char **argv);
/****************************************************************************
 * Name: nxsem_wait and nxsem_post
 *
 * Description:
 *   These functions hijacks by the way of a compile time wrapper the systems
 *   sem_wait and sem_post functions.
 *
 ****************************************************************************/

int __wrap_nxsem_wait_uninterruptible(void *sem)
{
	return 0;

}

int __wrap_nxsem_wait(void *sem)
{
	return 0;
}

int __wrap_nxsem_post(void *sem)
{
	return 0;
}

/****************************************************************************
 * Public Functions
 ****************************************************************************/
void timer_init(void);
int __wrap_arm_svcall(int irq, FAR void *context);
__EXPORT void __wrap_exit(int status);


/****************************************************************************
 * Name: os_start
 *
 * Description:
 *   This function hijacks the entry point of the OS. Normally called by the
 *   statup code for a given architecture
 *
 ****************************************************************************/
#ifdef CONFIG_STACK_COLORATION
inline static void irq_stack_collor(void *pv, unsigned int nwords)
{
#ifdef CONFIG_ARCH_INTERRUPTSTACK
	/* Set the Interrupt stack to the stack coloration value then jump to
	 * os_start().  We take extreme care here because were currently
	 * executing on this stack.
	 *
	 * We want to avoid sneak stack access generated by the compiler.
	 */

	__asm__ __volatile__
	(
		"\tmov  r0, %[addr]\n"      /* R0 = Address */
		"\tmovs r1, %[nwords]\n"    /* R1 = nbyts*/
		"\tbeq  2f\n"               /* (should not happen) */

		"\tbic  r0, r0, #3\n"       /* R0 = Aligned stackptr */
		"\tmovw r2, #0xbeef\n"      /* R2 = STACK_COLOR = 0xdeadbeef */
		"\tmovt r2, #0xdead\n"

		"1:\n"                      /* Top of the loop */
		"\tsub  r1, r1, #1\n"       /* R1 nwords-- */
		"\tcmp  r1, #0\n"           /* Check (nwords == 0) */
		"\tstr  r2, [r0], #4\n"     /* Save stack color word, increment stackptr */
		"\tbne  1b\n"               /* Bottom of the loop */

		"2:\n"
		:
		: [addr]"r"(pv), [nwords]"r"(nwords)     /* Address*/
		: /* No clobbers */

	);
#endif
}
#endif

void __wrap_nx_start(void)
{
#ifdef CONFIG_STACK_COLORATION
	/* Color the irq stack */
	irq_stack_collor((void *)&g_intstackalloc, CONFIG_ARCH_INTERRUPTSTACK / sizeof(uint32_t));
#endif

	/* Initialize the timer software subsystem */
	timer_init();

	/* Initialize the interrupt subsystem */
	up_irqinitialize();


#if !defined(CONFIG_SUPPRESS_INTERRUPTS) && !defined(CONFIG_SUPPRESS_TIMER_INTS) && \
    !defined(CONFIG_SYSTEMTICK_EXTCLK)
	/* Initialize the OS's timer subsystem */
	up_timer_initialize();
#endif

	/* Keep the compiler happy for a no return function */
	while (1) {
		main(0, 0);
	}
}

/****************************************************************************
 * Name: malloc
 *
 * Description:
 *   This function hijacks the OS's malloc and provides no or small
 *   allocation
 *
 ****************************************************************************/

FAR void *malloc(size_t size)
{
#if defined(OPT_SIMPLE_MALLOC_HEAP_SIZE)
	static int aloc = 0;
	static uint8_t mem[OPT_SIMPLE_MALLOC_HEAP_SIZE];
	void *rv = &mem[aloc];
	aloc += size;

	if (aloc > OPT_SIMPLE_MALLOC_HEAP_SIZE) {
		PANIC();
	}

	return rv;
#else
	return NULL;
#endif
}

/****************************************************************************
 * Name: malloc
 *
 * Description:
 *   This function hijacks the systems exit
 *
 ****************************************************************************/
void __wrap_exit(int status)
{
	while (1);
}

/****************************************************************************
 * Name: sched_ufree
 *
 * Description:
 *   This function hijacks the systems sched_ufree that my be called during
 *   exception processing.
 *
 ****************************************************************************/

void sched_ufree(FAR void *address)
{

}

/****************************************************************************
 * Name: up_svcall
 *
 * Description:
 *   This function hijacks by the way of a compile time wrapper the systems
 *   up_svcall
 *
 ****************************************************************************/

int __wrap_arm_svcall(int irq, FAR void *context)
{
	return 0;
}
