//----------------------------------*-C++-*----------------------------------//
// Copyright 2024 UT-Battelle, LLC, and other Celeritas developers.
// See the top-level COPYRIGHT file for details.
// SPDX-License-Identifier: (Apache-2.0 OR MIT)
//---------------------------------------------------------------------------//
//! \file celeritas/em/params/WentzelOKVIParams.cc
//---------------------------------------------------------------------------//
#include "WentzelOKVIParams.hh"

#include <algorithm>
#include <utility>

#include "corecel/io/Logger.hh"
#include "corecel/sys/ScopedMem.hh"
#include "celeritas/em/xs/NuclearFormFactors.hh"
#include "celeritas/io/ImportData.hh"
#include "celeritas/mat/IsotopeView.hh"
#include "celeritas/mat/MaterialParams.hh"

namespace celeritas
{
//---------------------------------------------------------------------------//
/*!
 * Construct if Wentzel VI or Coulomb is present, else return nullptr.
 */
std::shared_ptr<WentzelOKVIParams>
WentzelOKVIParams::from_import(ImportData const& data,
                               SPConstMaterials materials)
{
    CELER_EXPECT(materials);

    bool wentzel = has_msc_model(data, ImportModelClass::wentzel_vi_uni);
    bool coulomb = has_model(data, ImportModelClass::e_coulomb_scattering);
    if (!(wentzel || coulomb))
    {
        // No Wentzel VI MSC or Coulomb scattering present
        return nullptr;
    }

    Options opts;
    opts.is_combined = wentzel && coulomb;
    opts.polar_angle_limit = [&]() -> real_type {
        if (!coulomb)
        {
            // Set the maximum scattering angle for Wentzel VI MSC
            return constants::pi;
        }
        if (!wentzel)
        {
            // Set the minimum scattering angle for Coulomb single scattering
            return real_type(0);
        }
        // Polar angle limit between single and multiple scattering if both
        // models are present
        return data.em_params.msc_theta_limit;
    }();
    opts.screening_factor = data.em_params.screening_factor;
    opts.angle_limit_factor = data.em_params.angle_limit_factor;
    opts.form_factor = data.em_params.form_factor;

    return std::make_shared<WentzelOKVIParams>(materials, opts);
}

//---------------------------------------------------------------------------//
/*!
 * Construct from cross section data and material properties.
 */
WentzelOKVIParams::WentzelOKVIParams(SPConstMaterials materials,
                                     Options options)
{
    CELER_EXPECT(materials);

    ScopedMem record_mem("WentzelOKVIParams.construct");

    HostVal<WentzelOKVIData> host_data;

    host_data.params.is_combined = options.is_combined;
    host_data.params.costheta_limit = std::cos(options.polar_angle_limit);
    host_data.params.a_sq_factor
        = real_type(0.5)
          * ipow<2>(native_value_to<units::MevEnergy>(
                        options.angle_limit_factor * constants::hbar_planck
                        * constants::c_light / units::femtometer)
                        .value());
    host_data.params.screening_factor = options.screening_factor;
    host_data.params.form_factor_type = options.form_factor;

    // Load Mott coefficients
    build_data(host_data, *materials);
    CELER_ASSERT(host_data);

    // Move to mirrored data, copying to device
    data_ = CollectionMirror<WentzelOKVIData>{std::move(host_data)};
    CELER_ENSURE(data_);
}

//---------------------------------------------------------------------------//
/*!
 * Load Mott coefficients and construct per-element data.
 */
void WentzelOKVIParams::build_data(HostVal<WentzelOKVIData>& host_data,
                                   MaterialParams const& materials)
{
    // Build element data
    size_type const num_elements = materials.num_elements();
    auto mott_coeffs = make_builder(&host_data.mott_coeffs);
    mott_coeffs.reserve(num_elements);

    for (auto el_id : range(ElementId{num_elements}))
    {
        // Load Mott coefficients
        MottElementData z_data;
        z_data.electron
            = get_electron_mott_coeffs(materials.get(el_id).atomic_number());
        z_data.positron
            = get_positron_mott_coeffs(materials.get(el_id).atomic_number());
        mott_coeffs.push_back(z_data);
    }

    auto prefactors = make_builder(&host_data.nuclear_form_prefactor);
    prefactors.reserve(materials.num_isotopes());
    for (auto iso_id : range(IsotopeId{materials.num_isotopes()}))
    {
        using InvMomSq = Quantity<UnitInverse<units::MevMomentumSq::unit_type>>;

        auto&& iso_view = materials.get(iso_id);
        prefactors.push_back(value_as<InvMomSq>(
            ExpNuclearFormFactor{iso_view.atomic_mass_number()}.prefactor()));
    }
    CELER_ENSURE(host_data.nuclear_form_prefactor.size()
                 == materials.num_isotopes());

    // Build material data
    if (host_data.params.is_combined)
    {
        std::vector<real_type> inv_mass_cbrt_sq(materials.num_materials(), 0);
        for (auto mat_id : range(MaterialId(materials.num_materials())))
        {
            auto mat = materials.get(mat_id);
            for (auto elcomp_id : range(ElementComponentId(mat.num_elements())))
            {
                auto const& el_comp = mat.elements()[elcomp_id.get()];
                auto atomic_mass
                    = mat.make_element_view(elcomp_id).atomic_mass();
                inv_mass_cbrt_sq[mat_id.get()]
                    += el_comp.fraction
                       / std::pow(atomic_mass.value(), real_type(2) / 3);
            }
        }
        make_builder(&host_data.inv_mass_cbrt_sq)
            .insert_back(inv_mass_cbrt_sq.begin(), inv_mass_cbrt_sq.end());
    }
}

//---------------------------------------------------------------------------//
/*!
 * Get interpolated Mott coefficients for electrons for an element.
 *
 * These coefficients are for electrons interacting with nuclei with 1 <= Z <=
 * 118 and taken from table 3 in M.J. Boschini et al, Radiat. Phys. Chem. 90
 * (2013), 39-66. Geant4 provides the same data in G4MottData.hh, but only for
 * 1 <= Z <= 92.
 */
auto WentzelOKVIParams::get_electron_mott_coeffs(AtomicNumber z) -> CoeffMat
{
    CELER_EXPECT(z);

    // clang-format off
    static CoeffMat const electron_mott_coeffs[]
        = {{{{1.00000e+00, 2.67363e-08, 7.11530e-08, -9.77030e-08, -6.69132e-07, -3.09263e-07},
             {1.17182e-02, 1.62222e-02, -5.90397e-05, -1.05585e-04, 4.17873e-04, 9.13843e-04},
             {-2.65955e-01, -7.29531e-01, -4.99796e-01, 2.83507e-04, -9.09042e-04, -2.20244e-03},
             {-1.82348e-04, -8.86355e-05, -1.90554e-04, -2.49708e-04, 6.35004e-04, 1.73523e-03},
             {4.70966e-05, -4.09705e-06, 3.75218e-05, 8.05645e-05, -1.90534e-04, -5.42847e-04}}},
           {{{1.00000e+00, 3.76476e-08, -3.05313e-07, -3.27422e-07, 2.44235e-06, 4.08754e-06},
             {2.35767e-02, 3.24642e-02, -6.37269e-04, -7.69160e-04, 5.28004e-03, 9.45642e-03},
             {-2.73743e-01, -7.40767e-01, -4.98195e-01, 1.74337e-03, -1.25798e-02, -2.24046e-02},
             {-7.79128e-04, -4.14495e-04, -1.62657e-03, -1.37286e-03, 1.04319e-02, 1.83488e-02},
             {2.02855e-04, 1.94598e-06, 4.30102e-04, 4.32180e-04, -3.31526e-03, -5.81788e-03}}},
           {{{1.00000e+00, 7.00357e-08, -3.15076e-07, -4.24915e-07, 2.45516e-06, 4.90187e-06},
             {3.55657e-02, 4.87956e-02, -1.95525e-03, -2.78660e-03, 1.65490e-02, 3.11496e-02},
             {-2.81171e-01, -7.52015e-01, -4.95329e-01, 5.83548e-03, -3.39830e-02, -6.55379e-02},
             {-1.83452e-03, -8.12746e-04, -3.84675e-03, -4.44467e-03, 2.55871e-02, 4.99483e-02},
             {4.79031e-04, -3.89615e-05, 1.01022e-03, 1.39133e-03, -7.99398e-03, -1.56366e-02}}},
           {{{1.00000e+00, 7.58881e-08, 4.70500e-08, 2.48041e-07, -2.06053e-06, -1.97319e-06},
             {4.76788e-02, 6.52200e-02, -4.54331e-03, -6.50318e-03, 3.76564e-02, 7.17176e-02},
             {-2.88203e-01, -7.63217e-01, -4.90337e-01, 1.22839e-02, -6.86398e-02, -1.35769e-01},
             {-3.37733e-03, -1.36514e-03, -7.51614e-03, -8.78592e-03, 4.78572e-02, 9.69021e-02},
             {8.81822e-04, -1.02577e-04, 1.99797e-03, 2.72661e-03, -1.48296e-02, -3.01060e-02}}},
           {{{9.99999e-01, 7.91498e-08, 1.84164e-06, 2.68534e-06, -1.81630e-05, -2.69021e-05},
             {5.98818e-02, 8.17654e-02, -7.70811e-03, -1.12378e-02, 6.38329e-02, 1.25339e-01},
             {-2.94716e-01, -7.74405e-01, -4.86220e-01, 1.77367e-02, -9.46825e-02, -2.01789e-01},
             {-5.52375e-03, -2.05348e-03, -9.44915e-03, -1.08135e-02, 5.41024e-02, 1.25257e-01},
             {1.44555e-03, -1.99404e-04, 2.36742e-03, 3.29655e-03, -1.64122e-02, -3.83750e-02}}},
           {{{9.99999e-01, 7.68158e-08, 5.18185e-06, 7.34245e-06, -4.94780e-05, -7.71923e-05},
             {7.21461e-02, 9.84618e-02, -1.06535e-02, -1.62358e-02, 8.59238e-02, 1.78727e-01},
             {-3.00622e-01, -7.85616e-01, -4.85735e-01, 1.91563e-02, -8.10204e-02, -2.15386e-01},
             {-8.34809e-03, -2.85241e-03, -7.03252e-03, -7.56786e-03, 1.44975e-02, 8.79093e-02},
             {2.18964e-03, -3.42022e-04, 1.32930e-03, 2.20108e-03, -3.57927e-03, -2.59280e-02}}},
           {{{9.99999e-01, 8.36312e-08, 1.09116e-05, 1.47812e-05, -1.02733e-04, -1.62724e-04},
             {8.44142e-02, 1.15310e-01, -1.17230e-02, -1.94732e-02, 8.92604e-02, 2.09303e-01},
             {-3.05743e-01, -7.96809e-01, -4.93957e-01, 1.01607e-02, 1.67761e-02, -1.05909e-01},
             {-1.20090e-02, -3.80678e-03, 4.51195e-03, 6.93472e-03, -1.12405e-01, -8.15484e-02},
             {3.16048e-03, -5.22237e-04, -2.58261e-03, -2.38303e-03, 3.63393e-02, 2.75127e-02}}},
           {{{9.99998e-01, 1.57323e-08, 1.77595e-05, 2.56082e-05, -1.67537e-04, -2.73755e-04},
             {9.66438e-02, 1.32264e-01, -9.53841e-03, -1.83707e-02, 6.01664e-02, 1.93357e-01},
             {-3.09969e-01, -8.07790e-01, -5.14392e-01, -1.67153e-02, 2.33870e-01, 1.91600e-01},
             {-1.65906e-02, -5.11585e-03, 2.80424e-02, 3.94663e-02, -3.55720e-01, -4.39251e-01},
             {4.37866e-03, -6.81795e-04, -1.01520e-02, -1.24875e-02, 1.11484e-01, 1.38105e-01}}},
           {{{9.99997e-01, -8.06132e-08, 2.49797e-05, 3.85120e-05, -2.37451e-04, -3.99607e-04},
             {1.08782e-01, 1.49306e-01, -2.50975e-03, -1.05471e-02, -1.64831e-02, 1.05733e-01},
             {-3.13165e-01, -8.18489e-01, -5.50832e-01, -6.74447e-02, 6.06357e-01, 7.39717e-01},
             {-2.21976e-02, -6.84023e-03, 6.65411e-02, 9.48702e-02, -7.43989e-01, -1.03582e+00},
             {5.87088e-03, -8.10480e-04, -2.21731e-02, -2.94422e-02, 2.29540e-01, 3.19669e-01}}},
           {{{9.99997e-01, -1.87404e-07, 3.10276e-05, 5.20007e-05, -2.98132e-04, -5.19259e-04},
             {1.20783e-01, 1.66407e-01, 1.06608e-02, 6.48772e-03, -1.53031e-01, -7.59354e-02},
             {-3.15222e-01, -8.28793e-01, -6.05740e-01, -1.47812e-01, 1.15760e+00, 1.58565e+00},
             {-2.89055e-02, -9.08096e-03, 1.21467e-01, 1.77575e-01, -1.29110e+00, -1.90333e+00},
             {7.65342e-03, -8.85417e-04, -3.89092e-02, -5.44040e-02, 3.93087e-01, 5.79439e-01}}},
           {{{9.99996e-01, -2.44548e-07, 3.31019e-05, 6.29483e-05, -3.24667e-04, -5.95527e-04},
             {1.32615e-01, 1.83566e-01, 3.04158e-02, 3.40925e-02, -3.54681e-01, -3.63044e-01},
             {-3.16092e-01, -8.38704e-01, -6.78558e-01, -2.59346e-01, 1.88547e+00, 2.73632e+00},
             {-3.67233e-02, -1.18139e-02, 1.91089e-01, 2.87408e-01, -1.98397e+00, -3.03075e+00},
             {9.72033e-03, -9.26380e-04, -5.95654e-02, -8.69829e-02, 5.95744e-01, 9.10242e-01}}},
           {{{9.99995e-01, -2.12227e-07, 2.95645e-05, 6.92848e-05, -3.02153e-04, -6.05145e-04},
             {1.44258e-01, 2.00775e-01, 5.67845e-02, 7.35166e-02, -6.22861e-01, -7.62213e-01},
             {-3.15754e-01, -8.48196e-01, -7.67318e-01, -4.02984e-01, 2.77477e+00, 4.18114e+00},
             {-4.56307e-02, -1.50425e-02, 2.72232e-01, 4.23528e-01, -2.79606e+00, -4.38863e+00},
             {1.20560e-02, -9.44637e-04, -8.28738e-02, -1.26564e-01, 8.26726e-01, 1.29882e+00}}},
           {{{9.99995e-01, -4.03407e-08, 1.86047e-05, 6.85201e-05, -2.14503e-04, -5.22528e-04},
             {1.55704e-01, 2.18048e-01, 8.88994e-02, 1.24878e-01, -9.51331e-01, -1.26824e+00},
             {-3.14244e-01, -8.57322e-01, -8.67190e-01, -5.75787e-01, 3.78571e+00, 5.87052e+00},
             {-5.55526e-02, -1.86861e-02, 3.58860e-01, 5.81094e-01, -3.67623e+00, -5.90800e+00},
             {1.46269e-02, -9.79742e-04, -1.06652e-01, -1.71226e-01, 1.06737e+00, 1.71918e+00}}},
           {{{9.99994e-01, 3.00267e-07, -1.11840e-06, 5.88256e-05, -4.78456e-05, -3.25731e-04},
             {1.66960e-01, 2.35405e-01, 1.25215e-01, 1.87646e-01, -1.32685e+00, -1.86549e+00},
             {-3.11630e-01, -8.66152e-01, -9.71254e-01, -7.72715e-01, 4.85654e+00, 7.72150e+00},
             {-6.63778e-02, -2.26481e-02, 4.42898e-01, 7.53182e-01, -4.55172e+00, -7.48670e+00},
             {1.73883e-02, -1.07669e-03, -1.28075e-01, -2.18389e-01, 1.29217e+00, 2.13475e+00}}},
           {{{9.99994e-01, 8.94829e-07, -2.98434e-05, 3.82193e-05, 2.00584e-04, -6.40482e-06},
             {1.78039e-01, 2.52912e-01, 1.63761e-01, 2.60132e-01, -1.73287e+00, -2.53185e+00},
             {-3.08007e-01, -8.74905e-01, -1.07146e+00, -9.85062e-01, 5.91697e+00, 9.63265e+00},
             {-7.79747e-02, -2.66797e-02, 5.15288e-01, 9.29261e-01, -5.34252e+00, -9.00574e+00},
             {2.02892e-02, -1.33011e-03, -1.44039e-01, -2.64330e-01, 1.47360e+00, 2.50398e+00}}},
           {{{9.99994e-01, 1.75397e-06, -6.73310e-05, 6.29524e-06, 5.29623e-04, 4.35288e-04},
             {1.88968e-01, 2.70612e-01, 2.01975e-01, 3.40574e-01, -2.14737e+00, -3.23836e+00},
             {-3.03499e-01, -8.83717e-01, -1.15816e+00, -1.20414e+00, 6.88176e+00, 1.14841e+01},
             {-9.01806e-02, -3.06202e-02, 5.65581e-01, 1.09902e+00, -5.95552e+00, -1.03302e+01},
             {2.32694e-02, -1.80614e-03, -1.51041e-01, -3.05449e-01, 1.58037e+00, 2.78083e+00}}},
           {{{9.99994e-01, 3.07931e-06, -1.11876e-04, -4.10164e-05, 9.17095e-04, 9.80145e-04},
             {1.99765e-01, 2.88611e-01, 2.37501e-01, 4.25803e-01, -2.55105e+00, -3.95585e+00},
             {-2.98206e-01, -8.92930e-01, -1.22279e+00, -1.41690e+00, 7.67836e+00, 1.31601e+01},
             {-1.02865e-01, -3.40967e-02, 5.84677e-01, 1.24786e+00, -6.31301e+00, -1.13328e+01},
             {2.62800e-02, -2.63995e-03, -1.46076e-01, -3.36795e-01, 1.58677e+00, 2.92251e+00}}},
           {{{9.99993e-01, 4.49776e-06, -1.65136e-04, -9.76754e-05, 1.39664e-03, 1.66293e-03},
             {2.10469e-01, 3.06924e-01, 2.66793e-01, 5.13797e-01, -2.90958e+00, -4.63816e+00},
             {-2.92294e-01, -9.02560e-01, -1.25307e+00, -1.61470e+00, 8.18574e+00, 1.44912e+01},
             {-1.15831e-01, -3.70891e-02, 5.59807e-01, 1.36619e+00, -6.28824e+00, -1.18327e+01},
             {2.92513e-02, -3.84903e-03, -1.24976e-01, -3.55149e-01, 1.45127e+00, 2.86925e+00}}},
           {{{9.99993e-01, 6.01488e-06, -2.22125e-04, -1.61774e-04, 1.92058e-03, 2.41975e-03},
             {2.21091e-01, 3.25648e-01, 2.87732e-01, 6.01632e-01, -3.20436e+00, -5.25724e+00},
             {-2.85814e-01, -9.12889e-01, -1.24213e+00, -1.78635e+00, 8.34196e+00, 1.53776e+01},
             {-1.29005e-01, -3.92986e-02, 4.84255e-01, 1.44206e+00, -5.81999e+00, -1.17275e+01},
             {3.21555e-02, -5.54272e-03, -8.56572e-02, -3.56589e-01, 1.15480e+00, 2.58829e+00}}},
           {{{9.99993e-01, 8.01467e-06, -2.79242e-04, -2.36820e-04, 2.45459e-03, 3.21683e-03},
             {2.31651e-01, 3.44948e-01, 2.97820e-01, 6.85187e-01, -3.41294e+00, -5.77715e+00},
             {-2.78858e-01, -9.24428e-01, -1.18215e+00, -1.91691e+00, 8.07489e+00, 1.56969e+01},
             {-1.42276e-01, -4.01888e-02, 3.50466e-01, 1.45983e+00, -4.83806e+00, -1.08936e+01},
             {3.49529e-02, -7.90933e-03, -2.58002e-02, -3.36028e-01, 6.75740e-01, 2.04052e+00}}},
           {{{9.99992e-01, 1.04277e-05, -3.35126e-04, -3.21042e-04, 2.98507e-03, 4.03325e-03},
             {2.42172e-01, 3.64954e-01, 2.94606e-01, 7.60693e-01, -3.51409e+00, -6.16460e+00},
             {-2.71512e-01, -9.37543e-01, -1.06570e+00, -1.99328e+00, 7.31863e+00, 1.53396e+01},
             {-1.55540e-01, -3.93862e-02, 1.51477e-01, 1.40614e+00, -3.28024e+00, -9.22338e+00},
             {3.76066e-02, -1.10812e-02, 5.66831e-02, -2.89210e-01, -4.60274e-03, 1.19273e+00}}},
           {{{9.99992e-01, 1.30838e-05, -3.84070e-04, -4.09294e-04, 3.47025e-03, 4.81071e-03},
             {2.52646e-01, 3.85718e-01, 2.76330e-01, 8.25665e-01, -3.48888e+00, -6.39017e+00},
             {-2.63758e-01, -9.52326e-01, -8.88179e-01, -2.00720e+00, 6.01960e+00, 1.42157e+01},
             {-1.68806e-01, -3.68095e-02, -1.16429e-01, 1.27309e+00, -1.09991e+00, -6.63409e+00},
             {4.01184e-02, -1.50938e-02, 1.62740e-01, -2.13760e-01, -8.99142e-01, 2.08129e-02}}},
           {{{9.99991e-01, 1.59363e-05, -4.27315e-04, -5.01461e-04, 3.91365e-03, 5.55096e-03},
             {2.63096e-01, 4.07357e-01, 2.40649e-01, 8.76450e-01, -3.31641e+00, -6.42069e+00},
             {-2.55682e-01, -9.69090e-01, -6.43149e-01, -1.94678e+00, 4.11915e+00, 1.22255e+01},
             {-1.81974e-01, -3.21462e-02, -4.58817e-01, 1.04913e+00, 1.75388e+00, -3.03434e+00},
             {4.24541e-02, -2.00595e-02, 2.93916e-01, -1.06138e-01, -2.02203e+00, -1.50181e+00}}},
           {{{9.99990e-01, 1.88950e-05, -4.59994e-04, -5.93663e-04, 4.27684e-03, 6.20090e-03},
             {2.73504e-01, 4.29990e-01, 1.86473e-01, 9.09921e-01, -2.98441e+00, -6.23344e+00},
             {-2.47225e-01, -9.88118e-01, -3.28759e-01, -1.80252e+00, 1.59090e+00, 9.30968e+00},
             {-1.95100e-01, -2.51242e-02, -8.76244e-01, 7.25592e-01, 5.29650e+00, 1.62177e+00},
             {4.46307e-02, -2.60754e-02, 4.50054e-01, 3.61638e-02, -3.37524e+00, -3.38601e+00}}},
           {{{9.99989e-01, 2.18906e-05, -4.79199e-04, -6.83164e-04, 4.53467e-03, 6.72485e-03},
             {2.83854e-01, 4.53722e-01, 1.12877e-01, 9.23203e-01, -2.48247e+00, -5.80855e+00},
             {-2.38338e-01, -1.00965e+00, 5.61453e-02, -1.56605e+00, -1.58353e+00, 5.42138e+00},
             {-2.08226e-01, -1.55138e-02, -1.36846e+00, 2.95168e-01, 9.53392e+00, 7.36550e+00},
             {4.66619e-02, -3.32255e-02, 6.30711e-01, 2.15167e-01, -4.95748e+00, -5.63744e+00}}},
           {{{9.99987e-01, 2.48200e-05, -4.82895e-04, -7.67488e-04, 4.66900e-03, 7.09581e-03},
             {2.94123e-01, 4.78653e-01, 1.93256e-02, 9.13569e-01, -1.80354e+00, -5.13040e+00},
             {-2.28945e-01, -1.03390e+00, 5.11210e-01, -1.22996e+00, -5.40939e+00, 5.31490e-01},
             {-2.21426e-01, -3.12679e-03, -1.93353e+00, -2.48179e-01, 1.44576e+01, 1.42077e+01},
             {4.85718e-02, -4.15796e-02, 8.34879e-01, 4.32418e-01, -6.76244e+00, -8.25464e+00}}},
           {{{9.99985e-01, 2.76168e-05, -4.67522e-04, -8.43978e-04, 4.65008e-03, 7.27476e-03},
             {3.04280e-01, 5.04856e-01, -9.42256e-02, 8.78905e-01, -9.44256e-01, -4.18847e+00},
             {-2.18939e-01, -1.06098e+00, 1.03431e+00, -7.89123e-01, -9.87815e+00, -5.36997e+00},
             {-2.34805e-01, 1.21281e-02, -2.56765e+00, -9.08015e-01, 2.00436e+01, 2.21379e+01},
             {5.03942e-02, -5.11768e-02, 1.06090e+00, 6.88631e-01, -8.77867e+00, -1.12289e+01}}},
           {{{9.99982e-01, 3.00792e-05, -4.33447e-04, -9.09366e-04, 4.47786e-03, 7.25072e-03},
             {3.14283e-01, 5.32444e-01, -2.27528e-01, 8.16951e-01, 9.53704e-02, -2.97730e+00},
             {-2.08186e-01, -1.09112e+00, 1.62237e+00, -2.38394e-01, -1.49699e+01, -1.22743e+01},
             {-2.48493e-01, 3.04543e-02, -3.26601e+00, -1.68760e+00, 2.62568e+01, 3.11259e+01},
             {5.21709e-02, -6.20908e-02, 1.30683e+00, 9.84353e-01, -1.09909e+01, -1.45449e+01}}},
           {{{9.99979e-01, 3.24569e-05, -3.76717e-04, -9.64000e-04, 4.11997e-03, 6.98761e-03},
             {3.24082e-01, 5.61534e-01, -3.79993e-01, 7.25313e-01, 1.31342e+00, -1.49163e+00},
             {-1.96521e-01, -1.12457e+00, 2.27091e+00, 4.27678e-01, -2.06558e+01, -2.01690e+01},
             {-2.62655e-01, 5.20812e-02, -4.02224e+00, -2.59048e+00, 3.30508e+01, 4.11350e+01},
             {5.39556e-02, -7.44077e-02, 1.57015e+00, 1.32022e+00, -1.33802e+01, -1.81848e+01}}},
           {{{9.99976e-01, 3.39628e-05, -2.98845e-04, -9.99651e-04, 3.58523e-03, 6.47782e-03},
             {3.33647e-01, 5.92068e-01, -5.51020e-01, 6.03630e-01, 2.70712e+00, 2.67853e-01},
             {-1.83849e-01, -1.16095e+00, 2.97559e+00, 1.20728e+00, -2.69053e+01, -2.90214e+01},
             {-2.77375e-01, 7.65838e-02, -4.83023e+00, -3.61238e+00, 4.03786e+01, 5.21084e+01},
             {5.57745e-02, -8.79952e-02, 1.84848e+00, 1.69425e+00, -1.59274e+01, -2.21242e+01}}},
           {{{9.99972e-01, 3.48473e-05, -1.97828e-04, -1.01659e-03, 2.85509e-03, 5.69725e-03},
             {3.42904e-01, 6.24164e-01, -7.39023e-01, 4.50306e-01, 4.26553e+00, 2.29299e+00},
             {-1.69940e-01, -1.20051e+00, 3.72868e+00, 2.10276e+00, -3.36594e+01, -3.87726e+01},
             {-2.92882e-01, 1.04194e-01, -5.68033e+00, -4.75339e+00, 4.81633e+01, 6.39614e+01},
             {5.77008e-02, -1.02941e-01, 2.13826e+00, 2.10589e+00, -1.86034e+01, -2.63294e+01}}},
           {{{9.99968e-01, 3.47804e-05, -7.11898e-05, -1.01028e-03, 1.90919e-03, 4.61426e-03},
             {3.51793e-01, 6.57815e-01, -9.42175e-01, 2.64940e-01, 5.97606e+00, 4.57222e+00},
             {-1.54595e-01, -1.24305e+00, 4.52170e+00, 3.11205e+00, -4.08539e+01, -4.93518e+01},
             {-3.09367e-01, 1.34661e-01, -6.56214e+00, -6.00882e+00, 5.63228e+01, 7.65965e+01},
             {5.97948e-02, -1.19174e-01, 2.43570e+00, 2.55300e+00, -2.13779e+01, -3.07628e+01}}},
           {{{9.99963e-01, 3.37519e-05, 8.13037e-05, -9.78638e-04, 7.41412e-04, 3.21498e-03},
             {3.60246e-01, 6.93065e-01, -1.15880e+00, 4.68519e-02, 7.82662e+00, 7.09456e+00},
             {-1.37615e-01, -1.28855e+00, 5.34673e+00, 4.23402e+00, -4.84263e+01, -6.06897e+01},
             {-3.27017e-01, 1.67941e-01, -7.46593e+00, -7.37492e+00, 6.47773e+01, 8.99181e+01},
             {6.21158e-02, -1.36692e-01, 2.73726e+00, 3.03374e+00, -2.42209e+01, -3.53873e+01}}},
           {{{9.99958e-01, 3.14983e-05, 2.59741e-04, -9.19008e-04, -6.49202e-04, 1.49008e-03},
             {3.68196e-01, 7.29888e-01, -1.38664e+00, -2.03807e-01, 9.80062e+00, 9.84155e+00},
             {-1.18788e-01, -1.33674e+00, 6.19400e+00, 5.46446e+00, -5.62994e+01, -7.26915e+01},
             {-3.46033e-01, 2.03729e-01, -8.38012e+00, -8.84466e+00, 7.34319e+01, 1.03805e+02},
             {6.47255e-02, -1.55409e-01, 3.03879e+00, 3.54516e+00, -2.70980e+01, -4.01572e+01}}},
           {{{9.99952e-01, 2.79961e-05, 4.60479e-04, -8.33486e-04, -2.23214e-03, -5.16285e-04},
             {3.75580e-01, 7.68292e-01, -1.62392e+00, -4.87674e-01, 1.18854e+01, 1.28025e+01},
             {-9.79355e-02, -1.38749e+00, 7.05560e+00, 6.80198e+00, -6.44113e+01, -8.52915e+01},
             {-3.66572e-01, 2.41863e-01, -9.29524e+00, -1.04141e+01, 8.22093e+01, 1.18166e+02},
             {6.76714e-02, -1.75289e-01, 3.33691e+00, 4.08538e+00, -2.99809e+01, -4.50381e+01}}},
           {{{9.99947e-01, 2.29520e-05, 6.82639e-04, -7.17139e-04, -4.00522e-03, -2.81601e-03},
             {3.82332e-01, 8.08194e-01, -1.86848e+00, -8.03415e-01, 1.40640e+01, 1.59560e+01},
             {-7.48735e-02, -1.44031e+00, 7.92233e+00, 8.23820e+00, -7.26858e+01, -9.83865e+01},
             {-3.88797e-01, 2.81837e-01, -1.02005e+01, -1.20716e+01, 9.10175e+01, 1.32873e+02},
             {7.10017e-02, -1.96182e-01, 3.62780e+00, 4.65000e+00, -3.28365e+01, -4.99823e+01}}},
           {{{9.99941e-01, 1.63607e-05, 9.28242e-04, -5.69257e-04, -5.98245e-03, -5.42476e-03},
             {3.88363e-01, 8.49548e-01, -2.11701e+00, -1.15003e+00, 1.63119e+01, 1.92737e+01},
             {-4.93364e-02, -1.49491e+00, 8.78129e+00, 9.76596e+00, -8.10212e+01, -1.11851e+02},
             {-4.12953e-01, 3.23322e-01, -1.10814e+01, -1.38073e+01, 9.97386e+01, 1.47775e+02},
             {7.47911e-02, -2.18001e-01, 3.90642e+00, 5.23509e+00, -3.56231e+01, -5.49352e+01}}},
           {{{9.99935e-01, 8.31520e-06, 1.19244e-03, -3.95258e-04, -8.12525e-03, -8.28836e-03},
             {3.93603e-01, 8.92358e-01, -2.36688e+00, -1.52786e+00, 1.86071e+01, 2.27314e+01},
             {-2.11345e-02, -1.55112e+00, 9.62205e+00, 1.13827e+01, -8.93266e+01, -1.25574e+02},
             {-4.39199e-01, 3.66161e-01, -1.19262e+01, -1.56159e+01, 1.08267e+02, 1.62736e+02},
             {7.90853e-02, -2.40719e-01, 4.16872e+00, 5.83839e+00, -3.83040e+01, -5.98484e+01}}},
           {{{9.99929e-01, -1.65608e-06, 1.47476e-03, -1.86250e-04, -1.04363e-02, -1.14318e-02},
             {3.97989e-01, 9.36430e-01, -2.61569e+00, -1.93387e+00, 2.09305e+01, 2.63021e+01},
             {9.88744e-03, -1.60813e+00, 1.04351e+01, 1.30740e+01, -9.75207e+01, -1.39437e+02},
             {-4.67657e-01, 4.09494e-01, -1.27240e+01, -1.74797e+01, 1.16508e+02, 1.77614e+02},
             {8.39169e-02, -2.64077e-01, 4.41092e+00, 6.45344e+00, -4.08455e+01, -6.46698e+01}}},
           {{{9.99922e-01, -1.37624e-05, 1.77335e-03, 5.99400e-05, -1.29013e-02, -1.48443e-02},
             {4.01450e-01, 9.81600e-01, -2.86053e+00, -2.36562e+00, 2.32591e+01, 2.99551e+01},
             {4.39150e-02, -1.66522e+00, 1.12093e+01, 1.48279e+01, -1.05512e+02, -1.53310e+02},
             {-4.98475e-01, 4.52607e-01, -1.34628e+01, -1.93836e+01, 1.24357e+02, 1.92256e+02},
             {8.93258e-02, -2.87869e-01, 4.62891e+00, 7.07473e+00, -4.32116e+01, -6.93461e+01}}},
           {{{9.99916e-01, -2.78975e-05, 2.08753e-03, 3.41607e-04, -1.55141e-02, -1.85156e-02},
             {4.03889e-01, 1.02783e+00, -3.09762e+00, -2.82183e+00, 2.55629e+01, 3.36548e+01},
             {8.12122e-02, -1.72210e+00, 1.19310e+01, 1.66362e+01, -1.13185e+02, -1.67048e+02},
             {-5.31880e-01, 4.95236e-01, -1.41275e+01, -2.13166e+01, 1.31687e+02, 2.06495e+02},
             {9.53781e-02, -3.12041e-01, 4.81765e+00, 7.69813e+00, -4.53587e+01, -7.38182e+01}}},
           {{{9.99910e-01, -4.39880e-05, 2.41439e-03, 6.54636e-04, -1.82489e-02, -2.24062e-02},
             {4.05249e-01, 1.07495e+00, -3.32445e+00, -3.30073e+00, 2.78221e+01, 3.73760e+01},
             {1.21896e-01, -1.77813e+00, 1.25907e+01, 1.84893e+01, -1.20461e+02, -1.80542e+02},
             {-5.67942e-01, 5.36741e-01, -1.47080e+01, -2.32664e+01, 1.38408e+02, 2.20203e+02},
             {1.02086e-01, -3.36419e-01, 4.97373e+00, 8.31905e+00, -4.72562e+01, -7.80409e+01}}},
           {{{9.99904e-01, -6.22073e-05, 2.75066e-03, 1.00063e-03, -2.10817e-02, -2.64933e-02},
             {4.05472e-01, 1.12279e+00, -3.53825e+00, -3.79964e+00, 3.00132e+01, 4.10871e+01},
             {1.66089e-01, -1.83255e+00, 1.31785e+01, 2.03743e+01, -1.27252e+02, -1.93664e+02},
             {-6.06729e-01, 5.76392e-01, -1.51936e+01, -2.52171e+01, 1.44423e+02, 2.33232e+02},
             {1.09463e-01, -3.60803e-01, 5.09358e+00, 8.93178e+00, -4.88706e+01, -8.19632e+01}}},
           {{{9.99898e-01, -8.26232e-05, 3.09353e-03, 1.37924e-03, -2.39902e-02, -3.07502e-02},
             {4.04498e-01, 1.17115e+00, -3.73628e+00, -4.31580e+00, 3.21142e+01, 4.47580e+01},
             {2.13904e-01, -1.88458e+00, 1.36848e+01, 2.22782e+01, -1.33471e+02, -2.06293e+02},
             {-6.48301e-01, 6.13439e-01, -1.55742e+01, -2.71531e+01, 1.49638e+02, 2.45444e+02},
             {1.17516e-01, -3.84990e-01, 5.17383e+00, 9.53070e+00, -5.01709e+01, -8.55368e+01}}},
           {{{9.99893e-01, -1.05293e-04, 3.43932e-03, 1.78981e-03, -2.69462e-02, -3.51436e-02},
             {4.02272e-01, 1.21982e+00, -3.91575e+00, -4.84621e+00, 3.41013e+01, 4.83563e+01},
             {2.65438e-01, -1.93341e+00, 1.40998e+01, 2.41876e+01, -1.39034e+02, -2.18300e+02},
             {-6.92691e-01, 6.47135e-01, -1.58398e+01, -2.90581e+01, 1.53960e+02, 2.56697e+02},
             {1.26243e-01, -4.08783e-01, 5.21125e+00, 1.01100e+01, -5.11262e+01, -8.87130e+01}}},
           {{{9.99888e-01, -1.30188e-04, 3.78436e-03, 2.22971e-03, -2.99185e-02, -3.96315e-02},
             {3.98750e-01, 1.26855e+00, -4.07427e+00, -5.38796e+00, 3.59550e+01, 5.18546e+01},
             {3.20748e-01, -1.97817e+00, 1.44153e+01, 2.60890e+01, -1.43866e+02, -2.29578e+02},
             {-7.39891e-01, 6.76679e-01, -1.59819e+01, -3.09162e+01, 1.57312e+02, 2.66867e+02},
             {1.35630e-01, -4.31972e-01, 5.20310e+00, 1.06642e+01, -5.17104e+01, -9.14493e+01}}},
           {{{9.99884e-01, -1.57459e-04, 4.12052e-03, 2.69984e-03, -3.28468e-02, -4.41512e-02},
             {3.93891e-01, 1.31710e+00, -4.20944e+00, -5.93691e+00, 3.76535e+01, 5.52188e+01},
             {3.79857e-01, -2.01791e+00, 1.46238e+01, 2.79653e+01, -1.47893e+02, -2.39997e+02},
             {-7.89852e-01, 7.01196e-01, -1.59929e+01, -3.27077e+01, 1.59613e+02, 2.75813e+02},
             {1.45646e-01, -4.54328e-01, 5.14701e+00, 1.11863e+01, -5.18977e+01, -9.36987e+01}}},
           {{{9.99881e-01, -1.86816e-04, 4.45491e-03, 3.19574e-03, -3.57812e-02, -4.87457e-02},
             {3.87612e-01, 1.36524e+00, -4.31686e+00, -6.49005e+00, 3.91612e+01, 5.84032e+01},
             {4.42940e-01, -2.05189e+00, 1.47103e+01, 2.98033e+01, -1.50988e+02, -2.49389e+02},
             {-8.42683e-01, 7.20045e-01, -1.58579e+01, -3.44169e+01, 1.60733e+02, 2.83354e+02},
             {1.56312e-01, -4.75705e-01, 5.03810e+00, 1.16708e+01, -5.16454e+01, -9.53999e+01}}},
           {{{9.99878e-01, -2.18480e-04, 4.78376e-03, 3.71897e-03, -3.86926e-02, -5.33866e-02},
             {3.79879e-01, 1.41265e+00, -4.39391e+00, -7.04282e+00, 4.04550e+01, 6.13721e+01},
             {5.10001e-01, -2.07896e+00, 1.46666e+01, 3.15843e+01, -1.53072e+02, -2.57624e+02},
             {-8.98306e-01, 7.32213e-01, -1.55686e+01, -3.60229e+01, 1.60591e+02, 2.89348e+02},
             {1.67591e-01, -4.95837e-01, 4.87389e+00, 1.21106e+01, -5.09274e+01, -9.65060e+01}}},
           {{{9.99876e-01, -2.52173e-04, 5.09558e-03, 4.26518e-03, -4.14939e-02, -5.79712e-02},
             {3.70689e-01, 1.45908e+00, -4.43983e+00, -7.59192e+00, 4.15261e+01, 6.41080e+01},
             {5.80924e-01, -2.09831e+00, 1.44911e+01, 3.32945e+01, -1.54118e+02, -2.64634e+02},
             {-9.56518e-01, 7.37009e-01, -1.51243e+01, -3.75098e+01, 1.59159e+02, 2.93723e+02},
             {1.79397e-01, -5.14573e-01, 4.65426e+00, 1.25003e+01, -4.97362e+01, -9.69932e+01}}},
           {{{9.99874e-01, -2.87917e-04, 5.39066e-03, 4.83313e-03, -4.41852e-02, -6.24953e-02},
             {3.60001e-01, 1.50420e+00, -4.45150e+00, -8.13270e+00, 4.23476e+01, 6.65709e+01},
             {6.55716e-01, -2.10882e+00, 1.41741e+01, 3.49157e+01, -1.54034e+02, -2.70276e+02},
             {-1.01724e+00, 7.33501e-01, -1.45156e+01, -3.88572e+01, 1.56348e+02, 2.96328e+02},
             {1.91690e-01, -5.31690e-01, 4.37639e+00, 1.28328e+01, -4.80435e+01, -9.68125e+01}}},
           {{{9.99874e-01, -3.25847e-04, 5.67083e-03, 5.42945e-03, -4.67877e-02, -6.70100e-02},
             {3.47783e-01, 1.54767e+00, -4.42606e+00, -8.65995e+00, 4.28934e+01, 6.87183e+01},
             {7.34353e-01, -2.10941e+00, 1.37070e+01, 3.64273e+01, -1.52736e+02, -2.74401e+02},
             {-1.08036e+00, 7.20771e-01, -1.37344e+01, -4.00427e+01, 1.52073e+02, 2.97007e+02},
             {2.04417e-01, -5.46972e-01, 4.03784e+00, 1.31007e+01, -4.58224e+01, -9.59123e+01}}},
           {{{9.99875e-01, -3.65406e-04, 5.92115e-03, 6.03332e-03, -4.91674e-02, -7.12951e-02},
             {3.34062e-01, 1.58924e+00, -4.36361e+00, -9.17241e+00, 4.31638e+01, 7.05522e+01},
             {8.16604e-01, -2.09933e+00, 1.30915e+01, 3.78235e+01, -1.50231e+02, -2.77015e+02},
             {-1.14554e+00, 6.98298e-01, -1.27840e+01, -4.10591e+01, 1.46348e+02, 2.95770e+02},
             {2.17451e-01, -5.60344e-01, 3.63994e+00, 1.33015e+01, -4.30802e+01, -9.42981e+01}}},
           {{{9.99877e-01, -4.06637e-04, 6.14278e-03, 6.64937e-03, -5.13391e-02, -7.53881e-02},
             {3.18822e-01, 1.62854e+00, -4.26169e+00, -9.66459e+00, 4.31357e+01, 7.20331e+01},
             {9.02373e-01, -2.07743e+00, 1.23209e+01, 3.90832e+01, -1.46445e+02, -2.77984e+02},
             {-1.21259e+00, 6.65174e-01, -1.16583e+01, -4.18839e+01, 1.39104e+02, 2.92479e+02},
             {2.30710e-01, -5.71608e-01, 3.18104e+00, 1.34277e+01, -3.97963e+01, -9.19255e+01}}},
           {{{9.99881e-01, -4.49175e-04, 6.32631e-03, 7.27184e-03, -5.32297e-02, -7.91972e-02},
             {3.02086e-01, 1.66529e+00, -4.11996e+00, -1.01330e+01, 4.28031e+01, 7.31464e+01},
             {9.91429e-01, -2.04292e+00, 1.13959e+01, 4.01931e+01, -1.41369e+02, -2.77260e+02},
             {-1.28117e+00, 6.20870e-01, -1.03596e+01, -4.25024e+01, 1.30338e+02, 2.87090e+02},
             {2.44065e-01, -5.80702e-01, 2.66217e+00, 1.34745e+01, -3.59721e+01, -8.87824e+01}}},
           {{{9.99886e-01, -4.93136e-04, 6.47914e-03, 7.90029e-03, -5.48909e-02, -8.27736e-02},
             {2.83840e-01, 1.69905e+00, -3.93534e+00, -1.05724e+01, 4.21392e+01, 7.38521e+01},
             {1.08366e+00, -1.99451e+00, 1.03076e+01, 4.11340e+01, -1.34917e+02, -2.74710e+02},
             {-1.35107e+00, 5.64397e-01, -8.87993e+00, -4.28945e+01, 1.19972e+02, 2.79470e+02},
             {2.57431e-01, -5.87417e-01, 2.08111e+00, 1.34353e+01, -3.15844e+01, -8.48268e+01}}},
           {{{9.99892e-01, -5.38122e-04, 6.59206e-03, 8.52848e-03, -5.62508e-02, -8.60251e-02},
             {2.64130e-01, 1.72952e+00, -3.70812e+00, -1.09794e+01, 4.11430e+01, 7.41410e+01},
             {1.17876e+00, -1.93141e+00, 9.05928e+00, 4.18928e+01, -1.27099e+02, -2.70309e+02},
             {-1.42185e+00, 4.95292e-01, -7.22427e+00, -4.30461e+01, 1.08024e+02, 2.69601e+02},
             {2.70647e-01, -5.91727e-01, 1.43982e+00, 1.33056e+01, -2.66423e+01, -8.00557e+01}}},
           {{{9.99900e-01, -5.83896e-04, 6.66373e-03, 9.15389e-03, -5.72988e-02, -8.89368e-02},
             {2.42961e-01, 1.75635e+00, -3.43619e+00, -1.13482e+01, 3.97928e+01, 7.39747e+01},
             {1.27654e+00, -1.85263e+00, 7.64589e+00, 4.24492e+01, -1.17853e+02, -2.63933e+02},
             {-1.49322e+00, 4.12926e-01, -5.38909e+00, -4.29363e+01, 9.44419e+01, 2.57363e+02},
             {2.83597e-01, -5.93557e-01, 7.37540e-01, 1.30785e+01, -2.11316e+01, -7.44329e+01}}},
           {{{9.99909e-01, -6.30100e-04, 6.69295e-03, 9.77077e-03, -5.80220e-02, -9.14799e-02},
             {2.20373e-01, 1.77919e+00, -3.11907e+00, -1.16754e+01, 3.80818e+01, 7.33384e+01},
             {1.37668e+00, -1.75734e+00, 6.06819e+00, 4.27905e+01, -1.07169e+02, -2.55543e+02},
             {-1.56475e+00, 3.16885e-01, -3.37687e+00, -4.25518e+01, 7.92269e+01, 2.42726e+02},
             {2.96124e-01, -5.92915e-01, -2.45072e-02, 1.27501e+01, -1.50562e+01, -6.79514e+01}}},
           {{{9.99920e-01, -6.77211e-04, 6.67917e-03, 1.03866e-02, -5.84208e-02, -9.36775e-02},
             {1.96406e-01, 1.79753e+00, -2.75530e+00, -1.19533e+01, 3.59929e+01, 7.21943e+01},
             {1.47887e+00, -1.64416e+00, 4.32381e+00, 4.28906e+01, -9.50034e+01, -2.45021e+02},
             {-1.63600e+00, 2.06169e-01, -1.18704e+00, -4.18661e+01, 6.23476e+01, 2.25579e+02},
             {3.08067e-01, -5.89622e-01, -8.46080e-01, 1.23116e+01, -8.40939e+00, -6.05788e+01}}},
           {{{9.99932e-01, -7.23543e-04, 6.61026e-03, 1.09756e-02, -5.83892e-02, -9.53462e-02},
             {1.71135e-01, 1.81127e+00, -2.34665e+00, -1.21818e+01, 3.35368e+01, 7.05547e+01},
             {1.58269e+00, -1.51310e+00, 2.42140e+00, 4.27487e+01, -8.14091e+01, -2.32422e+02},
             {-1.70642e+00, 8.13302e-02, 1.16945e+00, -4.08790e+01, 4.38728e+01, 2.05995e+02},
             {3.19226e-01, -5.84023e-01, -1.72313e+00, 1.17636e+01, -1.21776e+00, -5.23434e+01}}},
           {{{9.99946e-01, -7.69368e-04, 6.48923e-03, 1.15425e-02, -5.79523e-02, -9.65284e-02},
             {1.44613e-01, 1.81998e+00, -1.89196e+00, -1.23548e+01, 3.06998e+01, 6.83894e+01},
             {1.68776e+00, -1.36316e+00, 3.59801e-01, 4.23443e+01, -6.63559e+01, -2.17658e+02},
             {-1.77552e+00, -5.81387e-02, 3.69184e+00, -3.95701e+01, 2.37855e+01, 1.83897e+02},
             {3.29420e-01, -5.76119e-01, -2.65496e+00, 1.10995e+01, 6.52027e+00, -4.32241e+01}}},
           {{{9.99962e-01, -8.13654e-04, 6.31217e-03, 1.20763e-02, -5.70782e-02, -9.71683e-02},
             {1.16915e-01, 1.82343e+00, -1.39208e+00, -1.24689e+01, 2.74833e+01, 6.56914e+01},
             {1.79364e+00, -1.19400e+00, -1.85513e+00, 4.16665e+01, -4.98680e+01, -2.00724e+02},
             {-1.84272e+00, -2.12012e-01, 6.37189e+00, -3.79293e+01, 2.12742e+00, 1.59295e+02},
             {3.38449e-01, -5.66167e-01, -3.63838e+00, 1.03165e+01, 1.47866e+01, -3.32296e+01}}},
           {{{9.99980e-01, -8.57331e-04, 6.07736e-03, 1.25819e-02, -5.57471e-02, -9.72475e-02},
             {8.81540e-02, 1.82105e+00, -8.47718e-01, -1.25186e+01, 2.38891e+01, 6.24510e+01},
             {1.89976e+00, -1.00428e+00, -4.21799e+00, 4.06970e+01, -3.19703e+01, -1.81604e+02},
             {-1.90731e+00, -3.81080e-01, 9.20190e+00, -3.59386e+01, -2.10594e+01, 1.32192e+02},
             {3.46059e-01, -5.54093e-01, -4.67039e+00, 9.40924e+00, 2.35631e+01, -2.23653e+01}}},
           {{{9.99999e-01, -8.99343e-04, 5.78680e-03, 1.30512e-02, -5.39764e-02, -9.67723e-02},
             {5.84024e-02, 1.81266e+00, -2.59193e-01, -1.25002e+01, 1.99139e+01, 5.86545e+01},
             {2.00567e+00, -7.93885e-01, -6.72461e+00, 3.94244e+01, -1.26714e+01, -1.60273e+02},
             {-1.96873e+00, -5.64782e-01, 1.21753e+01, -3.35880e+01, -4.57487e+01, 1.02578e+02},
             {3.52050e-01, -5.40282e-01, -5.74837e+00, 8.37487e+00, 3.28364e+01, -1.06341e+01}}},
           {{{1.00002e+00, -9.39311e-04, 5.43607e-03, 1.34763e-02, -5.17249e-02, -9.56765e-02},
             {2.78005e-02, 1.79795e+00, 3.70737e-01, -1.24114e+01, 1.55759e+01, 5.43184e+01},
             {2.11070e+00, -5.62473e-01, -9.36239e+00, 3.78424e+01, 7.94516e+00, -1.36806e+02},
             {-2.02617e+00, -7.62769e-01, 1.52768e+01, -3.08728e+01, -7.18354e+01, 7.05554e+01},
             {3.56140e-01, -5.25061e-01, -6.86679e+00, 7.21247e+00, 4.25672e+01, 1.92463e+00}}},
           {{{1.00004e+00, -9.76747e-04, 5.02266e-03, 1.38530e-02, -4.89737e-02, -9.39340e-02},
             {-3.54744e-03, 1.77664e+00, 1.04132e+00, -1.22473e+01, 1.08741e+01, 4.94269e+01},
             {2.21429e+00, -3.09743e-01, -1.21254e+01, 3.59359e+01, 2.98600e+01, -1.11177e+02},
             {-2.07893e+00, -9.74603e-01, 1.84979e+01, -2.77795e+01, -9.92800e+01, 3.61173e+01},
             {3.58090e-01, -5.08796e-01, -8.02233e+00, 5.91833e+00, 5.27374e+01, 1.53069e+01}}},
           {{{1.00007e+00, -1.01117e-03, 4.54349e-03, 1.41720e-02, -4.56907e-02, -9.14834e-02},
             {-3.54805e-02, 1.74848e+00, 1.74921e+00, -1.20063e+01, 5.83106e+00, 4.40029e+01},
             {2.31570e+00, -3.56472e-02, -1.49988e+01, 3.37013e+01, 5.29727e+01, -8.34879e+01},
             {-2.12613e+00, -1.19953e+00, 2.18208e+01, -2.43067e+01, -1.27958e+02, -6.05797e-01},
             {3.57593e-01, -4.91966e-01, -9.20870e+00, 4.49274e+00, 6.33009e+01, 2.94631e+01}}},
           {{{1.00009e+00, -1.04219e-03, 3.99979e-03, 1.44301e-02, -4.18849e-02, -8.83308e-02},
             {-6.78832e-02, 1.71323e+00, 2.49363e+00, -1.16834e+01, 4.45978e-01, 3.80313e+01},
             {2.41433e+00, 2.59886e-01, -1.79765e+01, 3.11249e+01, 7.72617e+01, -5.37152e+01},
             {-2.16704e+00, -1.43680e+00, 2.52369e+01, -2.04429e+01, -1.57828e+02, -3.96140e+01},
             {3.54400e-01, -4.75067e-01, -1.04225e+01, 2.93269e+00, 7.42385e+01, 4.43866e+01}}},
           {{{1.00012e+00, -1.06927e-03, 3.39091e-03, 1.46208e-02, -3.75471e-02, -8.44506e-02},
             {-1.00611e-01, 1.67068e+00, 3.27246e+00, -1.12756e+01, -5.27058e+00, 3.15152e+01},
             {2.50948e+00, 5.76648e-01, -2.10477e+01, 2.81988e+01, 1.02665e+02, -2.18989e+01},
             {-2.20084e+00, -1.68528e+00, 2.87325e+01, -1.61828e+01, -1.88802e+02, -8.08394e+01},
             {3.48226e-01, -4.58722e-01, -1.16589e+01, 1.23735e+00, 8.55166e+01, 6.00477e+01}}},
           {{{1.00015e+00, -1.09216e-03, 2.71745e-03, 1.47404e-02, -3.26779e-02, -7.98343e-02},
             {-1.33495e-01, 1.62059e+00, 4.08307e+00, -1.07793e+01, -1.13042e+01, 2.44607e+01},
             {2.60038e+00, 9.14519e-01, -2.42000e+01, 2.49142e+01, 1.29108e+02, 1.19101e+01},
             {-2.22661e+00, -1.94391e+00, 3.22924e+01, -1.15206e+01, -2.20784e+02, -1.24202e+02},
             {3.38764e-01, -4.43546e-01, -1.29122e+01, -5.94280e-01, 9.70972e+01, 7.64128e+01}}},
           {{{1.00018e+00, -1.11005e-03, 1.98019e-03, 1.47814e-02, -2.72814e-02, -7.44728e-02},
             {-1.66392e-01, 1.56288e+00, 4.92359e+00, -1.01915e+01, -1.76472e+01, 1.68656e+01},
             {2.68631e+00, 1.27279e+00, -2.74231e+01, 2.12642e+01, 1.56537e+02, 4.76851e+01},
             {-2.24352e+00, -2.21096e+00, 3.59035e+01, -6.45262e+00, -2.53694e+02, -1.69646e+02},
             {3.25734e-01, -4.30391e-01, -1.41779e+01, -2.56236e+00, 1.08949e+02, 9.34555e+01}}},
           {{{1.00021e+00, -1.12270e-03, 1.18750e-03, 1.47467e-02, -2.14258e-02, -6.84575e-02},
             {-1.99145e-01, 1.49739e+00, 5.79226e+00, -9.50812e+00, -2.42938e+01, 8.72312e+00},
             {2.76656e+00, 1.65088e+00, -3.07077e+01, 1.72391e+01, 1.84905e+02, 8.54178e+01},
             {-2.25073e+00, -2.48478e+00, 3.95541e+01, -9.72452e-01, -2.87462e+02, -2.17133e+02},
             {3.08850e-01, -4.20088e-01, -1.54515e+01, -4.66797e+00, 1.21043e+02, 1.11156e+02}}},
           {{{1.00025e+00, -1.12974e-03, 3.25083e-04, 1.46196e-02, -1.49818e-02, -6.15916e-02},
             {-2.31499e-01, 1.42403e+00, 6.68301e+00, -8.72894e+00, -3.12005e+01, 8.13039e-02},
             {2.84004e+00, 2.04799e+00, -3.40292e+01, 1.28426e+01, 2.14035e+02, 1.24907e+02},
             {-2.24701e+00, -2.76343e+00, 4.32160e+01, 4.91177e+00, -3.21880e+02, -2.66423e+02},
             {2.87708e-01, -4.13579e-01, -1.67235e+01, -6.90724e+00, 1.33305e+02, 1.29426e+02}}},
           {{{1.00028e+00, -1.13018e-03, -5.85208e-04, 1.44008e-02, -8.12721e-03, -5.40793e-02},
             {-2.63328e-01, 1.34285e+00, 7.59541e+00, -7.85118e+00, -3.83723e+01, -9.07534e+00},
             {2.90612e+00, 2.46267e+00, -3.73829e+01, 8.06978e+00, 2.43918e+02, 1.66172e+02},
             {-2.23164e+00, -3.04426e+00, 4.68824e+01, 1.12007e+01, -3.56918e+02, -3.17506e+02},
             {2.62068e-01, -4.12056e-01, -1.79911e+01, -9.27922e+00, 1.45719e+02, 1.48255e+02}}},
           {{{1.00032e+00, -1.12442e-03, -1.55891e-03, 1.40881e-02, -7.32260e-04, -4.57754e-02},
             {-2.94323e-01, 1.25374e+00, 8.52167e+00, -6.87354e+00, -4.57537e+01, -1.86910e+01},
             {2.96357e+00, 2.89403e+00, -4.07385e+01, 2.92128e+00, 2.74335e+02, 2.08986e+02},
             {-2.20324e+00, -3.32515e+00, 5.05197e+01, 1.78893e+01, -3.92323e+02, -3.70113e+02},
             {2.31474e-01, -4.16536e-01, -1.92429e+01, -1.17810e+01, 1.58197e+02, 1.67545e+02}}},
           {{{1.00036e+00, -1.11107e-03, -2.58076e-03, 1.36692e-02, 7.08911e-03, -3.67730e-02},
             {-3.24316e-01, 1.15695e+00, 9.45947e+00, -5.79675e+00, -5.33328e+01, -2.87528e+01},
             {3.01159e+00, 3.33981e+00, -4.40850e+01, -2.59468e+00, 3.05217e+02, 2.53266e+02},
             {-2.16093e+00, -3.60244e+00, 5.41141e+01, 2.49638e+01, -4.28002e+02, -4.24126e+02},
             {1.95645e-01, -4.28567e-01, -2.04742e+01, -1.44066e+01, 1.70702e+02, 1.87249e+02}}},
           {{{1.00039e+00, -1.09011e-03, -3.65347e-03, 1.31413e-02, 1.53649e-02, -2.70335e-02},
             {-3.53028e-01, 1.05257e+00, 1.04028e+01, -4.62034e+00, -6.10681e+01, -3.92186e+01},
             {3.04907e+00, 3.79831e+00, -4.73979e+01, -8.47391e+00, 3.36394e+02, 2.98829e+02},
             {-2.10347e+00, -3.87306e+00, 5.76385e+01, 3.24147e+01, -4.63751e+02, -4.79318e+02},
             {1.54174e-01, -4.49506e-01, -2.16756e+01, -1.71516e+01, 1.83161e+02, 2.07283e+02}}},
           {{{1.00043e+00, -1.06114e-03, -4.77340e-03, 1.25001e-02, 2.40713e-02, -1.65728e-02},
             {-3.80207e-01, 9.40778e-01, 1.13464e+01, -3.34423e+00, -6.89259e+01, -5.00549e+01},
             {3.07493e+00, 4.26732e+00, -5.06567e+01, -1.47102e+01, 3.67721e+02, 3.45521e+02},
             {-2.02970e+00, -4.13334e+00, 6.10696e+01, 4.02298e+01, -4.99399e+02, -5.35495e+02},
             {1.06694e-01, -4.80932e-01, -2.28393e+01, -2.00106e+01, 1.95512e+02, 2.27574e+02}}},
           {{{1.00047e+00, -1.02377e-03, -5.93877e-03, 1.17393e-02, 3.32021e-02, -5.38025e-03},
             {-4.05577e-01, 8.21834e-01, 1.22846e+01, -1.96909e+00, -7.68675e+01, -6.12207e+01},
             {3.08808e+00, 4.74434e+00, -5.38383e+01, -2.12945e+01, 3.99035e+02, 3.93158e+02},
             {-1.93842e+00, -4.37923e+00, 6.43816e+01, 4.83938e+01, -5.34750e+02, -5.92428e+02},
             {5.28167e-02, -5.24544e-01, -2.39565e+01, -2.29768e+01, 2.07685e+02, 2.48037e+02}}},
           {{{1.00051e+00, -9.77691e-04, -7.14416e-03, 1.08563e-02, 4.27196e-02, 6.50699e-03},
             {-4.28882e-01, 6.96049e-01, 1.32126e+01, -4.94649e-01, -8.48638e+01, -7.26907e+01},
             {3.08745e+00, 5.22659e+00, -5.69235e+01, -2.82201e+01, 4.30208e+02, 4.41611e+02},
             {-1.82849e+00, -4.60634e+00, 6.75530e+01, 5.68935e+01, -5.69648e+02, -6.49946e+02},
             {-7.81506e-03, -5.82177e-01, -2.50201e+01, -2.60444e+01, 2.19621e+02, 2.68607e+02}}},
           {{{1.00055e+00, -9.22411e-04, -8.38967e-03, 9.83942e-03, 5.26394e-02, 1.91419e-02},
             {-4.49833e-01, 5.63865e-01, 1.41246e+01, 1.07680e+00, -9.28737e+01, -8.44177e+01},
             {3.07189e+00, 5.71077e+00, -5.98888e+01, -3.54715e+01, 4.61066e+02, 4.90670e+02},
             {-1.69869e+00, -4.80968e+00, 7.05579e+01, 6.57062e+01, -6.03887e+02, -7.07793e+02},
             {-7.55851e-02, -6.55871e-01, -2.60213e+01, -2.92042e+01, 2.31249e+02, 2.89188e+02}}},
           {{{1.00059e+00, -8.58152e-04, -9.66020e-03, 8.70229e-03, 6.28328e-02, 3.23376e-02},
             {-4.68192e-01, 4.25668e-01, 1.50174e+01, 2.74763e+00, -1.00883e+02, -9.64012e+01},
             {3.04043e+00, 6.19366e+00, -6.27207e+01, -4.30487e+01, 4.91532e+02, 5.40291e+02},
             {-1.54799e+00, -4.98431e+00, 7.33810e+01, 7.48252e+01, -6.37365e+02, -7.65885e+02},
             {-1.50810e-01, -7.47659e-01, -2.69551e+01, -3.24524e+01, 2.42528e+02, 3.09744e+02}}},
           {{{1.00062e+00, -7.84582e-04, -1.09553e-02, 7.43533e-03, 7.33106e-02, 4.61358e-02},
             {-4.83646e-01, 2.82050e-01, 1.58842e+01, 4.51493e+00, -1.08846e+02, -1.08589e+02},
             {2.99185e+00, 6.67128e+00, -6.53938e+01, -5.09329e+01, 5.21419e+02, 5.90246e+02},
             {-1.37512e+00, -5.12439e+00, 7.59949e+01, 8.42238e+01, -6.69862e+02, -8.23945e+02},
             {-2.33878e-01, -8.59879e-01, -2.78125e+01, -3.57783e+01, 2.53381e+02, 3.30173e+02}}},
           {{{1.00066e+00, -7.01484e-04, -1.22684e-02, 6.03454e-03, 8.40298e-02, 6.05018e-02},
             {-4.95899e-01, 1.33690e-01, 1.67195e+01, 6.37574e+00, -1.16723e+02, -1.20936e+02},
             {2.92499e+00, 7.13921e+00, -6.78861e+01, -5.91051e+01, 5.50562e+02, 6.40337e+02},
             {-1.17889e+00, -5.22361e+00, 7.83760e+01, 9.38748e+01, -7.01185e+02, -8.81728e+02},
             {-3.25146e-01, -9.95032e-01, -2.85858e+01, -3.91713e+01, 2.63738e+02, 3.50385e+02}}},
           {{{1.00070e+00, -6.08729e-04, -1.35873e-02, 4.50061e-03, 9.48991e-02, 7.53277e-02},
             {-5.04647e-01, -1.86445e-02, 1.75175e+01, 8.32729e+00, -1.24477e+02, -1.33405e+02},
             {2.83871e+00, 7.59269e+00, -7.01766e+01, -6.75463e+01, 5.78809e+02, 6.90388e+02},
             {-9.58143e-01, -5.27523e+00, 8.05020e+01, 1.03751e+02, -7.31155e+02, -9.39013e+02},
             {-4.24964e-01, -1.15578e+00, -2.92679e+01, -4.26205e+01, 2.73537e+02, 3.70298e+02}}},
           {{{1.00073e+00, -5.06594e-04, -1.49195e-02, 2.82752e-03, 1.05984e-01, 9.07179e-02},
             {-5.09498e-01, -1.74114e-01, 1.82691e+01, 1.03640e+01, -1.32042e+02, -1.45912e+02},
             {2.73154e+00, 8.02658e+00, -7.22315e+01, -7.62278e+01, 6.05902e+02, 7.40063e+02},
             {-7.11399e-01, -5.27209e+00, 8.23375e+01, 1.13815e+02, -7.59479e+02, -9.95409e+02},
             {-5.33773e-01, -1.34491e+00, -2.98474e+01, -4.61115e+01, 2.82675e+02, 3.89772e+02}}},
           {{{1.00076e+00, -3.94938e-04, -1.62461e-02, 1.01616e-03, 1.17156e-01, 1.06529e-01},
             {-5.10179e-01, -3.31690e-01, 1.89693e+01, 1.24811e+01, -1.39382e+02, -1.58412e+02},
             {2.60247e+00, 8.43503e+00, -7.40330e+01, -8.51228e+01, 6.31692e+02, 7.89166e+02},
             {-4.37673e-01, -5.20620e+00, 8.38642e+01, 1.24030e+02, -7.85984e+02, -1.05067e+03},
             {-6.51850e-01, -1.56551e+00, -3.03186e+01, -4.96307e+01, 2.91091e+02, 4.08719e+02}}},
           {{{1.00079e+00, -2.74014e-04, -1.75614e-02, -9.33046e-04, 1.28369e-01, 1.22714e-01},
             {-5.06317e-01, -4.90282e-01, 1.96102e+01, 1.46725e+01, -1.46440e+02, -1.70838e+02},
             {2.45014e+00, 8.81187e+00, -7.55530e+01, -9.42005e+01, 6.55963e+02, 8.37419e+02},
             {-1.35635e-01, -5.06921e+00, 8.50532e+01, 1.34356e+02, -8.10425e+02, -1.10448e+03},
             {-7.79579e-01, -1.82077e+00, -3.06725e+01, -5.31628e+01, 2.98700e+02, 4.27020e+02}}},
           {{{1.00082e+00, -1.43895e-04, -1.88590e-02, -3.02404e-03, 1.39581e-01, 1.39236e-01},
             {-4.97537e-01, -6.48585e-01, 2.01838e+01, 1.69296e+01, -1.53157e+02, -1.83108e+02},
             {2.27321e+00, 9.15011e+00, -7.67630e+01, -1.03420e+02, 6.78484e+02, 8.84496e+02},
             {1.95990e-01, -4.85189e+00, 8.58760e+01, 1.44742e+02, -8.32543e+02, -1.15645e+03},
             {-9.17319e-01, -2.11420e+00, -3.09003e+01, -5.66889e+01, 3.05413e+02, 4.44542e+02}}},
           {{{1.00085e+00, -5.30411e-06, -2.01239e-02, -5.24780e-03, 1.50683e-01, 1.55963e-01},
             {-4.83526e-01, -8.05262e-01, 2.06846e+01, 1.92461e+01, -1.59496e+02, -1.95177e+02},
             {2.07059e+00, 9.44256e+00, -7.76442e+01, -1.12749e+02, 6.99103e+02, 9.30193e+02},
             {5.58224e-01, -4.54471e+00, 8.63141e+01, 1.55146e+02, -8.52163e+02, -1.20633e+03},
             {-1.06534e+00, -2.44938e+00, -3.09967e+01, -6.01932e+01, 3.11168e+02, 4.61192e+02}}},
           {{{1.00087e+00, 1.42049e-04, -2.13455e-02, -7.61031e-03, 1.61591e-01, 1.72814e-01},
             {-4.63920e-01, -9.58725e-01, 2.11056e+01, 2.16134e+01, -1.65402e+02, -2.06974e+02},
             {1.84100e+00, 9.68106e+00, -7.81727e+01, -1.22145e+02, 7.17614e+02, 9.74218e+02},
             {9.52222e-01, -4.13715e+00, 8.63446e+01, 1.65515e+02, -8.69054e+02, -1.25378e+03},
             {-1.22395e+00, -2.83024e+00, -3.09548e+01, -6.36564e+01, 3.15887e+02, 4.76849e+02}}},
           {{{1.00089e+00, 2.96049e-04, -2.25199e-02, -1.00947e-02, 1.72304e-01, 1.89770e-01},
             {-4.38356e-01, -1.10758e+00, 2.14388e+01, 2.40250e+01, -1.70813e+02, -2.18421e+02},
             {1.58322e+00, 9.85807e+00, -7.83221e+01, -1.31575e+02, 7.33783e+02, 1.01626e+03},
             {1.37908e+00, -3.61927e+00, 8.59420e+01, 1.75804e+02, -8.82957e+02, -1.29844e+03},
             {-1.39342e+00, -3.26049e+00, -3.07672e+01, -6.70618e+01, 3.19480e+02, 4.91383e+02}}},
           {{{1.00090e+00, 4.56346e-04, -2.36386e-02, -1.27003e-02, 1.82771e-01, 2.06789e-01},
             {-4.06465e-01, -1.24997e+00, 2.16774e+01, 2.64711e+01, -1.75677e+02, -2.29447e+02},
             {1.29603e+00, 9.96437e+00, -7.80698e+01, -1.40992e+02, 7.47404e+02, 1.05603e+03},
             {1.83985e+00, -2.97940e+00, 8.50852e+01, 1.85958e+02, -8.93641e+02, -1.33995e+03},
             {-1.57401e+00, -3.74444e+00, -3.04281e+01, -7.03888e+01, 3.21870e+02, 5.04671e+02}}},
           {{{1.00091e+00, 6.22359e-04, -2.46850e-02, -1.54269e-02, 1.92829e-01, 2.23669e-01},
             {-3.67940e-01, -1.38391e+00, 2.18175e+01, 2.89415e+01, -1.79969e+02, -2.40018e+02},
             {9.78442e-01, 9.99015e+00, -7.74041e+01, -1.50349e+02, 7.58380e+02, 1.09335e+03},
             {2.33533e+00, -2.20518e+00, 8.37655e+01, 1.95915e+02, -9.00998e+02, -1.37812e+03},
             {-1.76589e+00, -4.28658e+00, -2.99358e+01, -7.36156e+01, 3.23018e+02, 5.16638e+02}}},
           {{{1.00091e+00, 7.92875e-04, -2.56477e-02, -1.82659e-02, 2.02416e-01, 2.40347e-01},
             {-3.22419e-01, -1.50732e+00, 2.18520e+01, 3.14257e+01, -1.83632e+02, -2.50054e+02},
             {6.29314e-01, 9.92529e+00, -7.63035e+01, -1.59597e+02, 7.66494e+02, 1.12791e+03},
             {2.86640e+00, -1.28398e+00, 8.19642e+01, 2.05617e+02, -9.04788e+02, -1.41257e+03},
             {-1.96925e+00, -4.89150e+00, -2.92855e+01, -7.67206e+01, 3.22845e+02, 5.27151e+02}}},
           {{{1.00091e+00, 9.66485e-04, -2.65182e-02, -2.12089e-02, 2.11471e-01, 2.56746e-01},
             {-2.69552e-01, -1.61793e+00, 2.17751e+01, 3.39120e+01, -1.86616e+02, -2.59484e+02},
             {2.47591e-01, 9.75896e+00, -7.47505e+01, -1.68681e+02, 7.71562e+02, 1.15941e+03},
             {3.43387e+00, -2.02402e-01, 7.96675e+01, 2.14997e+02, -9.04813e+02, -1.44297e+03},
             {-2.18422e+00, -5.56401e+00, -2.84744e+01, -7.96806e+01, 3.21284e+02, 5.36093e+02}}},
           {{{1.00090e+00, 1.14163e-03, -2.72849e-02, -2.42457e-02, 2.19912e-01, 2.72759e-01},
             {-2.08991e-01, -1.71333e+00, 2.15806e+01, 3.63880e+01, -1.88871e+02, -2.68235e+02},
             {-1.67743e-01, 9.47970e+00, -7.27286e+01, -1.77547e+02, 7.73399e+02, 1.18758e+03},
             {4.03843e+00, 1.05357e+00, 7.68633e+01, 2.23990e+02, -9.00876e+02, -1.46897e+03},
             {-2.41092e+00, -6.30912e+00, -2.75000e+01, -8.24716e+01, 3.18270e+02, 5.43346e+02}}},
           {{{1.00088e+00, 1.31654e-03, -2.79362e-02, -2.73643e-02, 2.27648e-01, 2.88269e-01},
             {-1.40385e-01, -1.79089e+00, 2.12627e+01, 3.88400e+01, -1.90344e+02, -2.76230e+02},
             {-6.17665e-01, 9.07535e+00, -7.02219e+01, -1.86133e+02, 7.71818e+02, 1.21209e+03},
             {4.68070e+00, 2.49864e+00, 7.35409e+01, 2.32521e+02, -8.92777e+02, -1.49023e+03},
             {-2.64940e+00, -7.13202e+00, -2.63610e+01, -8.50681e+01, 3.13739e+02, 5.48788e+02}}},
           {{{1.00086e+00, 1.48934e-03, -2.84596e-02, -3.05545e-02, 2.34591e-01, 3.03168e-01},
             {-6.33861e-02, -1.84775e+00, 2.08152e+01, 4.12512e+01, -1.90978e+02, -2.83376e+02},
             {-1.10309e+00, 8.53286e+00, -6.72151e+01, -1.94369e+02, 7.66617e+02, 1.23259e+03},
             {5.36119e+00, 4.14839e+00, 6.96913e+01, 2.40511e+02, -8.80309e+02, -1.50636e+03},
             {-2.89970e+00, -8.03816e+00, -2.50563e+01, -8.74417e+01, 3.07622e+02, 5.52280e+02}}},
           {{{1.00083e+00, 1.65771e-03, -2.88470e-02, -3.38011e-02, 2.40678e-01, 3.17362e-01},
             {2.23629e-02, -1.88091e+00, 2.02316e+01, 4.36050e+01, -1.90711e+02, -2.89575e+02},
             {-1.62493e+00, 7.83872e+00, -6.36928e+01, -2.02185e+02, 7.57583e+02, 1.24872e+03},
             {6.08035e+00, 6.01881e+00, 6.53055e+01, 2.47876e+02, -8.63252e+02, -1.51693e+03},
             {-3.16179e+00, -9.03311e+00, -2.35856e+01, -8.95638e+01, 2.99851e+02, 5.53685e+02}}},
           {{{1.00079e+00, 1.81924e-03, -2.90832e-02, -3.70879e-02, 2.45791e-01, 3.30698e-01},
             {1.17187e-01, -1.88709e+00, 1.95065e+01, 4.58826e+01, -1.89486e+02, -2.94734e+02},
             {-2.18391e+00, 6.97844e+00, -5.96443e+01, -2.09503e+02, 7.44530e+02, 1.26012e+03},
             {6.83839e+00, 8.12682e+00, 6.03812e+01, 2.54528e+02, -8.41421e+02, -1.52156e+03},
             {-3.43558e+00, -1.01226e+01, -2.19506e+01, -9.14044e+01, 2.90368e+02, 5.52871e+02}}},
           {{{1.00075e+00, 1.97121e-03, -2.91599e-02, -4.03994e-02, 2.49868e-01, 3.43083e-01},
             {2.21414e-01, -1.86285e+00, 1.86345e+01, 4.80646e+01, -1.87245e+02, -2.98750e+02},
             {-2.78070e+00, 5.93700e+00, -5.50600e+01, -2.16243e+02, 7.27260e+02, 1.26641e+03},
             {7.63543e+00, 1.04897e+01, 5.49176e+01, 2.60377e+02, -8.14622e+02, -1.51985e+03},
             {-3.72092e+00, -1.13126e+01, -2.01538e+01, -9.29324e+01, 2.79114e+02, 5.49699e+02}}},
           {{{1.00069e+00, 2.11063e-03, -2.90649e-02, -4.37166e-02, 2.52811e-01, 3.54381e-01},
             {3.35343e-01, -1.80450e+00, 1.76110e+01, 5.01296e+01, -1.83930e+02, -3.01520e+02},
             {-3.41584e+00, 4.69858e+00, -4.99338e+01, -2.22321e+02, 7.05590e+02, 1.26721e+03},
             {8.47134e+00, 1.31256e+01, 4.89188e+01, 2.65326e+02, -7.82678e+02, -1.51137e+03},
             {-4.01757e+00, -1.26092e+01, -1.81997e+01, -9.41154e+01, 2.66036e+02, 5.44032e+02}}},
           {{{1.00062e+00, 2.23438e-03, -2.87912e-02, -4.70251e-02, 2.54570e-01, 3.64523e-01},
             {4.59241e-01, -1.70813e+00, 1.64324e+01, 5.20553e+01, -1.79492e+02, -3.02947e+02},
             {-4.08966e+00, 3.24663e+00, -4.42655e+01, -2.27646e+02, 6.79364e+02, 1.26217e+03},
             {9.34577e+00, 1.60530e+01, 4.23955e+01, 2.69276e+02, -7.45446e+02, -1.49575e+03},
             {-4.32520e+00, -1.40184e+01, -1.60952e+01, -9.49201e+01, 2.51093e+02, 5.35741e+02}}},
           {{{1.00054e+00, 2.33856e-03, -2.83172e-02, -5.02884e-02, 2.54953e-01, 3.73216e-01},
             {5.93303e-01, -1.56969e+00, 1.50988e+01, 5.38223e+01, -1.73905e+02, -3.02969e+02},
             {-4.80218e+00, 1.56424e+00, -3.80667e+01, -2.32140e+02, 6.48516e+02, 1.25106e+03},
             {1.02579e+01, 1.92906e+01, 3.53725e+01, 2.72140e+02, -7.02882e+02, -1.47274e+03},
             {-4.64333e+00, -1.55463e+01, -1.38520e+01, -9.53176e+01, 2.34279e+02, 5.24749e+02}}},
           {{{1.00045e+00, 2.41947e-03, -2.76379e-02, -5.34904e-02, 2.53921e-01, 3.80403e-01},
             {7.37729e-01, -1.38486e+00, 1.36088e+01, 5.54064e+01, -1.67129e+02, -3.01495e+02},
             {-5.55335e+00, -3.66307e-01, -3.13473e+01, -2.35709e+02, 6.12931e+02, 1.23355e+03},
             {1.12069e+01, 2.28579e+01, 2.78734e+01, 2.73815e+02, -6.54893e+02, -1.44199e+03},
             {-4.97143e+00, -1.71992e+01, -1.14816e+01, -9.52743e+01, 2.15573e+02, 5.10941e+02}}},
           {{{1.00035e+00, 2.47268e-03, -2.67441e-02, -5.66022e-02, 2.51390e-01, 3.85940e-01},
             {8.92647e-01, -1.14912e+00, 1.19639e+01, 5.67848e+01, -1.59141e+02, -2.98457e+02},
             {-6.34279e+00, -2.56322e+00, -2.41272e+01, -2.38265e+02, 5.72562e+02, 1.20938e+03},
             {1.21914e+01, 2.67741e+01, 1.99334e+01, 2.74204e+02, -6.01463e+02, -1.40324e+03},
             {-5.30880e+00, -1.89832e+01, -8.99983e+00, -9.47588e+01, 1.94979e+02, 4.94237e+02}}},
           {{{1.00024e+00, 2.49375e-03, -2.56255e-02, -5.95976e-02, 2.47266e-01, 3.89678e-01},
             {1.05812e+00, -8.57709e-01, 1.01681e+01, 5.79340e+01, -1.49935e+02, -2.93803e+02},
             {-7.16980e+00, -5.04529e+00, -1.64365e+01, -2.39716e+02, 5.27420e+02, 1.17838e+03},
             {1.32096e+01, 3.10591e+01, 1.15998e+01, 2.73212e+02, -5.42641e+02, -1.35631e+03},
             {-5.65461e+00, -2.09042e+01, -6.42647e+00, -9.37406e+01, 1.72527e+02, 4.74581e+02}}},
           {{{1.00011e+00, 2.47775e-03, -2.42766e-02, -6.24493e-02, 2.41487e-01, 3.91511e-01},
             {1.23421e+00, -5.05710e-01, 8.22428e+00, 5.88279e+01, -1.39487e+02, -2.87452e+02},
             {-8.03353e+00, -7.83160e+00, -8.30429e+00, -2.39965e+02, 4.77471e+02, 1.14028e+03},
             {1.42597e+01, 3.57324e+01, 2.91990e+00, 2.70736e+02, -4.78440e+02, -1.30094e+03},
             {-6.00791e+00, -2.29682e+01, -3.78184e+00, -9.21874e+01, 1.48234e+02, 4.51893e+02}}},
           {{{9.99975e-01, 2.41925e-03, -2.26882e-02, -6.51219e-02, 2.33958e-01, 3.91263e-01},
             {1.42085e+00, -8.80256e-02, 6.13985e+00, 5.94426e+01, -1.27806e+02, -2.79367e+02},
             {-8.93265e+00, -1.09415e+01, 2.24068e-01, -2.38922e+02, 4.22794e+02, 1.09495e+03},
             {1.53388e+01, 4.08131e+01, -6.04035e+00, 2.66685e+02, -4.08996e+02, -1.23702e+03},
             {-6.36755e+00, -2.51806e+01, -1.09260e+00, -9.00713e+01, 1.22161e+02, 4.26151e+02}}},
           {{{9.99822e-01, 2.31254e-03, -2.08550e-02, -6.75815e-02, 2.24607e-01, 3.88796e-01},
             {1.61799e+00, 4.00698e-01, 3.92095e+00, 5.97493e+01, -1.14882e+02, -2.69475e+02},
             {-9.86566e+00, -1.43946e+01, 9.10466e+00, -2.36487e+02, 3.63422e+02, 1.04216e+03},
             {1.64443e+01, 4.63204e+01, -1.52151e+01, 2.60954e+02, -3.34406e+02, -1.16435e+03},
             {-6.73227e+00, -2.75466e+01, 1.61430e+00, -8.73612e+01, 9.43594e+01, 3.97300e+02}}},
           {{{9.99655e-01, 2.15142e-03, -1.87711e-02, -6.97883e-02, 2.13350e-01, 3.83936e-01},
             {1.82545e+00, 9.65942e-01, 1.57686e+00, 5.97207e+01, -1.00725e+02, -2.57723e+02},
             {-1.08305e+01, -1.82107e+01, 1.82808e+01, -2.32560e+02, 2.99460e+02, 9.81786e+02},
             {1.75726e+01, 5.22723e+01, -2.45237e+01, 2.53451e+02, -2.54850e+02, -1.08283e+03},
             {-7.10059e+00, -3.00710e+01, 4.30673e+00, -8.40296e+01, 6.49087e+01, 3.65326e+02}}},
           {{{9.99473e-01, 1.92924e-03, -1.64318e-02, -7.17008e-02, 2.00109e-01, 3.76517e-01},
             {2.04302e+00, 1.61334e+00, -8.82149e-01, 5.93264e+01, -8.53384e+01, -2.44043e+02},
             {-1.18250e+01, -2.24092e+01, 2.76900e+01, -2.27039e+02, 2.31010e+02, 9.13629e+02},
             {1.87198e+01, 5.86859e+01, -3.38779e+01, 2.44074e+02, -1.70518e+02, -9.92335e+02},
             {-7.47086e+00, -3.27576e+01, 6.94978e+00, -8.00487e+01, 3.38953e+01, 3.30205e+02}}},
           {{{9.99276e-01, 1.63895e-03, -1.38361e-02, -7.32738e-02, 1.84822e-01, 3.66376e-01},
             {2.27037e+00, 2.34860e+00, -3.44417e+00, 5.85364e+01, -6.87344e+01, -2.28373e+02},
             {-1.28460e+01, -2.70092e+01, 3.72616e+01, -2.19825e+02, 1.58211e+02, 8.37547e+02},
             {1.98811e+01, 6.55768e+01, -4.31793e+01, 2.32735e+02, -8.16486e+01, -8.92777e+02},
             {-7.84119e+00, -3.56097e+01, 9.50496e+00, -7.53943e+01, 1.42399e+00, 2.91936e+02}}},
           {{{9.99064e-01, 1.27324e-03, -1.09916e-02, -7.44696e-02, 1.67497e-01, 3.53455e-01},
             {2.50713e+00, 3.17754e+00, -6.09661e+00, 5.73172e+01, -5.09157e+01, -2.10626e+02},
             {-1.38905e+01, -3.20289e+01, 4.69203e+01, -2.10810e+02, 8.11857e+01, 7.53321e+02},
             {2.10515e+01, 7.29586e+01, -5.23231e+01, 2.19337e+02, 1.15276e+01, -7.84022e+02},
             {-8.20951e+00, -3.86295e+01, 1.19315e+01, -7.00414e+01, -3.23996e+01, 2.50503e+02}}},
           {{{9.98836e-01, 8.23950e-04, -7.89491e-03, -7.52274e-02, 1.48028e-01, 3.37491e-01},
             {2.75270e+00, 4.10593e+00, -8.82051e+00, 5.56392e+01, -3.19293e+01, -1.90772e+02},
             {-1.49543e+01, -3.74855e+01, 5.65661e+01, -1.99903e+02, 2.22903e-01, 6.60953e+02},
             {2.22247e+01, 8.08420e+01, -6.11776e+01, 2.03805e+02, 1.08591e+02, -6.66179e+02},
             {-8.57337e+00, -4.18175e+01, 1.41794e+01, -6.39741e+01, -6.74045e+01, 2.05974e+02}}},
           {{{9.98592e-01, 2.82966e-04, -4.55740e-03, -7.55030e-02, 1.26429e-01, 3.18414e-01},
             {3.00639e+00, 5.13948e+00, -1.15960e+01, 5.34702e+01, -1.18151e+01, -1.68761e+02},
             {-1.60324e+01, -4.33944e+01, 6.60933e+01, -1.87010e+02, -8.43973e+01, 5.60386e+02},
             {2.33936e+01, 8.92345e+01, -6.96034e+01, 1.86068e+02, 2.09119e+02, -5.39313e+02},
             {-8.93007e+00, -4.51728e+01, 1.61962e+01, -5.71780e+01, -1.03415e+02, 1.58410e+02}}}};
    // clang-format on
    static_assert(
        std::size(electron_mott_coeffs) == MottElementData::num_elements,
        "wrong number of Mott coefficient elements");

    int index = z.unchecked_get() - 1;
    CELER_VALIDATE(index >= 0 && index < int{MottElementData::num_elements},
                   << "atomic number " << z.get()
                   << " is out of range for Coulomb scattering model Mott "
                      "coefficients (must be less than "
                   << MottElementData::num_elements << ")");

    return electron_mott_coeffs[index];
}

//---------------------------------------------------------------------------//
/*!
 * Get interpolated Mott coefficients for positrons for an element.
 *
 * These coefficients are for positrons interacting with nuclei with 1 <= Z <=
 * 118 and taken from table 4 in M.J. Boschini et al, Radiat. Phys. Chem. 90
 * (2013), 39-66.
 */
auto WentzelOKVIParams::get_positron_mott_coeffs(AtomicNumber z) -> CoeffMat
{
    CELER_EXPECT(z);

    // clang-format off
    static CoeffMat const positron_mott_coeffs[]
        = {{{{1.00000e+00, -2.63961e-08, -1.05983e-07, 6.98859e-08, 9.77498e-07, 7.22885e-07},
             {-1.15697e-02, -1.62210e-02, 2.56754e-05, 5.94651e-05, -1.25966e-04, -4.19100e-04},
             {-2.49475e-01, -7.06540e-01, -5.00006e-01, -1.93135e-04, 3.28434e-04, 1.24240e-03},
             {-2.01733e-04, -1.52457e-04, -3.45949e-05, 1.75794e-04, -1.53568e-04, -9.52819e-04},
             {5.43511e-05, 1.80300e-05, -9.61947e-06, -5.72438e-05, 3.75348e-05, 2.95392e-04}}},
           {{{1.00000e+00, -3.71563e-08, 2.54023e-07, 2.73203e-07, -1.93946e-06, -3.30422e-06},
             {-2.29788e-02, -3.24498e-02, 2.27671e-04, 1.90496e-04, -1.58887e-03, -3.00192e-03},
             {-2.40858e-01, -6.94758e-01, -5.00597e-01, -7.88898e-04, 6.37319e-03, 1.15722e-02},
             {-7.54571e-04, -5.41978e-04, 4.55243e-04, 6.88365e-04, -5.90329e-03, -1.04595e-02},
             {2.02061e-04, 5.13597e-05, -2.37821e-04, -2.22551e-04, 1.92190e-03, 3.39201e-03}}},
           {{{1.00000e+00, -6.61242e-08, 3.14624e-07, 3.78658e-07, -2.45678e-06, -4.73353e-06},
             {-3.42080e-02, -4.87208e-02, 1.75679e-04, 1.55745e-04, -6.27212e-04, -2.79906e-03},
             {-2.32061e-01, -6.82670e-01, -5.00715e-01, -1.62390e-03, 9.13871e-03, 2.08325e-02},
             {-1.60732e-03, -1.28113e-03, 4.24244e-04, 1.48822e-03, -8.54039e-03, -1.89911e-02},
             {4.28858e-04, 1.40228e-04, -3.39983e-04, -4.85420e-04, 2.79129e-03, 6.17637e-03}}},
           {{{1.00000e+00, -8.54294e-08, 3.60088e-07, 3.06775e-07, -1.60440e-06, -3.89561e-06},
             {-4.52455e-02, -6.50181e-02, -1.35878e-04, -4.75424e-04, 4.01616e-03, 3.49910e-03},
             {-2.23117e-01, -6.70328e-01, -5.00843e-01, -2.03032e-03, 8.97348e-03, 2.62014e-02},
             {-2.73626e-03, -2.31306e-03, 5.37447e-04, 2.12418e-03, -9.99203e-03, -2.62119e-02},
             {7.29502e-04, 2.68752e-04, -5.42321e-04, -7.05686e-04, 3.34604e-03, 8.64321e-03}}},
           {{{1.00000e+00, -1.34641e-07, -2.19322e-07, -8.96986e-08, 3.06738e-06, 9.15255e-07},
             {-5.60680e-02, -8.13384e-02, -1.25850e-03, -2.47621e-03, 1.57853e-02, 2.15886e-02},
             {-2.14102e-01, -6.57738e-01, -4.99411e-01, -1.62861e-05, -3.08306e-03, 1.42231e-02},
             {-4.07182e-03, -3.63517e-03, -7.11399e-04, 7.88606e-04, -2.22086e-03, -2.05660e-02},
             {1.08339e-03, 4.39517e-04, -3.48395e-04, -2.96024e-04, 9.43883e-04, 7.02341e-03}}},
           {{{9.99999e-01, -2.01536e-07, -2.43353e-07, -6.26168e-07, 4.50746e-06, 3.19698e-06},
             {-6.66775e-02, -9.77011e-02, -2.92288e-03, -5.72193e-03, 3.38850e-02, 5.16453e-02},
             {-2.05015e-01, -6.44838e-01, -4.97373e-01, 3.79528e-03, -2.40521e-02, -1.38486e-02},
             {-5.62473e-03, -5.31084e-03, -2.23196e-03, -1.69424e-03, 1.07593e-02, -5.15036e-03},
             {1.49631e-03, 6.75768e-04, -1.21222e-04, 4.73516e-04, -3.07173e-03, 2.36262e-03}}},
           {{{9.99999e-01, -3.29166e-07, -5.33244e-07, -1.11684e-06, 8.57093e-06, 8.69030e-06},
             {-7.70567e-02, -1.14122e-01, -5.26696e-03, -1.01861e-02, 5.89303e-02, 9.46008e-02},
             {-1.95925e-01, -6.31578e-01, -4.93974e-01, 9.47957e-03, -5.87513e-02, -6.39687e-02},
             {-7.32710e-03, -7.39261e-03, -4.84392e-03, -5.32266e-03, 3.41970e-02, 2.60213e-02},
             {1.94765e-03, 9.97847e-04, 4.19732e-04, 1.60667e-03, -1.05245e-02, -7.42775e-03}}},
           {{{9.99999e-01, -4.66583e-07, -7.22192e-07, -1.98661e-06, 1.23013e-05, 1.47981e-05},
             {-8.72102e-02, -1.30583e-01, -7.85695e-03, -1.58723e-02, 8.72272e-02, 1.46473e-01},
             {-1.86831e-01, -6.18034e-01, -4.90257e-01, 1.75633e-02, -9.87667e-02, -1.28102e-01},
             {-9.18418e-03, -9.79790e-03, -7.53141e-03, -1.06985e-02, 5.99496e-02, 6.51713e-02},
             {2.44097e-03, 1.38055e-03, 9.44623e-04, 3.31625e-03, -1.87816e-02, -1.98735e-02}}},
           {{{9.99999e-01, -6.33597e-07, -9.21708e-07, -3.04874e-06, 1.68020e-05, 2.25753e-05},
             {-9.71347e-02, -1.47092e-01, -1.04691e-02, -2.24563e-02, 1.16763e-01, 2.04465e-01},
             {-1.77760e-01, -6.04187e-01, -4.86471e-01, 2.76011e-02, -1.42274e-01, -2.03878e-01},
             {-1.11684e-02, -1.25461e-02, -1.01695e-02, -1.74501e-02, 8.73345e-02, 1.11298e-01},
             {2.96875e-03, 1.83334e-03, 1.42155e-03, 5.48622e-03, -2.77099e-02, -3.47727e-02}}},
           {{{9.99999e-01, -8.31912e-07, -8.95264e-07, -4.13450e-06, 1.94766e-05, 2.85275e-05},
             {-1.06829e-01, -1.63663e-01, -1.28280e-02, -2.93936e-02, 1.44582e-01, 2.63884e-01},
             {-1.68734e-01, -5.90005e-01, -4.83100e-01, 3.85352e-02, -1.83863e-01, -2.82527e-01},
             {-1.32601e-02, -1.56645e-02, -1.23736e-02, -2.46471e-02, 1.11881e-01, 1.56900e-01},
             {3.52612e-03, 2.36734e-03, 1.72495e-03, 7.82216e-03, -3.58623e-02, -4.97178e-02}}},
           {{{9.99998e-01, -1.05528e-06, -6.99187e-07, -5.48613e-06, 2.11682e-05, 3.43426e-05},
             {-1.16292e-01, -1.80299e-01, -1.46747e-02, -3.64327e-02, 1.68178e-01, 3.21368e-01},
             {-1.59771e-01, -5.75495e-01, -4.80630e-01, 5.03005e-02, -2.19171e-01, -3.59389e-01},
             {-1.54415e-02, -1.91445e-02, -1.37650e-02, -3.24205e-02, 1.30291e-01, 1.98954e-01},
             {4.10873e-03, 2.98228e-03, 1.73164e-03, 1.03773e-02, -4.21798e-02, -6.37748e-02}}},
           {{{9.99998e-01, -1.34039e-06, -1.96569e-07, -6.23873e-06, 2.01170e-05, 3.57585e-05},
             {-1.25521e-01, -1.97018e-01, -1.59161e-02, -4.29269e-02, 1.86135e-01, 3.73582e-01},
             {-1.50896e-01, -5.60600e-01, -4.79096e-01, 6.14058e-02, -2.46683e-01, -4.28835e-01},
             {-1.76863e-02, -2.30408e-02, -1.44138e-02, -3.93907e-02, 1.42116e-01, 2.33294e-01},
             {4.70933e-03, 3.69893e-03, 1.46355e-03, 1.26977e-02, -4.65229e-02, -7.55944e-02}}},
           {{{9.99998e-01, -1.66526e-06, 2.86994e-07, -7.06381e-06, 1.99241e-05, 3.90205e-05},
             {-1.34513e-01, -2.13815e-01, -1.64697e-02, -4.87511e-02, 1.97477e-01, 4.19240e-01},
             {-1.42127e-01, -5.45349e-01, -4.78603e-01, 7.21116e-02, -2.65433e-01, -4.91123e-01},
             {-1.99772e-02, -2.73217e-02, -1.42989e-02, -4.60473e-02, 1.47300e-01, 2.61800e-01},
             {5.32352e-03, 4.50917e-03, 9.11673e-04, 1.49536e-02, -4.88699e-02, -8.58220e-02}}},
           {{{9.99998e-01, -2.02835e-06, 1.05777e-06, -7.72567e-06, 1.69130e-05, 3.88243e-05},
             {-1.43268e-01, -2.30700e-01, -1.62160e-02, -5.33915e-02, 2.00710e-01, 4.55181e-01},
             {-1.33480e-01, -5.29716e-01, -4.79482e-01, 8.12550e-02, -2.71756e-01, -5.38863e-01},
             {-2.22979e-02, -3.20101e-02, -1.31294e-02, -5.13703e-02, 1.42703e-01, 2.78031e-01},
             {5.94727e-03, 5.42309e-03, -2.46612e-05, 1.68127e-02, -4.81535e-02, -9.23173e-02}}},
           {{{9.99997e-01, -2.45028e-06, 1.90684e-06, -7.87635e-06, 1.26832e-05, 3.60947e-05},
             {-1.51781e-01, -2.47668e-01, -1.52167e-02, -5.66269e-02, 1.95686e-01, 4.80404e-01},
             {-1.24981e-01, -5.13712e-01, -4.81543e-01, 8.84855e-02, -2.65878e-01, -5.70906e-01},
             {-2.46237e-02, -3.70970e-02, -1.11407e-02, -5.51224e-02, 1.29134e-01, 2.81832e-01},
             {6.57359e-03, 6.44093e-03, -1.27284e-03, 1.81886e-02, -4.46216e-02, -9.50057e-02}}},
           {{{9.99997e-01, -2.90852e-06, 2.74924e-06, -8.07366e-06, 8.59057e-06, 3.41872e-05},
             {-1.60050e-01, -2.64711e-01, -1.35619e-02, -5.84638e-02, 1.82876e-01, 4.95464e-01},
             {-1.16647e-01, -4.97365e-01, -4.84589e-01, 9.42103e-02, -2.49196e-01, -5.90241e-01},
             {-2.69371e-02, -4.25516e-02, -8.53860e-03, -5.78939e-02, 1.08283e-01, 2.77047e-01},
             {7.19786e-03, 7.55474e-03, -2.77282e-03, 1.92789e-02, -3.87745e-02, -9.51075e-02}}},
           {{{9.99997e-01, -3.43342e-06, 3.66943e-06, -7.57715e-06, 3.33705e-06, 2.94070e-05},
             {-1.68070e-01, -2.81840e-01, -1.13145e-02, -5.83113e-02, 1.62350e-01, 4.98702e-01},
             {-1.08498e-01, -4.80624e-01, -4.88568e-01, 9.66929e-02, -2.21203e-01, -5.91460e-01},
             {-2.92193e-02, -4.84272e-02, -5.36121e-03, -5.79946e-02, 7.96746e-02, 2.58455e-01},
             {7.81488e-03, 8.78493e-03, -4.51941e-03, 1.95226e-02, -3.03955e-02, -9.08335e-02}}},
           {{{9.99997e-01, -4.19813e-06, 5.18039e-06, -1.81063e-06, 2.61123e-06, 2.14096e-05},
             {-1.75845e-01, -2.99065e-01, -8.45957e-03, -5.55301e-02, 1.35545e-01, 4.90797e-01},
             {-1.00533e-01, -4.63446e-01, -4.93718e-01, 9.38312e-02, -1.85980e-01, -5.76684e-01},
             {-3.14706e-02, -5.47751e-02, -1.34067e-03, -5.32597e-02, 4.77261e-02, 2.28681e-01},
             {8.42573e-03, 1.01520e-02, -6.60649e-03, 1.81850e-02, -2.09323e-02, -8.30172e-02}}},
           {{{9.99996e-01, -4.78732e-06, 6.44595e-06, -8.20875e-07, -5.43702e-06, 1.27126e-05},
             {-1.83367e-01, -3.16296e-01, -5.28081e-03, -5.18988e-02, 1.01190e-01, 4.72879e-01},
             {-9.27851e-02, -4.46119e-01, -4.99342e-01, 9.18166e-02, -1.37216e-01, -5.48138e-01},
             {-3.36579e-02, -6.12952e-02, 2.85727e-03, -5.02898e-02, 5.32032e-03, 1.89771e-01},
             {9.02033e-03, 1.15589e-02, -8.82479e-03, 1.74522e-02, -7.90938e-03, -7.21991e-02}}},
           {{{9.99996e-01, -4.99651e-06, 9.85067e-06, -3.06013e-06, -3.75834e-05, -1.99197e-05},
             {-1.90639e-01, -3.33499e-01, -1.46206e-03, -4.67782e-02, 5.67814e-02, 4.40289e-01},
             {-8.52545e-02, -4.28745e-01, -5.06772e-01, 8.85811e-02, -6.46795e-02, -4.88733e-01},
             {-3.57823e-02, -6.78790e-02, 8.68379e-03, -4.70307e-02, -5.85800e-02, 1.23822e-01},
             {9.59982e-03, 1.29714e-02, -1.16619e-02, 1.66507e-02, 1.23891e-02, -5.24059e-02}}},
           {{{9.99996e-01, -5.64027e-06, 1.12089e-05, -1.39130e-06, -4.81007e-05, -3.31794e-05},
             {-1.97654e-01, -3.50787e-01, 2.15113e-03, -3.92435e-02, 1.05595e-02, 4.01060e-01},
             {-7.79716e-02, -4.10908e-01, -5.13292e-01, 8.07516e-02, 4.25834e-03, -4.25183e-01},
             {-3.78112e-02, -7.49813e-02, 1.32882e-02, -3.98722e-02, -1.14896e-01, 5.93661e-02},
             {1.01540e-02, 1.45455e-02, -1.41660e-02, 1.45535e-02, 3.02220e-02, -3.31854e-02}}},
           {{{9.99996e-01, -6.18069e-06, 1.16691e-05, -1.56787e-06, -5.44130e-05, -3.81630e-05},
             {-2.04406e-01, -3.68067e-01, 5.48049e-03, -3.01211e-02, -3.74218e-02, 3.56766e-01},
             {-7.09656e-02, -3.92903e-01, -5.18975e-01, 7.10868e-02, 7.22547e-02, -3.59705e-01},
             {-3.97147e-02, -8.22933e-02, 1.68611e-02, -3.17533e-02, -1.67420e-01, -2.32943e-03},
             {1.06736e-02, 1.61803e-02, -1.64165e-02, 1.21314e-02, 4.70242e-02, -1.47295e-02}}},
           {{{9.99996e-01, -6.90601e-06, 1.09846e-05, 2.41652e-07, -5.22234e-05, -3.81638e-05},
             {-2.10895e-01, -3.85368e-01, 8.29726e-03, -1.88946e-02, -8.53325e-02, 3.08009e-01},
             {-6.42450e-02, -3.74600e-01, -5.23304e-01, 5.76144e-02, 1.34881e-01, -2.92287e-01},
             {-4.14842e-02, -8.99614e-02, 1.89268e-02, -2.05684e-02, -2.11877e-01, -6.16277e-02},
             {1.11563e-02, 1.79278e-02, -1.82607e-02, 8.67045e-03, 6.14040e-02, 3.12672e-03}}},
           {{{9.99996e-01, -7.61454e-06, 1.17901e-05, 2.55920e-06, -5.99752e-05, -4.97952e-05},
             {-2.17141e-01, -4.02610e-01, 1.11161e-02, -6.14320e-03, -1.37177e-01, 2.51307e-01},
             {-5.77524e-02, -3.56238e-01, -5.28278e-01, 4.20781e-02, 2.07581e-01, -2.08928e-01},
             {-4.31798e-02, -9.77438e-02, 2.16529e-02, -8.10625e-03, -2.64892e-01, -1.33648e-01},
             {1.16226e-02, 1.97110e-02, -2.04207e-02, 4.74691e-03, 7.88928e-02, 2.54391e-02}}},
           {{{9.99996e-01, -8.30050e-06, 1.28467e-05, 4.76135e-06, -7.10175e-05, -6.53661e-05},
             {-2.23135e-01, -4.19786e-01, 1.36188e-02, 8.10272e-03, -1.90880e-01, 1.88778e-01},
             {-5.15229e-02, -3.37823e-01, -5.33034e-01, 2.44228e-02, 2.85150e-01, -1.14661e-01},
             {-4.47635e-02, -1.05641e-01, 2.41923e-02, 5.67830e-03, -3.21525e-01, -2.13608e-01},
             {1.20602e-02, 2.15324e-02, -2.26173e-02, 3.45116e-04, 9.78965e-02, 5.06733e-02}}},
           {{{9.99995e-01, -9.07299e-06, 1.22647e-05, 7.72193e-06, -6.89733e-05, -6.83607e-05},
             {-2.28867e-01, -4.36894e-01, 1.50926e-02, 2.37363e-02, -2.40614e-01, 1.27076e-01},
             {-4.56095e-02, -3.19317e-01, -5.35345e-01, 4.68063e-03, 3.49262e-01, -3.00149e-02},
             {-4.61790e-02, -1.13706e-01, 2.42606e-02, 2.08436e-02, -3.62865e-01, -2.80428e-01},
             {1.24508e-02, 2.34125e-02, -2.40936e-02, -4.57624e-03, 1.12143e-01, 7.18756e-02}}},
           {{{9.99995e-01, -9.78027e-06, 1.22471e-05, 1.00363e-05, -7.20555e-05, -7.50524e-05},
             {-2.34348e-01, -4.53885e-01, 1.58318e-02, 4.03231e-02, -2.88712e-01, 6.47722e-02},
             {-3.99817e-02, -3.00861e-01, -5.36419e-01, -1.58753e-02, 4.09540e-01, 5.18828e-02},
             {-4.74575e-02, -1.21798e-01, 2.31934e-02, 3.60665e-02, -3.99528e-01, -3.41811e-01},
             {1.28049e-02, 2.53069e-02, -2.52955e-02, -9.58190e-03, 1.25184e-01, 9.16313e-02}}},
           {{{9.99995e-01, -1.04899e-05, 1.23373e-05, 1.28647e-05, -7.66918e-05, -8.52106e-05},
             {-2.39585e-01, -4.70739e-01, 1.59910e-02, 5.80188e-02, -3.36548e-01, -7.46874e-04},
             {-3.46232e-02, -2.82488e-01, -5.36952e-01, -3.81406e-02, 4.72111e-01, 1.42141e-01},
             {-4.86143e-02, -1.29896e-01, 2.17785e-02, 5.23996e-02, -4.38375e-01, -4.10085e-01},
             {1.31278e-02, 2.72121e-02, -2.64838e-02, -1.50435e-02, 1.39303e-01, 1.14084e-01}}},
           {{{9.99995e-01, -1.11960e-05, 1.22987e-05, 1.58216e-05, -8.07622e-05, -9.52147e-05},
             {-2.44577e-01, -4.87435e-01, 1.51539e-02, 7.64811e-02, -3.80560e-01, -6.45690e-02},
             {-2.95563e-02, -2.64238e-01, -5.35677e-01, -6.12348e-02, 5.26192e-01, 2.25957e-01},
             {-4.96243e-02, -1.37968e-01, 1.87361e-02, 6.89830e-02, -4.68533e-01, -4.70288e-01},
             {1.34111e-02, 2.91202e-02, -2.72355e-02, -2.06877e-02, 1.50921e-01, 1.34325e-01}}},
           {{{9.99995e-01, -1.18594e-05, 1.19562e-05, 1.84543e-05, -8.33692e-05, -1.03387e-04},
             {-2.49326e-01, -5.03959e-01, 1.34275e-02, 9.56498e-02, -4.21777e-01, -1.27967e-01},
             {-2.47795e-02, -2.46127e-01, -5.33031e-01, -8.51586e-02, 5.75930e-01, 3.08568e-01},
             {-5.04870e-02, -1.46003e-01, 1.45363e-02, 8.58268e-02, -4.94391e-01, -5.27872e-01},
             {1.36549e-02, 3.10289e-02, -2.76968e-02, -2.65148e-02, 1.61418e-01, 1.54058e-01}}},
           {{{9.99995e-01, -1.25029e-05, 1.03925e-05, 2.03923e-05, -7.56190e-05, -9.81458e-05},
             {-2.53834e-01, -5.20282e-01, 1.04736e-02, 1.15144e-01, -4.57140e-01, -1.86638e-01},
             {-2.03038e-02, -2.28223e-01, -5.27977e-01, -1.08978e-01, 6.11931e-01, 3.77036e-01},
             {-5.11885e-02, -1.53946e-01, 8.13661e-03, 1.02071e-01, -5.06445e-01, -5.69832e-01},
             {1.38546e-02, 3.29229e-02, -2.75224e-02, -3.22619e-02, 1.67650e-01, 1.69018e-01}}},
           {{{9.99995e-01, -1.32957e-05, 8.54079e-06, 2.43056e-05, -6.73005e-05, -9.66759e-05},
             {-2.58106e-01, -5.36422e-01, 6.44176e-03, 1.35448e-01, -4.87849e-01, -2.43033e-01},
             {-1.61263e-02, -2.10432e-01, -5.21143e-01, -1.34601e-01, 6.39567e-01, 4.41608e-01},
             {-5.17296e-02, -1.61902e-01, 2.50372e-04, 1.19767e-01, -5.10901e-01, -6.07646e-01},
             {1.40102e-02, 3.48389e-02, -2.69503e-02, -3.86117e-02, 1.71722e-01, 1.83077e-01}}},
           {{{9.99995e-01, -1.40602e-05, 6.76125e-06, 2.82630e-05, -5.97299e-05, -9.63731e-05},
             {-2.62151e-01, -5.52312e-01, 1.41196e-03, 1.55882e-01, -5.14490e-01, -2.96596e-01},
             {-1.22226e-02, -1.92953e-01, -5.12829e-01, -1.60006e-01, 6.61216e-01, 5.01314e-01},
             {-5.21333e-02, -1.69673e-01, -8.79982e-03, 1.36882e-01, -5.10330e-01, -6.40524e-01},
             {1.41294e-02, 3.67137e-02, -2.60802e-02, -4.49061e-02, 1.74437e-01, 1.95949e-01}}},
           {{{9.99996e-01, -1.46234e-05, 5.78798e-06, 3.09303e-05, -5.94048e-05, -1.02715e-04},
             {-2.65984e-01, -5.67919e-01, -4.39481e-03, 1.76363e-01, -5.38797e-01, -3.49511e-01},
             {-8.56306e-03, -1.75876e-01, -5.03813e-01, -1.85163e-01, 6.83169e-01, 5.64370e-01},
             {-5.24276e-02, -1.77175e-01, -1.81830e-02, 1.53442e-01, -5.11442e-01, -6.77238e-01},
             {1.42214e-02, 3.85217e-02, -2.51813e-02, -5.11578e-02, 1.77976e-01, 2.10488e-01}}},
           {{{9.99996e-01, -1.51841e-05, 1.14760e-06, 3.01648e-05, -2.57597e-05, -5.94205e-05},
             {-2.69584e-01, -5.83246e-01, -1.19601e-02, 1.96030e-01, -5.51786e-01, -3.88462e-01},
             {-5.22711e-03, -1.59155e-01, -4.90729e-01, -2.07367e-01, 6.74785e-01, 5.85675e-01},
             {-5.25260e-02, -1.84466e-01, -3.14465e-02, 1.66654e-01, -4.81979e-01, -6.70364e-01},
             {1.42571e-02, 4.02834e-02, -2.30679e-02, -5.64446e-02, 1.71572e-01, 2.10893e-01}}},
           {{{9.99996e-01, -1.56599e-05, -1.00540e-06, 3.20395e-05, -1.58142e-05, -5.30333e-05},
             {-2.72984e-01, -5.98270e-01, -2.01789e-02, 2.16021e-01, -5.63685e-01, -4.29326e-01},
             {-2.11870e-03, -1.42849e-01, -4.77423e-01, -2.30820e-01, 6.71934e-01, 6.21234e-01},
             {-5.25271e-02, -1.91491e-01, -4.44960e-02, 1.81047e-01, -4.60173e-01, -6.79635e-01},
             {1.42694e-02, 4.19825e-02, -2.11011e-02, -6.22845e-02, 1.67959e-01, 2.17083e-01}}},
           {{{9.99996e-01, -1.60740e-05, -2.21886e-06, 3.40021e-05, -1.44508e-05, -5.69472e-05},
             {-2.76186e-01, -6.12968e-01, -2.91285e-02, 2.35616e-01, -5.73179e-01, -4.68306e-01},
             {7.56429e-04, -1.27008e-01, -4.63600e-01, -2.53305e-01, 6.70130e-01, 6.58462e-01},
             {-5.24212e-02, -1.98209e-01, -5.76436e-02, 1.94359e-01, -4.41348e-01, -6.91960e-01},
             {1.42547e-02, 4.36073e-02, -1.91709e-02, -6.79360e-02, 1.65550e-01, 2.24687e-01}}},
           {{{9.99996e-01, -1.64436e-05, -4.31347e-06, 3.54126e-05, -5.47913e-06, -5.11367e-05},
             {-2.79184e-01, -6.27336e-01, -3.93501e-02, 2.54555e-01, -5.75193e-01, -4.98863e-01},
             {3.36753e-03, -1.11627e-01, -4.47403e-01, -2.74189e-01, 6.52275e-01, 6.75819e-01},
             {-5.21730e-02, -2.04632e-01, -7.28294e-02, 2.06008e-01, -4.07720e-01, -6.85082e-01},
             {1.42012e-02, 4.51633e-02, -1.66308e-02, -7.32148e-02, 1.58444e-01, 2.26349e-01}}},
           {{{9.99996e-01, -1.66505e-05, -7.18581e-06, 3.41224e-05, 1.15080e-05, -3.00315e-05},
             {-2.81990e-01, -6.41333e-01, -5.06942e-02, 2.72474e-01, -5.70812e-01, -5.21648e-01},
             {5.73305e-03, -9.68200e-02, -4.29240e-01, -2.92343e-01, 6.21436e-01, 6.74903e-01},
             {-5.17977e-02, -2.10648e-01, -8.96697e-02, 2.14804e-01, -3.62155e-01, -6.60187e-01},
             {1.41135e-02, 4.66142e-02, -1.35922e-02, -7.77187e-02, 1.47466e-01, 2.22290e-01}}},
           {{{9.99996e-01, -1.67956e-05, -8.97030e-06, 3.29841e-05, 1.95944e-05, -1.94280e-05},
             {-2.84617e-01, -6.54968e-01, -6.26662e-02, 2.89739e-01, -5.64239e-01, -5.42316e-01},
             {7.88727e-03, -8.25453e-02, -4.10758e-01, -3.09242e-01, 5.92020e-01, 6.75568e-01},
             {-5.13278e-02, -2.16307e-01, -1.06433e-01, 2.22378e-01, -3.19951e-01, -6.38486e-01},
             {1.40021e-02, 4.79782e-02, -1.06256e-02, -8.19977e-02, 1.37688e-01, 2.19570e-01}}},
           {{{9.99996e-01, -1.69109e-05, -1.08484e-05, 3.25312e-05, 2.50532e-05, -1.51081e-05},
             {-2.87068e-01, -6.68206e-01, -7.53768e-02, 3.05977e-01, -5.55654e-01, -5.60855e-01},
             {9.82986e-03, -6.88940e-02, -3.91544e-01, -3.23913e-01, 5.64597e-01, 6.78390e-01},
             {-5.07599e-02, -2.21522e-01, -1.23543e-01, 2.27827e-01, -2.81795e-01, -6.20957e-01},
             {1.38657e-02, 4.92280e-02, -7.58711e-03, -8.57699e-02, 1.29323e-01, 2.18540e-01}}},
           {{{9.99997e-01, -1.70602e-05, -1.31470e-05, 3.30741e-05, 3.69378e-05, -4.74005e-06},
             {-2.89350e-01, -6.81125e-01, -8.91796e-02, 3.22195e-01, -5.38594e-01, -5.70742e-01},
             {1.15641e-02, -5.55927e-02, -3.70290e-01, -3.39787e-01, 5.16721e-01, 6.60586e-01},
             {-5.00932e-02, -2.26583e-01, -1.42412e-01, 2.34704e-01, -2.23983e-01, -5.83370e-01},
             {1.37037e-02, 5.04599e-02, -3.99826e-03, -9.01942e-02, 1.14460e-01, 2.11065e-01}}},
           {{{9.99997e-01, -1.71497e-05, -1.47293e-05, 3.39505e-05, 4.12114e-05, -4.67233e-06},
             {-2.91468e-01, -6.93627e-01, -1.03371e-01, 3.37345e-01, -5.21461e-01, -5.80338e-01},
             {1.31012e-02, -4.29546e-02, -3.49321e-01, -3.53555e-01, 4.76616e-01, 6.50725e-01},
             {-4.93364e-02, -2.31171e-01, -1.60612e-01, 2.39643e-01, -1.76058e-01, -5.55826e-01},
             {1.35187e-02, 5.15694e-02, -6.57013e-04, -9.41718e-02, 1.02838e-01, 2.07094e-01}}},
           {{{9.99997e-01, -1.71505e-05, -1.68161e-05, 3.37003e-05, 4.99482e-05, 2.81614e-06},
             {-2.93422e-01, -7.05724e-01, -1.18440e-01, 3.51330e-01, -4.99397e-01, -5.83705e-01},
             {1.44239e-02, -3.09324e-02, -3.26859e-01, -3.64987e-01, 4.27192e-01, 6.28206e-01},
             {-4.84672e-02, -2.35337e-01, -1.80043e-01, 2.42424e-01, -1.19897e-01, -5.16535e-01},
             {1.33028e-02, 5.25748e-02, 3.07617e-03, -9.76271e-02, 8.83904e-02, 1.99339e-01}}},
           {{{9.99997e-01, -1.71432e-05, -1.87684e-05, 3.43631e-05, 5.65598e-05, 5.14426e-06},
             {-2.95225e-01, -7.17422e-01, -1.34090e-01, 3.64444e-01, -4.74674e-01, -5.83985e-01},
             {1.55614e-02, -1.94920e-02, -3.03857e-01, -3.75242e-01, 3.76112e-01, 6.04065e-01},
             {-4.75129e-02, -2.39123e-01, -1.99702e-01, 2.44322e-01, -6.36892e-02, -4.77415e-01},
             {1.30651e-02, 5.34896e-02, 6.87014e-03, -1.00987e-01, 7.38466e-02, 1.91787e-01}}},
           {{{9.99997e-01, -1.70410e-05, -2.02072e-05, 3.43416e-05, 6.00349e-05, 5.10394e-06},
             {-2.96890e-01, -7.28701e-01, -1.50015e-01, 3.76542e-01, -4.49253e-01, -5.83174e-01},
             {1.65464e-02, -8.69112e-03, -2.81232e-01, -3.83919e-01, 3.29264e-01, 5.84405e-01},
             {-4.65037e-02, -2.42471e-01, -2.18674e-01, 2.44939e-01, -1.33235e-02, -4.44574e-01},
             {1.28151e-02, 5.42961e-02, 1.04332e-02, -1.04118e-01, 6.10792e-02, 1.86372e-01}}},
           {{{9.99997e-01, -1.68703e-05, -2.19148e-05, 3.36605e-05, 6.58777e-05, 9.29556e-06},
             {-2.98413e-01, -7.39556e-01, -1.66572e-01, 3.87266e-01, -4.19880e-01, -5.76264e-01},
             {1.73553e-02, 1.45800e-03, -2.57648e-01, -3.89849e-01, 2.75001e-01, 5.51655e-01},
             {-4.54112e-02, -2.45372e-01, -2.38393e-01, 2.43047e-01, 4.35941e-02, -3.99374e-01},
             {1.25429e-02, 5.49905e-02, 1.42497e-02, -1.06608e-01, 4.59332e-02, 1.76851e-01}}},
           {{{9.99998e-01, -1.63851e-05, -2.41130e-05, 3.06863e-05, 7.51386e-05, 2.04334e-05},
             {-2.99804e-01, -7.49931e-01, -1.83831e-01, 3.96114e-01, -3.86250e-01, -5.62334e-01},
             {1.80076e-02, 1.07684e-02, -2.32847e-01, -3.91521e-01, 2.12426e-01, 5.03397e-01},
             {-4.42535e-02, -2.47631e-01, -2.59102e-01, 2.37167e-01, 1.07684e-01, -3.39917e-01},
             {1.22546e-02, 5.55098e-02, 1.83927e-02, -1.07983e-01, 2.82674e-02, 1.62721e-01}}},
           {{{9.99998e-01, -1.61044e-05, -2.53291e-05, 2.98967e-05, 7.74839e-05, 2.04178e-05},
             {-3.01071e-01, -7.59936e-01, -2.01146e-01, 4.04409e-01, -3.53023e-01, -5.49009e-01},
             {1.85178e-02, 1.96270e-02, -2.08871e-01, -3.93442e-01, 1.56441e-01, 4.64481e-01},
             {-4.30409e-02, -2.49659e-01, -2.78724e-01, 2.31999e-01, 1.63731e-01, -2.91708e-01},
             {1.19531e-02, 5.59897e-02, 2.21905e-02, -1.09785e-01, 1.30173e-02, 1.52276e-01}}},
           {{{9.99998e-01, -1.56344e-05, -2.12485e-05, 2.74332e-05, 4.12295e-05, -1.77772e-05},
             {-3.02233e-01, -7.69486e-01, -2.18133e-01, 4.10915e-01, -3.22343e-01, -5.35716e-01},
             {1.89406e-02, 2.77392e-02, -1.87032e-01, -3.91506e-01, 1.14703e-01, 4.33675e-01},
             {-4.18293e-02, -2.51147e-01, -2.95835e-01, 2.23293e-01, 2.03236e-01, -2.54139e-01},
             {1.16569e-02, 5.63284e-02, 2.51602e-02, -1.10618e-01, 3.10746e-03, 1.45469e-01}}},
           {{{9.99998e-01, -1.52486e-05, -2.18685e-05, 2.63797e-05, 3.96735e-05, -2.19621e-05},
             {-3.03266e-01, -7.78634e-01, -2.35957e-01, 4.16585e-01, -2.86177e-01, -5.16301e-01},
             {1.91815e-02, 3.52883e-02, -1.63056e-01, -3.88945e-01, 5.77365e-02, 3.87668e-01},
             {-4.05134e-02, -2.52287e-01, -3.15080e-01, 2.14402e-01, 2.58325e-01, -2.01326e-01},
             {1.13302e-02, 5.65893e-02, 2.88772e-02, -1.11575e-01, -1.23904e-02, 1.33369e-01}}},
           {{{9.99998e-01, -1.49384e-05, -1.85668e-05, 2.64379e-05, 9.32927e-06, -5.90939e-05},
             {-3.04204e-01, -7.87379e-01, -2.53267e-01, 4.21300e-01, -2.54104e-01, -5.00591e-01},
             {1.93435e-02, 4.22782e-02, -1.41547e-01, -3.85439e-01, 1.84081e-02, 3.60351e-01},
             {-3.91987e-02, -2.53089e-01, -3.31589e-01, 2.05043e-01, 2.94128e-01, -1.69326e-01},
             {1.10080e-02, 5.67762e-02, 3.17179e-02, -1.12568e-01, -2.18340e-02, 1.28031e-01}}},
           {{{9.99998e-01, -1.42427e-05, -2.49446e-05, 1.53930e-05, 6.24366e-05, 3.34028e-05},
             {-3.05013e-01, -7.95643e-01, -2.72262e-01, 4.22919e-01, -2.07694e-01, -4.61381e-01},
             {1.92976e-02, 4.84288e-02, -1.14760e-01, -3.73536e-01, -6.77231e-02, 2.56912e-01},
             {-3.77449e-02, -2.53251e-01, -3.53610e-01, 1.87283e-01, 3.79081e-01, -5.75145e-02},
             {1.06430e-02, 5.67874e-02, 3.64408e-02, -1.10941e-01, -4.80690e-02, 9.58647e-02}}},
           {{{9.99998e-01, -1.32027e-05, -2.33027e-05, 9.14491e-06, 4.56416e-05, 2.27780e-05},
             {-3.05759e-01, -8.03445e-01, -2.90033e-01, 4.24185e-01, -1.72116e-01, -4.37185e-01},
             {1.92652e-02, 5.38043e-02, -9.27549e-02, -3.62839e-01, -1.13336e-01, 2.11304e-01},
             {-3.63847e-02, -2.52841e-01, -3.70455e-01, 1.71365e-01, 4.20403e-01, -8.30644e-03},
             {1.03130e-02, 5.66457e-02, 3.94761e-02, -1.10117e-01, -6.01102e-02, 8.44433e-02}}},
           {{{9.99999e-01, -1.26179e-05, -2.37033e-05, 6.58973e-06, 4.50540e-05, 2.42166e-05},
             {-3.06397e-01, -8.10913e-01, -3.08160e-01, 4.24872e-01, -1.33622e-01, -4.09408e-01},
             {1.90768e-02, 5.88503e-02, -6.98798e-02, -3.52526e-01, -1.67197e-01, 1.57681e-01},
             {-3.49340e-02, -2.52333e-01, -3.88188e-01, 1.56384e-01, 4.70278e-01, 4.87572e-02},
             {9.95585e-03, 5.65088e-02, 4.28601e-02, -1.09783e-01, -7.54609e-02, 7.01232e-02}}},
           {{{9.99999e-01, -1.20077e-05, -2.38736e-05, 4.05730e-06, 4.32277e-05, 2.42538e-05},
             {-3.06948e-01, -8.17977e-01, -3.26242e-01, 4.24366e-01, -9.49814e-02, -3.79922e-01},
             {1.87960e-02, 6.33103e-02, -4.74766e-02, -3.40500e-01, -2.19769e-01, 1.02759e-01},
             {-3.34577e-02, -2.51452e-01, -4.05385e-01, 1.40143e-01, 5.18458e-01, 1.06260e-01},
             {9.59321e-03, 5.62848e-02, 4.61175e-02, -1.09209e-01, -9.06716e-02, 5.54358e-02}}},
           {{{9.99999e-01, -1.07481e-05, -1.92539e-05, -2.25080e-06, 4.42593e-06, -1.23825e-05},
             {-3.07451e-01, -8.24542e-01, -3.43245e-01, 4.22445e-01, -6.46015e-02, -3.58740e-01},
             {1.85472e-02, 6.68500e-02, -2.90780e-02, -3.25947e-01, -2.42061e-01, 8.10462e-02},
             {-3.20856e-02, -2.49840e-01, -4.18293e-01, 1.21766e-01, 5.34064e-01, 1.27455e-01},
             {9.26834e-03, 5.58535e-02, 4.79916e-02, -1.08095e-01, -9.54003e-02, 5.26817e-02}}},
           {{{9.99999e-01, -1.00541e-05, -1.86994e-05, -4.85810e-06, -2.16427e-06, -1.77696e-05},
             {-3.07844e-01, -8.30789e-01, -3.61171e-01, 4.19383e-01, -2.56881e-02, -3.25272e-01},
             {1.81087e-02, 7.01009e-02, -7.58517e-03, -3.09910e-01, -2.93182e-01, 2.04700e-02},
             {-3.05781e-02, -2.48171e-01, -4.34502e-01, 1.02408e-01, 5.80158e-01, 1.89068e-01},
             {8.90064e-03, 5.54399e-02, 5.10269e-02, -1.06835e-01, -1.10754e-01, 3.62068e-02}}},
           {{{9.99999e-01, -9.79122e-06, -1.69151e-05, -3.50937e-06, -1.68765e-05, -3.82375e-05},
             {-3.08169e-01, -8.36725e-01, -3.78369e-01, 4.16165e-01, 8.30956e-03, -2.97683e-01},
             {1.76086e-02, 7.30739e-02, 1.14792e-02, -2.95652e-01, -3.27931e-01, -1.78743e-02},
             {-2.90664e-02, -2.46452e-01, -4.48258e-01, 8.54517e-02, 6.09592e-01, 2.27377e-01},
             {8.53336e-03, 5.50463e-02, 5.33261e-02, -1.06535e-01, -1.21188e-01, 2.70114e-02}}},
           {{{9.99999e-01, -9.13937e-06, -1.80755e-05, -5.16470e-06, -1.01788e-05, -3.14865e-05},
             {-3.08424e-01, -8.42217e-01, -3.95809e-01, 4.11420e-01, 4.50191e-02, -2.65066e-01},
             {1.70401e-02, 7.52877e-02, 3.14236e-02, -2.78353e-01, -3.71788e-01, -7.05787e-02},
             {-2.75426e-02, -2.44164e-01, -4.62996e-01, 6.57591e-02, 6.48570e-01, 2.80179e-01},
             {8.16401e-03, 5.44970e-02, 5.60168e-02, -1.05473e-01, -1.35245e-01, 1.27240e-02}}},
           {{{9.99999e-01, -8.98752e-06, -2.55104e-05, -8.43805e-06, 5.28526e-05, 5.37052e-05},
             {-3.08578e-01, -8.47428e-01, -4.14251e-01, 4.05335e-01, 9.19151e-02, -2.16792e-01},
             {1.62819e-02, 7.73058e-02, 5.49978e-02, -2.58700e-01, -4.51624e-01, -1.75559e-01},
             {-2.58749e-02, -2.41906e-01, -4.81676e-01, 4.40975e-02, 7.26016e-01, 3.88156e-01},
             {7.74836e-03, 5.39922e-02, 6.00930e-02, -1.03912e-01, -1.62654e-01, -2.03163e-02}}},
           {{{1.00000e+00, -8.26662e-06, -2.52336e-05, -1.12133e-05, 5.14447e-05, 5.60973e-05},
             {-3.08718e-01, -8.52175e-01, -4.30815e-01, 3.98540e-01, 1.24188e-01, -1.86131e-01},
             {1.56265e-02, 7.84801e-02, 7.23642e-02, -2.38748e-01, -4.82373e-01, -2.18764e-01},
             {-2.43703e-02, -2.38986e-01, -4.93934e-01, 2.25767e-02, 7.52214e-01, 4.31084e-01},
             {7.38841e-03, 5.33001e-02, 6.21117e-02, -1.02536e-01, -1.73599e-01, -3.21800e-02}}},
           {{{1.00000e+00, -8.34060e-06, -2.98627e-05, -8.01232e-06, 8.76510e-05, 9.03860e-05},
             {-3.08763e-01, -8.56707e-01, -4.48077e-01, 3.91822e-01, 1.63120e-01, -1.48303e-01},
             {1.47872e-02, 7.96639e-02, 9.24568e-02, -2.21201e-01, -5.37350e-01, -2.85347e-01},
             {-2.27224e-02, -2.36305e-01, -5.09218e-01, 4.07278e-03, 8.04499e-01, 4.98644e-01},
             {6.98190e-03, 5.27206e-02, 6.52177e-02, -1.02305e-01, -1.93777e-01, -5.26367e-02}}},
           {{{1.00000e+00, -7.64510e-06, -2.92089e-05, -1.00617e-05, 8.41945e-05, 8.89495e-05},
             {-3.08800e-01, -8.60757e-01, -4.63789e-01, 3.83454e-01, 1.91718e-01, -1.19542e-01},
             {1.40560e-02, 7.99159e-02, 1.07549e-01, -2.00082e-01, -5.57994e-01, -3.19260e-01},
             {-2.12387e-02, -2.32861e-01, -5.19348e-01, -1.77338e-02, 8.20853e-01, 5.31391e-01},
             {6.63116e-03, 5.19186e-02, 6.66840e-02, -1.01113e-01, -2.02513e-01, -6.18592e-02}}},
           {{{1.00000e+00, -6.94044e-06, -2.89150e-05, -1.25627e-05, 8.45092e-05, 9.35270e-05},
             {-3.08782e-01, -8.64460e-01, -4.79407e-01, 3.73899e-01, 2.21408e-01, -8.71143e-02},
             {1.32660e-02, 7.96962e-02, 1.22757e-01, -1.76859e-01, -5.84303e-01, -3.65358e-01},
             {-1.97397e-02, -2.29136e-01, -5.29774e-01, -4.13492e-02, 8.43816e-01, 5.77210e-01},
             {6.27617e-03, 5.10544e-02, 6.83314e-02, -9.94423e-02, -2.14011e-01, -7.58764e-02}}},
           {{{1.00000e+00, -6.24640e-06, -2.40805e-05, -1.21487e-05, 4.68365e-05, 4.79931e-05},
             {-3.08744e-01, -8.67831e-01, -4.93405e-01, 3.64221e-01, 2.39391e-01, -6.84646e-02},
             {1.25298e-02, 7.90388e-02, 1.32891e-01, -1.55210e-01, -5.72399e-01, -3.63560e-01},
             {-1.83419e-02, -2.25166e-01, -5.35025e-01, -6.28541e-02, 8.27047e-01, 5.72471e-01},
             {5.95549e-03, 5.01387e-02, 6.83405e-02, -9.86030e-02, -2.12849e-01, -7.34822e-02}}},
           {{{1.00000e+00, -5.50393e-06, -2.52916e-05, -1.48554e-05, 5.96989e-05, 6.70725e-05},
             {-3.08636e-01, -8.70881e-01, -5.08584e-01, 3.53063e-01, 2.70240e-01, -3.13652e-02},
             {1.16653e-02, 7.79734e-02, 1.47649e-01, -1.30344e-01, -6.06926e-01, -4.24928e-01},
             {-1.68508e-02, -2.20974e-01, -5.45358e-01, -8.73598e-02, 8.59865e-01, 6.34420e-01},
             {5.60415e-03, 4.91784e-02, 7.01237e-02, -9.68821e-02, -2.28707e-01, -9.36510e-02}}},
           {{{1.00000e+00, -4.85819e-06, -2.31489e-05, -1.50494e-05, 4.56408e-05, 4.90092e-05},
             {-3.08515e-01, -8.73647e-01, -5.22017e-01, 3.42249e-01, 2.87970e-01, -1.14362e-02},
             {1.08639e-02, 7.66037e-02, 1.57097e-01, -1.08398e-01, -5.99682e-01, -4.28645e-01},
             {-1.54650e-02, -2.16664e-01, -5.50347e-01, -1.08454e-01, 8.49779e-01, 6.36435e-01},
             {5.28790e-03, 4.82073e-02, 7.02259e-02, -9.63966e-02, -2.30958e-01, -9.44806e-02}}},
           {{{1.00000e+00, -4.20134e-06, -2.16789e-05, -1.60894e-05, 3.80435e-05, 4.12939e-05},
             {-3.08352e-01, -8.76114e-01, -5.35381e-01, 3.30437e-01, 3.07894e-01, 1.37055e-02},
             {1.00202e-02, 7.48660e-02, 1.66956e-01, -8.46903e-02, -6.02531e-01, -4.50663e-01},
             {-1.40726e-02, -2.12164e-01, -5.55986e-01, -1.31086e-01, 8.50981e-01, 6.57958e-01},
             {4.96956e-03, 4.71995e-02, 7.06296e-02, -9.54982e-02, -2.37492e-01, -1.02212e-01}}},
           {{{1.00000e+00, -3.56688e-06, -2.00685e-05, -1.67957e-05, 2.98774e-05, 3.24015e-05},
             {-3.08156e-01, -8.78296e-01, -5.48324e-01, 3.17992e-01, 3.26759e-01, 3.89863e-02},
             {9.15883e-03, 7.27945e-02, 1.76059e-01, -6.04885e-02, -6.04528e-01, -4.73749e-01},
             {-1.26976e-02, -2.07506e-01, -5.61052e-01, -1.53887e-01, 8.52004e-01, 6.80843e-01},
             {4.65698e-03, 4.61659e-02, 7.09273e-02, -9.46480e-02, -2.44507e-01, -1.10820e-01}}},
           {{{1.00000e+00, -2.96667e-06, -1.87695e-05, -1.74281e-05, 2.49741e-05, 2.73867e-05},
             {-3.07933e-01, -8.80219e-01, -5.60632e-01, 3.05337e-01, 3.42808e-01, 6.12281e-02},
             {8.29085e-03, 7.04640e-02, 1.83875e-01, -3.70395e-02, -6.01099e-01, -4.89035e-01},
             {-1.13475e-02, -2.02764e-01, -5.65099e-01, -1.75647e-01, 8.48873e-01, 6.96932e-01},
             {4.35207e-03, 4.51294e-02, 7.09953e-02, -9.42244e-02, -2.50865e-01, -1.17856e-01}}},
           {{{1.00000e+00, -2.38117e-06, -1.71196e-05, -1.76206e-05, 1.75818e-05, 1.86146e-05},
             {-3.07688e-01, -8.81871e-01, -5.72312e-01, 2.92226e-01, 3.56380e-01, 8.18537e-02},
             {7.43188e-03, 6.78190e-02, 1.90320e-01, -1.35192e-02, -5.92277e-01, -4.99818e-01},
             {-1.00390e-02, -1.97874e-01, -5.67945e-01, -1.97188e-01, 8.40799e-01, 7.08540e-01},
             {4.06067e-03, 4.40685e-02, 7.07499e-02, -9.39650e-02, -2.56104e-01, -1.23813e-01}}},
           {{{1.00000e+00, -1.64094e-06, -1.55193e-05, -1.89774e-05, 1.04852e-05, 1.16829e-05},
             {-3.07422e-01, -8.83241e-01, -5.83531e-01, 2.78552e-01, 3.68475e-01, 1.01668e-01},
             {6.57878e-03, 6.47988e-02, 1.95926e-01, 1.05680e-02, -5.81061e-01, -5.08678e-01},
             {-8.76825e-03, -1.92764e-01, -5.70119e-01, -2.19093e-01, 8.30673e-01, 7.18276e-01},
             {3.78161e-03, 4.29575e-02, 7.03587e-02, -9.36606e-02, -2.61130e-01, -1.29544e-01}}},
           {{{1.00000e+00, -9.06217e-07, -1.75273e-05, -2.63276e-05, 3.91616e-05, 7.05587e-05},
             {-3.07119e-01, -8.84377e-01, -5.94807e-01, 2.63602e-01, 3.84677e-01, 1.31208e-01},
             {5.67301e-03, 6.15609e-02, 2.02707e-01, 3.78304e-02, -5.88336e-01, -5.53948e-01},
             {-7.46868e-03, -1.87597e-01, -5.73876e-01, -2.44177e-01, 8.41510e-01, 7.67774e-01},
             {3.49199e-03, 4.18500e-02, 7.05980e-02, -9.23631e-02, -2.73795e-01, -1.49135e-01}}},
           {{{1.00000e+00, -3.58948e-07, -1.62823e-05, -2.71290e-05, 3.72118e-05, 7.14444e-05},
             {-3.06807e-01, -8.85301e-01, -6.05065e-01, 2.49307e-01, 3.94729e-01, 1.51179e-01},
             {4.80860e-03, 5.81619e-02, 2.06801e-01, 6.15127e-02, -5.77208e-01, -5.66864e-01},
             {-6.24053e-03, -1.82429e-01, -5.75024e-01, -2.65216e-01, 8.33338e-01, 7.82915e-01},
             {3.22560e-03, 4.07640e-02, 7.00473e-02, -9.24940e-02, -2.80574e-01, -1.57600e-01}}},
           {{{1.00000e+00, 2.73211e-07, -1.17490e-05, -2.58038e-05, 6.36158e-06, 3.32743e-05},
             {-3.06487e-01, -8.86009e-01, -6.14422e-01, 2.35359e-01, 3.99566e-01, 1.64284e-01},
             {3.97998e-03, 5.45694e-02, 2.08689e-01, 8.29080e-02, -5.51489e-01, -5.58007e-01},
             {-5.07592e-03, -1.77216e-01, -5.74108e-01, -2.83686e-01, 8.10490e-01, 7.75556e-01},
             {2.97963e-03, 3.96831e-02, 6.88939e-02, -9.35395e-02, -2.82965e-01, -1.59015e-01}}},
           {{{1.00000e+00, 7.63461e-07, -1.03728e-05, -2.61459e-05, 3.81382e-06, 3.27945e-05},
             {-3.06146e-01, -8.86511e-01, -6.23541e-01, 2.20869e-01, 4.05684e-01, 1.80618e-01},
             {3.13512e-03, 5.08041e-02, 2.10764e-01, 1.05164e-01, -5.34247e-01, -5.62996e-01},
             {-3.91682e-03, -1.71979e-01, -5.73714e-01, -3.02939e-01, 7.97776e-01, 7.83848e-01},
             {2.73430e-03, 3.86138e-02, 6.80101e-02, -9.43784e-02, -2.89324e-01, -1.66201e-01}}},
           {{{1.00000e+00, 1.51657e-06, -7.09538e-06, -2.84408e-05, -1.27698e-05, 2.01646e-05},
             {-3.05804e-01, -8.86765e-01, -6.31784e-01, 2.05913e-01, 4.07649e-01, 1.93502e-01},
             {2.34793e-03, 4.67022e-02, 2.10764e-01, 1.28101e-01, -5.06111e-01, -5.58114e-01},
             {-2.84140e-03, -1.66534e-01, -5.71402e-01, -3.22806e-01, 7.74285e-01, 7.82225e-01},
             {2.51586e-03, 3.74936e-02, 6.65720e-02, -9.50616e-02, -2.92587e-01, -1.70530e-01}}},
           {{{1.00000e+00, 1.98683e-06, -6.48918e-06, -2.96293e-05, -7.69468e-06, 3.13394e-05},
             {-3.05435e-01, -8.86882e-01, -6.40039e-01, 1.91062e-01, 4.12813e-01, 2.10658e-01},
             {1.51474e-03, 4.26317e-02, 2.11891e-01, 1.49893e-01, -4.93374e-01, -5.71304e-01},
             {-1.73665e-03, -1.61267e-01, -5.70637e-01, -3.41415e-01, 7.68388e-01, 8.00987e-01},
             {2.28617e-03, 3.64503e-02, 6.57491e-02, -9.62013e-02, -3.02313e-01, -1.82243e-01}}},
           {{{1.00000e+00, 2.39732e-06, -5.11645e-06, -2.96111e-05, -9.22427e-06, 3.19320e-05},
             {-3.05062e-01, -8.86831e-01, -6.47453e-01, 1.76573e-01, 4.13633e-01, 2.22198e-01},
             {7.17189e-04, 3.84725e-02, 2.11135e-01, 1.69653e-01, -4.69442e-01, -5.67717e-01},
             {-6.88895e-04, -1.56047e-01, -5.68178e-01, -3.57792e-01, 7.51537e-01, 8.02701e-01},
             {2.07418e-03, 3.54391e-02, 6.44469e-02, -9.81311e-02, -3.08896e-01, -1.88736e-01}}},
           {{{1.00000e+00, 2.78244e-06, -3.64471e-06, -2.95014e-05, -1.12274e-05, 3.20414e-05},
             {-3.04686e-01, -8.86615e-01, -6.54170e-01, 1.62242e-01, 4.11585e-01, 2.30563e-01},
             {-5.16907e-05, 3.42119e-02, 2.09082e-01, 1.88283e-01, -4.40015e-01, -5.56825e-01},
             {3.11515e-04, -1.50852e-01, -5.64706e-01, -3.72992e-01, 7.30202e-01, 7.98110e-01},
             {1.87637e-03, 3.44517e-02, 6.29055e-02, -1.00477e-01, -3.14592e-01, -1.93758e-01}}},
           {{{1.00000e+00, 3.11912e-06, -2.09434e-06, -2.88816e-05, -1.39980e-05, 2.99387e-05},
             {-3.04309e-01, -8.86252e-01, -6.60070e-01, 1.48310e-01, 4.05729e-01, 2.33997e-01},
             {-7.79204e-04, 2.98994e-02, 2.05239e-01, 2.04916e-01, -4.00938e-01, -5.31269e-01},
             {1.25147e-03, -1.45730e-01, -5.59635e-01, -3.86047e-01, 6.99409e-01, 7.78559e-01},
             {1.69719e-03, 3.35036e-02, 6.09158e-02, -1.03573e-01, -3.17614e-01, -1.94234e-01}}},
           {{{1.00000e+00, 3.35749e-06, 3.97719e-06, -2.34572e-05, -5.63032e-05, -3.28419e-05},
             {-3.03936e-01, -8.85747e-01, -6.65065e-01, 1.34864e-01, 3.95346e-01, 2.31317e-01},
             {-1.45687e-03, 2.55398e-02, 1.99275e-01, 2.19322e-01, -3.49457e-01, -4.86677e-01},
             {2.12267e-03, -1.40682e-01, -5.52596e-01, -3.96718e-01, 6.56053e-01, 7.39183e-01},
             {1.53950e-03, 3.25940e-02, 5.83474e-02, -1.07501e-01, -3.16870e-01, -1.88461e-01}}},
           {{{1.00000e+00, 3.37161e-06, 6.41700e-06, -1.65495e-05, -7.10841e-05, -6.55183e-05},
             {-3.03544e-01, -8.85146e-01, -6.70190e-01, 1.21911e-01, 3.88427e-01, 2.32509e-01},
             {-2.16580e-03, 2.12729e-02, 1.94779e-01, 2.31585e-01, -3.13173e-01, -4.57701e-01},
             {3.01211e-03, -1.35849e-01, -5.47388e-01, -4.05136e-01, 6.29372e-01, 7.16760e-01},
             {1.37421e-03, 3.17686e-02, 5.64658e-02, -1.12206e-01, -3.22112e-01, -1.88731e-01}}},
           {{{1.00000e+00, 3.59419e-06, 8.62223e-06, -1.48197e-05, -7.92650e-05, -7.68342e-05},
             {-3.03155e-01, -8.84379e-01, -6.74772e-01, 1.08333e-01, 3.80917e-01, 2.35349e-01},
             {-2.83453e-03, 1.68369e-02, 1.89414e-01, 2.45715e-01, -2.77890e-01, -4.36776e-01},
             {3.84532e-03, -1.30946e-01, -5.41534e-01, -4.15613e-01, 6.04263e-01, 7.03374e-01},
             {1.22604e-03, 3.09317e-02, 5.44454e-02, -1.16240e-01, -3.28295e-01, -1.92431e-01}}},
           {{{1.00000e+00, 3.87146e-06, 4.80070e-06, -1.93015e-05, -3.42929e-05, -5.92706e-06},
             {-3.02743e-01, -8.83477e-01, -6.79905e-01, 9.38015e-02, 3.81334e-01, 2.51330e-01},
             {-3.55395e-03, 1.23259e-02, 1.87105e-01, 2.62907e-01, -2.74237e-01, -4.65042e-01},
             {4.72067e-03, -1.26071e-01, -5.39258e-01, -4.29411e-01, 6.13810e-01, 7.43311e-01},
             {1.06194e-03, 3.01154e-02, 5.37051e-02, -1.19185e-01, -3.46564e-01, -2.14447e-01}}},
           {{{1.00000e+00, 4.01379e-06, 5.49499e-06, -1.80611e-05, -3.07892e-05, -1.66001e-06},
             {-3.02339e-01, -8.82490e-01, -6.83951e-01, 8.05148e-02, 3.74652e-01, 2.55883e-01},
             {-4.22029e-03, 7.90066e-03, 1.82095e-01, 2.75606e-01, -2.49450e-01, -4.56602e-01},
             {5.52838e-03, -1.21388e-01, -5.34417e-01, -4.38535e-01, 6.01732e-01, 7.45118e-01},
             {9.18625e-04, 2.93730e-02, 5.21879e-02, -1.23704e-01, -3.58082e-01, -2.24204e-01}}},
           {{{1.00000e+00, 3.92136e-06, 6.47993e-06, -1.22539e-05, -3.36111e-05, -1.60510e-05},
             {-3.01948e-01, -8.81444e-01, -6.86996e-01, 6.86217e-02, 3.62375e-01, 2.51482e-01},
             {-4.81937e-03, 3.63292e-03, 1.74815e-01, 2.83496e-01, -2.09497e-01, -4.21244e-01},
             {6.25635e-03, -1.16966e-01, -5.27541e-01, -4.42744e-01, 5.74838e-01, 7.19912e-01},
             {7.99621e-04, 2.87267e-02, 5.00857e-02, -1.29858e-01, -3.65227e-01, -2.25574e-01}}},
           {{{1.00000e+00, 3.99266e-06, 6.47777e-06, -1.12429e-05, -2.38328e-05, -2.96736e-06},
             {-3.01550e-01, -8.80274e-01, -6.90202e-01, 5.59307e-02, 3.55180e-01, 2.56033e-01},
             {-5.42427e-03, -7.21339e-04, 1.69071e-01, 2.94138e-01, -1.89681e-01, -4.18927e-01},
             {6.98019e-03, -1.12541e-01, -5.22534e-01, -4.50001e-01, 5.69554e-01, 7.29976e-01},
             {6.80286e-04, 2.80878e-02, 4.86714e-02, -1.35006e-01, -3.79924e-01, -2.39042e-01}}},
           {{{1.00000e+00, 4.05101e-06, 1.00604e-05, -8.55053e-06, -3.80760e-05, -1.96756e-05},
             {-3.01180e-01, -8.79037e-01, -6.92166e-01, 4.44478e-02, 3.41177e-01, 2.50586e-01},
             {-5.91516e-03, -4.97546e-03, 1.60075e-01, 3.00765e-01, -1.49408e-01, -3.85113e-01},
             {7.57436e-03, -1.08305e-01, -5.14356e-01, -4.53218e-01, 5.43200e-01, 7.07528e-01},
             {6.02309e-04, 2.75190e-02, 4.62704e-02, -1.41493e-01, -3.88012e-01, -2.42136e-01}}},
           {{{1.00000e+00, 4.03802e-06, 1.22961e-05, -5.22107e-06, -4.92913e-05, -3.72393e-05},
             {-3.00797e-01, -8.77717e-01, -6.93950e-01, 3.31074e-02, 3.26970e-01, 2.44923e-01},
             {-6.43446e-03, -9.20205e-03, 1.51488e-01, 3.06993e-01, -1.11177e-01, -3.52850e-01},
             {8.19071e-03, -1.04178e-01, -5.06875e-01, -4.56147e-01, 5.19430e-01, 6.87083e-01},
             {5.14989e-04, 2.69938e-02, 4.41697e-02, -1.48089e-01, -3.97338e-01, -2.46248e-01}}},
           {{{1.00000e+00, 3.98480e-06, 1.30067e-05, -2.91293e-06, -4.64269e-05, -3.57834e-05},
             {-3.00419e-01, -8.76338e-01, -6.95427e-01, 2.20192e-02, 3.13702e-01, 2.40846e-01},
             {-6.92027e-03, -1.33345e-02, 1.42861e-01, 3.12565e-01, -7.88611e-02, -3.28521e-01},
             {8.76448e-03, -1.00225e-01, -4.99621e-01, -4.58571e-01, 5.02398e-01, 6.75598e-01},
             {4.39843e-04, 2.65321e-02, 4.22136e-02, -1.54855e-01, -4.09309e-01, -2.53760e-01}}},
           {{{1.00000e+00, 3.94823e-06, 1.35143e-05, -1.28843e-06, -4.26710e-05, -3.21558e-05},
             {-3.00039e-01, -8.74873e-01, -6.96826e-01, 1.08115e-02, 3.02127e-01, 2.39645e-01},
             {-7.40220e-03, -1.74666e-02, 1.34916e-01, 3.18716e-01, -5.45344e-02, -3.15813e-01},
             {9.32695e-03, -9.63453e-02, -4.93297e-01, -4.61732e-01, 4.93897e-01, 6.76357e-01},
             {3.66650e-04, 2.61010e-02, 4.06208e-02, -1.61393e-01, -4.24410e-01, -2.65594e-01}}},
           {{{1.00000e+00, 3.85274e-06, 1.67850e-05, 3.33082e-06, -6.43261e-05, -6.78587e-05},
             {-2.99678e-01, -8.73363e-01, -6.97176e-01, 7.68305e-04, 2.84062e-01, 2.28542e-01},
             {-7.81242e-03, -2.14852e-02, 1.24401e-01, 3.21238e-01, -1.07947e-02, -2.72085e-01},
             {9.80818e-03, -9.26511e-02, -4.84534e-01, -4.61294e-01, 4.65541e-01, 6.45226e-01},
             {3.18398e-04, 2.57363e-02, 3.82849e-02, -1.69124e-01, -4.33328e-01, -2.67276e-01}}},
           {{{1.00000e+00, 3.49119e-06, 1.77989e-05, 8.54007e-06, -5.94406e-05, -6.78762e-05},
             {-2.99325e-01, -8.71876e-01, -6.97466e-01, -8.34232e-03, 2.69771e-01, 2.21876e-01},
             {-8.18361e-03, -2.51618e-02, 1.14723e-01, 3.21013e-01, 1.72141e-02, -2.46552e-01},
             {1.02437e-02, -8.93779e-02, -4.76932e-01, -4.58213e-01, 4.54434e-01, 6.34192e-01},
             {2.83157e-04, 2.55154e-02, 3.64039e-02, -1.77726e-01, -4.48417e-01, -2.76123e-01}}},
           {{{1.00000e+00, 3.13119e-06, 2.04848e-05, 1.36397e-05, -6.54463e-05, -7.62514e-05},
             {-2.98982e-01, -8.70370e-01, -6.97309e-01, -1.67516e-02, 2.56138e-01, 2.16214e-01},
             {-8.51482e-03, -2.86591e-02, 1.04405e-01, 3.18798e-01, 4.13936e-02, -2.25977e-01},
             {1.06317e-02, -8.63510e-02, -4.68845e-01, -4.53229e-01, 4.47295e-01, 6.28363e-01},
             {2.61843e-04, 2.53800e-02, 3.44113e-02, -1.86967e-01, -4.65095e-01, -2.86964e-01}}},
           {{{1.00000e+00, 2.68238e-06, 1.95022e-05, 1.79662e-05, -4.64560e-05, -5.83468e-05},
             {-2.98635e-01, -8.68822e-01, -6.97267e-01, -2.50651e-02, 2.43906e-01, 2.11454e-01},
             {-8.85891e-03, -3.20692e-02, 9.55507e-02, 3.16895e-01, 5.76849e-02, -2.12199e-01},
             {1.10308e-02, -8.34676e-02, -4.62600e-01, -4.48895e-01, 4.49398e-01, 6.31098e-01},
             {2.34384e-04, 2.52948e-02, 3.31036e-02, -1.95975e-01, -4.85340e-01, -3.01238e-01}}},
           {{{1.00000e+00, 2.19243e-06, 1.83802e-05, 2.25939e-05, -2.74383e-05, -4.14375e-05},
             {-2.98291e-01, -8.67262e-01, -6.97067e-01, -3.27556e-02, 2.32469e-01, 2.06983e-01},
             {-9.18726e-03, -3.52986e-02, 8.70894e-02, 3.13464e-01, 6.93485e-02, -2.01573e-01},
             {1.14097e-02, -8.08214e-02, -4.56981e-01, -4.43207e-01, 4.56556e-01, 6.37500e-01},
             {2.11626e-04, 2.52905e-02, 3.20551e-02, -2.05432e-01, -5.07586e-01, -3.17064e-01}}},
           {{{1.00000e+00, 1.90824e-06, 1.75337e-05, 2.40067e-05, -1.39157e-05, -2.42798e-05},
             {-2.97948e-01, -8.65640e-01, -6.96905e-01, -4.09559e-02, 2.23681e-01, 2.08110e-01},
             {-9.51279e-03, -3.85358e-02, 7.96190e-02, 3.12379e-01, 7.07528e-02, -2.11292e-01},
             {1.17820e-02, -7.82100e-02, -4.52580e-01, -4.40225e-01, 4.74385e-01, 6.65212e-01},
             {1.89125e-04, 2.52991e-02, 3.14559e-02, -2.13995e-01, -5.33639e-01, -3.40216e-01}}},
           {{{1.00000e+00, 1.77000e-06, 2.62720e-05, 2.94930e-05, -1.05066e-04, -1.57983e-04},
             {-2.97652e-01, -8.63999e-01, -6.94236e-01, -4.71299e-02, 1.91590e-01, 1.74221e-01},
             {-9.67101e-03, -4.16520e-02, 6.44187e-02, 3.05069e-01, 1.50955e-01, -1.02197e-01},
             {1.19749e-02, -7.57620e-02, -4.40210e-01, -4.30966e-01, 4.09045e-01, 5.67673e-01},
             {2.24241e-04, 2.53622e-02, 2.82552e-02, -2.24641e-01, -5.32340e-01, -3.22038e-01}}},
           {{{1.00000e+00, 1.65023e-06, 1.70022e-05, 2.09197e-05, -1.78077e-05, -2.46650e-05},
             {-2.97304e-01, -8.62336e-01, -6.94643e-01, -5.55534e-02, 1.92543e-01, 1.89792e-01},
             {-1.00167e-02, -4.46503e-02, 6.08485e-02, 3.06312e-01, 1.14504e-01, -1.67135e-01},
             {1.23657e-02, -7.34762e-02, -4.40401e-01, -4.31040e-01, 4.67813e-01, 6.55299e-01},
             {1.91018e-04, 2.54799e-02, 2.92965e-02, -2.32178e-01, -5.72806e-01, -3.66007e-01}}},
           {{{1.00000e+00, 1.34186e-06, 1.69432e-05, 2.26628e-05, -1.33520e-05, -2.05400e-05},
             {-2.96996e-01, -8.60705e-01, -6.93244e-01, -6.16582e-02, 1.79649e-01, 1.83194e-01},
             {-1.02234e-02, -4.73718e-02, 5.18512e-02, 3.00343e-01, 1.25119e-01, -1.56522e-01},
             {1.26071e-02, -7.15120e-02, -4.34991e-01, -4.23759e-01, 4.76525e-01, 6.62845e-01},
             {2.05633e-04, 2.57040e-02, 2.85086e-02, -2.42168e-01, -5.96848e-01, -3.83555e-01}}},
           {{{1.00000e+00, 9.68943e-07, 1.51050e-05, 2.42782e-05, 4.87913e-06, -1.13536e-06},
             {-2.96688e-01, -8.59095e-01, -6.92020e-01, -6.72349e-02, 1.70148e-01, 1.80030e-01},
             {-1.04339e-02, -4.98599e-02, 4.42687e-02, 2.93348e-01, 1.23251e-01, -1.58755e-01},
             {1.28507e-02, -6.98228e-02, -4.31219e-01, -4.15631e-01, 4.98219e-01, 6.83791e-01},
             {2.17532e-04, 2.60191e-02, 2.83012e-02, -2.52454e-01, -6.25465e-01, -4.05797e-01}}},
           {{{1.00000e+00, 6.18000e-07, 1.58714e-05, 2.65990e-05, 9.51773e-07, -8.30010e-06},
             {-2.96402e-01, -8.57482e-01, -6.90151e-01, -7.21231e-02, 1.57438e-01, 1.72358e-01},
             {-1.05703e-02, -5.22097e-02, 3.52232e-02, 2.84934e-01, 1.32021e-01, -1.46479e-01},
             {1.30147e-02, -6.83011e-02, -4.26016e-01, -4.06312e-01, 5.08292e-01, 6.89268e-01},
             {2.54008e-04, 2.63882e-02, 2.76472e-02, -2.63140e-01, -6.50438e-01, -4.23135e-01}}},
           {{{9.99999e-01, 2.59474e-07, 1.66304e-05, 2.92739e-05, -4.13596e-06, -2.16221e-05},
             {-2.96127e-01, -8.55900e-01, -6.87860e-01, -7.59673e-02, 1.42366e-01, 1.59392e-01},
             {-1.06769e-02, -5.43151e-02, 2.57506e-02, 2.74014e-01, 1.46148e-01, -1.20052e-01},
             {1.31494e-02, -6.70578e-02, -4.20650e-01, -3.94725e-01, 5.13897e-01, 6.81679e-01},
             {2.97517e-04, 2.68481e-02, 2.70054e-02, -2.74569e-01, -6.74499e-01, -4.36824e-01}}},
           {{{9.99999e-01, -1.77820e-07, 1.81066e-05, 3.31456e-05, -1.69397e-05, -4.28294e-05},
             {-2.95860e-01, -8.54345e-01, -6.85493e-01, -7.93655e-02, 1.28453e-01, 1.48394e-01},
             {-1.07632e-02, -5.61996e-02, 1.67417e-02, 2.62468e-01, 1.56187e-01, -1.00805e-01},
             {1.32617e-02, -6.60643e-02, -4.15868e-01, -3.82710e-01, 5.23344e-01, 6.81217e-01},
             {3.46640e-04, 2.73888e-02, 2.65816e-02, -2.86163e-01, -7.00045e-01, -4.53062e-01}}},
           {{{9.99999e-01, -6.09064e-07, 1.75501e-05, 3.53367e-05, -1.26999e-05, -4.12248e-05},
             {-2.95596e-01, -8.52825e-01, -6.83326e-01, -8.23009e-02, 1.17972e-01, 1.40826e-01},
             {-1.08459e-02, -5.78446e-02, 9.15935e-03, 2.50321e-01, 1.54102e-01, -9.42181e-02},
             {1.33702e-02, -6.53381e-02, -4.12705e-01, -3.70314e-01, 5.45369e-01, 6.94045e-01},
             {3.95007e-04, 2.80161e-02, 2.67250e-02, -2.97907e-01, -7.30050e-01, -4.73991e-01}}},
           {{{9.99999e-01, -9.42759e-07, 1.72958e-05, 3.54784e-05, -9.69148e-06, -3.67223e-05},
             {-2.95349e-01, -8.51305e-01, -6.80689e-01, -8.49368e-02, 1.04962e-01, 1.29792e-01},
             {-1.08770e-02, -5.93862e-02, 7.36858e-04, 2.38278e-01, 1.59850e-01, -7.75931e-02},
             {1.34244e-02, -6.47286e-02, -4.08794e-01, -3.58341e-01, 5.59306e-01, 6.96865e-01},
             {4.59416e-04, 2.86787e-02, 2.66533e-02, -3.09522e-01, -7.57739e-01, -4.92019e-01}}},
           {{{9.99999e-01, -1.32996e-06, 1.59234e-05, 3.56739e-05, 1.06523e-06, -2.25545e-05},
             {-2.95093e-01, -8.49826e-01, -6.78848e-01, -8.76834e-02, 1.00610e-01, 1.30813e-01},
             {-1.09452e-02, -6.06841e-02, -4.28195e-03, 2.27644e-01, 1.36241e-01, -1.02336e-01},
             {1.35182e-02, -6.43838e-02, -4.08551e-01, -3.48083e-01, 6.03706e-01, 7.42793e-01},
             {5.08688e-04, 2.94262e-02, 2.78212e-02, -3.20603e-01, -7.95775e-01, -5.24547e-01}}},
           {{{9.99999e-01, -1.74682e-06, 1.57522e-05, 3.70429e-05, 4.12689e-07, -2.56009e-05},
             {-2.94861e-01, -8.48397e-01, -6.76273e-01, -8.91408e-02, 9.10447e-02, 1.22779e-01},
             {-1.09426e-02, -6.17213e-02, -1.11190e-02, 2.13700e-01, 1.30378e-01, -9.69320e-02},
             {1.35383e-02, -6.43198e-02, -4.06479e-01, -3.34582e-01, 6.29188e-01, 7.56942e-01},
             {5.80460e-04, 3.02636e-02, 2.84016e-02, -3.32795e-01, -8.27810e-01, -5.46804e-01}}},
           {{{9.99999e-01, -2.09294e-06, 1.43230e-05, 3.60782e-05, 1.07266e-05, -1.00341e-05},
             {-2.94623e-01, -8.46987e-01, -6.74321e-01, -9.10068e-02, 8.80040e-02, 1.24304e-01},
             {-1.09695e-02, -6.26224e-02, -1.52018e-02, 2.02357e-01, 1.02583e-01, -1.24641e-01},
             {1.35892e-02, -6.43956e-02, -4.07369e-01, -3.24105e-01, 6.77431e-01, 8.05852e-01},
             {6.40288e-04, 3.11424e-02, 2.99826e-02, -3.44011e-01, -8.67690e-01, -5.80897e-01}}},
           {{{9.99999e-01, -2.48040e-06, 1.34856e-05, 3.60818e-05, 1.43319e-05, -5.39346e-06},
             {-2.94407e-01, -8.45597e-01, -6.71717e-01, -9.23030e-02, 7.97576e-02, 1.18247e-01},
             {-1.09308e-02, -6.33722e-02, -2.07559e-02, 1.90204e-01, 9.18095e-02, -1.28190e-01},
             {1.35744e-02, -6.46357e-02, -4.06846e-01, -3.12984e-01, 7.08165e-01, 8.30043e-01},
             {7.19579e-04, 3.20730e-02, 3.11270e-02, -3.55492e-01, -9.02203e-01, -6.07246e-01}}},
           {{{9.99999e-01, -2.85883e-06, 1.25526e-05, 3.58006e-05, 1.83547e-05, 3.68727e-07},
             {-2.94193e-01, -8.44281e-01, -6.69393e-01, -9.28313e-02, 7.52932e-02, 1.15795e-01},
             {-1.08961e-02, -6.38078e-02, -2.47799e-02, 1.76646e-01, 6.89388e-02, -1.43931e-01},
             {1.35640e-02, -6.51987e-02, -4.07959e-01, -3.00617e-01, 7.51084e-01, 8.66622e-01},
             {7.95784e-04, 3.31054e-02, 3.28208e-02, -3.67433e-01, -9.40998e-01, -6.37918e-01}}},
           {{{9.99999e-01, -3.23089e-06, 1.14643e-05, 3.52274e-05, 2.27972e-05, 6.85460e-06},
             {-2.93995e-01, -8.43015e-01, -6.66747e-01, -9.27084e-02, 6.89020e-02, 1.09788e-01},
             {-1.08216e-02, -6.40252e-02, -2.91655e-02, 1.62218e-01, 5.20261e-02, -1.49534e-01},
             {1.35150e-02, -6.59834e-02, -4.08823e-01, -2.87623e-01, 7.88225e-01, 8.93534e-01},
             {8.82523e-04, 3.42062e-02, 3.44636e-02, -3.79623e-01, -9.78363e-01, -6.65926e-01}}},
           {{{9.99999e-01, -3.59368e-06, 1.03465e-05, 3.44987e-05, 2.66691e-05, 1.28457e-05},
             {-2.93801e-01, -8.41801e-01, -6.64287e-01, -9.22542e-02, 6.46779e-02, 1.06102e-01},
             {-1.07421e-02, -6.40140e-02, -3.23716e-02, 1.47883e-01, 2.86703e-02, -1.62837e-01},
             {1.34620e-02, -6.69992e-02, -4.10941e-01, -2.74897e-01, 8.31710e-01, 9.28176e-01},
             {9.68854e-04, 3.53788e-02, 3.65264e-02, -3.91796e-01, -1.01812e+00, -6.96738e-01}}},
           {{{9.99999e-01, -4.00634e-06, 1.34332e-05, 3.78284e-05, -7.14545e-06, -3.82071e-05},
             {-2.93636e-01, -8.40634e-01, -6.60935e-01, -9.08644e-02, 5.35291e-02, 9.20857e-02},
             {-1.05766e-02, -6.37990e-02, -3.80644e-02, 1.31539e-01, 2.97590e-02, -1.40756e-01},
             {1.33207e-02, -6.82218e-02, -4.10470e-01, -2.60169e-01, 8.49243e-01, 9.25370e-01},
             {1.08257e-03, 3.66156e-02, 3.77369e-02, -4.04722e-01, -1.04961e+00, -7.15408e-01}}},
           {{{9.99999e-01, -4.45767e-06, 1.33605e-05, 3.96345e-05, -1.55937e-05, -5.33269e-05},
             {-2.93460e-01, -8.39537e-01, -6.58549e-01, -8.94432e-02, 5.07890e-02, 8.85956e-02},
             {-1.04585e-02, -6.33154e-02, -4.00660e-02, 1.16361e-01, 4.17181e-03, -1.53315e-01},
             {1.32292e-02, -6.97106e-02, -4.13806e-01, -2.46814e-01, 8.93960e-01, 9.58183e-01},
             {1.17843e-03, 3.79349e-02, 4.01968e-02, -4.17279e-01, -1.09035e+00, -7.46081e-01}}},
           {{{9.99999e-01, -4.73699e-06, 1.33242e-05, 3.73673e-05, -2.10149e-05, -5.46306e-05},
             {-2.93297e-01, -8.38473e-01, -6.56026e-01, -8.84640e-02, 4.84392e-02, 8.76520e-02},
             {-1.03082e-02, -6.26988e-02, -4.19114e-02, 1.04068e-01, -2.26331e-02, -1.75588e-01},
             {1.31073e-02, -7.13217e-02, -4.17369e-01, -2.36640e-01, 9.40258e-01, 1.00169e+00},
             {1.28237e-03, 3.92889e-02, 4.27449e-02, -4.28876e-01, -1.13208e+00, -7.80735e-01}}}};
    // clang-format on
    static_assert(
        std::size(positron_mott_coeffs) == MottElementData::num_elements,
        "wrong number of Mott coefficient elements");

    int index = z.unchecked_get() - 1;
    CELER_VALIDATE(index >= 0 && index < int{MottElementData::num_elements},
                   << "atomic number " << z.get()
                   << " is out of range for Coulomb scattering model Mott "
                      "coefficients (must be less than "
                   << MottElementData::num_elements << ")");

    return positron_mott_coeffs[index];
}

//---------------------------------------------------------------------------//
}  // namespace celeritas
