import os
import csv


def read_dat_files(directory, delimiter="\t"):
    data = dict()  # List to hold all data from .dat files

    # Iterate over all files in the given directory
    for filename in os.listdir(directory):
        # Check if the file has a .dat extension
        if filename.endswith(".dat"):
            filepath = os.path.join(directory, filename)

            with open(filepath, "r", newline="") as file:
                # Read the TSV file
                tsv_reader = csv.reader(file, delimiter=delimiter)
                # Store each file's data as a list of lists
                file_data = [row for row in tsv_reader]
                data[filename] = file_data

    return data


labels = {
    "baseline": "Baseline",
    "cp": "Clopper-Pearson Interval",
    "small_supp": "\\smallsupport",
    "independence": "\\independence",
    "structure": "Structural improvements",
    "nwr": "$\\hookrightarrow$ \\equivalencestructures",
    "chains": "$\\hookrightarrow$ \\chainfragments",
    "full": "All improvements",
}

properties = {
    "consensus": "disagree",
    "csma": "all_before",
    "firewire_dl": "deadline",
    "mer": "k_equal_n",
    "pacman": "crash",
    "wlan": "collisions_max",
    "wlan_dl": "deadline",
    "zeroconf": "correct",
    "zeroconf_dl": "deadline",
}

precisions = {
    "consensus": "0.3",
    "csma": "0.1",
    "firewire_dl": "0.05",
    "mer": "0.55",
    "pacman": "0.1",
    "wlan": "0.7",
    "wlan_dl": "0.9",
    "zeroconf": "0.05",
    "zeroconf_dl": "0.05",
}

tt = 0
directory = "ablation"
dat_files_data = read_dat_files(directory, delimiter=",")

out = """\\documentclass{article}
    
\\newcommand{\\smallsupport}{\\texttt{Small Support}}
\\newcommand{\\independence}{\\texttt{Independence}}
\\newcommand{\\equivalencestructures}{\\texttt{Equivalence Structures}}
\\newcommand{\\chainfragments}{\\texttt{Chain Fragments}}

\\usepackage{adjustbox}
\\usepackage{booktabs}
\\usepackage{amsmath}
\\usepackage{array}
        
\\begin{document}"""


###########
# Table 1 #
###########

out += """\\begin{table}[t]
    \\centering
    \\caption{
       Summary of our improvement's effects on \\textbf{Property-SMC} instances.
    For each instance we report the number of transitions in the model; the number of samples needed for the precision $\\varepsilon$ for the baseline approach and with our improvements; and the improvement factor, i.e.\\ ratio of the previous two values.}\\label{tbl:obj-smc}
    \\setlength{\\tabcolsep}{5pt}
    \\begin{tabular}{>{\\ttfamily}l>{\\ttfamily}lrrrrr}
        model & {objective} &  precision & transitions & {baseline} & ours & ratio \\\\ \\midrule"""

for filename, data in sorted(dat_files_data.items()):
    if filename == "average.dat":
        continue
    filename = filename.replace(".dat", "")
    i = filename.find("-")
    param = filename[i + 1 :]
    filename = filename[:i]
    prop = properties[filename]
    precision = precisions[filename]
    num_transitions = int(float(data[1][1]))
    samples_std = float(data[1][2])
    samples_smart = float(data[2][2])
    factor = samples_std / samples_smart
    time_std = float(data[1][-1])
    time_smart = float(data[2][-1])
    filename = filename.replace("_", "{\\textunderscore}")
    prop = prop.replace("_", "{\\textunderscore}")
    out += f"\t\t{filename} & {prop} & {precision} & {num_transitions} & {samples_std} & {samples_smart} & {factor:.1f} \\\\"
out += "\t\\end{tabular}"
out += "\\end{table}"

out += "\n\n"

###########
# Table 2 #
###########

with open("ablation/average.dat", "r", newline="") as file:
    # Read the TSV file
    csv_reader = csv.reader(file, delimiter=",")
    # Store each file's data as a list of lists
    model_smc_data = dict()
    for row in csv_reader:
        model_smc_data[row[0]] = [float(x) for x in row[1:]]
    out += """\\begin{table}[t]
    \\centering
    \\caption{Minimum, average, and maximum sample reduction for each improvement\\label{tbl:model-smc}}
    \\setlength{\\tabcolsep}{5pt}
    \\begin{tabular}{lrrr}
        Ablated improvement & min & avg & max \\\\ \\midrule"""
    for row in labels.keys():
        if row == "full":
            continue
        minf, avgf, maxf = model_smc_data[row]
        out += f"\t\t{labels[row]} & {minf:.2f} & {avgf:.2f} & {maxf:.2f} \\\\"
    out += "\\midrule"
    out += "\t\\end{tabular}"
    out += "\\end{table}"

out += "\n\n"

###########
# Table 3 #
###########

out += """\\begin{table}[t]
    \\centering \\scriptsize
    \\caption{\\textbf{Property-SMC} results including parameters and runtimes} \\label{tbl:obj-smc-full}
    \\begin{tabular}
        { > {\\ttfamily}l > {\\ttfamily}lrrrrrrrr}
        & & & \multicolumn{2}{c}
        {Transitions} &  \multicolumn{3}{c}
        {Samples} & \multicolumn{2}{c}
        {Runtime}        \\\\\\cline{6 - 8}
        \\normalfont
        Model / Property & \\normalfont Parameters & precision & Base & Ours & Base & Ours & Factor & Base & Ours \\\\
        \\midrule"""

for filename, data in sorted(dat_files_data.items()):
    if filename == "average.dat":
        continue
    filename = filename.replace(".dat", "")
    i = filename.find("-")
    param = filename[i + 1 :]
    filename = filename[:i]
    prop = properties[filename]
    precision = precisions[filename]
    num_transitions_std = int(float(data[1][1]))
    num_transitions_smart = int(float(data[2][1]))
    samples_std = float(data[1][2])
    samples_smart = float(data[2][2])
    factor = samples_std / samples_smart
    time_std = float(data[1][-1])
    time_smart = float(data[2][-1])
    filename = filename.replace("_", "{\\textunderscore}")
    prop = prop.replace("_", "{\\textunderscore}")
    out += f"\t\t{filename}.{prop} & {param} & {precision} & {num_transitions_std} & {num_transitions_smart} & {samples_std} & {samples_smart} & {factor:.1f} & {time_std:.1f} & {time_smart:.1f} \\\\"

out += "    \\end{tabular}"
out += "\\end{table}"

###########
# Table 4 #
###########

out += """\\begin{table}[p]
    \\caption{Ablation study of \\textbf{Property-SMC} per problem instance\\label{tbl:model-smc-full}}
    \\centering
    \\begin{adjustbox}{angle=270}
        \\setlength\\tabcolsep{4pt}
        \\begin{tabular}{>{\\ttfamily}l>{\\hskip 2pt\\ttfamily}lr>{\\hskip 5pt}rrrrrrr}
            \\normalfont Model / Property                             & \\normalfont  Parameters & Transitions & Baseline  & CP    & \\smallsupport & \\independence  & Structure & NWR   & Chains \\\\ \\midrule"""
last_model = None
for filename, data in sorted(dat_files_data.items()):
    if filename == "average.dat":
        continue
    filename = filename.replace(".dat", "")
    i = filename.find("-")
    param = filename[i + 1 :]
    filename = filename[:i]
    prop = properties[filename]
    e = "" if last_model == filename or last_model is None else "\t\t\\midrule\n"
    filename_str = filename.replace("_", "{\\textunderscore}")
    prop_str = prop.replace("_", "{\\textunderscore}")
    m = "" if last_model == filename else f"{filename_str}.{prop_str}"
    last_model = filename
    num_transitions = int(float(data[1][1]))
    tt += int(float(num_transitions))
    nwr = "1" if float(data[7][3]) == 1 else f"{float(data[7][3]):.3f}"
    chains = "1" if float(data[6][3]) == 1 else f"{float(data[6][3]):.3f}"
    out += f"{e}\t\t{m} & {param} & {num_transitions} & {float(data[1][3]):.3f} & {float(data[3][3]):.3f}  & {float(data[4][3]):.3f}   & {float(data[5][3]):.3f} & {float(data[8][3]):.3f}  & {nwr} & {chains}  \\\\"
out += "\t\\end{tabular}"
out += "\t\\end{adjustbox}"
out += "\\end{table}"
out += "\n\n"

out += "\\end{document}"

with open("tables.tex", "w") as f:
    f.write(out)
