import os
import csv
import numpy as np


def process_folder(folder_path, env_name):
    # Get a list of all .dat files in the folder
    dat_files = [
        f
        for f in os.listdir(folder_path)
        if f.startswith(env_name) and f.endswith(".dat")
    ]

    if not dat_files:
        return  # No .dat files found, so skip this folder

    # Read the first file to get the structure (header and first column)
    with open(os.path.join(folder_path, dat_files[0]), "r") as file:
        reader = csv.reader(file, delimiter=",")  # Changed delimiter to comma
        header = next(reader)  # First row is the header
        first_column = [row[0] for row in reader]  # First column values (for retaining)

    # Prepare a 2D list to hold the numeric data for averaging
    num_rows = len(first_column)
    num_cols = len(header) - 1  # Exclude the first column
    data_matrix = np.zeros((num_rows, num_cols))  # Initialize with zeros
    file_count = 0

    # Read and accumulate the data from each .dat file
    for dat_file in dat_files:
        with open(os.path.join(folder_path, dat_file), "r") as file:
            reader = csv.reader(file, delimiter=",")  # Changed delimiter to comma
            next(reader)  # Skip the header

            # Verify that each row has the correct number of columns
            for i, row in enumerate(reader):
                if len(row) != num_cols + 1:  # +1 to account for the first column
                    print(
                        f"Column mismatch in file {dat_file}, row {i + 1}: Expected {num_cols + 1} columns but found {len(row)}."
                    )
                    continue  # Skip this row and print the mismatch info

                data_matrix[i] += np.array(
                    [float(val) for val in row[1:]]
                )  # Exclude the first column for averaging
        file_count += 1

    # If there are files processed, calculate the average
    if file_count > 0:
        data_matrix /= file_count  # Average across all files

        # There is one exception: For the improvement factor, we do not take the average but recompute it
        num_runs_full = data_matrix[1][1]
        for i, row in enumerate(data_matrix):
            data_matrix[i][2] = row[1] / num_runs_full

        with open(
            os.path.join(base_folder, env_name + ".dat"), "w", newline=""
        ) as file:
            writer = csv.writer(file, delimiter=",")  # Changed delimiter to comma

            writer.writerow(header)  # Include the first row as is
            for i, row in enumerate(data_matrix):
                writer.writerow(
                    [first_column[i]] + row.tolist()
                )  # Write first column and average values
            print(f"Processed folder {folder_path} and wrote average to {env_name}.dat")


def process_all_folders(base_folder):
    # Loop through all subfolders in the base folder
    for folder_name in os.listdir(base_folder):
        folder_path = os.path.join(base_folder, folder_name)
        if os.path.isdir(folder_path):  # Check if it's a folder
            print(f"Processing folder: {folder_name}")
            process_folder(folder_path, folder_name)


# Set the base folder where the folders with .dat files are located
base_folder = "."

process_all_folders(base_folder)
