# configs.py

"""
This file defines all configuration parameters for training and evaluation of the ContraBin model.
These parameters include paths, hyperparameters, and model-specific settings.
"""

import torch


class Config:
    """
    Centralized configuration class for the ContraBin project. It includes
    model-specific parameters, hyperparameters, and device setup.
    """
    # Debugging settings
    debug = False

    # Training settings
    batch_size = 32
    num_workers = 4
    head_lr = 1e-3
    code_encoder_lr = 1e-5
    text_encoder_lr = 1e-5
    weight_decay = 1e-3
    patience = 1
    factor = 0.8
    epochs = 30
    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")  # Use GPU if available

    # Model settings
    model_name = "microsoft/graphcodebert-base"  # Pretrained model name
    tokenizer_name = "microsoft/graphcodebert-base"  # Tokenizer name
    source_max_length = 512
    binary_max_length = 512
    comment_max_length = 32
    source_embedding = 768
    binary_embedding = 768
    comment_embedding = 768
    trainable = True  # Whether model parameters are trainable
    temperature = 1.0  # Contrastive learning temperature

    # Projection head settings
    num_projection_layers = 1
    projection_dim = 256
    dropout = 0.1  # Dropout rate for regularization


# Create an instance of the configuration
configs = Config()

# Print the configuration details for debugging purposes
if configs.debug:
    print("Configuration Settings:")
    for key, value in Config.__dict__.items():
        if not key.startswith("__"):
            print(f"{key}: {value}")