"""Tasks for simulating sidereal and time stream data.

A typical pattern would be to turn a map into a
:class:`containers.SiderealStream` with the :class:`SimulateSidereal` task, then
expand any redundant products with :class:`ExpandProducts` and finally generate
a set of time stream files with :class:`MakeTimeStream`.
"""

import inspect

import numpy as np
from caput import config, mpiarray, mpiutil, pipeline, tools
from cora.util import hputil

from ..core import containers, io, task
from ..util import regrid


class SimulateSidereal(task.SingleTask):
    """Create a simulated sidereal dataset from an input map.

    Attributes
    ----------
    stacked : bool
        When set, in the case that the beam transfer matrices are not full triangle
        treat them as having been generated by collating the baselines of full triangle
        data, and set appropriate `index_map/stack` and `reverse_map/stack` entries. If
        not set, treat the entries as having been generated by down selecting the full
        set of baselines, and thus don't create the `stack` entries. Default is `True`.
    """

    stacked = config.Property(proptype=bool, default=True)

    def setup(self, bt):
        """Setup the simulation.

        Parameters
        ----------
        bt : ProductManager or BeamTransfer
            Beam Transfer maanger.
        """
        self.beamtransfer = io.get_beamtransfer(bt)
        self.telescope = io.get_telescope(bt)

    def process(self, map_):
        """Simulate a SiderealStream.

        Parameters
        ----------
        map_ : :class:`containers.Map`
            The sky map to process to into a sidereal stream. Frequencies in the
            map, must match the Beam Transfer matrices.

        Returns
        -------
        ss : SiderealStream
            Stacked sidereal day.
        feeds : list of CorrInput
            Description of the feeds simulated.
        """
        # Read in telescope system
        bt = self.beamtransfer
        tel = self.telescope

        lmax = tel.lmax
        mmax = tel.mmax
        nfreq = tel.nfreq
        npol = tel.num_pol_sky

        lfreq, sfreq, efreq = mpiutil.split_local(nfreq)

        lm, sm, em = mpiutil.split_local(mmax + 1)

        # Set the minimum resolution required for the sky.
        ntime = 2 * mmax + 1

        freqmap = map_.index_map["freq"][:]
        row_map = map_.map[:]

        if (tel.frequencies != freqmap["centre"]).any():
            raise ValueError("Frequencies in map do not match those in Beam Transfers.")

        # Calculate the alm's for the local sections
        row_alm = hputil.sphtrans_sky(row_map, lmax=lmax).reshape(
            (lfreq, npol * (lmax + 1), lmax + 1)
        )

        # Trim off excess m's and wrap into MPIArray
        row_alm = row_alm[..., : (mmax + 1)]
        row_alm = mpiarray.MPIArray.wrap(row_alm, axis=0)

        # Perform the transposition to distribute different m's across processes. Neat
        # tip, putting a shorter value for the number of columns, trims the array at
        # the same time
        col_alm = row_alm.redistribute(axis=2)

        # Transpose and reshape to shift m index first.
        col_alm = col_alm.transpose((2, 0, 1)).reshape((None, nfreq, npol, lmax + 1))

        # Create storage for visibility data
        vis_data = mpiarray.MPIArray(
            (mmax + 1, nfreq, bt.ntel), axis=0, dtype=np.complex128
        )
        vis_data[:] = 0.0

        # Iterate over m's local to this process and generate the corresponding
        # visibilities
        for mp, mi in vis_data.enumerate(axis=0):
            vis_data[mp] = bt.project_vector_sky_to_telescope(
                mi, col_alm[mp].view(np.ndarray)
            )

        # Rearrange axes such that frequency is last (as we want to divide
        # frequencies across processors)
        row_vis = vis_data.transpose((0, 2, 1))

        # Parallel transpose to get all m's back onto the same processor
        col_vis_tmp = row_vis.redistribute(axis=2)
        col_vis_tmp = col_vis_tmp.reshape((mmax + 1, 2, tel.npairs, None))

        # Transpose the local section to make the m's the last axis and unwrap the
        # positive and negative m at the same time.
        col_vis = mpiarray.MPIArray(
            (tel.npairs, nfreq, ntime), axis=1, dtype=np.complex128
        )
        col_vis[:] = 0.0
        col_vis[..., 0] = col_vis_tmp[0, 0]
        for mi in range(1, mmax + 1):
            col_vis[..., mi] = col_vis_tmp[mi, 0]
            col_vis[..., -mi] = col_vis_tmp[
                mi, 1
            ].conj()  # Conjugate only (not (-1)**m - see paper)

        del col_vis_tmp

        # Fourier transform m-modes back to get final timestream.
        vis_stream = np.fft.ifft(col_vis, axis=-1) * ntime
        vis_stream = vis_stream.reshape((tel.npairs, lfreq, ntime))
        vis_stream = vis_stream.transpose((1, 0, 2)).copy()

        # Try and fetch out the feed index and info from the telescope object.
        try:
            feed_index = tel.input_index
        except AttributeError:
            feed_index = tel.nfeed

        kwargs = {}

        if tel.npairs != (tel.nfeed + 1) * tel.nfeed // 2 and self.stacked:
            # If we should treat this as stacked, then pull the information straight
            # from the telescope class
            kwargs["prod"] = tel.index_map_prod
            kwargs["stack"] = tel.index_map_stack
            kwargs["reverse_map_stack"] = tel.reverse_map_stack
        else:
            # Construct a product map as if this was a down selection
            prod_map = np.zeros(
                tel.uniquepairs.shape[0], dtype=[("input_a", int), ("input_b", int)]
            )
            prod_map["input_a"] = tel.uniquepairs[:, 0]
            prod_map["input_b"] = tel.uniquepairs[:, 1]

            kwargs["prod"] = prod_map

        # Construct container and set visibility data
        sstream = containers.SiderealStream(
            freq=freqmap,
            ra=ntime,
            input=feed_index,
            distributed=True,
            comm=map_.comm,
            **kwargs,
        )
        sstream.vis[:] = mpiarray.MPIArray.wrap(vis_stream, axis=0)
        sstream.weight[:] = 1.0

        return sstream


class ExpandProducts(task.SingleTask):
    """Un-wrap collated products to full triangle."""

    def setup(self, telescope):
        """Get a reference to the telescope class.

        Parameters
        ----------
        telescope : :class:`drift.core.TransitTelescope`
            Telescope object to use
        """
        self.telescope = io.get_telescope(telescope)

    def process(self, sstream):
        """Transform a sidereal stream to having a full product matrix.

        Parameters
        ----------
        sstream : :class:`containers.SiderealStream`
            Sidereal stream to unwrap.

        Returns
        -------
        new_sstream : :class:`containers.SiderealStream`
            Unwrapped sidereal stream.
        """
        sstream.redistribute("freq")

        ninput = len(sstream.input)
        prod = np.array(
            [(fi, fj) for fi in range(ninput) for fj in range(fi, ninput)],
            dtype=[("input_a", int), ("input_b", int)],
        )
        nprod = len(prod)

        new_stream = containers.SiderealStream(prod=prod, stack=None, axes_from=sstream)
        new_stream.redistribute("freq")
        new_stream.vis[:] = 0.0
        new_stream.weight[:] = 0.0

        # Create dummpy index and reverse map for the stack axis to match the behaviour
        # of reading in an N^2 file through andata
        fwd_stack = np.empty(nprod, dtype=[("prod", "<u4"), ("conjugate", "u1")])
        fwd_stack["prod"] = np.arange(nprod)
        fwd_stack["conjugate"] = False
        new_stream.create_index_map("stack", fwd_stack)
        rev_stack = np.empty(nprod, dtype=[("stack", "<u4"), ("conjugate", "u1")])
        rev_stack["stack"] = np.arange(nprod)
        rev_stack["conjugate"] = False
        new_stream.create_reverse_map("stack", rev_stack)

        # Iterate over all feed pairs and work out which is the correct index in the sidereal stack.
        for pi, (fi, fj) in enumerate(prod):
            unique_ind = self.telescope.feedmap[fi, fj]
            conj = self.telescope.feedconj[fi, fj]

            # unique_ind is less than zero it has masked out
            if unique_ind < 0:
                continue

            prod_stream = sstream.vis[:, unique_ind]
            new_stream.vis[:, pi] = prod_stream.conj() if conj else prod_stream

            new_stream.weight[:, pi] = 1.0

        return new_stream


class MakeTimeStream(task.SingleTask):
    """Generate a time stream from a sidereal stream.

    Requires an input dataset with a time axis to mimic.

    Attributes
    ----------
    lanczos_width : int
        Width of the Lanczos interpolation kernel.
    """

    lanczos_width = config.Property(proptype=int, default=5)

    def setup(self, observer):
        """Set up a telescope object.

        Parameters
        ----------
        observer : ProductManager
            Telescope manager
        """
        self.observer = io.get_telescope(observer)

    def process(self, sstream, tstream):
        """Sample the sidereal stream at a set of times.

        Parameters
        ----------
        sstream : containers.SiderealStream | containers.HybridVisStream
            Dataset sampled in RA
        tstream : containers.TODContainer | HybridVisStream
            Dataset sampled in time

        Returns
        -------
        timestream : containers.TimeStream | containers.HybridVisStream
            sstream data sampled in time
        """
        # Figure out the time-like axis of `tstream`
        if hasattr(tstream, "time"):
            time = tstream.time[:]
            tra = self.observer.unix_to_lsa(time)
        elif hasattr(tstream, "ra"):
            tra = tstream.ra[:]
            lsd = tstream.attrs.get("lsd", tstream.attrs.get("csd"))
            time = self.observer.lsd_to_unix(lsd + tra / 360.0)

        # Determine output container based on input container
        container_map = {
            containers.SiderealStream: containers.TimeStream,
            containers.HybridVisStream: containers.HybridVisStream,
        }

        # Make the new timestream containers
        for cls in inspect.getmro(sstream.__class__):
            OutputContainer = container_map.get(cls)

            if OutputContainer is not None:
                break

        if OutputContainer is None:
            raise TypeError(
                f"No valid container mapping.\nGot {sstream.__class__}.\n"
                f"Mappings exist for {list(container_map.keys())}."
            )

        # Provide both the time and ra axes - whichever one isn't
        # used will be ignored
        out = OutputContainer(axes_from=sstream, attrs_from=sstream, time=time, ra=tra)

        # Make sure datasets are distributed correctly
        sstream.redistribute("freq")
        out.redistribute("freq")

        # Make the interpolation array
        R = regrid.lanczos_forward_matrix(
            sstream.ra, tra % 360, self.lanczos_width, periodic=True
        ).T.copy()

        # Move the RA axis to the back
        axind = list(sstream.data.attrs["axis"]).index("ra")
        data = np.moveaxis(sstream.data[:].local_array, axind, -1)

        # Apply the interpolation matrix and
        # move the time-like axis back to where it belongs
        out.data[:].local_array[:] = np.moveaxis(data @ R, -1, axind)

        # Propagate the weights as well
        waxind = list(sstream.weight.attrs["axis"]).index("ra")
        var = np.moveaxis(
            tools.invert_no_zero(sstream.weight[:].local_array), waxind, -1
        )
        wout = tools.invert_no_zero(var @ (R**2))

        out.weight[:].local_array[:] = np.moveaxis(wout, -1, waxind)

        return out


class MakeTimeStreamFixedInput(MakeTimeStream):
    """Make multiple time streams from a single input."""

    def setup(self, observer, sstream):
        """Set up a telescope object and save the input stream.

        Parameters
        ----------
        observer : ProductManager
            Telescope manager
        sstream : containers.SiderealStream | containers.HybridVisStream
            Input sidereal dataset
        """
        self.sstream = sstream
        super().setup(observer)

    def process(self, tstream):
        """Sample the saved sidereal stream at `tstream` time samples.

        Parameters
        ----------
        tstream : containers.TODContainer
            Dataset sampled in time

        Returns
        -------
        timestream : containers.TimeStream | containers.HybridVisStream
            sstream data sampled in time
        """
        return super().process(self.sstream, tstream)


class MakeTimeStreamFixedTime(MakeTimeStream):
    """Make multiple time streams for fixed time samples."""

    def setup(self, observer, tstream):
        """Set up a telescope object and save the input timestream.

        Parameters
        ----------
        observer : ProductManager
            Telescope manager
        tstream : containers.TODContainer
            Input sidereal dataset
        """
        self.tstream = tstream
        super().setup(observer)

    def process(self, sstream):
        """Sample the sidereal stream at saved time samples.

        Parameters
        ----------
        sstream : containers.SiderealSteam | containers.HybridVisStream
            Dataset sampled in time

        Returns
        -------
        timestream : containers.TimeStream | containers.HybridVisStream
            sstream data sampled in time
        """
        return super().process(sstream, self.tstream)


class MakeMultipleTimeStreams(MakeTimeStreamFixedInput):
    """Generate a series of time streams files from a sidereal stream.

    Parameters
    ----------
    start_time, end_time : float or datetime
        Start and end times of the timestream to simulate. Needs to be either a
        `float` (UNIX time) or a `datetime` objects in UTC.
    integration_time : float, optional
        Integration time in seconds. Takes precedence over `integration_frame_exp`.
    integration_frame_exp: int, optional
        Specify the integration time in frames. The integration time is
        `2**integration_frame_exp * 2.56 us`.
    samples_per_file : int, optional
        Number of samples per file.
    """

    start_time = config.utc_time()
    end_time = config.utc_time()

    integration_time = config.Property(proptype=float, default=None)
    integration_frame_exp = config.Property(proptype=int, default=23)

    samples_per_file = config.Property(proptype=int, default=1024)

    _cur_time = None  # Hold the current file start time

    def process(self):
        """Create a timestream file.

        Returns
        -------
        tstream : containers.TimeStream | containers.HybridVisStream
            Time stream object.
        """
        if self._cur_time is None:
            self._cur_time = self.start_time

        # First check to see if we have reached the end of the requested time,
        # and if so stop the iteration.
        if self._cur_time >= self.end_time:
            raise pipeline.PipelineStopIteration

        # Produce a new timestream with the target time axis
        tstream = self._next_time_axis()

        return super().process(self.sstream, tstream)

    def _next_time_axis(self):
        # Calculate the integration time
        if self.integration_time is not None:
            int_time = self.integration_time
        else:
            int_time = 2.56e-6 * 2**self.integration_frame_exp

        # Calculate number of samples in file and timestamps
        nsamp = min(
            int(np.ceil((self.end_time - self._cur_time) / int_time)),
            self.samples_per_file,
        )
        timestamps = (
            self._cur_time + (np.arange(nsamp) + 1) * int_time
        )  # +1 as timestamps are at the end of each sample

        # Construct the time axis index map
        if self.integration_time is not None:
            time = timestamps
        else:
            _time_dtype = [("fpga_count", np.uint64), ("ctime", np.float64)]
            time = np.zeros(nsamp, _time_dtype)
            time["ctime"] = timestamps
            time["fpga_count"] = (
                (timestamps - self.start_time)
                / int_time
                * 2**self.integration_frame_exp
            ).astype(np.uint64)

        # Increment the current start time for the next iteration
        self._cur_time += nsamp * int_time

        # Return an empty, un-initialised TODContainer
        # for compatibility with parent tasks
        return containers.TODContainer(time=time, skip_datasets=True)


class MakeSiderealDayStream(task.SingleTask):
    """Task for simulating a set of sidereal days from a given stream.

    This creates a copy of the base stream for every LSD within the provided time
    range.

    Attributes
    ----------
    start_time, end_time : float or datetime
        Start and end times of the sidereal streams to simulate. Needs to be either a
        `float` (UNIX time) or a `datetime` objects in UTC.
    """

    start_time = config.utc_time()
    end_time = config.utc_time()

    def setup(self, bt, sstream):
        """Set up an observer and the data to use for this simulation.

        Parameters
        ----------
        bt : beamtransfer.BeamTransfer or manager.ProductManager
            Sets up an observer holding the geographic location of the telscope.
        sstream : containers.SiderealStream
            The base sidereal data to use for this simulation.
        """
        self.observer = io.get_telescope(bt)
        self.lsd_start = self.observer.unix_to_lsd(self.start_time)
        self.lsd_end = self.observer.unix_to_lsd(self.end_time)

        self.log.info(
            "Sidereal period requested: LSD=%i to LSD=%i",
            int(self.lsd_start),
            int(self.lsd_end),
        )

        # Initialize the current lsd time
        self._current_lsd = None
        self.sstream = sstream

    def process(self):
        """Generate a sidereal stream for the specific sidereal day.

        Returns
        -------
        ss : :class:`containers.SiderealStream`
            Simulated sidereal day stream.
        """
        # If current_lsd is None then this is the first time we've run
        if self._current_lsd is None:
            # Check if lsd is an integer, if not add an lsd
            if isinstance(self.lsd_start, int):
                self._current_lsd = int(self.lsd_start)
            else:
                self._current_lsd = int(self.lsd_start + 1)

        # Check if we have reached the end of the requested time
        if self._current_lsd >= self.lsd_end:
            raise pipeline.PipelineStopIteration

        ss = self.sstream.copy()
        ss.attrs["tag"] = f"lsd_{self._current_lsd}"
        ss.attrs["lsd"] = self._current_lsd

        self._current_lsd += 1

        return ss
