# Vulnerability Details
A locking issue was found in the Linux kernel netfilter/nftables subsystem (`net/netfilter/nf_tables_api.c`), 
it leads to breaking the assumption in set asynchronous GC, which can be use to cause double free.

The asynchronous set GC (`nft_rhash_gc` for example) does not acquire commit lock while doing the work, 
instead, it use GC sequence (`gc_seq`) mechanism to protect it from racing with the transaction.
At the begin of `nft_rhash_gc` It will save the current GC sequence and allocate a GC transaction 
to store information, then traverse the set to record all expired set element into GC transaction, 
and finally call `nft_trans_gc_queue_async_done(gc)`.

```c
static void nft_rhash_gc(struct work_struct *work)
{
	// ...
	gc_seq = READ_ONCE(nft_net->gc_seq); // save GC sequence

	if (nft_set_gc_is_pending(set))
		goto done;

	gc = nft_trans_gc_alloc(set, gc_seq, GFP_KERNEL);
	if (!gc)
		goto done;

	// ...
	while ((he = rhashtable_walk_next(&hti))) {
		// check if setelem expired
		if (!nft_set_elem_expired(&he->ext))
			continue;
	
		// ...
		nft_trans_gc_elem_add(gc, he);
	}

	if (gc)
		nft_trans_gc_queue_async_done(gc);

	// ...
}
```

The function `nft_trans_gc_queue_async_done(gc)` saves the GC transaction into a global list and eventually schedules 
`nft_trans_gc_work()` to run. `nft_trans_gc_work()` then retrieves the gc transaction and calls `nft_trans_gc_work_done()` 
to perform check on GC sequence.

```c
static void nft_trans_gc_work(struct work_struct *work)
{
	// ...
	list_for_each_entry_safe(trans, next, &trans_gc_list, list) {
		list_del(&trans->list);
		if (!nft_trans_gc_work_done(trans)) { // do the check here
			nft_trans_gc_destroy(trans);
			continue;
		}
		call_rcu(&trans->rcu, nft_trans_gc_trans_free);
	}
}
```

The function `nft_trans_gc_work_done()` will first acquire the commit lock, and compare the saved GC sequence 
with current GC sequence, if they are different, means we race with the transaction, since all critical section 
which modify the control plane are surrounded by `nft_gc_seq_begin()` and `nft_gc_seq_end()` which both increase 
the current GC sequence (`nft_net->gc_seq`), so if it's the case, it means the state of the set may have been changed,
and the function will return false to stop processing this GC transaction.


```c
static bool nft_trans_gc_work_done(struct nft_trans_gc *trans)
{
	struct nftables_pernet *nft_net;
	struct nft_ctx ctx = {};

	nft_net = nft_pernet(trans->net);

	mutex_lock(&nft_net->commit_mutex); // acquire global mutex

	/* Check for race with transaction, otherwise this batch refers to
	 * stale objects that might not be there anymore. Skip transaction if
	 * set has been destroyed from control plane transaction in case gc
	 * worker loses race.
	 */
	if (READ_ONCE(nft_net->gc_seq) != trans->seq || trans->set->dead) { // check gc sequence to prevent race
		mutex_unlock(&nft_net->commit_mutex);
		return false;
	}

	ctx.net = trans->net;
	ctx.table = trans->set->table;

	nft_trans_gc_setelem_remove(&ctx, trans);
	mutex_unlock(&nft_net->commit_mutex);

	return true;
}
```

However, the GC sequence mechanism only works under the assumption that the commit lock should not be released
during the critical section between `nft_gc_seq_begin()` and `nft_gc_seq_end()`. Otherwise, a GC thread
may record the expired object and obtain the released commit lock within the same `gc_seq`, thus bypassing the GC sequence check.

`__nf_tables_abort()` is the one does it wrong, the function is surrounded by `nft_gc_seq_begin()` and `nft_gc_seq_end()`, 
if it received the action `NFNL_ABORT_AUTOLOAD`, `nf_tables_module_autoload()` will be called to process the module requests,
however, the function release the commit lock before processing the module request, which breaks the assumption of GC
sequence and leads to double free.

```c
static int nf_tables_abort(struct net *net, struct sk_buff *skb,
			   enum nfnl_abort_action action)
{
	gc_seq = nft_gc_seq_begin(nft_net);   // gc_seq++
	ret = __nf_tables_abort(net, action);
	nft_gc_seq_end(nft_net, gc_seq);      // gc_seq++
	mutex_unlock(&nft_net->commit_mutex);

	return ret;
}

static int __nf_tables_abort(struct net *net, enum nfnl_abort_action action)
{
	// ...

	if (action == NFNL_ABORT_AUTOLOAD)
		nf_tables_module_autoload(net); // load modules
	else
		nf_tables_module_autoload_cleanup(net);

	return 0;
}

static void nf_tables_module_autoload(struct net *net)
{
	struct nftables_pernet *nft_net = nft_pernet(net);
	struct nft_module_request *req, *next;
	LIST_HEAD(module_list);

	list_splice_init(&nft_net->module_list, &module_list);
	mutex_unlock(&nft_net->commit_mutex); // BUG: release mutex lock inside GC sequence critical section
	list_for_each_entry_safe(req, next, &module_list, list) {
		request_module("%s", req->module);
		req->done = true;
	}
	mutex_lock(&nft_net->commit_mutex);
	list_splice(&module_list, &nft_net->module_list);
}
```

## Requirements to trigger the vulnerability
- Capabilities: `CAP_NET_ADMIN` capability is required.
- Kernel configuration: `CONFIG_NETFILTER`, `CONFIG_NF_TABLES`
- User namespace: As this vulnerability requires `CAP_NET_ADMIN`, which is not usually given to the normal user, we used the unprivileged user namespace to achieve this capability.

## Commit which introduced the vulnerability
- The vulnerability was introduced in Linux v6.5, with commit [720344340fb9be2765bbaab7b292ece0a4570eae](https://git.kernel.org/pub/scm/linux/kernel/git/torvalds/linux.git/commit/?id=720344340fb9be2765bbaab7b292ece0a4570eae)
- An incomplete fix to new GC transaction API introduced this vulnerability.

## Commit which fixed the vulnerability
- The vulnerability was fixed in Linux v6.9-rc3, with commit [0d459e2ffb541841714839e8228b845458ed3b27](https://git.kernel.org/pub/scm/linux/kernel/git/torvalds/linux.git/commit/?id=0d459e2ffb541841714839e8228b845458ed3b27)
- The commit move the call to  `nf_tables_module_autoload()` after `nft_gc_seq_end()` to fix the check.

## Affected kernel versions
- Linux version v6.5 ~ v6.9-rc2 affects to this vulnerability
- For LTS versions
	- v5.15.134 ~
	- v6.1.56 ~

## Affected component, subsystem
- netfilter/nf_tables

## Cause (UAF, BoF, race condition, double free, refcount overflow, etc)
- Locking issue leads to double free

## Which syscalls or syscall parameters are needed to be blocked to prevent triggering the vulnerability? (If there is any easy way to block it.)
- Disable syscalls for netfilter (specifically, nftables) subsystem (ex. `socket`, `sendmsg` with netlink socket) to prevent this vulnerability.
- Disable syscalls for unprivileged user namespace (ex. `clone`, `unshare`) can reduce the attack surface since the netfilter subsystem requires `CAP_NET_ADMIN` to use.
