# Exploit details
The vulnerability is a locking issue that resides in __nf_tables_abort() during the call to nf_tables_module_autoload(), 
which releases the mutex lock and causes GC sequence protection to be bypassed. To exploit this vulnerability, 
we must find a reliable way to race the abort thread and the set GC thread. By doing so, we can turn this locking 
issue into a double-free primitive.

## Module Autoload

Since nftables object types (e.g., `nft_tunnel_obj_type`, `nft_quota_obj_type`, etc.) might reside in external kernel modules, 
encountering a type currently not present in nftables will first add the typename to `nft_net->module_list` and attempt 
to load the corresponding type's kernel module during the abort phase via `nf_tables_module_autoload()`. 

Therefore, `nf_tables_module_autoload()` can be triggered by requesting a non-existent object type in a batch commit.

```c
static const struct nft_object_type *
nft_obj_type_get(struct net *net, u32 objtype, u8 family)
{
    const struct nft_object_type *type;

    rcu_read_lock();
    type = __nft_obj_type_get(objtype, family);

    // ...

#ifdef CONFIG_MODULES
    if (type == NULL) { // if type does not exist
        if (nft_request_module(net, "nft-obj-%u", objtype) == -EAGAIN)
            return ERR_PTR(-EAGAIN);
    }
#endif
    return ERR_PTR(-ENOENT);
}
```

```c
__printf(2, 3) int nft_request_module(struct net *net, const char *fmt, ...)
{
    // ...

    nft_net = nft_pernet(net);
    list_for_each_entry(req, &nft_net->module_list, list) {
        if (!strcmp(req->module, module_name)) {
            if (req->done)
                return 0;

            /* A request to load this module already exists. */
            return -EAGAIN;
        }
    }

    req = kmalloc(sizeof(*req), GFP_KERNEL);
    if (!req)
        return -ENOMEM;

    req->done = false;
    strscpy(req->module, module_name, MODULE_NAME_LEN);
    list_add_tail(&req->list, &nft_net->module_list); // add to request list

    return -EAGAIN;
}
```

## Race to Double Free

With this vulnerability, we can make the set GC procedure record the GC sequence and acquire the mutex lock 
within the call to  `__nf_tables_abort()` to bypass the GC sequence check. The goal is to free 
the same set element twice during the handling of a batch commit.

The race process is as follows:
- New `setelem A`
- New unknown type object (trigger `__nf_tables_abort()`)
    - `nft_rhash_gc()` records expired `setelem A`
    - `setelem A `unlinks from the set, `kfree`'d
    - module autoload releases mutex lock
    - `nft_trans_gc_work_done()` acquires mutex lock, bypassing GC sequence check
    - `setelem A` `kfree`'d second time

To increase the success rate of the race (and capture the kernelctf slot), we need to enlarge the two time windows in the race process:
1. For the GC thread, we want its timer to wake up and record `setelem A` after `__nf_tables_abort()` starts but before `setelem A` is removed from the set. Otherwise, `setelem A` cannot be recorded.
2. The module loading time should be long enough to ensure that `nft_trans_gc_work_done()` can acquire the mutex lock.

To delay the removal of `setelem A` in `__nf_tables_abort()` from the set, we can add many operations after the creation of 
`setelem A` in the batch commit. Since `__nf_tables_abort()` processes batch commit operations in reverse order, 
these operations will be processed before removing `setelem A`. 

In order to maximize this delay, we pre-allocate multiple anonymous sets (`NFT_SET_MAP`) with many elements, 
reference them through `dynset` expressions, and delete these expressions via `NFT_MSG_DELRULE` at the end of the batch commit. 
This ensures that `nft_map_activate()` is called to traverse all set elements during the abort process, delaying the removal of `setelem A`.

For the module autoload part, since all types waiting for autoload will not be removed from `nft_net->module_list` 
even after autoloading finishes, and `nf_tables_module_autoload()` will always try to load the type in the list without 
checking if `req->done` is set, each autoload trigger will reload all previously autoloaded types.

```c
static void nf_tables_module_autoload(struct net *net)
{
    struct nftables_pernet *nft_net = nft_pernet(net);
    struct nft_module_request *req, *next;
    LIST_HEAD(module_list);

    list_splice_init(&nft_net->module_list, &module_list);
    mutex_unlock(&nft_net->commit_mutex);
    list_for_each_entry_safe(req, next, &module_list, list) {
        request_module("%s", req->module);
        req->done = true;
    }
    mutex_lock(&nft_net->commit_mutex);
    list_splice(&module_list, &nft_net->module_list);
}
```

Therefore, we only need to attempt autoloading non-existent and non-repeating object types N times before 
the new `setelem A` operation. Finally, triggering autoload with a non-existent object type after the new `setelem A` will trigger `request_module()` N+1 times.

With the above adjustments, the batch commit used in the exploit includes the following operations:
- New unknown type object (1)
- ...
- New unknown type object (N)
- New `setelem A` (kmalloc-cg-256)
- Delete all dynset expressions (deactivate all pre-allocated setelems)
- New unknown type object (N+1)

By extending the two race windows mentioned above, we should be able to reliably trigger the race condition and cause a double free, right? 

Actually, **no**. We were surprised to find that even if we extend the processing time of pre-allocated set elements to the scale of seconds, 
`nft_rhash_gc()` still doesn't race with `__nf_tables_abort()`.

We later found that for some reason, `nft_rhash_gc()` will not be scheduled by `system_power_efficient_wq` during high CPU usage.

```c
static void nft_rhash_gc_init(const struct nft_set *set)
{
    struct nft_rhash *priv = nft_set_priv(set);

    queue_delayed_work(system_power_efficient_wq, &priv->gc_work,
        nft_set_gc_interval(set));
}
```

In our case, re-activating all pre-allocated set elements 
in `__nf_tables_abort()` will cause high CPU usage, thus `nft_rhash_gc()` will not be scheduled.
To solve this problem, we switch the main thread to a different CPU using `set_cpu()` before the race. 
Additionally, this provides a bonus: the slab allocator will not detect our double free because the same object is freed by two different CPUs.

Once we can reliably trigger the race to cause double free, the free list in kmalloc-cg-256 will be `[A, A]`.

## KASLR Bypass

After obtaining the double free primitive, I used the same exploit method as in [CVE-2023-4004](https://github.com/google/security-research/blob/master/pocs/linux/kernelctf/CVE-2023-4004_lts_cos_mitigation/docs/exploit.md). 
Since there are now two elements A that have been double freed in the kmalloc-cg-256 free list, we can overlap `nft_table`'s `table->udata` with the `nft_object` object to leak `obj->ops` (address of `nft_ct_expect_obj_ops`).
- New `table A` (with `NFTA_TABLE_USERDATA` data length equals 256)
- New `object B` (`nft_ct_expect_obj`)
- Dump `table A` (leaking `object B` structure)

After leaking the kernel address, restore the free list state to `[A, B, A]` to facilitate subsequent operations.

## Control RIP

At this stage, we again overlap `nft_table`'s `table->udata` with the `nft_object` object to control the `obj->ops` function table pointer, thereby controlling the RIP.

We will first leak kernel heap address which we used to store fake `obj->ops` function pointer table.
- New `table A` (with `NFTA_TABLE_USERDATA` data length equals 256)
- New `table B` (with `NFTA_TABLE_USERDATA` data length equals 256)
- New `object C` (providing `NFTA_OBJ_USERDATA`, later used for faking `obj->ops`)
- Dump `table A` (leaking `obj->udata`)

Then reallocate table to modify the overlapped `object C` and call to `obj->ops->dump` to trigger ROP
- Delete `table A`
- New `table D` (setting `obj->ops` to `obj->udata`, setting ROP chain)
- Dump `object C` (triggering ROP chain)

## Container Escape

We reuse the the exploit technique from [CVE-2023-4622](https://github.com/google/security-research/blob/master/pocs/linux/kernelctf/CVE-2023-4622_lts/docs/exploit.md#achieve-container-escape). 

By rewriting `core_pattern` to `|/proc/%P/fd/<fd number>` and placing the binary in the corresponding fd via `memfd_create()`, 
we can execute any binary outside the container when a coredump is triggered.
