#!/usr/bin/env python3
from time import sleep
import subprocess
import nclib
import sys
import os

QEMU_SERIAL_PORT=1337
QEMU_MONITOR_PORT=1338

def error(message, vm):
    os.killpg(os.getpgid(vm.pid), 9)
    print(message)
    exit(1)

def start_vm(vmlinuz_path, arch):
    if arch in ("x86", "X86", "x86_64", "amd64"):
        args = [
            "qemu-system-x86_64",
            "-m", "4G",
            "-nographic",
            "-cpu", "max",
            "-kernel", vmlinuz_path,
            "-initrd", "./rootfs/initramfs-x86.cpio.gz",
            "-append", "\"root=/dev/sda rw console=ttyS0 panic=0 quiet net.ifnames=0 nokaslr no5lvl\"",
            f"-serial", f"tcp::{QEMU_SERIAL_PORT},server",
            f"-monitor", f"tcp::{QEMU_MONITOR_PORT},server,nowait",
        ]
    elif arch in ("aarch64", "arm64", "arm"):
        args = [
            "qemu-system-aarch64",
            "-m", "4G",
            "-nographic",
            "-machine", "virt",
            "-cpu", "max",
            "-kernel", vmlinuz_path,
            "-initrd", "./rootfs/initramfs-arm.cpio.gz",
            "-append", "\"root=/dev/vda rw quiet nokaslr\"",
            f"-serial", f"tcp::{QEMU_SERIAL_PORT},server",
            f"-monitor", f"tcp::{QEMU_MONITOR_PORT},server,nowait",
            "-bios /usr/share/qemu-efi-aarch64/QEMU_EFI.fd", # Fedora ships PE32+ EFI apps instead of compressed kernel images, they will not boot without this
        ]
    else:
        print("invalid arch:", arch)
        exit(1)

    print(args)

    vm = subprocess.Popen(
        " ".join(args),
        stdin=subprocess.PIPE,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
        preexec_fn=os.setsid,
        shell=True
    )
    return vm

def process_serial_out(vm):
    serial = nclib.Netcat(('localhost', QEMU_SERIAL_PORT), retry=10)
    serial_out = serial.recvuntil('~ #', timeout=20)

    text_addr = None
    etext_addr = None

    print("OUT:", serial_out)

    for line in serial_out.splitlines():
        if line.endswith(b"_stext") or line.endswith(b"_text"):
            text_addr = int(line.split()[0], 16)
        if line.endswith(b"_etext"):
            etext_addr = int(line.split()[0], 16)

    if text_addr is None or etext_addr is None:
        error("Could not find one of '_text' or '_etext' in boot log", vm)

    return text_addr, etext_addr

def extract_memory(start_addr, end_addr, path):
    monitor = nclib.Netcat(('localhost', QEMU_MONITOR_PORT), retry=10)
    monitor.recvuntil('(qemu)')
    monitor.sendline(f"memsave {start_addr} {end_addr-start_addr} {path}")
    #monitor.recvuntil('(qemu)')
    print(monitor.recvall(timeout=10).decode())

def usage():
    print("create_dump.py <vmlinuz_path> <arch>")
    exit(1)

def main(argc, argv):
    if argc != 3:
        usage()
    vmlinuz_path = argv[1]
    arch = argv[2]

    vm = start_vm(vmlinuz_path, arch)

    text_addr, etext_addr = process_serial_out(vm)

    # print("addrs:", hex(text_addr), hex(etext_addr))
    extract_memory(text_addr, etext_addr, "vmemdump")
    print("Memory dump written to ./vmemdump")
    os.killpg(os.getpgid(vm.pid), 9)


if __name__ == "__main__":
    main(len(sys.argv), sys.argv)
