#!/usr/bin/env python3

from IPython.display import display
from multiprocessing import Pool
import angrop.rop_utils as rop_utility
from IPython import embed
from capstone import *
import angr, angrop
import pandas as pd
import sys
import os

def analyze_dump(task):
    csv_path = task['csv_path']
    dump_path = task['dump_path']
    out_path = task['out_path']

    if os.path.exists(out_path):
        print("Already analyzed:", csv_path)
        return
    else:
        print("Analyzing:", csv_path)

    proj = angr.Project(dump_path, main_opts={'backend': 'blob', 'arch': 'aarch64'}, auto_load_libs=False)
    rop = proj.analyses.ROP(fast_mode=False, only_check_near_rets=False, max_block_size=800)

    df = pd.read_csv(csv_path)

    addrs = [int(a, 16) - 0xffff800080010000 for a in df['address']]
    #print(addrs)

    res_validity = []
    res_asm = []
    for addr in addrs:
        print(hex(addr))

        # Initialize the disassembler
        md = Cs(CS_ARCH_ARM64, CS_MODE_ARM)

        #print("==========================================")
        # Disassemble the code
        insn = next(md.disasm(proj.loader.memory.load(addr, 16), 16))
        #print(f"0x{i.address:x}:\t{i.mnemonic}\t{i.op_str}")
        #print("==========================================")

        try:
            g = rop.gadget_finder.gadget_analyzer.analyze_gadget(addr + insn.size, allow_conditional_branches=True)
        except Exception as e:
            res_asm += [""]
            res_validity += [False]
            print(e)
            continue
        #if len(g) > 0:
        #    embed()


        if len(g) > 0:
            print(hex(addr), f"{insn.mnemonic}\t{insn.op_str}", rop_utility.gadget_to_asmstring(project=proj, gadget=g[0]))
            res_asm += [f"{insn.mnemonic} {insn.op_str}; {rop_utility.gadget_to_asmstring(project=proj, gadget=g[0])}"]
            res_validity += [True]
        else:
            res_asm += [""]
            res_validity += [False]

    df['validity'] = res_validity
    df['asm'] = res_asm
    display(df)
    df.to_csv(out_path)

def main(argc, argv):
    csvs_path = argv[1]
    dumps_path = argv[2]
    out_path = argv[3]

    tasks = []
    for dump_file in os.listdir(dumps_path):
        dump_path = os.path.join(dumps_path, dump_file)
        csv_path = csvs_path + '/' + dump_file + '.csv'
        new_csv_path = out_path + '/' + dump_file + '_new.csv'
        tasks += [{'csv_path': csv_path, 'dump_path': dump_path, 'out_path': new_csv_path}]
        #print("ANALYZING:", csv_path)
        #analyze_dump(csv_path, dump_path, new_csv_path)

    print(tasks)
    #analyze_dump(tasks[0])
    with Pool(2) as p:
        p.map(analyze_dump, tasks)

if __name__ == "__main__":
    main(len(sys.argv), sys.argv)
