#!/bin/bash

set -x

JSON_FILE=$1
ARCH=$2

usage() {
  echo "Usage: $0 <JSON_FILE> <ARCH>"
  echo "  JSON_FILE: Path to a JSON file containing kernel sources (one of the files from ./kernel_sources)."
  echo "  ARCH: Architecture to pull images for ('x86' or 'arm')."
  exit 1
}

if [[ -z "$JSON_FILE" ]]; then
  echo "Error: JSON_FILE is required."
  usage
elif [[ ! -f "$JSON_FILE" ]]; then
  echo "Error: JSON_FILE '$JSON_FILE' does not exist."
  exit 1
elif [[ ! -r "$JSON_FILE" ]]; then
  echo "Error: JSON_FILE '$JSON_FILE' is not readable."
  exit 1
fi

if [[ -z "$ARCH" ]]; then
  echo "Error: ARCH is required."
  usage
elif [[ "$ARCH" != "x86" && "$ARCH" != "arm" ]]; then
  echo "Error: ARCH must be 'x86' or 'arm'."
  exit 1
fi

# Directories for downloads and extracted files
DOWNLOAD_DIR=$ARCH"_out/kernel_images"
VMLINUZ_DIR=$ARCH"_out/extracted_vmlinuz"
VMLINUX_DIR=$ARCH"_out/extracted_vmlinux"
DUMP_DIR=$ARCH"_out/dumps"

mkdir -p "$DOWNLOAD_DIR" "$VMLINUZ_DIR" "$VMLINUX_DIR" "$DUMP_DIR"

# Check for required tools
for tool in jq wget rpm2cpio cpio dpkg-deb; do
    if ! command -v "$tool" &>/dev/null; then
        echo "$tool is required but not installed. Please install $tool and try again."
        exit 1
    fi
done

# Function to extract vmlinuz from an RPM package
extract_rpm() {
    local rpm_file="$1"
    local dest_dir="$2"
    local base_name=$(basename "$rpm_file" .rpm)
    local vmlinuz_out_path="$VMLINUZ_DIR/${base_name}_vmlinuz"
    # local vmlinux_out_path="$VMLINUX_DIR/${base_name}_vmlinux"
    local dump_out_path="$DUMP_DIR/${base_name}_dump"

    echo "BASE_NAME $base_name"

    if [[ ! -f "$vmlinuz_out_path" ]]; then
        echo "Extracting vmlinuz from $rpm_file..."
        rpm2cpio "$rpm_file" | cpio -idmv -D "$dest_dir" 2>/dev/null

        find "$dest_dir" -name "vmlinuz*" -exec mv {} $vmlinuz_out_path \;
    else
        echo "File $vmlinuz_out_path already exists, skipping extraction."
    fi

    # if [[ ! -f "$vmlinux_out_path" ]]; then
    #     vmlinux-to-elf $vmlinuz_out_path $vmlinux_out_path
    # else
    #     echo "File $vmlinux_out_path already exists, skipping vmlinux-to-elf."
    # fi

    if [[ ! -f "$dump_out_path" ]]; then
        ./scripts/create_dump.py $vmlinuz_out_path $ARCH
        mv ./vmemdump $dump_out_path
    fi
}

# Function to extract vmlinuz from a DEB package
extract_deb() {
    local deb_file="$1"
    local dest_dir="$2"
    local base_name=$(basename "$deb_file" .deb)
    local vmlinuz_out_path="$VMLINUZ_DIR/${base_name}_vmlinuz"
    # local vmlinux_out_path="$VMLINUX_DIR/${base_name}_vmlinux"
    local dump_out_path="$DUMP_DIR/${base_name}_dump"

    echo "BASE_NAME $base_name"

    if [[ ! -f "$vmlinuz_out_path" ]]; then
        echo "Extracting vmlinuz from $deb_file..."
        dpkg-deb -x "$deb_file" "$dest_dir"

        find "$dest_dir" -name "vmlinuz*" -exec mv {} "$vmlinuz_out_path" \;
    else
        echo "File $vmlinuz_out_path already exists, skipping extraction."
    fi

    # if [[ ! -f "$vmlinux_out_path" ]]; then
    #     vmlinux-to-elf $vmlinuz_out_path $vmlinux_out_path
    # else
    #     echo "File $vmlinux_out_path already exists, skipping vmlinux-to-elf."
    # fi

    if [[ ! -f "$dump_out_path" ]]; then
        ./scripts/create_dump.py $vmlinuz_out_path $ARCH
        mv ./vmemdump $dump_out_path
    fi
}

# Process JSON data and extract kernel images
echo "Starting download and extraction of kernel images..."
jq -r 'to_entries[] | [.key][]' "$JSON_FILE" | while read -r distro; do
    echo "Processing distribution: $distro"

    jq -r ".\"$distro\"[]" "$JSON_FILE" | while read -r url; do
        filename=$(basename "$url")
        dest_file="$DOWNLOAD_DIR/$filename"

        # Download the file if it doesn't already exist
        if [[ ! -f "$dest_file" ]]; then
            echo "Downloading $url..."
            wget -P "$DOWNLOAD_DIR" "$url"
        else
            echo "File $filename already exists, skipping download."
        fi

        # Extract the package based on its format
        temp_dir=$(mktemp -d)
        case "$filename" in
            *.rpm)
                extract_rpm "$dest_file" "$temp_dir"
                ;;
            *.deb)
                extract_deb "$dest_file" "$temp_dir"
                ;;
            *)
                echo "Unsupported file format for $filename, skipping extraction."
                ;;
        esac
        rm -rf "$temp_dir"

    done
done

echo "Extraction completed. vmlinuz files saved in $VMLINUZ_DIR. vmlinux files saved in $VMLINUX_DIR."

