"""Environment variables used by the TAPML."""

import os
import sys
from pathlib import Path
from typing import List

TAPML_CHAT_CONFIG_VERSION = "0.1.0"


def _check():
    if TAPML_JIT_POLICY not in ["ON", "OFF", "REDO", "READONLY"]:
        raise ValueError(
            'Invalid TAPML_JIT_POLICY. It has to be one of "ON", "OFF", "REDO", "READONLY"'
            f"but got {TAPML_JIT_POLICY}."
        )

    if TAPML_DOWNLOAD_CACHE_POLICY not in ["ON", "OFF", "REDO", "READONLY"]:
        raise ValueError(
            "Invalid TAPML_AUTO_DOWNLOAD_POLICY. "
            'It has to be one of "ON", "OFF", "REDO", "READONLY"'
            f"but got {TAPML_DOWNLOAD_CACHE_POLICY}."
        )


def _get_cache_dir() -> Path:
    if "TAPML_HOME" in os.environ:
        result = Path(os.environ["TAPML_HOME"])
    elif sys.platform == "win32":
        result = Path(os.environ["LOCALAPPDATA"])
        result = result / "tapml"
    elif os.getenv("XDG_CACHE_HOME", None) is not None:
        result = Path(os.getenv("XDG_CACHE_HOME"))
        result = result / "tapml"
    else:
        result = Path(os.path.expanduser("~/.cache"))
        result = result / "tapml"
    result.mkdir(parents=True, exist_ok=True)
    if not result.is_dir():
        raise ValueError(
            f"The default cache directory is not a directory: {result}. "
            "Use environment variable TAPML_HOME to specify a valid cache directory."
        )
    (result / "model_weights").mkdir(parents=True, exist_ok=True)
    (result / "model_lib").mkdir(parents=True, exist_ok=True)
    return result


def _get_dso_suffix() -> str:
    if "TAPML_DSO_SUFFIX" in os.environ:
        return os.environ["TAPML_DSO_SUFFIX"]
    if sys.platform == "win32":
        return "dll"
    if sys.platform == "darwin":
        return "dylib"
    return "so"


def _get_test_model_path() -> List[Path]:
    paths = []
    if "TAPML_TEST_MODEL_PATH" in os.environ:
        paths += [Path(p) for p in os.environ["TAPML_TEST_MODEL_PATH"].split(os.pathsep)]
    # by default, we reuse the cache dir via tapml chat
    # note that we do not auto download for testcase
    # to avoid networking dependencies
    base_list = ["hf"]
    paths += [_get_cache_dir() / "model_weights" / base / "tapml-ai" for base in base_list] + [
        Path(os.path.abspath(os.path.curdir)),
        Path(os.path.abspath(os.path.curdir)) / "dist",
    ]
    return paths


def _get_read_only_weight_caches() -> List[Path]:
    if "TAPML_READONLY_WEIGHT_CACHE" in os.environ:
        return [Path(p) for p in os.environ["TAPML_READONLY_WEIGHT_CACHE"].split(os.pathsep)]
    return []


TAPML_TEMP_DIR = os.getenv("TAPML_TEMP_DIR", None)
TAPML_MULTI_ARCH = os.environ.get("TAPML_MULTI_ARCH", None)
TAPML_JIT_POLICY = os.environ.get("TAPML_JIT_POLICY", "ON")
TAPML_DSO_SUFFIX = _get_dso_suffix()
TAPML_TEST_MODEL_PATH: List[Path] = _get_test_model_path()

TAPML_DOWNLOAD_CACHE_POLICY = os.environ.get("TAPML_DOWNLOAD_CACHE_POLICY", "ON")
TAPML_HOME: Path = _get_cache_dir()
TAPML_READONLY_WEIGHT_CACHE = _get_read_only_weight_caches()

_check()
