"""Command line entrypoint of package."""

import os
from pathlib import Path
from typing import Union

from tapml.interface.help import HELP
from tapml.interface.package import package
from tapml.support.argparse import ArgumentParser


def main(argv):
    """Parse command line arguments and call `tapml.interface.package`."""
    parser = ArgumentParser("TAPML Package CLI")

    def _parse_package_config(path: Union[str, Path]) -> Path:
        path = Path(path)
        if not path.exists():
            raise ValueError(
                f"Path {str(path)} is expected to be a JSON file, but the file does not exist."
            )
        if not path.is_file():
            raise ValueError(f"Path {str(path)} is expected to be a JSON file.")
        return path

    def _parse_tapml_source_dir(path: str) -> Path:
        os.environ["TAPML_SOURCE_DIR"] = path
        return Path(path)

    def _parse_output(path: Union[str, Path]) -> Path:
        path = Path(path)
        if not path.is_dir():
            path.mkdir(parents=True, exist_ok=True)
        return path

    parser.add_argument(
        "--package-config",
        type=_parse_package_config,
        default="tapml-package-config.json",
        help=HELP["config_package"] + ' (default: "%(default)s")',
    )
    parser.add_argument(
        "--tapml-source-dir",
        type=_parse_tapml_source_dir,
        default=os.environ.get("TAPML_SOURCE_DIR", None),
        help=HELP["tapml_source_dir"]
        + " (default: the $TAPML_SOURCE_DIR environment variable)",
    )
    parser.add_argument(
        "--output",
        "-o",
        type=_parse_output,
        default="dist",
        help=HELP["output_package"] + ' (default: "%(default)s")',
    )
    parsed = parser.parse_args(argv)
    if parsed.tapml_source_dir is None:
        raise ValueError(
            "TAPML home is not specified. "
        )
    package(
        package_config_path=parsed.package_config,
        tapml_source_dir=parsed.tapml_source_dir,
        output=parsed.output,
    )
