"use client"

import type React from "react"
import { useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Clock, Loader2, CheckCircle, XCircle } from "lucide-react";

const models = [
  { id: "google/gemini-2.0-flash-001", name: "Google: Gemini Flash 2.0" },
  { id: "meta-llama/llama-3.3-70b-instruct", name: "Meta: Llama 3.3 70B Instruct" },
  { id: "openai/gpt-4o-mini", name: "GPT-4oOpenAI: GPT-4o-mini" },
  { id: "anthropic/claude-3.5-sonnet", name: "Anthropic: Claude 3.5 Sonnet" },
]

interface QuestionStatus {
  question: string
  status: 'pending' | 'processing' | 'completed' | 'error'
}

interface ModelResult {
  modelId: string
  modelName: string
  response: string
  error?: string
}

interface BatchResult {
  question: string
  results: ModelResult[]
}

interface ExecutionResult {
  originalPrompt: string
  starPrompt: string
  originalResponse: string
  starResponse: string
  modelName: string
  error?: string
}

interface BatchExecutionResult {
  question: string
  executions: ExecutionResult[]
}

const StatusIcon = ({ status }: { status: QuestionStatus['status'] }) => {
  switch (status) {
    case "pending": return <Clock className="w-5 h-5 text-yellow-500" />
    case "processing": return <Loader2 className="w-5 h-5 animate-spin" />
    case "completed": return <CheckCircle className="w-5 h-5 text-green-500" />
    case "error": return <XCircle className="w-5 h-5 text-red-500" />
  }
}

export default function Home() {
  const [input, setInput] = useState("")
  const [isLoading, setIsLoading] = useState(false)
  const [error, setError] = useState<string | null>(null)
  const [questionStatuses, setQuestionStatuses] = useState<QuestionStatus[]>([])
  const [selectedModels, setSelectedModels] = useState<string[]>([models[0].id])
  const [batchResults, setBatchResults] = useState<BatchResult[]>([])
  const [executionResults, setExecutionResults] = useState<BatchExecutionResult[]>([])

  const executePrompt = async (prompt: string, modelId: string, isDirectExecution: boolean) => {
    const response = await fetch("/api", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ prompt, model: modelId, isDirectExecution }),
    })

    const data = await response.json()

    if (!response.ok) {
      throw new Error(data.error || `Error: ${response.status}`)
    }

    return data.text || "No response"
  }

  const processQuestions = async (questions: string[]) => {
    const results: BatchResult[] = []
    const executionResults: BatchExecutionResult[] = []

    setQuestionStatuses(questions.map(q => ({
      question: q,
      status: 'pending'
    })))

    for (let i = 0; i < questions.length; i++) {
      const question = questions[i]

      setQuestionStatuses(prev => prev.map((status, idx) =>
        idx === i ? { ...status, status: 'processing' } : status
      ))

      const result: BatchResult = {
        question,
        results: []
      }

      const executionResult: BatchExecutionResult = {
        question,
        executions: []
      }

      try {
        for (const modelId of selectedModels) {
          const model = models.find(m => m.id === modelId)
          if (!model) continue

          try {
            // Get STAR formatted prompt
            const starPrompt = await executePrompt(question, modelId, false)

            // Execute original prompt
            const originalResponse = await executePrompt(question, modelId, true)

            // Execute STAR formatted prompt
            const starResponse = await executePrompt(starPrompt, modelId, true)

            result.results.push({
              modelId,
              modelName: model.name,
              response: starPrompt
            })

            executionResult.executions.push({
              originalPrompt: question,
              starPrompt,
              originalResponse,
              starResponse,
              modelName: model.name
            })
          } catch (error) {
            result.results.push({
              modelId,
              modelName: model.name,
              response: "",
              error: error instanceof Error ? error.message : "Unknown error occurred"
            })

            executionResult.executions.push({
              originalPrompt: question,
              starPrompt: "",
              originalResponse: "",
              starResponse: "",
              modelName: model.name,
              error: error instanceof Error ? error.message : "Unknown error occurred"
            })
          }
        }

        results.push(result)
        executionResults.push(executionResult)

        setQuestionStatuses(prev => prev.map((status, idx) =>
          idx === i ? { ...status, status: 'completed' } : status
        ))
      } catch (error) {
        setError(error instanceof Error ? error.message : "Unknown error occurred")
        setQuestionStatuses(prev => prev.map((status, idx) =>
          idx === i ? { ...status, status: 'error' } : status
        ))
      }
    }

    setBatchResults(results)
    setExecutionResults(executionResults)
    return results
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!input.trim() || selectedModels.length === 0) return

    setIsLoading(true)
    setError(null)
    setBatchResults([])

    const prompts = input.split('\n').filter(line => line.trim())
    await processQuestions(prompts)
    setIsLoading(false)
  }

  const formattedResults = batchResults.map(result =>
    `Prompt: ${result.question}\n\n${result.results
      .map(r => `${r.modelName}: ${r.error || r.response}`)
      .join('\n\n')}`
  ).join('\n\n---\n\n')

  const formattedExecutionResults = executionResults.map(result =>
    `Prompt: ${result.question}\n\n${result.executions
      .map(e => `${e.modelName}:\nOriginal Response: ${e.error || e.originalResponse}\n\nSTAR Response: ${e.error || e.starResponse}`)
      .join('\n\n')}`
  ).join('\n\n---\n\n')

  return (
    <Card className="max-w-2xl mt-10 mx-auto border-zinc-300 bg-white shadow-lg">
      <CardHeader>
        <CardTitle className="text-3xl font-bold">STAR-Prompting Converter
          <span className="text-[10px] border rounded-md px-2 py-1 bg-zinc-900 text-white">v1.0.0</span>
        </CardTitle>
        <p className="text-xs font-light">developed within the scope part of the paper: <i>&quot;STAR-Prompting: Sistema Estruturado de Prompts através do método STAR aplicados para a Defesa e Segurança&quot;</i></p>
      </CardHeader>
      <CardContent>
        <form onSubmit={handleSubmit}>
          <div className="flex flex-col">
            {models.map((model) => (
              <label key={model.id} className="flex flex-row gap-2 mb-1 align-middle font-light">
                <input
                  type="checkbox"
                  className="mr-2 accent-zinc-900"
                  checked={selectedModels.includes(model.id)}
                  onChange={(e) => {
                    if (e.target.checked) {
                      setSelectedModels([...selectedModels, model.id])
                    } else {
                      setSelectedModels(selectedModels.filter(id => id !== model.id))
                    }
                  }}
                />
                {model.name}
              </label>
            ))}
          </div>

          <div className="flex flex-row gap-2 mt-2 align-middle font-light text-xs">
            <i>
              &quot;Você é um especialista em engenharia de prompts e sua função é transformar
              perguntas simples em instruções estruturadas no formato STAR - Situação, Tarefa, Ação, Resultado,
              mas apresentadas de forma contínua em um único parágrafo. Certifique-se de preservar o contexto
              original e garantir que a reformulação seja clara, objetiva e bem organizada. Reescreva o seguinte
              prompt de maneira estruturada, utilizando o método STAR sem mencionar explicitamente seus títulos.
              Formule o texto começando com &lsquo;Você é...&rsquo; para estabelecer a situação, seguido de &lsquo;Você precisa...&rsquo;
              para detalhar a tarefa, &lsquo;Faça...&rsquo; para descrever a ação necessária e finalize com &lsquo;Como resultado,
              quero...&rsquo; para especificar o resultado desejado. Aqui está o prompt original: <strong>$prompt</strong>.
              Retorne apenas o prompt reformulado, sem explicações adicionais.&quot;
            </i>
          </div>

          <textarea
            className="w-full p-2 mt-4 border border-zinc-300 rounded-md font-light"
            placeholder="Write your prompts here, one per line..."
            rows={6}
            value={input}
            onChange={(e) => setInput(e.target.value)}
          />

          <div>
            <button
              type="submit"
              className="p-2 px-8 mt-4 border border-zinc-300 rounded-md bg-zinc-900 text-white"
              disabled={isLoading || !input.trim()}
            >
              {isLoading ? "Processing..." : "Process"}
            </button>
          </div>
        </form>

        {error && (
          <div className="text-red-500 mt-4 font-bold">{error}</div>
        )}

        {questionStatuses.length > 0 && (
          <div className="flex flex-col mt-4 mb-4">
            {questionStatuses.map((status, index) => (
              <div key={index} className="flex flex-row gap-2 mb-1 align-middle">
                <StatusIcon status={status.status} />
                <div className="text-1xl font-bold">Prompt {index + 1}: {status.question}</div>
              </div>
            ))}
          </div>
        )}

        {formattedResults && !error && (
          <div className="mt-8">
            <h2 className="text-xl font-bold mb-4">STAR-Prompting Applied:</h2>
            <div className="flex flex-col border border-zinc-300 rounded-lg bg-zinc-50 font-light text-xs p-4 whitespace-pre-wrap">{formattedResults}</div>
          </div>
        )}

        {formattedExecutionResults && !error && (
          <div className="mt-8">
            <h2 className="text-xl font-bold mb-4">Execution Results:</h2>
            <div className="flex flex-col border border-zinc-300 rounded-lg bg-zinc-50 font-light text-xs p-4 whitespace-pre-wrap">{formattedExecutionResults}</div>
          </div>
        )}
      </CardContent>
    </Card>
  );
}
