import { NextResponse, NextRequest } from "next/server";

const API_URL = "https://openrouter.ai/api/v1/chat/completions";
const apiKey = process.env.OPENROUTER_API_KEY;

export async function POST(req: NextRequest) {
  try {
    const { prompt, model = "anthropic/claude-3-opus:beta", isDirectExecution = false } = await req.json();
    
    if (!prompt) return NextResponse.json({ error: "Prompt inválido" }, { status: 400 });
    if (!apiKey) return NextResponse.json({ error: "Configuração de API ausente" }, { status: 500 });

    const requestBody = {
      model,
      messages: isDirectExecution 
        ? [{ role: "user", content: prompt }]
        : [
            { 
              role: "system", 
              content: "Você é um especialista em engenharia de prompts e sua função é transformar perguntas simples em instruções estruturadas no formato STAR (Situação, Tarefa, Ação, Resultado), mas apresentadas de forma contínua em um único parágrafo. Certifique-se de preservar o contexto original e garantir que a reformulação seja clara, objetiva e bem organizada." 
            },
            { 
              role: "user", 
              content: `Reescreva o seguinte prompt de maneira estruturada, utilizando o método STAR sem mencionar explicitamente seus títulos. Formule o texto começando com 'Você é...' para estabelecer a situação, seguido de 'Você precisa...' para detalhar a tarefa, 'Faça...' para descrever a ação necessária e finalize com 'Como resultado, quero...' para especificar o resultado esperado. Aqui está o prompt original:\n${prompt}. Retorne apenas o prompt reformulado, sem explicações adicionais.` 
            },
          ],
      temperature: 0.7,
      max_tokens: 1000,
    };

    const response = await fetch(API_URL, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        Authorization: `Bearer ${apiKey}`,
      },
      body: JSON.stringify(requestBody),
    });

    const data = await response.json();

    return response.ok
      ? NextResponse.json({ text: data.choices?.[0]?.message?.content || "" })
      : NextResponse.json({ error: data.error?.message || "Erro na API" }, { status: response.status });

  } catch (error) {
    return NextResponse.json({ error: error instanceof Error ? error.message : "Erro interno" }, { status: 500 });
  }
}
