library(tidyverse)
library(ggplot2)
library(ggpubr)
library(viridis)
library(readr)


data <- read_csv("data_cleaned.csv")

##### Figure 1 Profiles ##### 
# Extract the dive profile columns
dive_profile_columns <- grep("^diveprofile_", colnames(data), value = TRUE)

# Create a data frame in long format for all dives
long_format_data <- data %>%
  select(person_id, measure_id,bubble_grade, starts_with("diveprofile_")) %>%
  pivot_longer(cols = starts_with("diveprofile_"), 
               names_to = "Time", 
               values_to = "Depth", 
               names_prefix = "diveprofile_") %>%
  mutate(Time = as.numeric(sub("_min", "", Time)))


# Add the zero row for each unique combination of person_id and measure_id
data_0 <- long_format_data %>%
  group_by(person_id, measure_id,bubble_grade) %>%
  summarise(Time = 0, Depth = 0) %>%  ungroup()

plot_data = rbind(long_format_data,data_0)


# Define custom labels for the facets
custom_labels <- c("0" = "EB Grade 0", "1" = "EB Grade 1", "2" = "EB Grade 2", "3" = "EB Grade 3", "4" = "EB Grade 4", "5" = "EB Grade 5")

figure_1 = ggplot(plot_data, aes(x = Time, y = Depth, group = measure_id, color = bubble_grade)) +
  geom_line(alpha = 0.3, size = 0.6) +  # Set line transparency to 70%
  labs(x = "divetime (minutes)", y = "depth (meters)", title = "Dive Profiles by EB Grade") +
  scale_y_reverse(limits = c(45, 0)) +  # Fix y-axis
  scale_color_viridis(discrete = FALSE, begin = 0.8, end = 0) +  # Use Viridis colors for EB grade
  theme_minimal() +
  theme(legend.position = "none",  # Remove the legend
        strip.text.y = element_text(angle = 270)) +  # Ensure labels are horizontal
  facet_wrap(~bubble_grade, ncol = 1, labeller = as_labeller(custom_labels), strip.position = "right")  # Create panels by EB grade with custom labels on the right

ggsave("figures/figure_1.jpg",plot = figure_1,units = "cm",height = 15,width = 15, dpi = 600)
figure_1
rm(list=ls())

