! Land-mask-dependent smoothing operators imply Neumann (no-flux) b.c.
! on the coastline from the water side: that is any point adjacent to
! the coast line is allowed to be changed without regard of the value
! of its nearest neighbor inside the land.   This is accomplished by
! applying the usual U- and V-type masking rules to the straight
! fluxes, FX and FE.  Masking for diagonal fluxes FX1 and FE1 imposes
! an additional condition: its end points must be both water points,
! AND MUST NOT BE SEPARATED BY LAND, e.g.,
!
!             FX1(i,j)=h(i,j)-h(i-1,j-1)
!
! is unmasked only if (1) both (i,j) and (i-1,j-1) are water points;
!                 AND (2) at least one of (i-1,j) or (i,j-1) is water
!                         as well -- thus there is exist a two-step
!                         90-degree passage from (i,j) to (i-1,j-1)
!                         by water points.
!
! This explains the somewhat awkward-looking expressions like
!
!         FX1(i,j)=(h(i,j)-h(i-1,j-1)) *dble( msk(i,j)*msk(i-1,j-1)
!    &                                 *max(msk(i-1,j),msk(i,j-1)) )

! Because ROMS relies to some degree on continuity of the topography
! across coastline (spatial interpolations may reach grid-box heights
! at least one point inside land), each routine below is accompanied
! by a land_smooth_something routine generated by CPP from nearly the
! same source code), which changes values inside land only.
! These routines use essentially Dirichlet-type boundary conditions:
! their fluxes are unmasked, and from the land-side point of view the
! points outside the coastline (water points nearest to the coastline)
! are fixed and used as source for Dirichlet side boundaries.
! So, as a rule of thumb, when processing water points h-values inside
! land have no influence; in contrast, points inside land are
! completely slaved to the adjacent water points.

! Without masking the stencil of the smoothing operators have the
! following weights, depending on parameter settings in the code below
!
!         1/8               1/32  1/8  1/32          1/16  1/8  1/16
!
!    1/8  1/2  1/8          1/8   3/8  1/8           1/8   1/4  1/8
!
!         1/8               1/32  1/8  1/32          1/16  1/8  1/16
!
!       5-point                isotropic             two-dimensional
!      Laplacian               Laplacian                  1-2-1
!       smoother               smoother                Hann window
!
! Note that all 5 or 9 coefficients add up to 1.  All three operators
! suppress the checkerboard mode in just a single iteration, however,
! only the last one eliminates flat-front 2dx-mode in one iteration;
! the first and the second attenuate 1D 2dx-mode by factors of 1/2 and
! 1/4 respectively.

!!! LOG_SMOOTHING !!!
# ifndef LAND_SMOOTH
      subroutine lsmooth_tile(istr,iend,jstr,jend, Lm,Mm, r_max,
     &                            msk, Lgh,Lgh_new, FX,FE,FX1,FE1)
# else
      subroutine land_lsmth_tile(istr,iend,jstr,jend, Lm,Mm, r_max,
     &                            msk, Lgh,Lgh_new, FX,FE,FX1,FE1)
# endif
      implicit none
      integer istr,iend,jstr,jend, Lm,Mm, i,j
      integer(kind=2), dimension(0:Lm+1,0:Mm+1) :: msk
      real(kind=8), dimension(0:Lm+1,0:Mm+1) :: Lgh,Lgh_new
      real(kind=8), dimension(istr-2:iend+2,jstr-2:jend+2) ::
     &                                          FX,FE,FX1,FE1
      real(kind=8) r_max, lgr_max, lgr1_max, grad
      real(kind=8), parameter :: OneEights=0.125D0

      if (r_max > 0.D0) then                      ! Set threshold
        lgr_max=log((1.D0+r_max)/(1.D0-r_max))    ! for logarithmic
        lgr1_max=lgr_max*sqrt(2.D0)               ! slopes
      else
        lgr_max=0.D0 ; lgr1_max=0.D0
      endif

      do j=jstr,jend
        do i=istr,iend+1
# ifndef LAND_SMOOTH
          grad=(Lgh(i,j)-Lgh(i-1,j))*dble(msk(i,j)*msk(i-1,j))
# else
          grad=Lgh(i,j)-Lgh(i-1,j)
# endif
          if (grad > lgr_max) then
            FX(i,j)=grad -lgr_max
          elseif (grad < -lgr_max) then
            FX(i,j)=grad +lgr_max
          else
            FX(i,j)=0.D0
          endif
        enddo
      enddo
      do j=jstr,jend+1
        do i=istr,iend
# ifndef LAND_SMOOTH
          grad=(Lgh(i,j)-Lgh(i,j-1))*dble(msk(i,j)*msk(i,j-1))
# else
          grad= Lgh(i,j)-Lgh(i,j-1)
# endif
          if (grad > lgr_max) then
            FE(i,j)=grad -lgr_max
          elseif (grad < -lgr_max) then
            FE(i,j)=grad +lgr_max
          else
            FE(i,j)=0.D0
          endif
        enddo
        do i=istr,iend+1
# ifndef LAND_SMOOTH
          grad=(Lgh(i,j)-Lgh(i-1,j-1)) *dble( msk(i,j)*msk(i-1,j-1)
     &                                 *max(msk(i-1,j),msk(i,j-1)) )
# else
          grad= Lgh(i,j)-Lgh(i-1,j-1)
# endif
          if (grad > lgr1_max) then
            FX1(i,j)=grad-lgr1_max
          elseif (grad < -lgr1_max) then
            FX1(i,j)=grad+lgr1_max
          else
            FX1(i,j)=0.D0
          endif
        enddo
        do i=istr-1,iend
# ifndef LAND_SMOOTH
          grad=(Lgh(i,j)-Lgh(i+1,j-1)) *dble( msk(i,j)*msk(i+1,j-1)
     &                                 *max(msk(i+1,j),msk(i,j-1)) )
# else
          grad= Lgh(i,j)-Lgh(i+1,j-1)
# endif
          if (grad > lgr1_max) then
            FE1(i,j)=grad-lgr1_max
          elseif (grad < -lgr1_max) then
            FE1(i,j)=grad+lgr1_max
          else
            FE1(i,j)=0.D0
          endif
        enddo
      enddo
      do j=jstr,jend
        do i=istr,iend
# ifndef LAND_SMOOTH
          if (msk(i,j) > 0) then !--> water points
# else
          if (msk(i,j) < 1) then !--> on land only
# endif
            Lgh_new(i,j)=Lgh(i,j) + OneEights*( FX(i+1,j)-FX(i,j)
     &                                          +FE(i,j+1)-FE(i,j)
     &                          +0.25D0*( FX1(i+1,j+1)-FX1(i,j)
     &                                   +FE1(i-1,j+1)-FE1(i,j) ))
          else
            Lgh_new(i,j)=Lgh(i,j)
          endif
        enddo
      enddo
      call neumann_bc_tile(istr,iend,jstr,jend, Lm,Mm,Lgh_new)
      end

! Same as above, but instead of explicitly computing diagonal fluxes 
! FX1,FE1 apply cross-averaging to 90-degree fluxes. Linear version of
! the resultant operator is equivalent to isotropic Laplacian, however
! r_max threshold is applied differently, and only to construct the
! initial 90-degree slops. 



!!! LOG_SMOOTH_2 !!!
# ifndef LAND_SMOOTH
      subroutine lsmooth_2_tile(istr,iend,jstr,jend, Lm,Mm, r_max,
     &                                 msk, Lgh,Lgh_new, FX,FE,FE1)
# else
      subroutine land_lsmth_2_tile(istr,iend,jstr,jend, Lm,Mm, r_max,
     &                                 msk, Lgh,Lgh_new, FX,FE,FE1)
# endif
      implicit none
      integer istr,iend,jstr,jend, Lm,Mm, i,j
      integer(kind=2), dimension(0:Lm+1,0:Mm+1) :: msk
      real(kind=8),    dimension(0:Lm+1,0:Mm+1) :: Lgh,Lgh_new
      real(kind=8), dimension(istr-2:iend+2,jstr-2:jend+2) :: FX,FE,FE1
      real(kind=8) r_max, lgr_max, grad
      real(kind=8), parameter :: ThreeSixteenth=3.D0/16.D0,
     &                               OneTwelfth=1.D0/12.D0

      if (r_max > 0.D0) then                      ! Set threshold
        lgr_max=log((1.D0+r_max)/(1.D0-r_max))    ! for logarithmic
      else                                        ! slopes
        lgr_max=0.D0
      endif

      do j=jstr-1,jend+1
        do i=istr,iend+1
# ifndef LAND_SMOOTH
          grad=(Lgh(i,j)-Lgh(i-1,j))*dble(msk(i,j)*msk(i-1,j))
# else
          grad= Lgh(i,j)-Lgh(i-1,j)
# endif
          if (grad > lgr_max) then
            FX(i,j)=grad -lgr_max
          elseif (grad < -lgr_max) then
            FX(i,j)=grad +lgr_max
          else
            FX(i,j)=0.D0
          endif
        enddo
      enddo
      do j=jstr,jend+1
        do i=istr-1,iend+1
# ifndef LAND_SMOOTH
          grad=(Lgh(i,j)-Lgh(i,j-1))*dble(msk(i,j)*msk(i,j-1))
# else
          grad= Lgh(i,j)-Lgh(i,j-1)
# endif
          if (grad > lgr_max) then
            FE1(i,j)=grad-lgr_max
          elseif (grad < -lgr_max) then
            FE1(i,j)=grad+lgr_max
          else
            FE1(i,j)=0.D0
          endif
        enddo
        do i=istr,iend
          FE(i,j)=FE1(i,j) + OneTwelfth*( FX(i+1,j)+FX(i  ,j-1)
     &                                   -FX(i  ,j)-FX(i+1,j-1))
        enddo
      enddo
      do j=jstr,jend
        do i=istr,iend+1
          FX(i,j)=FX(i,j) + OneTwelfth*( FE1(i,j+1)+FE1(i-1,j  )
     &                                  -FE1(i,j  )-FE1(i-1,j+1))
        enddo
        do i=istr,iend
# ifndef LAND_SMOOTH
          if (msk(i,j) > 0) then !--> water points
# else
          if (msk(i,j) < 1) then !--> on land only
# endif
            Lgh_new(i,j)=Lgh(i,j) +ThreeSixteenth*( FX(i+1,j)-FX(i,j)
     &                                             +FE(i,j+1)-FE(i,j))
          else
            Lgh_new(i,j)=Lgh(i,j)
          endif
        enddo
      enddo
      call neumann_bc_tile(istr,iend,jstr,jend, Lm,Mm,Lgh_new)
      end


! The same as above, but without diagonal fluxes.

!!! LOG_SMOOTH_1 !!!
# ifndef LAND_SMOOTH
      subroutine lsmooth_1_tile(istr,iend,jstr,jend, Lm,Mm, r_max,
     &                                      msk, Lgh,Lgh_new, FX,FE)
# else
      subroutine land_lsmth_1_tile(istr,iend,jstr,jend, Lm,Mm, r_max,
     &                                      msk, Lgh,Lgh_new, FX,FE)
# endif
      implicit none
      integer istr,iend,jstr,jend, Lm,Mm, i,j
      integer(kind=2), dimension(0:Lm+1,0:Mm+1) :: msk
      real(kind=8),    dimension(0:Lm+1,0:Mm+1) :: Lgh, Lgh_new
      real(kind=8), dimension(istr-2:iend+2,jstr-2:jend+2) :: FX,FE
      real(kind=8) r_max, lgr_max, grad
      real(kind=8), parameter :: OneEighth=0.125D0

      if (r_max > 0.D0) then                      ! Set threshold
        lgr_max=log((1.D0+r_max)/(1.D0-r_max))    ! for logarithmic
      else                                        ! slopes
        lgr_max=0.D0
      endif

      do j=jstr,jend+1
        do i=istr,iend
# ifndef LAND_SMOOTH
          grad=(Lgh(i,j)-Lgh(i,j-1))*dble(msk(i,j)*msk(i,j-1))
# else
          grad= Lgh(i,j)-Lgh(i,j-1)
# endif
          if (grad > lgr_max) then
            FE(i,j)=grad-lgr_max
          elseif (grad < -lgr_max) then
            FE(i,j)=grad+lgr_max
          else
            FE(i,j)=0.D0
          endif
        enddo
      enddo
      do j=jstr,jend
        do i=istr,iend+1
# ifndef LAND_SMOOTH
          grad=(Lgh(i,j)-Lgh(i-1,j))*dble(msk(i,j)*msk(i-1,j))
# else
          grad= Lgh(i,j)-Lgh(i-1,j)
# endif
          if (grad > lgr_max) then
            FX(i,j)=grad -lgr_max
          elseif (grad < -lgr_max) then
            FX(i,j)=grad +lgr_max
          else
            FX(i,j)=0.D0
          endif
        enddo
        do i=istr,iend
# ifndef LAND_SMOOTH
          if (msk(i,j) > 0) then !--> water points
# else
          if (msk(i,j) < 1) then !--> on land only
# endif
            Lgh_new(i,j)=Lgh(i,j) + OneEighth*( FX(i+1,j)-FX(i,j)
     &                                         +FE(i,j+1)-FE(i,j))
          else
            Lgh_new(i,j)=Lgh(i,j)
          endif
        enddo
      enddo
      call neumann_bc_tile(istr,iend,jstr,jend, Lm,Mm, Lgh_new)
      end




!!! LEGACY_LOG_SMOOTH !!!
# ifndef LAND_SMOOTH

! Mask-independent logarithmic r-factor limiting algorithm.  Somewhat
! similar to the one from RomsTools by Pierrick Penven, the code below
! uses both straight and diagonal fluxes arranged into isotropic
! Laplacian operator to counter the diamond-shaped spreading patterns
! of single point spikes.

      subroutine lsmth_legacy_tile(istr,iend,jstr,jend, Lm,Mm, r_max,
     &                                  Lgh,Lgh_new, FX,FE, FX1,FE1)
      implicit none
      integer istr,iend,jstr,jend, Lm,Mm, i,j
      real(kind=8), dimension(0:Lm+1,0:Mm+1) :: Lgh,Lgh_new
      real(kind=8), dimension(istr-2:iend+2,jstr-2:jend+2) ::
     &                                          FX,FE,FX1,FE1
      real(kind=8) r_max, lgr_max, lgr1_max, grad
      real(kind=8), parameter :: OneEights=0.125D0

      if (r_max > 0.D0) then                      ! Set threshold
        lgr_max=log((1.D0+r_max)/(1.D0-r_max))    ! for logarithmic
        lgr1_max=lgr_max*sqrt(2.D0)               ! slopes
      else
        lgr_max=0.D0 ; lgr1_max=0.D0
      endif

      do j=jstr,jend
        do i=istr,iend+1
          grad=Lgh(i,j)-Lgh(i-1,j)
          if (grad > lgr_max) then
            FX(i,j)=grad -lgr_max
          elseif (grad < -lgr_max) then
            FX(i,j)=grad +lgr_max
          else
            FX(i,j)=0.D0
          endif
        enddo
      enddo
      do j=jstr,jend+1
        do i=istr,iend
          grad=Lgh(i,j)-Lgh(i,j-1)
          if (grad > lgr_max) then
            FE(i,j)=grad -lgr_max
          elseif (grad < -lgr_max) then
            FE(i,j)=grad +lgr_max
          else
            FE(i,j)=0.D0
          endif
        enddo
        do i=istr,iend+1
          grad=Lgh(i,j)-Lgh(i-1,j-1)
          if (grad > lgr1_max) then
            FX1(i,j)=grad-lgr1_max
          elseif (grad < -lgr1_max) then
            FX1(i,j)=grad+lgr1_max
          else
            FX1(i,j)=0.D0
          endif
        enddo
        do i=istr-1,iend
          grad=Lgh(i,j)-Lgh(i+1,j-1)
          if (grad > lgr1_max) then
            FE1(i,j)=grad-lgr1_max
          elseif (grad < -lgr1_max) then
            FE1(i,j)=grad+lgr1_max
          else
            FE1(i,j)=0.D0
          endif
        enddo
      enddo
      do j=jstr,jend
        do i=istr,iend
          Lgh_new(i,j)=Lgh(i,j) + OneEights*( FX(i+1,j)-FX(i,j)
     &                                        +FE(i,j+1)-FE(i,j)
     &                        +0.25D0*( FX1(i+1,j+1)-FX1(i,j)
     &                                 +FE1(i-1,j+1)-FE1(i,j) ))
        enddo
      enddo
      call neumann_bc_tile(istr,iend,jstr,jend, Lm,Mm,Lgh_new)
      end
# endif

!!! ELSE !!!

! Physical-space (non-logarithmic) mask-dependent r-factor restriction
! algorithm.   It is somewhat suggested in Mellor, Ezer, and Oey (1994)
! "conundrum" article (page 1130, Fig. 6, text on the right, however no
! detail about the specific procedure are provided).  Algorithm below
! computes both 90-degree (FX,FE) and diagonal (FX1,FE1) "fluxes" as
! the excesses of gradient over the maximum allowed threshold, e.g.,
!
!       h(i,j)-h(i-1,j)
!      ----------------- > r_max    [assuming that h(i,j)-h(i-1,j) > 0]
!       h(i,j)+h(i-1,j)
!
! results in   FX(i,j) = h(i,j)-h(i-1,j) -r_max*[h(i,j)+h(i-1,j)]
!
! while FX(i,j) = 0 if   |h(i,j)-h(i-1,j)|/[h(i,j)+h(i-1,j)] < r_max
!
! so this procedure will not alter topography where the slopes are
! already within the limits.


# ifndef LAND_SMOOTH
      subroutine cond_smth_tile(istr,iend,jstr,jend, Lm,Mm, r_max,
     &                               msk, h,h_new, FX,FE, FX1,FE1)
# else
      subroutine land_cnd_smth_tile(istr,iend,jstr,jend, Lm,Mm,
     &                        r_max, msk, h,h_new, FX,FE, FX1,FE1)
# endif
      implicit none
      integer istr,iend,jstr,jend, Lm,Mm, i,j
      integer(kind=2), dimension(0:Lm+1,0:Mm+1) :: msk
      real(kind=8), dimension(0:Lm+1,0:Mm+1) :: h,h_new
      real(kind=8), dimension(istr-2:iend+2,jstr-2:jend+2)
     &                                   :: FX,FE, FX1,FE1
      real(kind=8) r_max, r1_max, grad, cr
      real(kind=8), parameter :: OneEights=0.125D0

      r1_max=r_max*sqrt(2.D0)

      do j=jstr,jend
        do i=istr,iend+1
# ifndef LAND_SMOOTH
          grad=(h(i,j)-h(i-1,j))*dble(msk(i,j)*msk(i-1,j))
# else
          grad= h(i,j)-h(i-1,j)
# endif
          cr=r_max*(h(i,j)+h(i-1,j))
          if (grad > cr) then
            FX(i,j)=grad -cr
          elseif (grad < -cr) then
            FX(i,j)=grad +cr
          else
            FX(i,j)=0.D0
          endif
        enddo
      enddo
      do j=jstr,jend+1
        do i=istr,iend
# ifndef LAND_SMOOTH
          grad=(h(i,j)-h(i,j-1))*dble(msk(i,j)*msk(i,j-1))
# else
          grad= h(i,j)-h(i,j-1)
# endif
          cr=r_max*(h(i,j)+h(i,j-1))
          if (grad > cr) then
            FE(i,j)=grad -cr
          elseif (grad < -cr) then
            FE(i,j)=grad +cr
          else
            FE(i,j)=0.D0
          endif
        enddo
        do i=istr,iend+1
# ifndef LAND_SMOOTH
          grad=(h(i,j)-h(i-1,j-1)) *dble( msk(i,j)*msk(i-1,j-1)
     &                             *max(msk(i-1,j),msk(i,j-1)) )
# else
          grad= h(i,j)-h(i-1,j-1)
# endif
          cr=r1_max*(h(i,j)+h(i-1,j-1))
          if (grad > cr) then
            FX1(i,j)=grad -cr
          elseif (grad < -cr) then
            FX1(i,j)=grad +cr
          else
            FX1(i,j)=0.D0
          endif
        enddo
        do i=istr-1,iend
# ifndef LAND_SMOOTH
          grad=(h(i,j)-h(i+1,j-1)) *dble( msk(i,j)*msk(i+1,j-1)
     &                             *max(msk(i+1,j),msk(i,j-1)) )
# else
          grad=h(i,j)-h(i+1,j-1)
# endif
          cr=r1_max*(h(i,j)+h(i+1,j-1))
          if (grad > cr) then
            FE1(i,j)=grad -cr
          elseif (grad < -cr) then
            FE1(i,j)=grad +cr
          else
            FE1(i,j)=0.D0
          endif
        enddo
      enddo
      do j=jstr,jend
        do i=istr,iend
# ifndef LAND_SMOOTH
          if (msk(i,j) > 0) then !--> water points
# else
          if (msk(i,j) < 1) then !--> on land only
# endif
            h_new(i,j)=h(i,j) + OneEights*( FX(i+1,j)-FX(i,j)
     &                                      +FE(i,j+1)-FE(i,j)
     &                      +0.25D0*( FX1(i+1,j+1)-FX1(i,j)
     &                               +FE1(i-1,j+1)-FE1(i,j) ))
          else
            h_new(i,j)=h(i,j)
          endif
        enddo
      enddo
      call neumann_bc_tile(istr,iend,jstr,jend, Lm,Mm,h_new)
      end
!#endif


#ifndef LAND_SMOOTH
      subroutine smooth_tile(istr,iend,jstr,jend, Lm,Mm, msk,
     &                                      h,h_new, FX,FE,FE1)
#else
      subroutine land_smth_tile(istr,iend,jstr,jend, Lm,Mm, msk,
     &                                      h,h_new, FX,FE,FE1)
#endif
      implicit none
      integer istr,iend,jstr,jend, Lm,Mm, i,j
      integer(kind=2), dimension(0:Lm+1,0:Mm+1) :: msk
      real(kind=8),    dimension(0:Lm+1,0:Mm+1) :: h,h_new
      real(kind=8), dimension(istr-2:iend+2,jstr-2:jend+2)
     &                                         :: FX,FE,FE1
      real(kind=8), parameter :: ThreeSixteenth=3.D0/16.D0,
     &                               OneTwelfth=1.D0/12.D0

      do j=jstr-1,jend+1
        do i=istr,iend+1
#ifndef LAND_SMOOTH
          FX(i,j)=(h(i,j)-h(i-1,j))*dble(msk(i,j)*msk(i-1,j))
#else
          FX(i,j)= h(i,j)-h(i-1,j)
#endif
        enddo
      enddo
      do j=jstr,jend+1
        do i=istr-1,iend+1
#ifndef LAND_SMOOTH
          FE1(i,j)=(h(i,j)-h(i,j-1))*dble(msk(i,j)*msk(i,j-1))
#else
          FE1(i,j)=h(i,j)-h(i,j-1)
#endif
        enddo
        do i=istr,iend
          FE(i,j)=FE1(i,j) + OneTwelfth*( FX(i+1,j)+FX(i  ,j-1)
     &                                   -FX(i  ,j)-FX(i+1,j-1))
        enddo
      enddo
      do j=jstr,jend
        do i=istr,iend+1
          FX(i,j)=FX(i,j) + OneTwelfth*( FE1(i,j+1)+FE1(i-1,j  )
     &                                  -FE1(i,j  )-FE1(i-1,j+1))
        enddo
        do i=istr,iend
#ifndef LAND_SMOOTH
          if (msk(i,j) > 0) then !--> water points
#else
          if (msk(i,j) < 1) then !--> on land only
#endif
            h_new(i,j)=h(i,j) + ThreeSixteenth*( FX(i+1,j)-FX(i,j)
     &                                          +FE(i,j+1)-FE(i,j))
          else
            h_new(i,j)=h(i,j)
          endif
        enddo
      enddo
      call neumann_bc_tile(istr,iend,jstr,jend, Lm,Mm,h_new)
      end

#ifndef LAND_SMOOTH
# define LAND_SMOOTH
# include "tools_topo.F"
#endif

