import copy
import subprocess
import gmsh
import os
from fiqus.pro_assemblers.ProAssembler import ASS_PRO as aP
from fiqus.data import DataFiQuS as dF
import fiqus.utils.Utils as utl


def check_for_event():  # pragma: no cover
    action = gmsh.onelab.getString("ONELAB/Action")
    if len(action) and action[0] == "check":
        gmsh.onelab.setString("ONELAB/Action", [""])
        gmsh.fltk.update()
        gmsh.graphics.draw()
    if len(action) and action[0] == "compute":
        gmsh.onelab.setString("ONELAB/Action", [""])
        gmsh.onelab.setChanged("Gmsh", 0)
        gmsh.onelab.setChanged("GetDP", 0)
        gmsh.fltk.update()
        gmsh.graphics.draw()
    return True


class SimpleThermalModel:
    def __init__(self,
                 magnet_name: str = None, gui: bool = False,
                 elements_along_width: int = None,
                 target_size_insulation: float = None,
                 target_size_conductor: float = None,
                 output_path: str = None,
                 GetDP_path: str = None):
        self.magnet_name = magnet_name
        self.gui = gui
        self.output_path = output_path

        self.GetDP_path = GetDP_path

        self.data = utl.FilesAndFolders.read_data_from_yaml(os.path.join('input', self.magnet_name, self.magnet_name + '.yaml'), dF.FDM)

        conductor = list(self.data.conductors.values())[0]
        self.conductor_ins_th_h = conductor.cable.th_insulation_along_height
        self.conductor_ins_th_w = conductor.cable.th_insulation_along_width
        self.cable_width = conductor.cable.bare_cable_width
        self.cable_height = conductor.cable.bare_cable_height_mean

        self.elements = elements_along_width if elements_along_width else round(self.cable_width / (self.cable_height / 2))
        self.target_size_insulation = target_size_insulation if target_size_insulation else 1
        self.target_size_conductor = target_size_conductor if target_size_conductor else self.cable_height / 3

        self.colors = {'intense_blue': [0, 115, 181],
                       'morning_orange': [242, 151, 43],
                       'october': [205, 93, 4],
                       'dark_red': [170, 0, 0],
                       'hopbush': [204, 121, 167],
                       'light_blue': [106, 200, 253],
                       'bluish_green': [0, 158, 115],
                       'dark_violet': [75, 0, 146]}

    def run_model(self):

        gu = utl.GmshUtils(self.output_path)
        gu.initialize()
        occ = gmsh.model.occ
        gmsh.option.setNumber("Geometry.OCCParallel", 1)
        gmsh.option.setNumber("Mesh.Algorithm", 8)
        gmsh.model.add(self.data.general.magnet_name)
        gmsh.option.setNumber("General.Verbosity", self.data.run.verbosity_Gmsh)

        ext = {}
        ex = self.data.magnet.solve.thermal.insulation_TSA.exterior
        for i, side in enumerate(ex.blocks):
            ext[side] = i

        qh = self.data.quench_protection.quench_heaters
        mid_first_i, mid_second_i = None, None
        external_qh_first, external_qh_second = None, None
        for i, turn in enumerate(qh.turns):  # assumes that first block has no inner quench heaters
            if turn == 1 and qh.turns_sides[i] == 'o':
                mid_first_o = i
            elif turn == 3 and qh.turns_sides[i] == 'i':
                mid_first_i = i
            elif turn == 2 and qh.turns_sides[i] == 'o':
                mid_second_o = i
            elif turn == 4 and qh.turns_sides[i] == 'i':
                mid_second_i = i
            elif turn == 3 and qh.turns_sides[i] == 'o':
                external_qh_first = i
            elif turn == 4 and qh.turns_sides[i] == 'o':
                external_qh_second = i
            else:
                raise ValueError('Change quench heater entries')

        cond_ins_mat = list(self.data.conductors.values())[0].cable.material_insulation
        ordered_mid_first_ths = [self.conductor_ins_th_w] + qh.h_ins[mid_first_o] + [qh.h[mid_first_o]] + qh.h_ground_ins[mid_first_o] + \
                          (qh.h_ground_ins[mid_first_i][::-1] + [qh.h[mid_first_i]] + qh.h_ins[mid_first_i][::-1] if mid_first_i else []) + [self.conductor_ins_th_w]
        ordered_mid_first_mat = [cond_ins_mat] + qh.type_ins[mid_first_o] + ['SS'] + qh.type_ground_ins[mid_first_o] + \
                          (qh.type_ground_ins[mid_first_i][::-1] + ['SS'] + qh.type_ins[mid_first_i][::-1] if mid_first_i else []) + [cond_ins_mat]
        # ordered_mid_first_qh[len(ordered_mid_first_ths) - len(qh.h_ins[mid_first_i]) - 2] = True  # enable if double mid qh

        ordered_mid_second_ths = [self.conductor_ins_th_w] + qh.h_ins[mid_second_o] + [qh.h[mid_second_o]] + qh.h_ground_ins[mid_second_o] + \
                          (qh.h_ground_ins[mid_second_i][::-1] + [qh.h[mid_second_i]] + qh.h_ins[mid_second_i][::-1] if mid_second_i else []) + [self.conductor_ins_th_w]
        ordered_mid_second_mat = [cond_ins_mat] + qh.type_ins[mid_second_o] + ['SS'] + qh.type_ground_ins[mid_second_o] + \
                          (qh.type_ground_ins[mid_second_i][::-1] + ['SS'] + qh.type_ins[mid_second_i][::-1] if mid_second_i else []) + [cond_ins_mat]
        # ordered_mid_second_qh[len(ordered_mid_second_ths) - len(qh.h_ins[mid_second_i]) - 2] = True  # enable if double mid qh

        ordered_ext_first_ths = [self.conductor_ins_th_w] + (qh.h_ins[external_qh_first] + [qh.h[external_qh_first]] + qh.h_ground_ins[external_qh_first] if external_qh_first else [])
        ordered_ext_first_mat = [cond_ins_mat] + (qh.type_ins[external_qh_first] + ['SS'] + qh.type_ground_ins[external_qh_first] if external_qh_first else [])
        ordered_ext_second_ths = [self.conductor_ins_th_w] + (qh.h_ins[external_qh_second] + [qh.h[external_qh_second]] + qh.h_ground_ins[external_qh_second] if external_qh_second else [])
        ordered_ext_second_mat = [cond_ins_mat] + (qh.type_ins[external_qh_second] + ['SS'] + qh.type_ground_ins[external_qh_second] if external_qh_second else [])

        ordered_ext1_ths = [self.conductor_ins_th_h] + (ex.thicknesses_append[ext['1l']] if '1l' in ext else [])
        ordered_ext2_ths = [self.conductor_ins_th_h] + (ex.thicknesses_append[ext['1h']] if '1h' in ext else [])
        ordered_ext3_ths = [self.conductor_ins_th_h] + (ex.thicknesses_append[ext['2l']] if '2l' in ext else [])
        ordered_ext4_ths = [self.conductor_ins_th_h] + (ex.thicknesses_append[ext['2h']] if '2h' in ext else [])
        ordered_ext1_mat = [cond_ins_mat] + (ex.materials_append[ext['1l']] if '1l' in ext else [])
        ordered_ext2_mat = [cond_ins_mat] + (ex.materials_append[ext['1h']] if '1h' in ext else [])
        ordered_ext3_mat = [cond_ins_mat] + (ex.materials_append[ext['2l']] if '2l' in ext else [])
        ordered_ext4_mat = [cond_ins_mat] + (ex.materials_append[ext['2h']] if '2h' in ext else [])

        tot_mid_ins_th = sum(ordered_mid_first_ths)  # assumes mid first and second same thickness
        lc = 0

        # Points
        ht1_pnts = {'il': occ.addPoint(0, 0, 0, lc),
                    'ol': occ.addPoint(self.cable_width, 0, 0, lc),
                    'oh': occ.addPoint(self.cable_width, self.cable_height, 0, lc),
                    'ih': occ.addPoint(0, self.cable_height, 0, lc)}
        ht2_pnts = {'il': occ.addPoint(0, self.cable_height + self.conductor_ins_th_h * 2, 0, lc),
                    'ol': occ.addPoint(self.cable_width, self.cable_height + self.conductor_ins_th_h * 2, 0, lc),
                    'oh': occ.addPoint(self.cable_width, (self.cable_height + self.conductor_ins_th_h) * 2, 0, lc),
                    'ih': occ.addPoint(0, (self.cable_height + self.conductor_ins_th_h) * 2, 0, lc)}
        ht3_pnts = {'il': occ.addPoint(self.cable_width + tot_mid_ins_th, 0, 0, lc),
                    'ol': occ.addPoint(2 * self.cable_width + tot_mid_ins_th, 0, 0, lc),
                    'oh': occ.addPoint(2 * self.cable_width + tot_mid_ins_th, self.cable_height, 0, lc),
                    'ih': occ.addPoint(self.cable_width + tot_mid_ins_th, self.cable_height, 0, lc)}
        ht4_pnts = {'il': occ.addPoint(self.cable_width + tot_mid_ins_th, self.cable_height + self.conductor_ins_th_h * 2, 0, lc),
                    'ol': occ.addPoint(2 * self.cable_width + tot_mid_ins_th, self.cable_height + self.conductor_ins_th_h * 2, 0, lc),
                    'oh': occ.addPoint(2 * self.cable_width + tot_mid_ins_th, (self.cable_height + self.conductor_ins_th_h) * 2, 0, lc),
                    'ih': occ.addPoint(self.cable_width + tot_mid_ins_th, (self.cable_height + self.conductor_ins_th_h) * 2, 0, lc)}
        ins1i_pnts = {'l': occ.addPoint(- self.conductor_ins_th_w, 0, 0, lc),
                      'h': occ.addPoint(- self.conductor_ins_th_w, self.cable_height, 0, lc)}
        ins2i_pnts = {'l': occ.addPoint(- self.conductor_ins_th_w, self.cable_height + self.conductor_ins_th_h * 2, 0, lc),
                      'h': occ.addPoint(- self.conductor_ins_th_w, (self.cable_height + self.conductor_ins_th_h) * 2, 0, lc)}

        mid_layers_first_pnts = [ht1_pnts]
        temp_th = self.cable_width
        for th in ordered_mid_first_ths[:-1]:
            mid_layers_first_pnts.append({'l': occ.addPoint(temp_th + th, 0, 0, lc),
                                          'h': occ.addPoint(temp_th + th, self.cable_height, 0, lc)})
            temp_th += th
        mid_layers_second_pnts = [ht2_pnts]
        temp_th = self.cable_width
        for th in ordered_mid_second_ths[:-1]:
            mid_layers_second_pnts.append({'l': occ.addPoint(temp_th + th, self.cable_height + self.conductor_ins_th_h * 2, 0, lc),
                                           'h': occ.addPoint(temp_th + th, (self.cable_height + self.conductor_ins_th_h) * 2, 0, lc)})
            temp_th += th
        external_layers_first_pnts = [ht3_pnts]
        temp_th = 2 * self.cable_width + tot_mid_ins_th
        for th in ordered_ext_first_ths:
            external_layers_first_pnts.append({'l': occ.addPoint(temp_th + th, 0, 0, lc),
                                               'h': occ.addPoint(temp_th + th, self.cable_height, 0, lc)})
            temp_th += th
        external_layers_second_pnts = [ht4_pnts]
        temp_th = 2 * self.cable_width + tot_mid_ins_th
        for th in ordered_ext_second_ths:
            external_layers_second_pnts.append({'l': occ.addPoint(temp_th + th, self.cable_height + self.conductor_ins_th_h * 2, 0, lc),
                                                'h': occ.addPoint(temp_th + th, (self.cable_height + self.conductor_ins_th_h) * 2, 0, lc)})
            temp_th += th

        external_layers1_pnts = [ht1_pnts]
        temp_th = 0
        for th in ordered_ext1_ths:
            external_layers1_pnts.append({'i': occ.addPoint(0, temp_th - th, 0, lc),
                                          'o': occ.addPoint(self.cable_width, temp_th - th, 0, lc)})
            temp_th -= th
        external_layers2_pnts = [ht2_pnts]
        temp_th = (self.cable_height + self.conductor_ins_th_h) * 2
        for th in ordered_ext2_ths:
            external_layers2_pnts.append({'i': occ.addPoint(0, temp_th + th, 0, lc),
                                          'o': occ.addPoint(self.cable_width, temp_th + th, 0, lc)})
            temp_th += th
        external_layers3_pnts = [ht3_pnts]
        temp_th = 0
        for th in ordered_ext3_ths:
            external_layers3_pnts.append({'i': occ.addPoint(self.cable_width + tot_mid_ins_th, temp_th - th, 0, lc),
                                          'o': occ.addPoint(2 * self.cable_width + tot_mid_ins_th, temp_th - th, 0, lc)})
            temp_th -= th
        external_layers4_pnts = [ht4_pnts]
        temp_th = (self.cable_height + self.conductor_ins_th_h) * 2
        for th in ordered_ext4_ths:
            external_layers4_pnts.append({'i': occ.addPoint(self.cable_width + tot_mid_ins_th, temp_th + th, 0, lc),
                                          'o': occ.addPoint(2 * self.cable_width + tot_mid_ins_th, temp_th + th, 0, lc)})
            temp_th += th

        # Lines
        ht1_lines = {'l': occ.addLine(ht1_pnts['il'], ht1_pnts['ol']),
                     'o': occ.addLine(ht1_pnts['ol'], ht1_pnts['oh']),
                     'h': occ.addLine(ht1_pnts['oh'], ht1_pnts['ih']),
                     'i': occ.addLine(ht1_pnts['ih'], ht1_pnts['il'])}
        ht2_lines = {'l': occ.addLine(ht2_pnts['il'], ht2_pnts['ol']),
                     'o': occ.addLine(ht2_pnts['ol'], ht2_pnts['oh']),
                     'h': occ.addLine(ht2_pnts['oh'], ht2_pnts['ih']),
                     'i': occ.addLine(ht2_pnts['ih'], ht2_pnts['il'])}
        ht3_lines = {'l': occ.addLine(ht3_pnts['il'], ht3_pnts['ol']),
                     'o': occ.addLine(ht3_pnts['ol'], ht3_pnts['oh']),
                     'h': occ.addLine(ht3_pnts['oh'], ht3_pnts['ih']),
                     'i': occ.addLine(ht3_pnts['ih'], ht3_pnts['il'])}
        ht4_lines = {'l': occ.addLine(ht4_pnts['il'], ht4_pnts['ol']),
                     'o': occ.addLine(ht4_pnts['ol'], ht4_pnts['oh']),
                     'h': occ.addLine(ht4_pnts['oh'], ht4_pnts['ih']),
                     'i': occ.addLine(ht4_pnts['ih'], ht4_pnts['il'])}
        ins1i_lines = {'h': occ.addLine(ht1_pnts['ih'], ins1i_pnts['h']),
                       'i': occ.addLine(ins1i_pnts['h'], ins1i_pnts['l']),
                       'l': occ.addLine(ins1i_pnts['l'], ht1_pnts['il'])}
        ins2i_lines = {'h': occ.addLine(ht2_pnts['ih'], ins2i_pnts['h']),
                       'i': occ.addLine(ins2i_pnts['h'], ins2i_pnts['l']),
                       'l': occ.addLine(ins2i_pnts['l'], ht2_pnts['il'])}
        ins12_lines = {'i': occ.addLine(ht1_pnts['ih'], ht2_pnts['il']),
                       'o': occ.addLine(ht1_pnts['oh'], ht2_pnts['ol'])}
        ins34_lines = {'i': occ.addLine(ht3_pnts['ih'], ht4_pnts['il']),
                       'o': occ.addLine(ht3_pnts['oh'], ht4_pnts['ol'])}

        mid_layers_first_lines = [ht1_lines]
        for i, lyr in enumerate(mid_layers_first_pnts[1:], 1):
            mid_layers_first_lines.append({'h': occ.addLine(mid_layers_first_pnts[i - 1][('o' if i == 1 else '') + 'h'], lyr['h']),
                                           'o': occ.addLine(lyr['h'], lyr['l']),
                                           'l': occ.addLine(lyr['l'], mid_layers_first_pnts[i - 1][('o' if i == 1 else '') + 'l'])})
        mid_layers_first_lines.append({'h': occ.addLine(mid_layers_first_pnts[-1]['h'], ht3_pnts['ih']),
                                       'o': ht3_lines['i'],
                                       'l': occ.addLine(ht3_pnts['il'], mid_layers_first_pnts[-1]['l'])})
        mid_layers_second_lines = [ht2_lines]
        for i, lyr in enumerate(mid_layers_second_pnts[1:], 1):
            mid_layers_second_lines.append({'h': occ.addLine(mid_layers_second_pnts[i - 1][('o' if i == 1 else '') + 'h'], lyr['h']),
                                            'o': occ.addLine(lyr['h'], lyr['l']),
                                            'l': occ.addLine(lyr['l'], mid_layers_second_pnts[i - 1][('o' if i == 1 else '') + 'l'])})
        mid_layers_second_lines.append({'h': occ.addLine(mid_layers_second_pnts[-1]['h'], ht4_pnts['ih']),
                                        'o': ht4_lines['i'],
                                        'l': occ.addLine(ht4_pnts['il'], mid_layers_second_pnts[-1]['l'])})
        external_layers_first_lines = [ht3_lines]
        for i, lyr in enumerate(external_layers_first_pnts[1:], 1):
            external_layers_first_lines.append({'h': occ.addLine(external_layers_first_pnts[i - 1][('o' if i == 1 else '') + 'h'], lyr['h']),
                                                'o': occ.addLine(lyr['h'], lyr['l']),
                                                'l': occ.addLine(lyr['l'], external_layers_first_pnts[i - 1][('o' if i == 1 else '') + 'l'])})
        external_layers_second_lines = [ht4_lines]
        for i, lyr in enumerate(external_layers_second_pnts[1:], 1):
            external_layers_second_lines.append({'h': occ.addLine(external_layers_second_pnts[i - 1][('o' if i == 1 else '') + 'h'], lyr['h']),
                                                 'o': occ.addLine(lyr['h'], lyr['l']),
                                                 'l': occ.addLine(lyr['l'], external_layers_second_pnts[i - 1][('o' if i == 1 else '') + 'l'])})
        external_layers1_lines = [ht1_lines]
        for i, lyr in enumerate(external_layers1_pnts[1:], 1):
            external_layers1_lines.append({'o': occ.addLine(external_layers1_pnts[i - 1]['o' + ('l' if i == 1 else '')], lyr['o']),
                                           'l': occ.addLine(lyr['o'], lyr['i']),
                                           'i': occ.addLine(lyr['i'], external_layers1_pnts[i - 1]['i' + ('l' if i == 1 else '')])})
        external_layers2_lines = [ht2_lines]
        for i, lyr in enumerate(external_layers2_pnts[1:], 1):
            external_layers2_lines.append({'o': occ.addLine(external_layers2_pnts[i - 1]['o' + ('h' if i == 1 else '')], lyr['o']),
                                           'h': occ.addLine(lyr['o'], lyr['i']),
                                           'i': occ.addLine(lyr['i'], external_layers2_pnts[i - 1]['i' + ('h' if i == 1 else '')])})
        external_layers3_lines = [ht3_lines]
        for i, lyr in enumerate(external_layers3_pnts[1:], 1):
            external_layers3_lines.append({'o': occ.addLine(external_layers3_pnts[i - 1]['o' + ('l' if i == 1 else '')], lyr['o']),
                                           'l': occ.addLine(lyr['o'], lyr['i']),
                                           'i': occ.addLine(lyr['i'], external_layers3_pnts[i - 1]['i' + ('l' if i == 1 else '')])})
        external_layers4_lines = [ht4_lines]
        for i, lyr in enumerate(external_layers4_pnts[1:], 1):
            external_layers4_lines.append({'o': occ.addLine(external_layers4_pnts[i - 1]['o' + ('h' if i == 1 else '')], lyr['o']),
                                           'h': occ.addLine(lyr['o'], lyr['i']),
                                           'i': occ.addLine(lyr['i'], external_layers4_pnts[i - 1]['i' + ('h' if i == 1 else '')])})

        # Apply transfinite curves
        occ.synchronize()
        def apply_trans_curves_o(i, lyr, ths):
            gmsh.model.mesh.setTransfiniteCurve(lyr['o'], max(1, round(self.cable_height / self.target_size_conductor)))
            if i == 0:
                gmsh.model.mesh.setTransfiniteCurve(lyr['i'], max(1, round(self.cable_height / self.target_size_conductor)))
                for side in ['h', 'l']:
                    gmsh.model.mesh.setTransfiniteCurve(lyr[side], self.elements)
            else:
                for side in ['h', 'l']:
                    gmsh.model.mesh.setTransfiniteCurve(lyr[side], max(1, round(ths[i - 1] / self.target_size_insulation)) + 1)
        def apply_trans_curves_lh(i, lyr, ths, tb):
            gmsh.model.mesh.setTransfiniteCurve(lyr[tb], self.elements)
            for side in ['i', 'o']:
                gmsh.model.mesh.setTransfiniteCurve(lyr[side], max(1, round(ths[i - 1] / self.target_size_insulation)) + 1)

        for i, lyr in enumerate([ins1i_lines, ins2i_lines]):
            gmsh.model.mesh.setTransfiniteCurve(lyr['i'], max(1, round(self.cable_height / self.target_size_conductor)))
            for side in ['h', 'l']:
                gmsh.model.mesh.setTransfiniteCurve(lyr[side], max(1, round(self.conductor_ins_th_w / self.target_size_insulation)) + 1)
        for i, lyr in enumerate([ins12_lines, ins34_lines]):
            for side in ['i', 'o']:
                gmsh.model.mesh.setTransfiniteCurve(lyr[side], max(1, 2 * round(self.conductor_ins_th_h / self.target_size_insulation)) + 1)
        for idx, ins_layer in enumerate(mid_layers_first_lines):
            apply_trans_curves_o(idx, ins_layer, ordered_mid_first_ths)
        for idx, ins_layer in enumerate(mid_layers_second_lines):
            apply_trans_curves_o(idx, ins_layer, ordered_mid_second_ths)
        for idx, ins_layer in enumerate(external_layers_first_lines):
            apply_trans_curves_o(idx, ins_layer, ordered_ext_first_ths)
        for idx, ins_layer in enumerate(external_layers_second_lines):
            apply_trans_curves_o(idx, ins_layer, ordered_ext_second_ths)
        for idx, ins_layer in enumerate(external_layers1_lines[1:], 1):
            apply_trans_curves_lh(idx, ins_layer, ordered_ext1_ths, 'l')
        for idx, ins_layer in enumerate(external_layers2_lines[1:], 1):
            apply_trans_curves_lh(idx, ins_layer, ordered_ext2_ths, 'h')
        for idx, ins_layer in enumerate(external_layers3_lines[1:], 1):
            apply_trans_curves_lh(idx, ins_layer, ordered_ext3_ths, 'l')
        for idx, ins_layer in enumerate(external_layers4_lines[1:], 1):
            apply_trans_curves_lh(idx, ins_layer, ordered_ext4_ths, 'h')

        # Loops
        ht1_loop = occ.addCurveLoop([ht1_lines['l'], ht1_lines['o'], ht1_lines['h'], ht1_lines['i']])
        ht2_loop = occ.addCurveLoop([ht2_lines['l'], ht2_lines['o'], ht2_lines['h'], ht2_lines['i']])
        ht3_loop = occ.addCurveLoop([ht3_lines['l'], ht3_lines['o'], ht3_lines['h'], ht3_lines['i']])
        ht4_loop = occ.addCurveLoop([ht4_lines['l'], ht4_lines['o'], ht4_lines['h'], ht4_lines['i']])
        ins1i_loop = occ.addCurveLoop([ht1_lines['i'], ins1i_lines['h'], ins1i_lines['i'], ins1i_lines['l']])
        ins2i_loop = occ.addCurveLoop([ht2_lines['i'], ins2i_lines['h'], ins2i_lines['i'], ins2i_lines['l']])
        ins12_loop = occ.addCurveLoop([ht1_lines['h'], ins12_lines['i'], ht2_lines['l'], ins12_lines['o']])
        ins34_loop = occ.addCurveLoop([ht3_lines['h'], ins34_lines['i'], ht4_lines['l'], ins34_lines['o']])

        mid_layers_first_loops = []
        for i, lyr in enumerate(mid_layers_first_lines[1:], 1):
            mid_layers_first_loops.append(occ.addCurveLoop([mid_layers_first_lines[i - 1]['o'], lyr['h'], lyr['o'], lyr['l']]))
        mid_layers_second_loops = []
        for i, lyr in enumerate(mid_layers_second_lines[1:], 1):
            mid_layers_second_loops.append(occ.addCurveLoop([mid_layers_second_lines[i - 1]['o'], lyr['h'], lyr['o'], lyr['l']]))
        external_layers_first_loops = []
        for i, lyr in enumerate(external_layers_first_lines[1:], 1):
            external_layers_first_loops.append(occ.addCurveLoop([external_layers_first_lines[i - 1]['o'], lyr['h'], lyr['o'], lyr['l']]))
        external_layers_second_loops = []
        for i, lyr in enumerate(external_layers_second_lines[1:], 1):
            external_layers_second_loops.append(occ.addCurveLoop([external_layers_second_lines[i - 1]['o'], lyr['h'], lyr['o'], lyr['l']]))
        external_layers1_loops = []
        for i, lyr in enumerate(external_layers1_lines[1:], 1):
            external_layers1_loops.append(occ.addCurveLoop([external_layers1_lines[i - 1]['l'], lyr['o'], lyr['l'], lyr['i']]))
        external_layers2_loops = []
        for i, lyr in enumerate(external_layers2_lines[1:], 1):
            external_layers2_loops.append(occ.addCurveLoop([external_layers2_lines[i - 1]['h'], lyr['o'], lyr['h'], lyr['i']]))
        external_layers3_loops = []
        for i, lyr in enumerate(external_layers3_lines[1:], 1):
            external_layers3_loops.append(occ.addCurveLoop([external_layers3_lines[i - 1]['l'], lyr['o'], lyr['l'], lyr['i']]))
        external_layers4_loops = []
        for i, lyr in enumerate(external_layers4_lines[1:], 1):
            external_layers4_loops.append(occ.addCurveLoop([external_layers4_lines[i - 1]['h'], lyr['o'], lyr['h'], lyr['i']]))

        # Areas
        ht1_area = occ.addPlaneSurface([ht1_loop])
        ht2_area = occ.addPlaneSurface([ht2_loop])
        ht3_area = occ.addPlaneSurface([ht3_loop])
        ht4_area = occ.addPlaneSurface([ht4_loop])
        ins1i_area = occ.addPlaneSurface([ins1i_loop])
        ins2i_area = occ.addPlaneSurface([ins2i_loop])
        ins12_area = occ.addPlaneSurface([ins12_loop])
        ins34_area = occ.addPlaneSurface([ins34_loop])
        mid_layers_first_areas, mid_layers_second_areas = [], []
        for lyr in mid_layers_first_loops: mid_layers_first_areas.append(occ.addPlaneSurface([lyr]))
        for lyr in mid_layers_second_loops: mid_layers_second_areas.append(occ.addPlaneSurface([lyr]))
        external_layers_first_areas, external_layers_second_areas = [], []
        for lyr in external_layers_first_loops: external_layers_first_areas.append(occ.addPlaneSurface([lyr]))
        for lyr in external_layers_second_loops: external_layers_second_areas.append(occ.addPlaneSurface([lyr]))
        external_layers1_areas, external_layers2_areas, external_layers3_areas, external_layers4_areas = [], [], [], []
        for lyr in external_layers1_loops: external_layers1_areas.append(occ.addPlaneSurface([lyr]))
        for lyr in external_layers2_loops: external_layers2_areas.append(occ.addPlaneSurface([lyr]))
        for lyr in external_layers3_loops: external_layers3_areas.append(occ.addPlaneSurface([lyr]))
        for lyr in external_layers4_loops: external_layers4_areas.append(occ.addPlaneSurface([lyr]))

        # Apply transfinite areas
        occ.synchronize()
        geom_path = os.path.join(self.output_path, self.data.general.magnet_name + ".brep")
        gmsh.write(geom_path)
        for area in [ht1_area, ht2_area, ht3_area, ht4_area, ins1i_area, ins2i_area, ins12_area, ins34_area]:
            gmsh.model.mesh.setTransfiniteSurface(area)
        for lyr in (mid_layers_first_areas + mid_layers_second_areas + external_layers_first_areas + external_layers_second_areas +
                    external_layers1_areas + external_layers2_areas + external_layers3_areas + external_layers4_areas):
            gmsh.model.mesh.setTransfiniteSurface(lyr)

        # Physical groups
        # Areas
        ht1_pg = gmsh.model.addPhysicalGroup(2, [ht1_area])
        gmsh.model.setPhysicalName(2, ht1_pg, "ht1")
        ht2_pg = gmsh.model.addPhysicalGroup(2, [ht2_area])
        gmsh.model.setPhysicalName(2, ht2_pg, "ht2")
        ht3_pg = gmsh.model.addPhysicalGroup(2, [ht3_area])
        gmsh.model.setPhysicalName(2, ht3_pg, "ht3")
        ht4_pg = gmsh.model.addPhysicalGroup(2, [ht4_area])
        gmsh.model.setPhysicalName(2, ht4_pg, "ht4")
        color = self.colors['light_blue']
        gmsh.model.setColor([(2, i) for i in [ht1_pg, ht2_pg, ht3_pg, ht4_pg]], color[0], color[1], color[2])
        ins1i_pg = gmsh.model.addPhysicalGroup(2, [ins1i_area])
        gmsh.model.setPhysicalName(2, ins1i_pg, "ins1i")
        ins2i_pg = gmsh.model.addPhysicalGroup(2, [ins2i_area])
        gmsh.model.setPhysicalName(2, ins2i_pg, "ins2i")
        ins12_pg = gmsh.model.addPhysicalGroup(2, [ins12_area])
        gmsh.model.setPhysicalName(2, ins12_pg, "ins12")
        ins34_pg = gmsh.model.addPhysicalGroup(2, [ins34_area])
        gmsh.model.setPhysicalName(2, ins34_pg, "ins34")
        color = self.colors['morning_orange']
        gmsh.model.setColor([(2, i) for i in [ins1i_pg, ins2i_pg, ins12_pg, ins34_pg]], color[0], color[1], color[2])
        qhs_pgs = []
        mid_layers_pgs = {'kapton': [], 'G10': []}
        for i, lyr in enumerate(mid_layers_first_areas):
            if ordered_mid_first_mat[i] == 'SS':
                qhs_pgs.append(gmsh.model.addPhysicalGroup(2, [lyr]))
                gmsh.model.setPhysicalName(2, qhs_pgs[-1], 'qh_mid_first' + str(qhs_pgs[-1]))
            else:
                mid_layers_pgs[ordered_mid_first_mat[i]].append(gmsh.model.addPhysicalGroup(2, [lyr]))
                gmsh.model.setPhysicalName(2, mid_layers_pgs[ordered_mid_first_mat[i]][-1], 'ins_mid_first' + str(mid_layers_pgs[ordered_mid_first_mat[i]][-1]))
        for i, lyr in enumerate(mid_layers_second_areas):
            if ordered_mid_second_mat[i] == 'SS':
                qhs_pgs.append(gmsh.model.addPhysicalGroup(2, [lyr]))
                gmsh.model.setPhysicalName(2, qhs_pgs[-1], 'qh_mid_second' + str(qhs_pgs[-1]))
            else:
                mid_layers_pgs[ordered_mid_second_mat[i]].append(gmsh.model.addPhysicalGroup(2, [lyr]))
                gmsh.model.setPhysicalName(2, mid_layers_pgs[ordered_mid_second_mat[i]][-1], 'ins_mid_second' + str(mid_layers_pgs[ordered_mid_second_mat[i]][-1]))
        color = self.colors['bluish_green']
        gmsh.model.setColor([(2, i) for i in [mid_layers_pgs['kapton'][1], mid_layers_pgs['kapton'][2],
                                              mid_layers_pgs['kapton'][5], mid_layers_pgs['kapton'][6]]], color[0], color[1], color[2])
        color = self.colors['morning_orange']
        gmsh.model.setColor([(2, i) for i in [mid_layers_pgs['kapton'][0], mid_layers_pgs['kapton'][3],
                                              mid_layers_pgs['kapton'][4], mid_layers_pgs['kapton'][7]]], color[0], color[1], color[2])
        external_layers_first_pgs = {'kapton': [], 'G10': []}
        for i, lyr in enumerate(external_layers_first_areas):
            if ordered_ext_first_mat[i] == 'SS':
                qhs_pgs.append(gmsh.model.addPhysicalGroup(2, [lyr]))
                gmsh.model.setPhysicalName(2, qhs_pgs[-1], 'qh_ext_first' + str(qhs_pgs[-1]))
            else:
                external_layers_first_pgs[ordered_ext_first_mat[i]].append(gmsh.model.addPhysicalGroup(2, [lyr]))
                gmsh.model.setPhysicalName(2, external_layers_first_pgs[ordered_ext_first_mat[i]][-1], 'ins_ext_first' + str(external_layers_first_pgs[ordered_ext_first_mat[i]][-1]))
        color = self.colors['bluish_green']
        gmsh.model.setColor([(2, i) for i in external_layers_first_pgs['kapton'][1:]], color[0], color[1], color[2])
        color = self.colors['morning_orange']
        gmsh.model.setColor([(2, external_layers_first_pgs['kapton'][0])], color[0], color[1], color[2])
        external_layers_second_pgs = {'kapton': [], 'G10': []}
        for i, lyr in enumerate(external_layers_second_areas):
            if ordered_ext_second_mat[i] == 'SS':
                qhs_pgs.append(gmsh.model.addPhysicalGroup(2, [lyr]))
                gmsh.model.setPhysicalName(2, qhs_pgs[-1], 'qh_ext_second' + str(qhs_pgs[-1]))
            else:
                external_layers_second_pgs[ordered_ext_second_mat[i]].append(gmsh.model.addPhysicalGroup(2, [lyr]))
                gmsh.model.setPhysicalName(2, external_layers_second_pgs[ordered_ext_second_mat[i]][-1], 'ins_ext_second' + str(external_layers_second_pgs[ordered_ext_second_mat[i]][-1]))
        color = self.colors['bluish_green']
        gmsh.model.setColor([(2, i) for i in external_layers_second_pgs['kapton'][1:]], color[0], color[1], color[2])
        color = self.colors['morning_orange']
        gmsh.model.setColor([(2, external_layers_second_pgs['kapton'][0])], color[0], color[1], color[2])
        color = self.colors['dark_red']
        gmsh.model.setColor([(2, i) for i in qhs_pgs], color[0], color[1], color[2])
        external_layers_pgs = {'kapton': [], 'G10': []}
        for lyr, mat in zip(external_layers1_areas + external_layers2_areas + external_layers3_areas + external_layers4_areas,
                            ordered_ext1_mat + ordered_ext2_mat + ordered_ext3_mat + ordered_ext4_mat):
            external_layers_pgs[mat].append(gmsh.model.addPhysicalGroup(2, [lyr]))
            gmsh.model.setPhysicalName(2, external_layers_pgs[mat][-1], 'ins_ext' + str(external_layers_pgs[mat][-1]))
        color = self.colors['bluish_green']
        gmsh.model.setColor([(2, i) for i in [external_layers_pgs['kapton'][1], external_layers_pgs['kapton'][5]]], color[0], color[1], color[2])
        color = self.colors['morning_orange']
        gmsh.model.setColor([(2, i) for i in [external_layers_pgs['kapton'][0], external_layers_pgs['kapton'][2],
                                              external_layers_pgs['kapton'][3], external_layers_pgs['kapton'][4]]], color[0], color[1], color[2])
        color = self.colors['dark_violet']
        gmsh.model.setColor([(2, i) for i in mid_layers_pgs['G10'] + external_layers_first_pgs['G10'] +
                             external_layers_second_pgs['G10'] + external_layers_pgs['G10']], color[0], color[1], color[2])
        # Boundaries
        bnd_pgs = {'i': [], 'o': [], 'l': [], 'h': []}
        for side, ht_ins in zip(['i', 'i', 'o', 'o', 'l', 'l', 'h', 'h'],
                                [ins1i_lines, ins2i_lines, external_layers_first_lines[-1], external_layers_second_lines[-1],
                                 external_layers1_lines[-1], external_layers3_lines[-1], external_layers2_lines[-1], external_layers4_lines[-1]]):
            bnd_pgs[side].append(gmsh.model.addPhysicalGroup(1, [ht_ins[side]]))
            gmsh.model.setPhysicalName(1, bnd_pgs[side][-1], 'bnd_' + side + str(bnd_pgs[side][-1]))

        occ.synchronize()
        gmsh.option.setNumber("Mesh.RecombineAll", 1)
        gmsh.model.mesh.generate(2)
        # gu.launch_interactive_GUI()
        mesh_path = os.path.join(self.output_path, self.data.general.magnet_name + ".msh")
        gmsh.write(mesh_path)

        # Assign regions
        region = {'names': [], 'numbers': []}
        bnd_region = {'names': [], 'numbers': [], 'values': []}
        rm = {'powered': copy.deepcopy(region), 'conducting': copy.deepcopy(region), 'insulator_kapton': copy.deepcopy(region), 'insulator_G10': copy.deepcopy(region),
              'boundaries': {'temperature': copy.deepcopy(bnd_region), 'heat_flux': copy.deepcopy(bnd_region), 'cooling': copy.deepcopy(bnd_region)}}
        rm['powered']['names'] = ["ht1", "ht2", "ht3", "ht4"]
        rm['powered']['numbers'] = [ht1_pg, ht2_pg, ht3_pg, ht4_pg]
        for pg in [ins1i_pg, ins2i_pg, ins12_pg, ins34_pg]:
            rm['insulator_' + cond_ins_mat]['names'].append('ins' + str(pg))
            rm['insulator_' + cond_ins_mat]['numbers'].append(pg)
        for mat in ['kapton', 'G10']:
            for pg in mid_layers_pgs[mat] + external_layers_first_pgs[mat] + external_layers_second_pgs[mat] + external_layers_pgs[mat]:
                rm['insulator_' + mat]['names'].append('ins' + str(pg))
                rm['insulator_' + mat]['numbers'].append(pg)
        for pg in qhs_pgs:
            rm['conducting']['names'].append('qh' + str(pg))
            rm['conducting']['numbers'].append(pg)

        # Boundary conditions
        # Initialize lists
        for bc_type, bc_rm in rm['boundaries'].items():  # b.c. type
            bc_rm['names'] = []
            bc_rm['numbers'] = []
            bc_rm['values'] = []

        # Apply general cooling if activated
        bnd_list_names = {'Robin': [], 'Neumann': []}
        bnd_list_numbers = {'Robin': [], 'Neumann': []}
        bnd_names = [['ht1i', 'ht2i'], ['ht3o', 'ht4o'], ['ht1l', 'ht3l'], ['ht2h', 'ht4h']]
        for side, tags, names in zip(list(bnd_pgs.keys()), list(bnd_pgs.values()), bnd_names):
            bnd_type = 'Robin' if (self.data.magnet.solve.thermal.He_cooling.sides == 'external' or
                                   ('inner' in self.data.magnet.solve.thermal.He_cooling.sides and side == 'i') or
                                   ('outer' in self.data.magnet.solve.thermal.He_cooling.sides and side == 'o')) else 'Neumann'
            bnd_list_numbers[bnd_type].extend(tags)
            bnd_list_names[bnd_type].extend(names)
        if self.data.magnet.solve.thermal.He_cooling.sides != 'adiabatic':
            rm['boundaries']['cooling']['names'].append(bnd_list_names['Robin'])
            rm['boundaries']['cooling']['numbers'].append(bnd_list_numbers['Robin'])
            rm['boundaries']['cooling']['values'].append([self.data.magnet.solve.thermal.He_cooling.heat_transfer_coefficient,
                                                          self.data.magnet.solve.thermal.init_temperature])
        if self.data.magnet.solve.thermal.He_cooling.sides != 'external':
            rm['boundaries']['heat_flux']['names'].append(bnd_list_names['Neumann'])
            rm['boundaries']['heat_flux']['numbers'].append(bnd_list_numbers['Neumann'])
            rm['boundaries']['heat_flux']['values'].append([0])

        # Apply specific boundary conditions
        for bc_data, bc_rm in zip(self.data.magnet.solve.thermal.overwrite_boundary_conditions, rm['boundaries']):  # b.c. type
            # bc_data is a tuple like: ('temperature', {'const_T1': boundaries, value)})
            # bc_rm is a tuple like: ('temperature', DirichletCondition(names, numbers, value))
            for bc_name, bc in bc_data[1].items():  # all boundary conditions of one b.c. type (e.g., Dirichlet with different temperatures)
                bnd_list_names = []
                bnd_list_numbers = []

                for bnd in bc.boundaries:  # all boundaries of one boundary condition
                    name = 'ht' + bnd
                    bnd_list_names.append(name)
                    bnd_list_numbers.append(bnd_pgs[bnd[-1]][0 if bnd in ['1i', '3o', '1l', '2h'] else 1])
                    # Overwrite general cooling and adiabatic condition
                    if self.data.magnet.solve.thermal.He_cooling.sides != 'adiabatic':
                        if name in rm['boundaries']['cooling']['names'][0]:
                            bnd_idx = rm['boundaries']['cooling']['names'][0].index(name)
                            rm['boundaries']['cooling']['names'][0].pop(bnd_idx)
                            rm['boundaries']['cooling']['numbers'][0].pop(bnd_idx)
                    if self.data.magnet.solve.thermal.He_cooling.sides != 'external':
                        if name in rm['boundaries']['heat_flux']['names'][0]:
                            bnd_idx = rm['boundaries']['heat_flux']['names'][0].index(name)
                            rm['boundaries']['heat_flux']['names'][0].pop(bnd_idx)
                            rm['boundaries']['heat_flux']['numbers'][0].pop(bnd_idx)

                rm['boundaries'][bc_rm]['names'].append(bnd_list_names)
                rm['boundaries'][bc_rm]['numbers'].append(bnd_list_numbers)
                if bc_data[0] == 'cooling':
                    rm['boundaries'][bc_rm]['values'].append([bc.heat_transfer_coefficient, self.data.magnet.solve.thermal.init_temperature])
                elif bc_data[0] == 'temperature':
                    rm['boundaries'][bc_rm]['values'].append([bc.const_temperature])
                elif bc_data[0] == 'heat_flux':
                    rm['boundaries'][bc_rm]['values'].append([bc.const_heat_flux])

        naming_conv = {'omega': 'Omega', 'boundary': 'Bd_', 'powered': '_p', 'insulator': '_ins', 'conducting': '_c', 'terms': 'Terms'}
        self.data.magnet.postproc.thermal.volumes = [naming_conv['omega'] + naming_conv[var]
                                             if naming_conv.get(var) is not None else naming_conv['omega'] for var in self.data.magnet.postproc.thermal.volumes]
        ap = aP(file_base_path=os.path.join(self.output_path, self.data.general.magnet_name), naming_conv=naming_conv)
        ap.assemble_combined_pro(external_templates_paths=[os.path.join(os.getcwd(), "surface_insulation_simple_model_utils")], template="simple_thermal_model.pro", rm=rm, dm=self.data)

        # Run GetDP
        resolution = 'Thermal_T'
        post_operation = 'Map_T'
        command = ["-solve", f"{resolution}", "-v2", "-pos", f"{post_operation}", "-verbose", str(self.data.run.verbosity_GetDP)]
        
        subprocess.call([f"{self.GetDP_path}",
                         os.path.join(self.output_path, f"{self.data.general.magnet_name}.pro")] + command + ["-msh", mesh_path])
        gmsh.onelab.setChanged("GetDP", 0)

        if self.gui:
            gmsh.fltk.initialize()
            while gmsh.fltk.isAvailable() and check_for_event(): gmsh.fltk.wait()
        else:
            gmsh.clear()
        gmsh.finalize()
