import os
import fiqus.utils.Utils as utl
from fiqus.data import DataFiQuS as dF
from fiqus import MainFiQuS as mf
from surface_insulation_simple_model_utils.simple_thermal_model import SimpleThermalModel as stm
from concurrent.futures import ProcessPoolExecutor, as_completed
from steam_sdk.analyses.AnalysisSTEAM import AnalysisSTEAM

# Input parameters/settings

# Target mesh sizes for the insulation
target_size_insulation = [1e-6, 10**(-5.66), 10**(-5.33), 1e-5, 10**(-4.66), 10**(-4.33), 1e-4]

# Number of parallel processes to run
max_parallel_processes = 3 # os.cpu_count()

run_ref = True # run reference models
run_tsa = True # run TSA models

# create analysis steam Object, just used to read the input yaml in this script
a = AnalysisSTEAM(file_name_analysis='input/FiQuS.yaml', verbose=False)

# Function to run the model for each size
def run_ref_model_for_size(output_path_ref, cable_width, cable_height, size, fds):
    output_path_ref_mesh = os.path.join(output_path_ref, str(size))
    utl.FilesAndFolders.prep_folder(output_path_ref_mesh)
    simple_thermal_model_ref = stm(
        magnet_name=magnet,
        elements_along_width=round(cable_width / (cable_height / 2)),
        target_size_insulation=size,
        target_size_conductor=cable_height / 3,
        output_path=output_path_ref_mesh,
        GetDP_path=a.settings.GetDP_path
    )
    print(f'Will run reference simple thermal model with insulation mesh size {size}')
    simple_thermal_model_ref.run_model()
    return(f'Finished running reference simple thermal model with insulation mesh size {size}')

# some names and paths
magnet = "simple_thermal_model"
output_path = os.path.join(os.getcwd(), 'output_thesis', magnet )
paths = {'Ref': os.path.join(output_path, magnet + '_ref'),
         'TSA': os.path.join(output_path, 'Geometry_TSA')}



if __name__ == '__main__':
        
    # Map futures to their inputs
    future_to_size = {}
    future_to_type = {}

    input_file = os.path.join(os.getcwd(), 'input', magnet, f'{magnet}.yaml')
    data = utl.FilesAndFolders.read_data_from_yaml(input_file, dF.FDM)

    if run_ref:
        # Read data from input file
        conductor = list(data.conductors.values())[0]
        cable_width = conductor.cable.bare_cable_width
        cable_height = conductor.cable.bare_cable_height_mean

        # run reference with surface insulation mesh
        output_path_ref = os.path.join(output_path, magnet + '_ref')
        utl.FilesAndFolders.prep_folder(output_path_ref, clear=True)

    if run_tsa:
        # TSA
        input_folder = os.path.join(os.getcwd(), 'input', magnet + '_mesh_sensitivity')
        if not os.path.exists(input_folder): 
            os.makedirs(input_folder)
        input_file_iter = os.path.join(input_folder, f'{magnet}.yaml')
        data.magnet.geometry.geom_file_path = os.path.join(os.getcwd(), 'input', magnet, f'{magnet}.geom')
        data.run.geometry = 'TSA'
        data.run.type = 'geometry_only'

        utl.FilesAndFolders.write_data_to_yaml(input_file_iter, data.dict())
        mf.MainFiQuS(input_file_path=input_file_iter, model_folder=output_path, fds=a.settings)


    # Parallelize the loop
    with ProcessPoolExecutor(max_workers=max_parallel_processes) as executor:

        if run_ref:
            for size in target_size_insulation:
                future = executor.submit(run_ref_model_for_size, output_path_ref, cable_width, cable_height, size, fds=a.settings)
                future_to_size[future] = size
                future_to_type[future] = 'Ref'

        if run_tsa:
            for i, size in enumerate(target_size_insulation[1:]):
                data.run.mesh = f'{size}'

                data.magnet.mesh.thermal.insulation.global_size = size
                data.magnet.mesh.thermal.insulation.TSA.global_size_QH = size
                input_file_iter = os.path.join(input_folder, f'{magnet}_{size}.yaml')

                data.run.type = 'mesh_only'

                # mesh sequentially to avoid OSHandle errors
                utl.FilesAndFolders.write_data_to_yaml(input_file_iter, data.dict())
                mf.MainFiQuS(input_file_path=input_file_iter, model_folder=output_path, fds=a.settings)

                data.run.type = 'solve_with_post_process_python'
                print(f'Will run TSA simple thermal model with insulation mesh size {size}')
                utl.FilesAndFolders.write_data_to_yaml(input_file_iter, data.dict())

                try:
                    future = executor.submit(mf.MainFiQuS, input_file_path=input_file_iter, model_folder=output_path, fds=a.settings)
                except Exception as e:
                   print(f"Failure for type {type} of size {size}: {e}")

                future_to_size[future] = size
                future_to_type[future] = 'TSA'

        for future in as_completed(future_to_size):
            size = future_to_size[future]
            type = future_to_type[future]
            try:
                result = future.result()
                pass
                print(f"Successfully ran {type} model with insulation size {size}.")
            except Exception as e:
                print(f"Failure for type {type} of size {size}: {e}")
