#!/bin/bash

# This script executes in Terminal 

################### Initialize #####################

# Change to the current directory
cd "$(dirname "$0")"

# Define absolute path 
DATA_PATH=$(pwd)

# Create 'output' folder if it does not exist
OUTPUT_FOLDER="output"
if [ ! -d "$OUTPUT_FOLDER" ]; then
    mkdir "$OUTPUT_FOLDER"
    echo "Created folder: $OUTPUT_FOLDER"
else
    echo "Folder '$OUTPUT_FOLDER' already exists."
fi


################### Helper Functions #####################

# Function to locate MATLAB
locate_matlab() {
    local matlab_exec

    # Check using 'which'
    matlab_exec=$(which matlab 2>/dev/null)
    if [ -x "$matlab_exec" ]; then
        echo "$matlab_exec"
        return
    fi

    # Search for MATLAB in /Applications, /usr/local/MATLAB, and other common locations
    for version in $(ls /Applications | grep -E 'MATLAB_R[0-9]+[a-z]'); do
        if [ -x "/Applications/$version/bin/matlab" ]; then
            echo "/Applications/$version/bin/matlab"
            return
        fi
    done

    for version in $(ls /usr/local/MATLAB 2>/dev/null); do
        if [ -x "/usr/local/MATLAB/$version/bin/matlab" ]; then
            echo "/usr/local/MATLAB/$version/bin/matlab"
            return
        fi
    done

    # Last fallback: check /usr/bin/matlab
    if [ -x "/usr/bin/matlab" ]; then
        echo "/usr/bin/matlab"
    else
        echo ""  # MATLAB not found
    fi
}

# Function to locate Python
locate_python() {
    local python_exec

    # Check using 'which'
    python_exec=$(which python3 2>/dev/null)
    if [ -x "$python_exec" ]; then
        echo "$python_exec"
        return
    fi

    # Check common installation paths
    if [ -x "/usr/bin/python3" ]; then
        echo "/usr/bin/python3"
    elif [ -x "/usr/local/bin/python3" ]; then
        echo "/usr/local/bin/python3"
    elif [ -x "/opt/homebrew/bin/python3" ]; then
        echo "/opt/homebrew/bin/python3"  # For macOS with Homebrew
    else
        echo ""  # Python not found
    fi
}

# Function to locate Stata
locate_stata() {
    local stata_exec

    # Check using 'which'
    stata_exec=$(which stata-se 2>/dev/null)
    if [ -x "$stata_exec" ]; then
        echo "$stata_exec"
        return
    fi

    # Check common installation paths
    if [ -x "/Applications/Stata/StataSE.app/Contents/MacOS/stata-se" ]; then
        echo "/Applications/Stata/StataSE.app/Contents/MacOS/stata-se"
    elif [ -x "/usr/local/stata/stata-se" ]; then
        echo "/usr/local/stata/stata-se"
    elif [ -x "/usr/bin/stata-se" ]; then
        echo "/usr/bin/stata-se"
    else
        echo ""  # Stata not found
    fi
}

# Function to set custom PATHs for MATLAB, Stata, and Python
set_custom_paths() {
    local matlab_exec
    local stata_exec
    local python_exec

    matlab_exec=$(locate_matlab)
    stata_exec=$(locate_stata)
    python_exec=$(locate_python)

    if [ -n "$matlab_exec" ]; then
        export PATH=$PATH:"$(dirname "$matlab_exec")"
        echo "MATLAB executable located: $matlab_exec"
    else
        echo "MATLAB executable not found! Please install MATLAB or specify its path manually."
        exit 1
    fi

    if [ -n "$stata_exec" ]; then
        export PATH=$PATH:"$(dirname "$stata_exec")"
        echo "Stata executable located: $stata_exec"
    else
        echo "Stata executable not found! Please install Stata or specify its path manually."
        exit 1
    fi

    if [ -n "$python_exec" ]; then
        export PATH=$PATH:"$(dirname "$python_exec")"
        echo "Python executable located: $python_exec"
    else
        echo "Python executable not found! Please install Python or specify its path manually."
        exit 1
    fi
}

# Function to install required Python packages 
install_python_packages() {
    REQUIRED_PACKAGES=(
        pandas
        seaborn
        matplotlib
        scikit-learn
        unidecode
    )

    for PACKAGE in "${REQUIRED_PACKAGES[@]}"; do
        pip show "$PACKAGE" > /dev/null 2>&1
        if [ $? -ne 0 ]; then
            echo "Installing missing package: $PACKAGE"
            pip3 install "$PACKAGE" -q > /dev/null 2>&1
        else
            echo "Package '$PACKAGE' is already installed."
        fi
    done
}

# Function to install Stata packages 
install_stata_packages() {
    echo "Checking required Stata packages..."

    REQUIRED_PACKAGES=(
    "estout"     # Provides esttab and eststo
    "coefplot"   # For coefficient plots
    "runtest"    # Runs test for serial correlation
    "labmask"    # Label masking for variables
    "reghdfe"    # Fixed effect estimator
    )

    for package in "${REQUIRED_PACKAGES[@]}"; do
        stata-se -q -b do <(echo "cap which $package; if _rc ssc install $package;") > /dev/null 2>&1
        if [ $? -ne 0 ]; then
            echo "Error installing package: $package"
            exit 1
        else
            echo "Package '$package' is installed."
        fi
    done

    echo "All required packages are installed."
}

# Function to run Python scripts 
run_python_script() {
    local script_path=$1

    if [ -f "$script_path" ]; then
        echo "Running Python script: $script_path"

        python3 "$script_path" "$DATA_PATH" 2>/dev/null
        if [ $? -ne 0 ]; then
            echo "Error executing $(basename $script_path)."
            exit 1
        else
            echo "$(basename $script_path) executed successfully."
        fi
    else
        echo "Python script '$script_path' not found!"
        exit 1
    fi
}

# Function to run Stata scripts and measure time
run_stata_script() {
    local script_path=$1
    local start_time end_time elapsed_time

    if [ -f "$script_path" ]; then
        echo "Running Stata script: $script_path"
        start_time=$(date +%s)  # Record start time

        stata-se -b do "$script_path" "$DATA_PATH" > /dev/null 2>&1
        if [ $? -ne 0 ]; then
            echo "Error executing $(basename $script_path)."
            exit 1
        else
            echo "$(basename $script_path) executed successfully."
        fi

        end_time=$(date +%s)  # Record end time
        elapsed_time=$((end_time - start_time))
        echo "Processing time for $(basename $script_path): ${elapsed_time}s"
    else
        echo "Stata script '$script_path' not found!"
        exit 1
    fi

    rm -f *.log
}


# Function to run MATLAB scripts and measure time
run_matlab_script() {
    local script_path=$1
    local start_time end_time elapsed_time

    if [ -f "$script_path" ]; then
        echo "Running MATLAB script: $script_path" 
        start_time=$(date +%s)  # Record start time

        matlab -batch "run('$script_path'); exit;" 2>/dev/null
        if [ $? -ne 0 ]; then
            echo "Error executing $(basename $script_path)."
            exit 1
        else
            echo "$(basename $script_path) executed successfully."
        fi

        end_time=$(date +%s)  # Record end time
        elapsed_time=$((end_time - start_time))
        echo "Processing time for $(basename $script_path): ${elapsed_time}s"
    else
        echo "MATLAB script '$script_path' not found!"
        exit 1
    fi
}

################### Run Scripts #####################

# Automatically locate and set paths for MATLAB, Stata, and Python
set_custom_paths

# # Python Script
install_python_packages
run_python_script "code/Heatmap.py"

# Stata Scripts
install_stata_packages
run_stata_script "code/Datasets.do"
run_stata_script "code/EventStudy.do"

# MATLAB Scripts
run_matlab_script "code/Estimation/Estimation.m" # run time: ~ 10min
run_matlab_script "code/Estimation/AppendixB.m"
run_matlab_script "code/Estimation/MonteCarlo.m" # run time: ~ 180min (14553s)

################### Conclusion #####################
echo "All scripts executed successfully. Check output folder for results."
