%% This script implements Monte Carlo experiments 
%% for MMSE and MLE estimators.

clear all
clc

% Initialize model parmeters
para.N  		= 3;
para.c0 	 	= ones(para.N,1).*0;
para.c1 	 	= ones(para.N,1)*2; 
para.gamma      = ones(para.N,1).*1;
para.beta0      = para.c0;
para.beta1      = para.c1 - para.gamma;
para.delta      = - para.gamma./(para.N - 1);
para.sigma      = 1;

S       = 500;
rhogrid = [0; 0.2; 0.5]; % correlation between private shocks
Ggrid   = [50; 100; 500]; % number of events

resultsNested   = zeros(length(rhogrid), length(Ggrid), 4, 2); % mean and std for c0, c1, rho, sigma
resultsMMSE     = zeros(length(rhogrid), length(Ggrid), 2, 2); % mean and std for c0, c1

% Monte Carlo Experiments
for k1 = 1:length(rhogrid)
    para.rho = rhogrid(k1);
    for k2 = 1:length(Ggrid)
        G = Ggrid(k2);

        % Generate random state variables X for all S simulations
        seed = RandStream("dsfmt19937"); 
        reset(seed); % Reset the seed to ensure reproducibility
        Xrand = rand(seed, [G para.N S]);

        fprintf('MONTE CARLO FOR MODEL WITH rho = %d, G = %d \n', para.rho, G);
        parfor_progress(S); % Initialize

        paraMMSE    = zeros(S, 2);
        paraNested  = zeros(S, 4);

        parfor s = 1:S

            rng(s,'twister') % fixes the seed for generated sample
            [Y, X] = simulateData(para, G, s, Xrand);

            paraMMSE(s, :)   = TwoStepMMSE(X, Y);
            paraNested(s, :) = NestedMLE(X, Y);

            parfor_progress; % Count
        end
        parfor_progress(0); % Clean up

        % Store results for Nested
        resultsNested(k1, k2, :, 1) = mean(paraNested);
        resultsNested(k1, k2, :, 2) = std(paraNested);

        % Store results for MMSE
        resultsMMSE(k1, k2, :, 1) = mean(paraMMSE);
        resultsMMSE(k1, k2, :, 2) = std(paraMMSE);

    end
end

% Write results to LaTeX table
f = fopen('../../output/table10.tex', 'w+');

fprintf(f, '\\begin{tabular}{lccc ccc ccc}\n');
fprintf(f, '\\toprule\n');

% Write Nested results
fprintf(f, '\\hline\n');
fprintf(f, '& \\multicolumn{9}{c}{Nested Fixed Point Estimation (MLE)} \\\\\n');

metrics = {'$\hat{c_0}$', '$\hat{c_1}$', '$\hat{\rho}$', '$\hat{\sigma}$'};
for m = 1:4
    fprintf(f, '%s ', metrics{m});
    for k1 = 1:length(rhogrid)
        for k2 = 1:length(Ggrid)
            fprintf(f, '& %4.3f ', resultsNested(k1, k2, m, 1));
        end
    end
    fprintf(f, '\\\\\n');
    fprintf(f, '& ');
    for k1 = 1:length(rhogrid)
        for k2 = 1:length(Ggrid)
            fprintf(f, '& (%4.3f) ', resultsNested(k1, k2, m, 2));
        end
    end
    fprintf(f, '\\\\\n');
end

% Write MMSE results
fprintf(f, '\\hline\n');
fprintf(f, '& \\multicolumn{9}{c}{Two-step MMSE Estimation} \\\\\n');

metricsMMSE = {'$\hat{c_0}$', '$\hat{c_1}$'};
for m = 1:2
    fprintf(f, '%s ', metricsMMSE{m});
    for k1 = 1:length(rhogrid)
        for k2 = 1:length(Ggrid)
            fprintf(f, '& %4.3f ', resultsMMSE(k1, k2, m, 1));
        end
    end
    fprintf(f, '\\\\\n');
    fprintf(f, '& ');
    for k1 = 1:length(rhogrid)
        for k2 = 1:length(Ggrid)
            fprintf(f, '& (%4.3f) ', resultsMMSE(k1, k2, m, 2));
        end
    end
    fprintf(f, '\\\\\n');
end

fprintf(f, '\\bottomrule\n');
fprintf(f, '\\end{tabular}\n');

fclose(f);
