%% This script estimates the structural model in Section 5 & Appendix D 
%% and computes the counterfactual results in Section 7 & Appendix F

clear all
clc

% read data
filename = '../../data/estimation.csv';
data = readtable(filename);

Y = table2array(data(:,2:4)); % decision
X = table2array(data(:,5:7)); % traffic

% --------------------------------------------------------------- %
% Nested Fixed Point Estimation (NFXP)
% --------------------------------------------------------------- % 

display("Nested Fixed Point Estimation")

s       = RandStream('mlfg6331_64'); % for reproducibility
reset(s);
opt     = statset('UseParallel',true, 'UseSubstreams', true,'Streams', s);
nboot   = 500;

paraNested = NestedMLE(X, Y);
NESTEDci   = bootci(nboot, {@NestedMLE, X,Y}, 'Options', opt);

% create LaTex table
f = fopen('../../output/table6.tex', 'w+');

    fprintf(f, '\\begin{tabular}{lccccc}\n');
    fprintf(f, '\\toprule\\hline\n');
    fprintf(f, '           & $\\hat{c_0}$ & $\\hat{c_1}$ & $(\\gamma)^{-\\alpha}$ & $\\hat{\\rho}$ & $\\hat{\\sigma}$ \\\\\n');
    fprintf(f, '    \\cmidrule(lr){2-6}\n');

    % Write NFXP row
    fprintf(f, 'NFXP & %4.4f & %4.4f & 1 & %4.4f & %4.4f \\\\\n',...
    paraNested(1), paraNested(2), paraNested(3), paraNested(4));

    % Write confidence intervals
    fprintf(f, '& [%4.4f, %4.4f] & [%4.4f, %4.4f] & [$\\cdot$] & [%4.4f, %4.4f] & [%4.4f, %4.4f] \\\\\n', ...
        NESTEDci(1, 1), NESTEDci(2, 1), NESTEDci(1, 2), NESTEDci(2, 2), ...
    NESTEDci(1, 3), NESTEDci(2, 3), NESTEDci(1, 4), NESTEDci(2, 4));

    fprintf(f, '\\bottomrule\n');
    fprintf(f, '\\end{tabular}\n');
fclose(f);

% --------------------------------------------------------------- %
% Model Fit (NFXP)
% --------------------------------------------------------------- % 
display ("Model Fit")

N              = size(X,2);
para.beta0     =  ones(N,1).* paraNested(1); 
para.beta1     =  ones(N,1).* (paraNested(2)-1);	
para.delta     =  - ones(N,1)./(N-1);
para.rho  	   =  paraNested(3);
para.sigma     =  paraNested(4);
epsilonEq 	   =  solveEq(para, X);
sigmahat       = 1 - normcdf(epsilonEq,0,para.sigma);
sigmaEst       = mean(sigmahat);
sigmaObs       = mean(Y);


f = fopen('../../output/table7.tex', 'w+');

    fprintf(f, '\\begin{tabular}{llcc}\n');
    fprintf(f, '\\toprule\\hline\n');
    fprintf(f, '& Big Platform & Medium Platform & Small Platform  \\\\\n');
    fprintf(f, '\\cmidrule(lr){2-4}\n');

    fprintf(f, 'Observed & %4.4f & %4.4f & %4.4f \\\\\n', sigmaObs);
    fprintf(f, 'Predicted (NFXP) & %4.4f & %4.4f & %4.4f \\\\\n', sigmaEst);

    fprintf(f, '\\bottomrule\n');
    fprintf(f, '\\end{tabular}\n');

fclose(f);

% --------------------------------------------------------------- %
% Counterfactual (NFXP)
% --------------------------------------------------------------- % 

display("NFXP COUNTERFACTUAL - SHUTDOWN")

s       = RandStream('mlfg6331_64'); % for reproducibility
reset(s);
opt     = statset('UseParallel', true, 'UseSubstreams', true, 'Streams', s);
nboot   = 500;

result = shutdown(X, Y);
SHUTDOWN_NFXP = @(X, Y) shutdown(X, Y);
resultci = bootci(nboot, {SHUTDOWN_NFXP, X, Y}, 'Options', opt);
reset(s);

delta_aM_shutdown = result(4);
delta_aB_shutdown = result(3);
delta_SC_shutdown = result(1);
delta_SC_noInter_shutdown = result(2);

ci_aM_shutdown = resultci(:, 4);
ci_aB_shutdown = resultci(:, 3);
ci_SC_shutdown = resultci(:, 1);
ci_SC_noInter_shutdown = resultci(:, 2);

display("NFXP COUNTERFACTUAL - MERGER")

reset(s);
result = merger(X, Y);
MERGER_NFXP = @(X, Y) merger(X, Y);
resultci = bootci(nboot, {MERGER_NFXP, X, Y}, 'Options', opt);

delta_aM_merger = result(4);
delta_aB_merger = result(3);
delta_SC_merger = result(1);
delta_SC_noInter_merger = result(2);

ci_aM_merger = resultci(:, 4);
ci_aB_merger = resultci(:, 3);
ci_SC_merger = resultci(:, 1);
ci_SC_noInter_merger = resultci(:, 2);

% Create LaTeX table
f = fopen('../../output/table8.tex', 'w+');

fprintf(f, '\\begin{tabular}{lcc}\n');
fprintf(f, '\\toprule\n');
fprintf(f, '& \\multicolumn{2}{c}{Nested Fixed Point Estimation} \\\\\n');
fprintf(f, '\\cmidrule{2-3}\n');
fprintf(f, '& Permanent Shutdown & Merging Two Platforms \\\\\n');
fprintf(f, '\\hline\n');

% $ \Delta a_M $
fprintf(f, '$ \\Delta a_M $ & %4.4f & %4.4f \\\\\n', delta_aM_shutdown, delta_aM_merger);
fprintf(f, '& [%4.4f, %4.4f] & [%4.4f, %4.4f] \\\\\n', ci_aM_shutdown(1), ci_aM_shutdown(2), ci_aM_merger(1), ci_aM_merger(2));

% $ \Delta a_B $
fprintf(f, '$ \\Delta a_B $ & %4.4f & %4.4f \\\\\n', delta_aB_shutdown, delta_aB_merger);
fprintf(f, '& [%4.4f, %4.4f] & [%4.4f, %4.4f] \\\\\n', ci_aB_shutdown(1), ci_aB_shutdown(2), ci_aB_merger(1), ci_aB_merger(2));

% $ \Delta SC $
fprintf(f, '$ \\Delta SC $ & %4.3f\\%% & %4.3f\\%% \\\\\n', delta_SC_shutdown, delta_SC_merger);
fprintf(f, '& [%4.3f\\%%, %4.3f\\%%] & [%4.3f\\%%, %4.3f\\%%] \\\\\n', ci_SC_shutdown(1), ci_SC_shutdown(2), ci_SC_merger(1), ci_SC_merger(2));

% $ \Delta \tilde{SC} $
fprintf(f, '$ \\Delta \\tilde{SC} $ & %4.3f\\%% & %4.3f\\%% \\\\\n', delta_SC_noInter_shutdown, delta_SC_noInter_merger);
fprintf(f, '(w/o strategic incentive) & [%4.3f\\%%, %4.3f\\%%] & [%4.3f\\%%, %4.3f\\%%] \\\\\n', ci_SC_noInter_shutdown(1), ci_SC_noInter_shutdown(2), ci_SC_noInter_merger(1), ci_SC_noInter_merger(2));

fprintf(f, '\\bottomrule\n');
fprintf(f, '\\end{tabular}\n');

fclose(f);

% --------------------------------------------------------------- %
% Appendix D: Two-step MMSE 
% --------------------------------------------------------------- % 

display("Two-step MMSE")

[paraMMSE,fval] = TwoStepMMSE(X, Y);

s       = RandStream('mlfg6331_64'); % for reproducibility
reset(s);
opt     = statset('UseParallel',true, 'UseSubstreams', true,'Streams', s);
nboot   = 500;
MMSEci 	= bootci(nboot, {@TwoStepMMSE, X,Y},'Options', opt);


f = fopen('../../output/table9.tex', 'w+');

    fprintf(f, '\\begin{tabular}{lccccc}\n');
    fprintf(f, '\\toprule\\hline\n');
    fprintf(f, '           & $\\hat{c_0}$ & $\\hat{c_1}$ & $(\\gamma)^{-\\alpha}$ & $\\hat{\\rho}$ & $\\hat{\\sigma}$ \\\\\n');
    fprintf(f, '    \\cmidrule(lr){2-6}\n');

    % Write Two-step MMSE row
    fprintf(f, '2-step MMSE & %4.4f & %4.4f & 1 & N/A & N/A \\\\\n', paraMMSE(1), paraMMSE(2));
    fprintf(f, '& [%4.4f, %4.4f] & [%4.4f, %4.4f] & [$\\cdot$] & N/A & N/A \\\\\n', MMSEci(1, 1), MMSEci(2, 1), MMSEci(1, 2), MMSEci(2, 2));
    
    % Write NFXP row
    fprintf(f, 'NFXP & %4.4f & %4.4f & 1 & %4.4f & %4.4f \\\\\n', paraNested(1), paraNested(2), paraNested(3), paraNested(4));
    fprintf(f, '& [%4.4f, %4.4f] & [%4.4f, %4.4f] & [$\\cdot$] & [%4.4f, %4.4f] & [%4.4f, %4.4f] \\\\\n', ...
        NESTEDci(1, 1), NESTEDci(2, 1), NESTEDci(1, 2), NESTEDci(2, 2), ...
        NESTEDci(1, 3), NESTEDci(2, 3), NESTEDci(1, 4), NESTEDci(2, 4));

    fprintf(f, '\\bottomrule\\n');
    fprintf(f, '\\end{tabular}\\n');

fclose(f);

% --------------------------------------------------------------- %
% Appendix F: Common Ownership 
% --------------------------------------------------------------- % 

display("NFXP (Common Ownership)")

nboot 	= 500;
s       = RandStream('mlfg6331_64'); % for reproducibility
reset(s);
opt     = statset('UseParallel',true, 'UseSubstreams', true,'Streams', s);

paraNested_JO = NestedMLE_JO(X, Y);
[NESTEDci_JO, bootstat]  = bootci(nboot, {@NestedMLE_JO, X,Y}, 'Options', opt);

% Export to LaTex
f = fopen('../../output/table11.tex', 'w+');

    fprintf(f, '\\begin{tabular}{lcccc}\n');
    fprintf(f, '\\toprule\\hline\n');
    fprintf(f, '& $\\hat{c_0}$ & $\\hat{c_1}$ & $\\hat{\\rho}$ & $\\hat{\\sigma}$ \\\\\n');
    fprintf(f, '\\cmidrule(lr){2-5}\n');

    % Common Ownership
    fprintf(f, 'NFXP (Common Ownership) & %4.4f & %4.4f & %4.4f & %4.4f \\\\\n', ...
            paraNested_JO(1), paraNested_JO(2), paraNested_JO(3), paraNested_JO(4));
    fprintf(f, '& [%4.4f, %4.4f] & [%4.4f, %4.4f] & [%4.4f, %4.4f] & [%4.4f, %4.4f] \\\\\n', ...
    NESTEDci_JO(1, 1), NESTEDci_JO(2, 1), NESTEDci_JO(1, 2), NESTEDci_JO(2, 2), ...
    NESTEDci_JO(1, 3), NESTEDci_JO(2, 3), NESTEDci_JO(1, 4), NESTEDci_JO(2, 4));

    % Baseline
    fprintf(f, 'NFXP (Baseline) & %4.4f & %4.4f & %4.4f & %4.4f \\\\\n', ...
            paraNested(1), paraNested(2), paraNested(3), paraNested(4));
    fprintf(f, '& [%4.4f, %4.4f] & [%4.4f, %4.4f] & [%4.4f, %4.4f] & [%4.4f, %4.4f] \\\\\n', ...
        NESTEDci(1, 1), NESTEDci(2, 1), NESTEDci(1, 2), NESTEDci(2, 2), ...
        NESTEDci(1, 3), NESTEDci(2, 3), NESTEDci(1, 4), NESTEDci(2, 4));

    fprintf(f, '\\bottomrule\n');
    fprintf(f, '\\end{tabular}\n');

fclose(f);

% --------------------------------------------------------------- % 

display("Counterfactual (Common Ownership)")

% shutdown
reset(s);
result = shutdown_JO(X,Y); 
SHUTDOWN_NFXP = @(X,Y) shutdown_JO(X,Y);
resultci = bootci(nboot, {SHUTDOWN_NFXP ,X ,Y}, 'Options', opt);

delta_aM_shutdown_JO = result(3);
delta_aB_shutdown_JO = result(2);
delta_SC_shutdown_JO = result(1);

ci_aM_shutdown_JO = resultci(:, 3);
ci_aB_shutdown_JO = resultci(:, 2);
ci_SC_shutdown_JO = resultci(:, 1);

% merger
reset(s);
result 		 = merger_JO(X,Y);
MERGER_NFXP  = @(X,Y) merger_JO(X,Y);
resultci 	 = bootci(nboot, {MERGER_NFXP, X,Y}, 'Options', opt);

delta_aM_merger_JO = result(3);
delta_aB_merger_JO = result(2);
delta_SC_merger_JO = result(1);

ci_aM_merger_JO = resultci(:, 3);
ci_aB_merger_JO = resultci(:, 2);
ci_SC_merger_JO = resultci(:, 1);

% create LaTex table
f = fopen('../../output/table12.tex', 'w+');

fprintf(f, '\\begin{center}\n');
fprintf(f, '\\begin{tabular}{lcccc}\n');
fprintf(f, '\\toprule\\hline\n');
fprintf(f, '& \\multicolumn{2}{c}{Permanent Shutdown} & \\multicolumn{2}{c}{Merging Two Platforms} \\\\ \n');
fprintf(f, '\\cmidrule{2-5}\n');
fprintf(f, '& common ownership & baseline & common ownership & baseline \\\\ \n');
fprintf(f, '\\cmidrule{2-5}\n');

% $ \Delta a_M $
fprintf(f, '$ \\Delta a_M $ & %4.4f & %4.4f & %4.4f & %4.4f \\\\ \n',...
    delta_aM_shutdown_JO, delta_aM_shutdown, delta_aM_merger_JO, delta_aM_merger);
fprintf(f, '& [%4.4f, %4.4f] & [%4.4f, %4.4f] & [%4.4f, %4.4f] & [%4.4f, %4.4f] \\\\ \n', ...
    ci_aM_shutdown_JO(1), ci_aM_shutdown_JO(2), ci_aM_shutdown(1), ci_aM_shutdown(2), ...
    ci_aM_merger_JO(1), ci_aM_merger_JO(2), ci_aM_merger(1), ci_aM_merger(2));

% $ \Delta a_B $
fprintf(f, '$ \\Delta a_B $ & %4.4f & %4.4f & %4.4f & %4.4f \\\\ \n',...
    delta_aB_shutdown_JO, delta_aB_shutdown, delta_aB_merger_JO, delta_aB_merger);
fprintf(f, '& [%4.4f, %4.4f] & [%4.4f, %4.4f] & [%4.4f, %4.4f] & [%4.4f, %4.4f] \\\\ \n', ...
    ci_aB_shutdown_JO(1), ci_aB_shutdown_JO(2), ci_aB_shutdown(1), ci_aB_shutdown(2), ...
    ci_aB_merger_JO(1), ci_aB_merger_JO(2), ci_aB_merger(1), ci_aB_merger(2));

% $ \Delta SC $
fprintf(f, '$ \\Delta SC $ & %4.3f\\%% & %4.3f\\%% & %4.3f\\%% & %4.3f\\%% \\\\ \n', delta_SC_shutdown_JO, delta_SC_shutdown, delta_SC_merger_JO, delta_SC_merger);
fprintf(f, '& [%4.3f\\%%, %4.3f\\%%] & [%4.3f\\%%, %4.3f\\%%] & [%4.3f\\%%, %4.3f\\%%] & [%4.3f\\%%, %4.3f\\%%] \\\\ \n', ...
    ci_SC_shutdown_JO(1), ci_SC_shutdown_JO(2), ci_SC_shutdown(1), ci_SC_shutdown(2), ...
    ci_SC_merger_JO(1), ci_SC_merger_JO(2), ci_SC_merger(1), ci_SC_merger(2));

fprintf(f, '\\bottomrule\n');
fprintf(f, '\\end{tabular}\n');
fprintf(f, '\\end{center}\n');

fclose(f);


