#------------------------------------------------------------------
#libraries

# tidyverse bliss
library(tidyverse)
library(magrittr)
library(timetk)
library(tidyquant)  # Loads tidyverse, financial pkgs, used to get data
library(tsibble)
library(dplyr)

#library(tidymodels)
library(feasts)
library(pracma)     # for repmat
library(tsbox)      # convert everything to anything
library(reshape2)
library(tibbletime)

library(quantreg)

# file navigation
library(here)

#raster
library(raster)

# access python
library(reticulate)

#visualisation
library(RColorBrewer)

###########################
## Working directories
dir <- here()
## path to SWOP maps exported from GEE in Step 1:
dir_surf_occ <- paste0(dirname(dir), "/STEP_1/SurfWaterFreq_S2_17to22_Lakes_combination_v5/")

# Please register for an Earthdata login profile: https://urs.earthdata.nasa.gov/users/new
# ------------------------------------------------------------------
uid = 'magosilvain'
pswd = 'Wegnia2Abijata!'
email = 'silvan.ragettli@gmail.com'
# ------------------------------------------------------------------

setwd(dir)
all_surface_occ_prob_maps <- list.files(dir_surf_occ)
existing_files <- list.files(paste0(dir,'Outputs_atl13'))
missing_files <- setdiff(all_surface_occ_prob_maps, paste0(existing_files,'.tif'))
#------------------------------------------------------------------
# python code for downloading icesat-2 data
# Please make sure that  miniconda or conda is installed on your machine!
# find the path to conda environment using "conda_list()" function before using function "use_condaenv()"
reticulate::use_condaenv(condaenv = "base", conda = "auto", required = TRUE)
# reticulate::use_condaenv(condaenv = paste0('C:/Users/',User,'/anaconda3/python.exe'))
#reticulate::use_condaenv("/usr/local/bin/python3")
reticulate::source_python('download_icesat2_atl13_per_groundtrack_par_adj.py')
sys = import('sys')
sys$stdout$flush()
#------------------------------------------------------------------
# functions
source('functions.R')

#------------------------------------------------------------------
#-------------------MAIN CODE--------------------------------------
#------------------------------------------------------------------
# tibble to summarize status of the DEM creation process for all lakes
message = tibble(f_map = all_surface_occ_prob_maps, status="" )

# iterate through surface occurence percentiles map
#for (i in missing_files){
#for (i in all_surface_occ_prob_maps){
for (i in all_surface_occ_prob_maps[c(1:length(all_surface_occ_prob_maps))]){
#i = "SurfWaterFreq_S2_17to22_49.tif"
print(i)
  # get water body ID
  id0 <- strsplit(i, "_")[[1]]
  id <-  strsplit(id0[length(id0)] , ".tif")[[1]][1]
  # load and plot surface occurence percentile map
  surface_occ_prob <- raster(paste0(dir_surf_occ, i))
  raster::plot(surface_occ_prob, col=brewer.pal(9, "Blues"), main=i)

  # get extents of the map to define the bounding box (as a string) for icesat data download
  extents <- extent(surface_occ_prob)
  bounding_box = paste(as.character(extents@xmin), as.character(extents@ymin), as.character(extents@xmax), as.character(extents@ymax), sep = ",")
  print(bounding_box)

  # call python function for icesat elevation data dowﬁnload
  print("Download ICESat-2 Data via NSIDC: Check Python Script")
  icesat_profiles_download(bounding_box, substring(i, 1, nchar(i)-4),uid, pswd, email)
  all_files_atl13 <- list.files(paste0(getwd(), "/Outputs_atl13/",substring(i, 1, nchar(i)-4)),pattern="processed_ATL13*")
  
  # skip iteration: if there is no icesat elevation data for this waterbody
  if (isempty(all_files_atl13)) {
    print("No ICESat-2 Data is available for this inland water body.")
    message$status[message$f_map==i] <- ("No ICESat-2 Data is available for this inland water body.")
    next}
  
  # tibble to store elevation points from all groundtracks
  t <- tibble()
  ind<-0
  # iterate through ascii tables: each table represents one groundtrack
  for (filename in all_files_atl13){
    date <- ISOdate(substring(filename, 17, 20), substring(filename, 21,22), substring(filename, 23, 24)) %>% 
      as.Date()
    gtx <- as.numeric(substring(filename, 49, 49)) * gregexpr("r",substring(filename, 50, 50))[[1]][1]
    text=read.csv(paste0(here(),"/Outputs_atl13/",paste0(substring(i, 1, nchar(i)-4)),"/", filename))
    
    # read downloaded ascii file
    t_temp<-read.table(paste0(here(),"/Outputs_atl13/", substring(i, 1, nchar(i)-4), "/", filename), 
                       colClasses ="character", header=TRUE, blank.lines.skip=TRUE,# fill = TRUE,
                       row.names=NULL,
                       
                       col.names=c('date','delta_time','ht_ortho', 'ht_water_surf', 'segment_lat','segment_lon', 'mean_lat', 'mean_lon', 'date','time')) 
    
    colnames=c('ht_ortho', 'ht_water_surf', 'mean_lat', 'mean_lon', 'date', 'mean_time') #  ,'slope', 'water_depth'
    t_temp=t_temp[,-c(1,2,5,6)]
    
    colnames(t_temp)=colnames
    
    # discard if less than 2 data points
    if (nrow(t_temp) < 2) {
      t_temp<-add_row(t_temp,t_temp)
      ind<-1
    }
    #convert ascii table to tibble
    t_temp <- as_tibble(apply(t_temp, 2, function(x) as.numeric(x)))
    if (ind==1) {
      t_temp<-t_temp[-c(2),]
      ind<-0
    }    
    #add date column
    t_temp$date<-date
    
    #add gtx column
    t_temp$gtx<-gtx
    
    # filter segments where lat and lon are defined
    t_temp <- t_temp %>% filter(!is.na(mean_lon)) %>% filter(!is.na(mean_lat))
    
    # coordinate vector for raster::extract
    coords<-cbind(t_temp$mean_lon, t_temp$mean_lat)
    
    # retrieve percentile for each elevation value from the contour map
    t_temp$percentile <- raster::extract(surface_occ_prob, coords)
    
    # add to the previous groundtracks
    t <- t %>% bind_rows(t_temp)
  }
 
#save data retrieved from icesat   
    write_csv(t, paste0(here(), "/Bathymetries/icesat_data_atl13/icesat_data_ID_", id, '.csv'),'append'=FALSE)
  
  removeTmpFiles(h=2)
}

# store csv with messages
#write_csv(message, paste0(getwd(), '/DEMs/', 'summary.csv'))

