#------------------------------------------------------------------
#libraries

# tidyverse bliss
library(tidyverse)
library(magrittr)
library(timetk)
library(tidyquant)  # Loads tidyverse, financial pkgs, used to get data
library(tsibble)
library(dplyr)

#library(tidymodels)
library(feasts)
library(pracma)     # for repmat
library(tsbox)      # convert everything to anything
library(reshape2)
library(tibbletime)

library(quantreg)

# file navigation
library(here)

#raster
library(raster)

# access python
library(reticulate)

#visualisation
library(RColorBrewer)

###########################
## Working directories
dir <- here()
## path to SWOP maps exported from GEE in Step 1:
dir_surf_occ <- paste0(dirname(dir), "/STEP_1/SurfWaterFreq_S2_17to22_Lakes_combination_v5/")

# Please register for an Earthdata login profile: https://urs.earthdata.nasa.gov/users/new
# ------------------------------------------------------------------
uid = 'magosilvain'
pswd = 'Wegnia2Abijata!'
email = 'silvan.ragettli@gmail.com'
# ------------------------------------------------------------------

setwd(dir)
all_surface_occ_prob_maps <- list.files(dir_surf_occ)
#------------------------------------------------------------------
# python code for downloading icesat-2 data
# Please make sure that  miniconda or conda is installed on your machine!
# find the path to conda environment using "conda_list()" function before using function "use_condaenv()"
reticulate::use_condaenv(condaenv = "base", conda = "auto", required = TRUE)
# reticulate::use_condaenv(condaenv = paste0('C:/Users/',User,'/anaconda3/python.exe'))
#reticulate::use_condaenv("/usr/local/bin/python3")
reticulate::source_python('download_icesat2_atl08_20m.py')
sys = import('sys')
sys$stdout$flush()
#------------------------------------------------------------------
# functions
source('functions.R')

#------------------------------------------------------------------
#-------------------MAIN CODE--------------------------------------
#------------------------------------------------------------------
# tibble to summarize status of the DEM creation process for all lakes
message = tibble(f_map = all_surface_occ_prob_maps, status="" )

# iterate through surface occurence percentiles map
for (i in all_surface_occ_prob_maps[c(1:length(all_surface_occ_prob_maps))]){
# i = "SurfWaterFreq_S2_17to22_77.tif"
  print(i)
  # get water body ID
  id0 <- strsplit(i, "_")[[1]]
  id <-  strsplit(id0[length(id0)] , ".tif")[[1]][1]

  # load and plot surface occurence percentile map
  surface_occ_prob <- raster(paste0(dir_surf_occ, i))
  raster::plot(surface_occ_prob, col=brewer.pal(9, "Blues"), main=i)
  
  # get extents of the map to define the bounding box (as a string) for icesat data download
  extents <- extent(surface_occ_prob)
  bounding_box = paste(as.character(extents@xmin), as.character(extents@ymin), as.character(extents@xmax), as.character(extents@ymax), sep = ",")
  print(bounding_box)
  
  # call python function for icesat elevation data download
  print("Download ICESat-2 Data via NSIDC: Check Python Script")
  icesat_profiles_download(bounding_box,paste0(substring(i, 1, nchar(i)-4),'_20m'),uid, pswd, email)
  all_files_atl08 <- list.files(paste0(getwd(), "/Outputs_atl08/",paste0(substring(i, 1, nchar(i)-4),'_20m')),pattern="processed_ATL08*")

  # skip iteration: if there is no icesat elevation data for this waterbody
  if (isempty(all_files_atl08)) {
    print("No ICESat-2 Data is available for this inland water body.")
    message$status[message$f_map==i] <- ("No ICESat-2 Data is available for this inland water body.")
    next}
  
  # tibble to store elevation points from all groundtracks
  t <- tibble()
  ind<-0
  # iterate through ascii tables: each table represents one groundtrack
  for (filename in all_files_atl08){
    date <- ISOdate(substring(filename, 17, 20), substring(filename, 21,22), substring(filename, 23, 24)) %>% 
      as.Date()
    
    gtx <- as.numeric(substring(filename, 49, 49)) * gregexpr("r",substring(filename, 50, 50))[[1]][1]
    
    # read csv file
    text=read.csv(paste0(here(),"/Outputs_atl08/",paste0(substring(i, 1, nchar(i)-4),'_20m'),"/", filename))
    # get the number of lines
    lineN=dim(text)[1]
    if (lineN<=3){
      next
    }
    
    orbit_id <- text[dim(text)[1],]
    matches <- regmatches(orbit_id, gregexpr("\\d+", orbit_id))[[1]]
    orbit_id <- as.numeric(tail(matches, 1))
    
    lineN = lineN - 2
    colnames=c('latitude_20m[1]', 'latitude_20m[2]', 'latitude_20m[3]', 'latitude_20m[4]', 'latitude_20m[5]', 'longitude_20m[1]', 'longitude_20m[2]', 'longitude_20m[3]', 'longitude_20m[4]', 'longitude_20m[5]', 'h_te_best_fit_20m[1]', 'h_te_best_fit_20m[2]', 'h_te_best_fit_20m[3]', 'h_te_best_fit_20m[4]', 'h_te_best_fit_20m[5]','h_te_uncertainty')
    
    # read downloaded ascii file
    t_temp<-read.table(paste0(here(),"/Outputs_atl08/",paste0(substring(i, 1, nchar(i)-4),'_20m'),"/", filename), 
                       colClasses ="character", header=TRUE, nrows = lineN, blank.lines.skip=TRUE,# fill = TRUE,
                       row.names=NULL) 
    t_temp=t_temp[,-c(1,2,3,4)]
    colnames(t_temp)=colnames
    
    
    # if (nrow(t_temp) < 2) {
    #   t_temp<-add_row(t_temp,t_temp)
    #   ind<-1
    # }
    #convert ascii table to tibble
    t_temp <- as_tibble(apply(t_temp, 2, function(x) as.numeric(x)))
    # if (ind==1) {
    #   t_temp<-t_temp[-c(2),]
    #   ind<-0
    # }   
    
    #one line for each 20m segment
    # colnames=names(t_temp)
    
    ttmp=data.frame()
    for (j in 0:4){#5 20m segments
      t_temp2<-t_temp[,colnames[c(6+j,1+j,11+j,16)]]
      names(t_temp2)<-c( 'mean_lon', 'mean_lat','h_te_best_fit','h_te_uncertainty')
      ttmp<-rbind(ttmp,t_temp2)
    }
    #add ORBIT column
    ttmp$orbit_id<-orbit_id
    
    # filter segments where lat and lon are defined
    t_temp<-ttmp
    t_temp <- t_temp %>% filter(!is.na(mean_lon)) %>% filter(!is.na(mean_lat)) %>% filter(!is.na(h_te_best_fit))
    
    #add date column
    t_temp$date<-date
    
    #add gtx column
    t_temp$gtx<-gtx
    
    # coordinate vector for raster::extract
    coords<-cbind(t_temp$mean_lon, t_temp$mean_lat)
    
    # retrieve percentile for each elevation value from the contour map
    t_temp$percentile <- raster::extract(surface_occ_prob, coords)
    
    # add to the previous groundtracks
    t <- t %>% bind_rows(t_temp)
    # t <- unique( t[ , ] )
  }
  
  #save data retrieved from icesat   
  write_csv(t, paste0(here(), "/Bathymetries/icesat_data_atl08/icesat_data_20m_ID_", id, '.csv'),'append'=FALSE)
  
  
  # remove the downloaded ascii files
  # for (j in (list.files(paste0(here(), "/Outputs_atl08/",paste0(substring(i, 1, nchar(i)-4),'_20m')))))
  # unlink(paste0(here(), "/Outputs_atl08/",paste0(substring(i, 1, nchar(i)-4),'_20m'),"/", j))
  
  removeTmpFiles(h=2)#  will remove tmp files older than 60 minutes
  
}

# # store csv with messages
# write_csv(message, paste0(here(), '/DEMs/', 'summary.csv'))