module Reaction_Redox_module

#include "petsc/finclude/petscsys.h"
  use petscsys

  use Reaction_Aux_module
  use Reactive_Transport_Aux_module
  use Global_Aux_module
  use Reaction_Gas_Aux_module

  use PFLOTRAN_Constants_module

  implicit none

  private

  public :: ReactionRedoxCalcpH, &
            ReactionRedoxCalcLogKEh, &
            ReactionRedoxCalcEhpe, &
            ReactionRedoxCalcLnFO2

contains

! ************************************************************************** !

subroutine ReactionRedoxCalcpH(rt_auxvar,global_auxvar,reaction,ph,option)

  ! Calculates pH

  ! Author: Glenn Hammond
  !         Based on Peter Lichtner's previous implementation
  ! Date: 06/25/21

  use Option_module

  type(reactive_transport_auxvar_type) :: rt_auxvar
  type(global_auxvar_type) :: global_auxvar
  class(reaction_rt_type) :: reaction
  PetscReal :: ph
  type(option_type) :: option

  ! pH
  if (reaction%species_idx%h_ion_id > 0) then
    ph = &
      -log10(rt_auxvar%pri_molal(reaction%species_idx%h_ion_id)* &
             rt_auxvar%pri_act_coef(reaction%species_idx%h_ion_id))
  else if (reaction%species_idx%h_ion_id < 0) then
    ph = &
      -log10(rt_auxvar%sec_molal(abs(reaction%species_idx%h_ion_id))* &
             rt_auxvar%sec_act_coef(abs(reaction%species_idx%h_ion_id)))
  endif

end subroutine ReactionRedoxCalcpH

! ************************************************************************** !

subroutine ReactionRedoxCalcEhpe(rt_auxvar,global_auxvar,reaction, &
                                 eh,pe,option)

  ! Calculates pH, Eh and pe given an hydrogen ion and oxygen concentration

  ! Author: Glenn Hammond
  !         Based on Peter Lichtner's previous implementation
  ! Date: 06/25/21

  use Option_module

  type(reactive_transport_auxvar_type) :: rt_auxvar
  type(global_auxvar_type) :: global_auxvar
  class(reaction_rt_type) :: reaction
  PetscReal :: eh
  PetscReal :: pe
  type(option_type) :: option

  PetscReal :: ph
  PetscReal :: lnQKo2
  PetscReal :: t_kelvin
  PetscReal :: ehfac

  ! pH
  call ReactionRedoxCalcpH(rt_auxvar,global_auxvar,reaction,ph,option)
  call ReactionRedoxCalcLnFO2(rt_auxvar,global_auxvar,reaction,lnQKo2,option)
  t_kelvin = global_auxvar%temp+T273K
  ehfac = IDEAL_GAS_CONSTANT*t_kelvin*LOG_TO_LN/FARADAY

  ! O2(aq) + 4 e- + 4 H+ -> 2 H2O

  ! Keq = {H2O}^2 / {O2(aq)} * {e-}^4 * {H+}^4

  ! e-^4 = {H2O}^2 / Keq * {O2(aq)} * {H+}^4

  ! e- = {H2O}^0.5 / (Keq * {O2(aq)})^0.25 * {H+}

  ! pe = (0.5 ln{H2O} - 0.25 lnKeq - 0.25 ln{O2(aq)} - ln{H+}) * -LN_TO_LOG

    !   ph = -log{H+} = -ln{H+} * LN_TO_LOG

  ! pe = (0.5 ln{H2O} - 0.25 lnKeq - 0.25 ln{O2(aq)}) * -LN_TO_LOG - ph

  ! pe = (-0.5 ln{H2O} + 0.25 lnKeq + 0.25 ln{O2(aq)}) * LN_TO_LOG - ph

  ! pe = ((-2 ln{H2O} + lnKeq + ln{O2(aq)}) * LN_TO_LOG - 4 ph) * 0.25d0
  !   ReactionRedoxCalcLogKEh = ln Keq*LN_TO_LOG
  !   lnQKo2 = ln{O2(aq)}

  eh = ehfac*(-4.d0*ph+(-2.d0*rt_auxvar%ln_act_h2o+lnQKo2)*LN_TO_LOG+ &
              ReactionRedoxCalcLogKEh(t_kelvin)) * 0.25d0
  pe = eh/ehfac

end subroutine ReactionRedoxCalcEhpe

! ************************************************************************** !

subroutine ReactionRedoxCalcLnFO2(rt_auxvar,global_auxvar,reaction, &
                                  lnfo2,option)

  ! Calculates the natural log of O2 partial pressure

  ! Author: Glenn Hammond
  !         Based on Peter Lichtner's previous implementation
  ! Date: 06/25/21

  use Option_module

  type(reactive_transport_auxvar_type) :: rt_auxvar
  type(global_auxvar_type) :: global_auxvar
  class(reaction_rt_type) :: reaction
  PetscReal :: lnfo2
  type(option_type) :: option

  PetscInt :: io2gas
  PetscReal :: lnQKo2
  PetscInt :: jcomp
  PetscInt :: comp_id

  io2gas = reaction%species_idx%o2_gas_id
    ! compute gas partial pressure
  lnQKo2 = -reaction%gas%paseqlogK(io2gas)*LOG_TO_LN
  ! activity of water
  if (reaction%gas%paseqh2oid(io2gas) > 0) then
    lnQKo2 = lnQKo2 + reaction%gas%paseqh2ostoich(io2gas) * &
                      rt_auxvar%ln_act_h2o
  endif
  do jcomp = 1, reaction%gas%paseqspecid(0,io2gas)
    comp_id = reaction%gas%paseqspecid(jcomp,io2gas)
    lnQKo2 = lnQKo2 + reaction%gas%paseqstoich(jcomp,io2gas)* &
                      log(rt_auxvar%pri_molal(comp_id)* &
                          rt_auxvar%pri_act_coef(comp_id))
  enddo
  lnfo2 = lnQKo2

end subroutine ReactionRedoxCalcLnFO2

! ************************************************************************** !

function ReactionRedoxCalcLogKEh(tk)
  !
  ! Function logKeh: Maier-Kelly fit to equilibrium constant half-cell
  ! reaction 2 H2O - 4 H+ - 4 e- = O2, to compute Eh and pe.
  !
  ! Author: Peter Lichtner
  ! Date: 04/27/13
  !
  PetscReal, intent(in) :: tk

  PetscReal :: ReactionRedoxCalcLogKEh

  PetscReal, parameter :: cm1 = 6.745529048112373d0
  PetscReal, parameter :: c0 = -48.295936593543715d0
  PetscReal, parameter :: c1 = 0.0005578156078778505d0
  PetscReal, parameter :: c2 = 27780.749538022003d0
  PetscReal, parameter :: c3 = 4027.3376948579394d0

  ReactionRedoxCalcLogKEh = cm1 * log(tk) + c0 + c1 * tk + c2 / tk + &
                            c3 / (tk * tk)

end function ReactionRedoxCalcLogKEh

end module Reaction_Redox_module
