module PM_Mphase_class
#include "petsc/finclude/petscsnes.h"
  use petscsnes
  use PM_Base_class
  use PM_Subsurface_Flow_class

  use PFLOTRAN_Constants_module

  implicit none

  private

  type, public, extends(pm_subsurface_flow_type) :: pm_mphase_type
  contains
    procedure, public :: ReadSimulationOptionsBlock => &
                           PMMphaseReadSimOptionsBlock
    procedure, public :: InitializeSolver => PMMphaseInitializeSolver
    procedure, public :: Setup => PMMphaseSetup
    procedure, public :: InitializeTimestep => PMMphaseInitializeTimestep
    procedure, public :: Residual => PMMphaseResidual
    procedure, public :: Jacobian => PMMphaseJacobian
    procedure, public :: UpdateTimestep => PMMphaseUpdateTimestep
    procedure, public :: PreSolve => PMMphasePreSolve
    procedure, public :: PostSolve => PMMphasePostSolve
#if 0
    procedure, public :: CheckUpdatePre => PMMphaseCheckUpdatePre
    procedure, public :: CheckUpdatePost => PMMphaseCheckUpdatePost
#endif
    procedure, public :: TimeCut => PMMphaseTimeCut
    procedure, public :: UpdateSolution => PMMphaseUpdateSolution
    procedure, public :: UpdateAuxVars => PMMphaseUpdateAuxVars
    procedure, public :: MaxChange => PMMphaseMaxChange
    procedure, public :: ComputeMassBalance => PMMphaseComputeMassBalance
    procedure, public :: InputRecord => PMMphaseInputRecord
    procedure, public :: Destroy => PMMphaseDestroy
  end type pm_mphase_type

  public :: PMMphaseCreate

contains

! ************************************************************************** !

function PMMphaseCreate()
  !
  ! Creates Mphase process models shell
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !

  implicit none

  class(pm_mphase_type), pointer :: PMMphaseCreate

  class(pm_mphase_type), pointer :: mphase_pm

  allocate(mphase_pm)

  call PMSubsurfaceFlowInit(mphase_pm)
  mphase_pm%name = 'Mphase CO2 Flow'
  mphase_pm%header = 'MPHASE CO2 FLOW'

  PMMphaseCreate => mphase_pm

end function PMMphaseCreate

! ************************************************************************** !

subroutine PMMphaseReadSimOptionsBlock(this,input)
  !
  ! Reads input file parameters associated with the Mphase process model
  !
  ! Author: Glenn Hammond
  ! Date: 01/29/15
  use Input_Aux_module
  use String_module
  use Utility_module
  use EOS_Water_module
  use Option_module
  use Mphase_Aux_module

  implicit none

  class(pm_mphase_type) :: this
  type(input_type), pointer :: input

  character(len=MAXWORDLENGTH) :: word
  character(len=MAXSTRINGLENGTH) :: error_string
  type(option_type), pointer :: option
  PetscBool :: found

  option => this%option

  error_string = 'Mphase Options'

  input%ierr = INPUT_ERROR_NONE
  call InputPushBlock(input,option)
  do

    call InputReadPflotranString(input,option)
    if (InputError(input)) exit
    if (InputCheckExit(input,option)) exit

    call InputReadCard(input,option,word)
    call InputErrorMsg(input,option,'keyword',error_string)
    call StringToUpper(word)

    found = PETSC_FALSE
    call PMSubsurfFlowReadSimOptionsSC(this,input,word,found, &
                                       error_string,option)
    if (found) cycle

    select case(trim(word))
      case default
        call InputKeywordUnrecognized(input,word,error_string,option)
    end select
  enddo
  call InputPopBlock(input,option)

end subroutine PMMphaseReadSimOptionsBlock

! ************************************************************************** !

subroutine PMMphaseInitializeSolver(this)
  !
  ! Author: Glenn Hammond
  ! Date: 04/06/20

  use Solver_module

  implicit none

  class(pm_mphase_type) :: this

  call PMBaseInitializeSolver(this)

  ! helps accommodate rise in residual due to change in state
  this%solver%newton_dtol = 1.d9

end subroutine PMMphaseInitializeSolver

! ************************************************************************** !

subroutine PMMphaseInitializeTimestep(this)
  !
  ! Should not need this as it is called in PreSolve.
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !

  use Mphase_module, only : MphaseInitializeTimestep
  use Option_module

  implicit none

  class(pm_mphase_type) :: this

  call PMSubsurfaceFlowInitializeTimestepA(this)
  call MphaseInitializeTimestep(this%realization)
  call PMSubsurfaceFlowInitializeTimestepB(this)

end subroutine PMMphaseInitializeTimestep

! ************************************************************************** !

subroutine PMMphaseSetup(this)
  !
  ! Sets up auxvars and parameters
  !
  ! Author: Glenn Hammond
  ! Date: 04/11/24

  use Material_module
  use Mphase_module
  use Mphase_Aux_module
  use co2_sw_module, only : init_span_wagner

  implicit none

  class(pm_mphase_type) :: this

  call this%SetRealization()
  call MaterialSetupThermal( &
         this%realization%patch%aux%Material%material_parameter, &
         this%realization%patch%material_property_array, &
         this%realization%option)
  if (mphase_co2_eos == EOS_SPAN_WAGNER) then
    call init_span_wagner(this%realization%option)
  endif
  call MphaseSetup(this%realization)
  call PMSubsurfaceFlowSetup(this)

end subroutine PMMphaseSetup

! ************************************************************************** !

subroutine PMMphasePreSolve(this)
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  use Option_module
  use Reaction_Aux_module
  use Reactive_Transport_Aux_module
  use Grid_module
  use Patch_module
  use Global_Aux_module
  use Coupler_module
  use Connection_module

  implicit none

  class(pm_mphase_type) :: this

  class(reaction_rt_type), pointer :: reaction
  type(patch_type), pointer :: patch
  type(option_type), pointer :: option
  type(grid_type), pointer :: grid
  type(reactive_transport_auxvar_type), pointer :: rt_auxvars(:)
  type(global_auxvar_type), pointer :: global_auxvars(:)
  type(coupler_type), pointer :: boundary_condition
  type(connection_set_type), pointer :: cur_connection_set
  PetscInt :: local_id
  PetscInt :: ghosted_id
  PetscInt :: sum_connection
  PetscInt :: iconn
  PetscInt :: na_id, cl_id

  reaction => this%realization%reaction
  option => this%realization%option

  call PMSubsurfaceFlowPreSolve(this)

#if 1
  if (associated(reaction)) then
    if (associated(reaction%species_idx)) then
      patch => this%realization%patch
      global_auxvars => patch%aux%Global%auxvars
      if (associated(global_auxvars(1)%m_nacl)) then
        na_id = reaction%species_idx%na_ion_id
        cl_id = reaction%species_idx%cl_ion_id

        grid => patch%grid
        rt_auxvars => patch%aux%RT%auxvars
        global_auxvars => patch%aux%Global%auxvars

        if (na_id > 0 .and. cl_id > 0) then
          do ghosted_id = 1, grid%ngmax
            if (grid%nG2L(ghosted_id) < 0) cycle ! bypass ghosted corner cells
            !geh - Ignore inactive cells with inactive materials
            if (patch%imat(ghosted_id) <= 0) cycle
            global_auxvars(ghosted_id)%m_nacl(1) = &
              rt_auxvars(ghosted_id)%pri_molal(na_id)
            global_auxvars(ghosted_id)%m_nacl(2) = &
              rt_auxvars(ghosted_id)%pri_molal(cl_id)
          enddo
        else
          do ghosted_id = 1, grid%ngmax
            if (grid%nG2L(ghosted_id) < 0) cycle ! bypass ghosted corner cells
            !geh - Ignore inactive cells with inactive materials
            if (patch%imat(ghosted_id) <= 0) cycle
            global_auxvars(ghosted_id)%m_nacl = option%m_nacl
          enddo
        endif

        rt_auxvars => this%realization%patch%aux%RT%auxvars_bc
        global_auxvars => this%realization%patch%aux%Global%auxvars_bc

        boundary_condition => patch%boundary_condition_list%first
        sum_connection = 0
        do
          if (.not.associated(boundary_condition)) exit
          cur_connection_set => boundary_condition%connection_set
          if (na_id > 0 .and. cl_id > 0) then
            do iconn = 1, cur_connection_set%num_connections
              sum_connection = sum_connection + 1
              local_id = cur_connection_set%id_dn(iconn)
              ghosted_id = grid%nL2G(local_id)
              if (patch%imat(ghosted_id) <= 0) cycle
              global_auxvars(sum_connection)%m_nacl(1) = &
                rt_auxvars(sum_connection)%pri_molal(na_id)
              global_auxvars(sum_connection)%m_nacl(2) = &
                rt_auxvars(sum_connection)%pri_molal(cl_id)
            enddo
          else
            do iconn = 1, cur_connection_set%num_connections
              sum_connection = sum_connection + 1
              local_id = cur_connection_set%id_dn(iconn)
              ghosted_id = grid%nL2G(local_id)
              if (patch%imat(ghosted_id) <= 0) cycle
              global_auxvars(sum_connection)%m_nacl = option%m_nacl
            enddo
          endif
          boundary_condition => boundary_condition%next
        enddo
      endif
    endif
  endif
#endif

end subroutine PMMphasePreSolve

! ************************************************************************** !

subroutine PMMphasePostSolve(this)
  !
  ! PMMphaseUpdatePostSolve:
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13

  implicit none

  class(pm_mphase_type) :: this

end subroutine PMMphasePostSolve

! ************************************************************************** !

subroutine PMMphaseUpdateTimestep(this,update_dt, &
                                  dt,dt_min,dt_max,iacceleration, &
                                  num_newton_iterations,tfac, &
                                  time_step_max_growth_factor)
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !
  use Realization_Subsurface_class, only : RealizationLimitDTByCFL

  implicit none

  class(pm_mphase_type) :: this
  PetscBool :: update_dt
  PetscReal :: dt
  PetscReal :: dt_min,dt_max
  PetscInt :: iacceleration
  PetscInt :: num_newton_iterations
  PetscReal :: tfac(:)
  PetscReal :: time_step_max_growth_factor

  PetscReal :: fac
  PetscReal :: ut
  PetscReal :: up
  PetscReal :: utmp
  PetscReal :: uc
  PetscReal :: uus
  PetscReal :: dtt
  PetscReal :: dt_p
  PetscReal :: dt_tfac
  PetscInt :: ifac

  if (update_dt .and. iacceleration /= 0) then
    if (iacceleration > 0) then
      fac = 0.5d0
      if (num_newton_iterations >= iacceleration) then
        fac = 0.33d0
        ut = 0.d0
      else
        up = this%pressure_change_governor/(this%max_pressure_change+0.1)
        utmp = this%temperature_change_governor/(this%max_temperature_change+1.d-5)
        uc = this%xmol_change_governor/(this%max_xmol_change+1.d-6)
        uus= this%saturation_change_governor/(this%max_saturation_change+1.d-6)
        ut = min(up,utmp,uc,uus)
      endif
      dtt = fac * dt * (1.d0 + ut)
    else
      ifac = max(min(num_newton_iterations,size(tfac)),1)
      dt_tfac = tfac(ifac) * dt

      fac = 0.5d0
      up = this%pressure_change_governor/(this%max_pressure_change+0.1)
      dt_p = fac * dt * (1.d0 + up)

      dtt = min(dt_tfac,dt_p)
    endif

    dtt = min(time_step_max_growth_factor*dt,dtt)
    if (dtt > dt_max) dtt = dt_max
    ! geh: There used to be code here that cut the time step if it is too
    !      large relative to the simulation time.  This has been removed.
    dtt = max(dtt,dt_min)
    dt = dtt
  endif

  call RealizationLimitDTByCFL(this%realization,this%cfl_governor,dt,dt_max)

end subroutine PMMphaseUpdateTimestep

! ************************************************************************** !

subroutine PMMphaseResidual(this,snes,xx,r,ierr)
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !

  use Mphase_module, only : MphaseResidual

  implicit none

  class(pm_mphase_type) :: this
  SNES :: snes
  Vec :: xx
  Vec :: r
  PetscErrorCode :: ierr

  call MphaseResidual(snes,xx,r,this%realization,ierr)

end subroutine PMMphaseResidual

! ************************************************************************** !

subroutine PMMphaseJacobian(this,snes,xx,A,B,ierr)
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !

  use Mphase_module, only : MphaseJacobian

  implicit none

  class(pm_mphase_type) :: this
  SNES :: snes
  Vec :: xx
  Mat :: A, B
  PetscErrorCode :: ierr

  call MphaseJacobian(snes,xx,A,B,this%realization,ierr)

end subroutine PMMphaseJacobian

#if 0

! ************************************************************************** !

subroutine PMMphaseCheckUpdatePre(this,snes,X,dX,changed,ierr)
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !

  use Mphase_module, only : MphaseCheckUpdatePre

  implicit none

  class(pm_mphase_type) :: this
  SNES :: snes
  Vec :: X
  Vec :: dX
  PetscBool :: changed
  PetscErrorCode :: ierr

  call MphaseCheckUpdatePre(snes,X,dX,changed,this%realization,ierr)

end subroutine PMMphaseCheckUpdatePre

! ************************************************************************** !

subroutine PMMphaseCheckUpdatePost(this,snes,X0,dX,X1,dX_changed, &
                                   X1_changed,ierr)
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !

  use Mphase_module, only : MphaseCheckUpdatePost

  implicit none

  class(pm_mphase_type) :: this
  SNES :: snes
  Vec :: X0
  Vec :: dX
  Vec :: X1
  PetscBool :: dX_changed
  PetscBool :: X1_changed
  PetscErrorCode :: ierr

  call MphaseCheckUpdatePost(snes,X0,dX,X1,dX_changed, &
                               X1_changed,this%realization,ierr)

end subroutine PMMphaseCheckUpdatePost
#endif

! ************************************************************************** !

subroutine PMMphaseTimeCut(this)
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !

  use Mphase_module, only : MphaseTimeCut

  implicit none

  class(pm_mphase_type) :: this

  call PMSubsurfaceFlowTimeCut(this)
  call MphaseTimeCut(this%realization)

end subroutine PMMphaseTimeCut

! ************************************************************************** !

subroutine PMMphaseUpdateSolution(this)
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !

  use Mphase_module, only : MphaseUpdateSolution

  implicit none

  class(pm_mphase_type) :: this

  call PMSubsurfaceFlowUpdateSolution(this)
  call MphaseUpdateSolution(this%realization)

end subroutine PMMphaseUpdateSolution

! ************************************************************************** !

subroutine PMMphaseUpdateAuxVars(this)
  !
  ! Author: Glenn Hammond
  ! Date: 04/21/14

  use Mphase_module, only : MphaseUpdateAuxVars

  implicit none

  class(pm_mphase_type) :: this

  call MphaseUpdateAuxVars(this%realization)

end subroutine PMMphaseUpdateAuxVars

! ************************************************************************** !

subroutine PMMphaseMaxChange(this)
  !
  ! Not needed given MphaseMaxChange is called in PostSolve
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !
  use Option_module
  use Mphase_module, only : MphaseMaxChange

  implicit none

  class(pm_mphase_type) :: this

  call MphaseMaxChange(this%realization,this%max_pressure_change, &
                       this%max_temperature_change, &
                       this%max_saturation_change, &
                       this%max_xmol_change)
  write(this%option%io_buffer,'("  --> max change: dpmx= ",1pe12.4,&
                              &" dtmpmx= ",1pe12.4)') &
        this%max_pressure_change,this%max_temperature_change
  call PrintMsg(this%option)
  write(this%option%io_buffer,'(17x," dcmx= ",1pe12.4," dsmx= ",1pe12.4)') &
        this%max_xmol_change,this%max_saturation_change
  call PrintMsg(this%option)

end subroutine PMMphaseMaxChange

! ************************************************************************** !

subroutine PMMphaseComputeMassBalance(this,mass_balance_array)
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !

  use Mphase_module, only : MphaseComputeMassBalance

  implicit none

  class(pm_mphase_type) :: this
  PetscReal :: mass_balance_array(:)

  !geh: currently does not include "trapped" mass
  !call MphaseComputeMassBalance(this%realization,mass_balance_array)

end subroutine PMMphaseComputeMassBalance

! ************************************************************************** !

subroutine PMMphaseInputRecord(this)
  !
  ! Writes ingested information to the input record file.
  !
  ! Author: Jenn Frederick, SNL
  ! Date: 03/21/2016
  !

  implicit none

  class(pm_mphase_type) :: this

  PetscInt :: id

  id = INPUT_RECORD_UNIT

  write(id,'(a29)',advance='no') 'pm: '
  write(id,'(a)') this%name
  write(id,'(a29)',advance='no') 'mode: '
  write(id,'(a)') 'mphase'

end subroutine PMMphaseInputRecord

! ************************************************************************** !

subroutine PMMphaseDestroy(this)
  !
  ! Destroys Mphase process model
  !
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  !

  use Mphase_module, only : MphaseDestroy

  implicit none

  class(pm_mphase_type) :: this

  if (associated(this%next)) then
    call this%next%Destroy()
  endif

  ! preserve this ordering
  call MphaseDestroy(this%realization)
  call PMSubsurfaceFlowDestroy(this)

end subroutine PMMphaseDestroy

end module PM_Mphase_class
