#!/usr/bin/env python
"""Python program to generate the pflotran_provenance.F90 file based
on a template and revision control information for pflotran and petsc.

Author: Ben Andre <bandre@lbl.gov>

"""
from __future__ import print_function

import argparse
import copy
import os
import os.path
import re
import socket
from string import Template
import subprocess
import sys
import textwrap
import time
import traceback

################################################################################
#
# Template
#
################################################################################

provenance_module_write_value = Template(
"""  write(*, '(a)') "${value}"
""")

provenance_module_write_key_value = Template(
"""  write(*, '(''${key} = '', a)') "${value}"
""")

provenance_module_write_empty = """  write(*, '()')
"""

provenance_module_warning_comment = \
"""! DO NOT EDIT! This file is automatically generated at build time by pflotran-provenance.py
! Changes will be overwritten!
!
! DO NOT add this file to revision control!
"""

provenance_module_write_detail_loop = Template(
"""
  write(*, '(''--- ${key} ---'')')
  do i = 1, detail_${key}_len
     write(*, '(a)') trim(detail_${key}(i))
  end do

""")

################################################################################
#
# Regular expressions
#
################################################################################
magic_comment_re = re.compile("!-_-![\s]*([\w]+)[\s]*!-_-!")
provenance_max_str_len_re = re.compile("(:: provenance_max_str_len = )([\d]+)")

################################################################################
#
# Functions
#
################################################################################
def write_provenance_module(provenance, details, max_string_length):
    """Write the provenance file by combining a series of strings and templates.
    """

    # determine the total number of lines of provenance information we
    # will be writing
    total_lines = 0
    for p in provenance:
        total_lines += 1
    for d in details:
        total_lines += len(details[d])

    with open("pflotran_no_provenance.F90", 'r') as template, \
          open("pflotran_provenance.F90", 'w') as outfile:

        for line in template.readlines():
            # only want to modify a line once!
            orig_line = copy.deepcopy(line)
            for p in provenance:
                if line == orig_line:
                    old = '{0} = "unknown"'.format(p)
                    new = '{0} = &\n    "{1}"'.format(p, provenance[p])
                    line = line.replace(old, new)

            for d in details:
                if line == orig_line:
                    old = 'detail_{0}_len = 1'.format(d)
                    new = 'detail_{0}_len = {1}'.format(d, len(details[d]))
                    line = line.replace(old, new)
                    old = 'detail_{0}(detail_{0}_len) = (/ "unknown" /)'.format(d)
                    new = 'detail_{0}(detail_{0}_len) = (/ &\n'.format(d)
                    for count, val in enumerate(details[d]):
                        new += '    "{0}"'.format(val)
                        if count < len(details[d]) - 1:
                            new += ", &\n"
                        else:
                            new += " /)\n"
                    line = line.replace(old, new)

            if line == orig_line:
                match = provenance_max_str_len_re.search(line)
                if match:
                    old = 'provenance_max_str_len = {0}'.format(match.group(2))
                    new = 'provenance_max_str_len = {0}'.format(max_string_length)[0:79]
                    line = line.replace(old, new)

            if line == orig_line:
                match = magic_comment_re.search(line)
                if match:
                    section = match.group(1)
                    if section == "write_warning_comment":
                        line = provenance_module_warning_comment
                    elif section == "write_provenance_details":
                        # generate the detailed provenance strings
                        write_details = ""
                        write_details += provenance_module_write_value.substitute(value=
                            "------------------------------ Provenance Detailed -----------------------------")
                        for d in details:
                            write_details += provenance_module_write_empty
                            write_details += provenance_module_write_detail_loop.substitute(key=d)
                        line = write_details

            outfile.write(line)


def clean_provenance(provenance, max_string_length):
    """Cleanup the lines returned by the shell so they will print in fortran.

    NOTE: these should only be short single line values, so we don't
    worry about breaking and continuning the line.

    We are using double quotes for the string literals in
    fortran. Thate means we need to escape any double quotes in our
    input string! Fortran quote escaping is: " --> "".

    """
    fmt_str = "{{0:{0}}}".format(max_string_length)

    for p in provenance:
        line = provenance[p]
        line = line.strip()
        line = line.replace('"', '""')
        provenance[p] = fmt_str.format(line)


def clean_details(details, max_string_length):
    """Cleanup the multi-line detailed provenance info

    Since this is potentially multi-line output, we have to split on
    newline characters and break very long lines.

    We are using double quotes for the string literals in
    fortran. Thate means we need to escape any double quotes in our
    input string! Fortran quote escaping is: " --> "".

    """
    fmt_str = "{{0:{0}}}".format(max_string_length)
    txtwrap = textwrap.TextWrapper(width=max_string_length,
                                   subsequent_indent=4*" ",
                                   drop_whitespace=False)

    for d in details:
        data = []
        info = details[d].strip()
#fix suggested by Kris Kuhlman 11-11-15
#        info = info.replace('"', '""')
	info = info.replace('"', "'")
        temp = info.split("\n")
        if isinstance(temp, basestring):
            # FIXME: still must run textwrap on this incase it is avery long line!
            data.append(info)
        else:
            for line in temp:
                line = txtwrap.wrap(line)
                if isinstance(line, basestring):
                    data.append(fmt_str.format(line))
                else:
                    for l in line:
                        data.append(fmt_str.format(l))
        if len(data) > 255:
            raise RuntimeError(
                "ERROR: can not generate provenance file.\n"
                "The fortran 2003 standard does not allow more than 255 continutation lines.\n"
                "Provenance information for '{0}' = {1} lines.\n"
                "Please commit your changes or disable provenance information by "
                "building with 'make no_provenance=1 pflotran'.".format(
                    d, len(data)))

        details[d] = data


def determine_pflotran_provenance(provenance, details, fflags):
    """
    """

    provenance["pflotran_compile_date_time"] = time.strftime("%Y-%m-%d %H:%M:%S")
    provenance["pflotran_compile_user"] = os.getenv("USER")
    provenance["pflotran_compile_hostname"] = socket.gethostname()

    cmd = ["hg", "parent", "--template", "{node}"]
    provenance["pflotran_changeset"] = subprocess.check_output(cmd)

    #X#cmd = ["hg", "parent"]
    cmd = ["hg", "parent", "--template",
           "changeset:\t{rev}:{node|short}\nuser:\t\t{author}\ndate:\t\t{date|rfc822date}\n"]
    details["pflotran_parent"] = subprocess.check_output(cmd)

    cmd = ["hg", "status", "-q"]
    data = subprocess.check_output(cmd)
    if data.strip() == "":
        provenance["pflotran_status"] = "No modifications"
    else:
        provenance["pflotran_status"] = "Modified"
        details["pflotran_status"] = data

    cmd = ["hg", "diff", "--stat"]
    data = subprocess.check_output(cmd)
    if data.strip() == "":
        details["pflotran_diff"] = "No modifications"
    else:
        # FIXME(bja, 2013-11-25) full diffs break gcc!"
        # NOTE(bja, 2014-02-28) use hg diff --stat instead
        details["pflotran_diff"] = data
        #X#details["pflotran_diff"] = "Diffs unavailable"

    if len(fflags) > 0:
        details["pflotran_fflags"] = " ".join(["PFLOTRAN fortran compiler flags : ", fflags])


def determine_petsc_provenance(provenance, details, petsc_config):
    """Extract provenance information about petsc.

    Since this is being called during the pflotran build process, the
    environment variable PETSC_DIR must be set and is obviously the
    correct one for the build.

    """
    orig_cwd = os.getcwd()
    
    petsc_dir = os.getenv("PETSC_DIR", None)
    if os.path.isdir(petsc_dir):
        os.chdir(petsc_dir)
        if os.path.isdir("{0}/.git".format(petsc_dir)):
            cmd = ["git", "log", "--pretty=%H", "-1", "HEAD"]
            provenance["petsc_changeset"] = subprocess.check_output(cmd)

            #X#cmd = ["git", "log", "-1", "HEAD"]
            cmd = ["git", "log", "-1", "HEAD", '--pretty=commit: %H%nAuthor: %an%nDate:   %ai%n']
            details["petsc_parent"] = subprocess.check_output(cmd)

            cmd = ["git", "status", "-u", "no"]
            data = subprocess.check_output(cmd)
            if data.find("working directory clean") > 0:
                provenance["petsc_status"] = "No modifications"
                details["petsc_status"] = "No modifications"
            else:
                provenance["petsc_status"] = "Modified"
                details["petsc_status"] = data

        elif os.path.isdir("{0}/.hg".format(petsc_dir)):
            cmd = ["hg", "parent", "--template", "'{node}'"]
            provenance["petsc_changeset"] = subprocess.check_output(cmd)

            cmd = ["hg", "parent"]
            details["petsc_parent"] = subprocess.check_output(cmd)

            cmd = ["hg", "status", "-q"]
            data = subprocess.check_output(cmd)
            if data.strip() == "":
                provenance["petsc_status"] = "No modifications"
            else:
                provenance["petsc_status"] = "Modified"
                details["petsc_status"] = data
        else:
            # FIXME(bja, 2013-11-26) could be a release tarball...?
            print("could not determine the repository type for PETSC_DIR={0}".format(petsc_dir))

    if len(petsc_config) > 0:
        details["petsc_config"] = " ".join(["PETSC configure command : ", petsc_config])


        
    os.chdir(orig_cwd)


def commandline_options():
    """
    Process the command line arguments and return them as a dict.
    """
    parser = argparse.ArgumentParser(description="generate the pflotran_provenance.F90 module")

    parser.add_argument("--backtrace", action="store_true", default=False,
                        help="show exception backtraces as extra debugging "
                        "output")

    parser.add_argument("--petsc-config", default=[],
                        help="petsc configure command")

    parser.add_argument("--pflotran-fflags", default=[],
                        help="fortran compiler flags used to compile pflotran.")

    options = parser.parse_args()
    return options

def rebuild_message():
    print("\nERROR: Generation of provenance information failed!")
    print("To build pflotran without provenance information run: ", file=sys.stderr)
    print("    make no_provenance=1 pflotran", file=sys.stderr)

def main(options):
    # Fortran 2003 standard limits line length to 132 characters. We
    # need two for initial indent, two for the begin/end quotes, and
    # three for the continutation ", &". ---> 132 - 7 = max length of
    # 125. We'll go slightly shorter just to be safe...
    max_string_length = 120

    # this is short data (single line) that can be dumped into a
    # single fixed length string and used as hdf5 attributes etc.
    provenance = {"pflotran_changeset" : "unknown",
                  "pflotran_status" : "unknown",
                  "petsc_changeset" : "unknown",
                  "petsc_status" : "unknown",
    }

    # this is potentially longer infomation like detailed parent
    # changeset info, status or diffs. It is saved to an array of
    # fixed length strings.
    details = { }

    determine_pflotran_provenance(provenance, details, options.pflotran_fflags)
    determine_petsc_provenance(provenance, details, options.petsc_config)
    clean_provenance(provenance, max_string_length)
    clean_details(details, max_string_length)

    write_provenance_module(provenance, details, max_string_length)

    return 0

if __name__ == "__main__":
    options = commandline_options()
    try:
        status = main(options)
        if status:
            rebuild_message()
        sys.exit(status)
    except Exception as error:
        print(str(error))
        if options.backtrace:
            traceback.print_exc()
        rebuild_message()
        sys.exit(1)
