module Material_Aux_module

#include "petsc/finclude/petscsys.h"
  use petscsys
  use PFLOTRAN_Constants_module

  implicit none

  private

  PetscInt, parameter, public :: perm_xx_index = 1
  PetscInt, parameter, public :: perm_yy_index = 2
  PetscInt, parameter, public :: perm_zz_index = 3
  PetscInt, parameter, public :: perm_xy_index = 4
  PetscInt, parameter, public :: perm_yz_index = 5
  PetscInt, parameter, public :: perm_xz_index = 6

  ! do not use 0 as an index as there is a case statement in material.F90
  ! designed to catch erroneous values outside [1,2].
  PetscInt, parameter, public :: POROSITY_CURRENT = 1
  PetscInt, parameter, public :: POROSITY_BASE = 2
  PetscInt, parameter, public :: POROSITY_INITIAL = 3

  ! Tensor to scalar conversion models
  ! default for structured grids = TENSOR_TO_SCALAR_LINEAR
  ! default for unstructured grids = TENSOR_TO_SCALAR_POTENTIAL
  ! Both are set in discretization.F90:DiscretizationReadRequiredCards()
  ! immediately after the GRID cards is read with a call to
  ! MaterialAuxSetPermTensorModel()
  PetscInt, parameter, public :: TENSOR_TO_SCALAR_LINEAR = 1
  PetscInt, parameter, public :: TENSOR_TO_SCALAR_FLOW = 2
  PetscInt, parameter, public :: TENSOR_TO_SCALAR_POTENTIAL = 3
  PetscInt, parameter, public :: TENSOR_TO_SCALAR_FLOW_FT = 4
  PetscInt, parameter, public :: TENSOR_TO_SCALAR_POTENTIAL_FT = 5

  ! flag to determine which model to use for tensor to scalar conversion
  ! of permeability
  PetscInt :: perm_tensor_to_scalar_model = TENSOR_TO_SCALAR_LINEAR
  PetscInt :: tort_tensor_to_scalar_model = TENSOR_TO_SCALAR_LINEAR

  ! when adding a new index, add it above max_material_index and grep on
  ! "ADD_SOIL_PROPERTY_INDEX_HERE" to see whereall you must update the code.
  PetscInt, public :: soil_compressibility_index
  PetscInt, public :: soil_reference_pressure_index
  PetscInt, public :: material_elec_conduct_index
  PetscInt, public :: archie_cementation_exp_index
  PetscInt, public :: archie_saturation_exp_index
  PetscInt, public :: archie_tortuosity_index
  PetscInt, public :: surf_elec_conduct_index
  PetscInt, public :: ws_clay_conduct_index
  PetscInt, public :: tortuosity_yy_index
  PetscInt, public :: tortuosity_zz_index
  PetscInt, public :: max_material_index

  type, public :: material_auxvar_type
    PetscInt :: id
    PetscReal :: volume
    PetscReal :: porosity_0 ! initial porosity as defined in input file or
                            ! initial conditon
    PetscReal :: porosity_base ! base porosity prescribed by pm outside flow
                               ! (e.g. geomechanics, mineral precip/diss)
    PetscReal :: porosity ! porosity used in calculation, which may be a
                          ! function of soil compressibity, etc.
    PetscReal :: dporosity_dp
    PetscReal :: tortuosity
    PetscReal :: soil_particle_density
    PetscReal, pointer :: permeability(:)
    PetscReal, pointer :: sat_func_prop(:)
    PetscReal, pointer :: soil_properties(:) ! den, therm. cond., heat cap.
    type(fracture_auxvar_type), pointer :: fracture
    type(secondary_auxvar_type), pointer :: secondary_prop
    PetscReal, pointer :: geomechanics_subsurface_prop(:)
    PetscInt :: creep_closure_id

!    procedure(SaturationFunction), nopass, pointer :: SaturationFunction
!  contains
!    procedure, public :: PermeabilityTensorToScalar
!    procedure, public :: PermeabilityTensorToScalarSafe
  end type material_auxvar_type

  type, public :: fracture_auxvar_type
    PetscBool :: fracture_is_on
    PetscReal :: initial_pressure
    PetscReal :: properties(4)
    PetscReal :: vector(3) ! < 0. 0. 0. >
  end type fracture_auxvar_type

  type, public :: secondary_auxvar_type
    PetscReal :: epsilon
    PetscReal :: half_matrix_width
    PetscInt :: ncells
  end type secondary_auxvar_type

  type, public :: material_parameter_type
    PetscReal, pointer :: soil_heat_capacity(:) ! MJ/kg rock-K
    PetscReal, pointer :: soil_thermal_conductivity(:,:) ! W/m-K
  end type material_parameter_type

  type, public :: material_type
    PetscReal :: time_t, time_tpdt
    PetscInt :: num_aux
    PetscInt, pointer :: soil_properties_ivar(:)
    character(len=MAXWORDLENGTH), pointer :: soil_properties_name(:)
    type(material_parameter_type), pointer :: material_parameter
    type(material_auxvar_type), pointer :: auxvars(:)
  end type material_type

  ! procedure pointer declarations
  procedure(MaterialCompressSoilDummy), pointer :: &
    MaterialCompressSoilPtr => null()

  ! interface blocks
  interface
    subroutine MaterialCompressSoilDummy(auxvar,pressure,compressed_porosity, &
                                         dcompressed_porosity_dp)
    import material_auxvar_type
    implicit none
    type(material_auxvar_type), intent(in) :: auxvar
    PetscReal, intent(in) :: pressure
    PetscReal, intent(out) :: compressed_porosity
    PetscReal, intent(out) :: dcompressed_porosity_dp
    end subroutine MaterialCompressSoilDummy
  end interface

  interface MaterialCompressSoil
    procedure MaterialCompressSoilPtr
  end interface

  public :: MaterialCompressSoilDummy, &
            MaterialCompressSoilPtr, &
            MaterialCompressSoil, &
            MaterialCompressSoilBRAGFLO, &
            MaterialCompressSoilPoroExp, &
            MaterialCompressSoilLeijnse, &
            MaterialCompressSoilLinear, &
            MaterialCompressSoilQuadratic

  public :: PermeabilityTensorToScalar
  Public :: TortuosityTensorToScalar

  public :: MaterialAuxCreate, &
            MaterialAuxVarInit, &
            MaterialAuxVarCopy, &
            MaterialAuxVarStrip, &
            MaterialAuxVarGetValue, &
            MaterialAuxVarSetValue, &
            MaterialAuxDestroy, &
            MaterialAuxVarFractureStrip, &
            MaterialAuxSetPermTensorModel, &
            MaterialAuxVarGetSoilPropIndex

  public :: MaterialAuxVarCompute

contains

! ************************************************************************** !

function MaterialAuxCreate(option)
  !
  ! Allocate and initialize auxiliary object
  !
  ! Author: Glenn Hammond
  ! Date: 01/09/14
  !
  use Option_module
  use String_module
  use Variables_module, only : SOIL_COMPRESSIBILITY, &
                               SOIL_REFERENCE_PRESSURE, &
                               MATERIAL_ELECTRICAL_CONDUCTIVITY, &
                               ARCHIE_CEMENTATION_EXPONENT, &
                               ARCHIE_SATURATION_EXPONENT, &
                               ARCHIE_TORTUOSITY_CONSTANT, &
                               SURFACE_ELECTRICAL_CONDUCTIVITY, &
                               WAXMAN_SMITS_CLAY_CONDUCTIVITY, &
                               NUMBER_SECONDARY_CELLS, &
                               TORTUOSITY_Y, TORTUOSITY_Z

  implicit none

  type(option_type) :: option

  type(material_type), pointer :: MaterialAuxCreate

  type(material_type), pointer :: aux
  PetscInt :: i, j

  allocate(aux)
  nullify(aux%auxvars)
  allocate(aux%material_parameter)
  nullify(aux%material_parameter%soil_heat_capacity)
  nullify(aux%material_parameter%soil_thermal_conductivity)
  if (max_material_index > 0) then
    allocate(aux%soil_properties_ivar(max_material_index))
    aux%soil_properties_ivar = UNINITIALIZED_INTEGER
    allocate(aux%soil_properties_name(max_material_index))
    aux%soil_properties_name = 'UNINITIALIZED_STRING'

    call MaterialAuxInitSoilPropertyMap(aux,soil_compressibility_index, &
                                        SOIL_COMPRESSIBILITY, &
                                        'Soil Compressibility')
    call MaterialAuxInitSoilPropertyMap(aux,soil_reference_pressure_index, &
                                        SOIL_REFERENCE_PRESSURE, &
                                        'Soil Reference Pressure')
    call MaterialAuxInitSoilPropertyMap(aux,material_elec_conduct_index, &
                                        MATERIAL_ELECTRICAL_CONDUCTIVITY, &
                                        'Electrical Conductivity')
    call MaterialAuxInitSoilPropertyMap(aux,archie_cementation_exp_index, &
                                        ARCHIE_CEMENTATION_EXPONENT, &
                                        'Archie Cementation Exponent')
    call MaterialAuxInitSoilPropertyMap(aux,archie_saturation_exp_index, &
                                        ARCHIE_SATURATION_EXPONENT, &
                                        'Archie Saturation Exponent')
    call MaterialAuxInitSoilPropertyMap(aux,archie_tortuosity_index, &
                                        ARCHIE_TORTUOSITY_CONSTANT, &
                                        'Archie Tortuosity Constant')
    call MaterialAuxInitSoilPropertyMap(aux,surf_elec_conduct_index, &
                                        SURFACE_ELECTRICAL_CONDUCTIVITY, &
                                        'Surface Electrical Conductivity')
    call MaterialAuxInitSoilPropertyMap(aux,ws_clay_conduct_index, &
                                        WAXMAN_SMITS_CLAY_CONDUCTIVITY, &
                                        'Waxman-Smits Clay Conductivity')
    call MaterialAuxInitSoilPropertyMap(aux,tortuosity_yy_index, &
                                        TORTUOSITY_Y, &
                                        'Anisotropic Tortuosity Y')
    call MaterialAuxInitSoilPropertyMap(aux,tortuosity_zz_index, &
                                        TORTUOSITY_Z, &
                                        'Anisotropic Tortuosity Z')
    ! ADD_SOIL_PROPERTY_INDEX_HERE
    do i = 1, max_material_index
      if (Uninitialized(aux%soil_properties_ivar(i))) then
        do j = 1, max_material_index
          option%io_buffer = StringWrite(j) // ' : ' // 'name = ' // &
            trim(aux%soil_properties_name(j)) // ', ivar = ' // &
            StringWrite(aux%soil_properties_ivar(j))
          call PrintMsg(option)
        enddo
        option%io_buffer = 'Uninitialized value(s) exist within &
            &material_auxvar%soil_properties_ivar in MaterialAuxCreate(). &
            &Please email your input deck to pflotran-dev@googlegroups.com.'
        call PrintErrMsg(option)
      endif
    enddo
  else
    nullify(aux%soil_properties_ivar)
    nullify(aux%soil_properties_name)
  endif
  aux%num_aux = 0
  aux%time_t = 0.d0
  aux%time_tpdt = 0.d0

  MaterialAuxCreate => aux

end function MaterialAuxCreate

! ************************************************************************** !

subroutine MaterialAuxVarInit(auxvar,option)
  !
  ! Initialize auxiliary object
  !
  ! Author: Glenn Hammond
  ! Date: 01/09/14
  !

  use Option_module

  implicit none

  type(material_auxvar_type) :: auxvar
  type(option_type) :: option

  auxvar%id = UNINITIALIZED_INTEGER
  auxvar%volume = UNINITIALIZED_DOUBLE
  auxvar%porosity_0 = UNINITIALIZED_DOUBLE
  auxvar%porosity_base = UNINITIALIZED_DOUBLE
  auxvar%porosity = UNINITIALIZED_DOUBLE
  auxvar%dporosity_dp = 0.d0
  auxvar%tortuosity = UNINITIALIZED_DOUBLE
  auxvar%soil_particle_density = UNINITIALIZED_DOUBLE
  if (option%iflowmode /= NULL_MODE) then
    if (option%flow%full_perm_tensor) then
      allocate(auxvar%permeability(6))
    else
      allocate(auxvar%permeability(3))
    endif
    auxvar%permeability = UNINITIALIZED_DOUBLE
  else
    nullify(auxvar%permeability)
  endif
  nullify(auxvar%sat_func_prop)
  nullify(auxvar%fracture)
  if (option%use_sc) then
    allocate(auxvar%secondary_prop)
    auxvar%secondary_prop%epsilon = UNINITIALIZED_DOUBLE
    auxvar%secondary_prop%half_matrix_width = UNINITIALIZED_DOUBLE
    auxvar%secondary_prop%ncells = UNINITIALIZED_INTEGER
  else
    nullify(auxvar%secondary_prop)
  endif
  auxvar%creep_closure_id = 1

  if (max_material_index > 0) then
    allocate(auxvar%soil_properties(max_material_index))
    ! initialize these to zero for now
    auxvar%soil_properties = UNINITIALIZED_DOUBLE
  else
    nullify(auxvar%soil_properties)
  endif

  nullify(auxvar%geomechanics_subsurface_prop)

end subroutine MaterialAuxVarInit

! ************************************************************************** !

subroutine MaterialAuxInitSoilPropertyMap(aux,index,ivar,name)
  !
  ! Initializes entry in soil property mapping arrays
  !
  ! Author: Glenn Hammond
  ! Date: 07/07/23
  !
  type(material_type) :: aux
  PetscInt :: index
  PetscInt :: ivar
  character(len=*) :: name

  if (index > 0) then
    aux%soil_properties_ivar(index) = ivar
    aux%soil_properties_name(index) = trim(name)
  endif

end subroutine MaterialAuxInitSoilPropertyMap

! ************************************************************************** !

subroutine MaterialAuxVarCopy(auxvar,auxvar2,option)
  !
  ! Copies an auxiliary variable
  !
  ! Author: Glenn Hammond
  ! Date: 01/09/14
  !

  use Option_module

  implicit none

  type(material_auxvar_type) :: auxvar, auxvar2
  type(option_type) :: option

  auxvar2%volume = auxvar%volume
  auxvar2%porosity_0 = auxvar%porosity_0
  auxvar2%porosity_base = auxvar%porosity_base
  auxvar2%porosity = auxvar%porosity
  auxvar2%tortuosity = auxvar%tortuosity
  auxvar2%soil_particle_density = auxvar%soil_particle_density
  if (associated(auxvar%permeability)) then
    auxvar2%permeability = auxvar%permeability
  endif
  if (associated(auxvar%sat_func_prop)) then
    auxvar2%sat_func_prop = auxvar%sat_func_prop
  endif
  if (associated(auxvar%soil_properties)) then
    auxvar2%soil_properties = auxvar%soil_properties
  endif
  auxvar2%creep_closure_id = auxvar%creep_closure_id

end subroutine MaterialAuxVarCopy

! ************************************************************************** !

subroutine MaterialAuxSetPermTensorModel(model,option)

  use Option_module

  implicit none

  PetscInt :: model
  type(option_type) :: option

  select case(model)
    case(TENSOR_TO_SCALAR_LINEAR,TENSOR_TO_SCALAR_FLOW, &
         TENSOR_TO_SCALAR_POTENTIAL,TENSOR_TO_SCALAR_FLOW_FT, &
         TENSOR_TO_SCALAR_POTENTIAL_FT)
      perm_tensor_to_scalar_model = model
    case default
      option%io_buffer  = 'MaterialAuxSetPermTensorModel: tensor to scalar &
                          &model type is not recognized.'
      call PrintErrMsg(option)
  end select

end subroutine MaterialAuxSetPermTensorModel

! ************************************************************************** !

subroutine MaterialAuxSetTortTensorModel(model,option)

  use Option_module

  implicit none

  PetscInt :: model
  type(option_type) :: option

  select case(model)
    case(TENSOR_TO_SCALAR_LINEAR,TENSOR_TO_SCALAR_FLOW, &
         TENSOR_TO_SCALAR_POTENTIAL,TENSOR_TO_SCALAR_FLOW_FT, &
         TENSOR_TO_SCALAR_POTENTIAL_FT)
      tort_tensor_to_scalar_model = model
    case default
      option%io_buffer  = 'MaterialAuxSetPermTensorModel: tensor to scalar &
                          &model type is not recognized.'
      call PrintErrMsg(option)
  end select

end subroutine MaterialAuxSetTortTensorModel

! ************************************************************************** !

subroutine PermeabilityTensorToScalar(material_auxvar,dist,scalar_permeability)
  !
  ! Transforms a diagonal permeability tensor to a scalar through a dot
  ! product.
  !
  ! Author: Glenn Hammond
  ! Date: 01/09/14
  !
  ! Modified by Moise Rousseau 09/04/19 for full tensor
  !
  use Utility_module, only : Equal

  implicit none

  type(material_auxvar_type) :: material_auxvar
  ! -1 = fraction upwind
  ! 0 = magnitude
  ! 1 = unit x-dir
  ! 2 = unit y-dir
  ! 3 = unit z-dir
  PetscReal, intent(in) :: dist(-1:3)
  PetscReal, intent(out) :: scalar_permeability

  PetscReal :: kx, ky, kz, kxy, kxz, kyz

  kx = material_auxvar%permeability(perm_xx_index)
  ky = material_auxvar%permeability(perm_yy_index)
  kz = material_auxvar%permeability(perm_zz_index)

  select case(perm_tensor_to_scalar_model)
    case(TENSOR_TO_SCALAR_LINEAR)
      scalar_permeability = DiagTensorToScalar_Linear(kx,ky,kz,dist)
    case(TENSOR_TO_SCALAR_FLOW)
      scalar_permeability = DiagTensorToScalar_Flow(kx,ky,kz,dist)
    case(TENSOR_TO_SCALAR_POTENTIAL)
      scalar_permeability = DiagTensortoScalar_Potential(kx,ky,kz,dist)
    case(TENSOR_TO_SCALAR_FLOW_FT)
      kxy = material_auxvar%permeability(perm_xy_index)
      kxz = material_auxvar%permeability(perm_xz_index)
      kyz = material_auxvar%permeability(perm_yz_index)
      scalar_permeability = FullTensorToScalar_Flow(kx,ky,kz,kxy,kxz,kyz,dist)
    case(TENSOR_TO_SCALAR_POTENTIAL_FT)
      kxy = material_auxvar%permeability(perm_xy_index)
      kxz = material_auxvar%permeability(perm_xz_index)
      kyz = material_auxvar%permeability(perm_yz_index)
      scalar_permeability = FullTensorToScalar_Pot(kx,ky,kz,kxy,kxz,kyz,dist)
    case default
      ! as default, just do linear
      !scalar_permeability = DiagTensorToScalar_Linear(kx,ky,kz,dist)
      ! as default, do perm in direction of flow
      !scalar_permeability = DiagTensorToScalar_Flow(kx,ky,kz,dist)
      ! as default, do perm in direction of potential gradient
      scalar_permeability = DiagTensorToScalar_Potential(kx,ky,kz,dist)
  end select


end subroutine PermeabilityTensorToScalar

! ************************************************************************** !

subroutine PermeabilityTensorToScalarSafe(material_auxvar,dist, &
                                          scalar_permeability)
  !
  ! Transforms a diagonal perm. tensor to a scalar through a dot product.
  ! This version will not generate NaNs for zero permeabilities
  !
  ! Author: Dave Ponting
  ! Date: 03/19/19
  !

  implicit none

  type(material_auxvar_type) :: material_auxvar

  PetscReal, intent(in) :: dist(-1:3)
  PetscReal, intent(out) :: scalar_permeability

  PetscReal :: kx, ky, kz, kxy, kxz, kyz

  kx = material_auxvar%permeability(perm_xx_index)
  ky = material_auxvar%permeability(perm_yy_index)
  kz = material_auxvar%permeability(perm_zz_index)

  select case(perm_tensor_to_scalar_model)
    case(TENSOR_TO_SCALAR_LINEAR)
      scalar_permeability = DiagTensorToScalar_Linear(kx,ky,kz,dist)
    case(TENSOR_TO_SCALAR_FLOW)
      scalar_permeability = DiagTensorToScalar_Flow(kx,ky,kz,dist)
    case(TENSOR_TO_SCALAR_POTENTIAL)
      scalar_permeability = DiagTensorToScalarPotSafe(kx,ky,kz,dist)
    case(TENSOR_TO_SCALAR_FLOW_FT)
      kxy = material_auxvar%permeability(perm_xy_index)
      kxz = material_auxvar%permeability(perm_xz_index)
      kyz = material_auxvar%permeability(perm_yz_index)
      scalar_permeability = FullTensorToScalar_Flow(kx,ky,kz,kxy,kxz,kyz,dist)
    case(TENSOR_TO_SCALAR_POTENTIAL_FT)
      kxy = material_auxvar%permeability(perm_xy_index)
      kxz = material_auxvar%permeability(perm_xz_index)
      kyz = material_auxvar%permeability(perm_yz_index)
      scalar_permeability = FullTensorToScalarPotSafe(kx,ky,kz,kxy,kxz,kyz,dist)
    case default
      scalar_permeability = DiagTensorToScalarPotSafe(kx,ky,kz,dist)
  end select

end subroutine PermeabilityTensorToScalarSafe

! ************************************************************************** !

function DiagTensorToScalar_Linear(valx,valy,valz,dist)
  implicit none
  PetscReal :: DiagTensorToScalar_Linear
  PetscReal, intent(in) :: dist(-1:3)
  PetscReal :: valx,valy,valz

  DiagTensorToScalar_Linear = valx*dabs(dist(1))+valy*dabs(dist(2))+&
                              valz*dabs(dist(3))

end function DiagTensorToScalar_Linear

! ************************************************************************** !

function DiagTensorToScalar_Flow(valx,valy,valz,dist)

  ! Tensor in the direction of flow

  implicit none
  PetscReal :: DiagTensorToScalar_Flow
  PetscReal, intent(in) :: dist(-1:3)
  PetscReal :: valx,valy,valz

  DiagTensorToScalar_Flow = valx*dabs(dist(1))**2.0 + &
                            valy*dabs(dist(2))**2.0 + &
                            valz*dabs(dist(3))**2.0

end function DiagTensorToScalar_Flow

! ************************************************************************** !

function FullTensorToScalar_Flow(valx,valy,valz,valxy,valxz,valyz,dist)

  ! Tensor in the direction of flow
  ! Include non diagonal term of the full symetric tensor
  !
  ! Author: Moise Rousseau
  ! Date: 08/26/19

  implicit none

  PetscReal :: FullTensorToScalar_Flow
  PetscReal, intent(in) :: dist(-1:3)
  PetscReal :: valx,valy,valz,valxy,valxz,valyz

  FullTensorToScalar_Flow = valx*dabs(dist(1))**2.0 + &
                            valy*dabs(dist(2))**2.0 + &
                            valz*dabs(dist(3))**2.0 + &
                            2*valxy*dist(1)*dist(2) + &
                            2*valxz*dist(1)*dist(3) + &
                            2*valyz*dist(2)*dist(3)

end function FullTensorToScalar_Flow

! ************************************************************************** !

function DiagTensorToScalar_Potential(valx,valy,valz,dist)

  ! Tensor in the direction of the potential gradient

  implicit none
  PetscReal :: DiagTensorToScalar_Potential
  PetscReal, intent(in) :: dist(-1:3)
  PetscReal :: valx,valy,valz

  DiagTensorToScalar_Potential = 1.d0/(dist(1)*dist(1)/valx + &
                                       dist(2)*dist(2)/valy + &
                                       dist(3)*dist(3)/valz)

end function DiagTensorToScalar_Potential

! ************************************************************************** !

function FullTensorToScalar_Pot(valx,valy,valz,valxy,valxz,valyz,dist)

  ! Tensor in the direction of the potential gradient
  ! Include off diagonal term
  ! Not working
  !
  ! Author: Moise Rousseau
  ! Date: 08/26/19

  implicit none
  PetscReal :: FullTensorToScalar_Pot
  PetscReal, intent(in) :: dist(-1:3)
  PetscReal :: valx,valy,valz,valxy,valxz,valyz

  FullTensorToScalar_Pot = 1.d0/(dist(1)*dist(1)/valx + &
                                 dist(2)*dist(2)/valy + &
                                 dist(3)*dist(3)/valz + &
                                 2*dist(1)*dist(2)/valxy + &
                                 2*dist(1)*dist(3)/valxz + &
                                 2*dist(2)*dist(3)/valyz)

end function FullTensorToScalar_Pot

! ************************************************************************** !

function DiagTensorToScalarPotSafe(valx,valy,valz,dist)

  ! Tensor in the direction of the potential gradient
  ! This version will not generate NaNs for zero terms
  !
  ! Author: Dave Ponting
  ! Date: 03/19/19
  !

  implicit none
  PetscReal :: DiagTensorToScalarPotSafe
  PetscReal, intent(in) :: dist(-1:3)
  PetscReal :: valx, valy, valz, valxi, valyi, valzi, den, deni

  !  Form safe inverse permeabilities

  valxi = 0.0
  valyi = 0.0
  valzi = 0.0

  if (valx>0.0) valxi = 1.0/valx
  if (valy>0.0) valyi = 1.0/valy
  if (valz>0.0) valzi = 1.0/valz

  !  Form denominator

  den = dist(1)*dist(1)*valxi + &
        dist(2)*dist(2)*valyi + &
        dist(3)*dist(3)*valzi

  !  Form safe inverse denominator

  deni = 0.0
  if (den>0.0) deni=1.0/den

  !  Store final value

  DiagTensorToScalarPotSafe = deni

end function DiagTensorToScalarPotSafe

! ************************************************************************** !

function FullTensorToScalarPotSafe(valx,valy,valz,valxy,valxz,valyz,dist)

  ! Tensor in the direction of the potential gradient
  ! This version will not generate NaNs for zero terms
  !
  ! Author: Dave Ponting
  ! Date: 03/19/19
  !
  ! Modify to include non diagonal term
  ! Not working for instance
  !
  ! Author: Moise Rousseau
  ! Date: 08/26/19

  implicit none

  PetscReal :: FullTensorToScalarPotSafe
  PetscReal, intent(in) :: dist(-1:3)
  PetscReal :: valx, valy, valz, valxi, valyi, valzi, den, deni
  PetscReal :: valxy, valxz, valyz, valxyi, valxzi, valyzi

  !  Form safe inverse permeabilities

  valxi = 0.0
  valyi = 0.0
  valzi = 0.0
  valxyi = 0.0
  valxzi = 0.0
  valyzi = 0.0

  if (valx>0.0) valxi = 1.0/valx
  if (valy>0.0) valyi = 1.0/valy
  if (valz>0.0) valzi = 1.0/valz
  if (valxy>0.0) valxyi = 1.0/valxy
  if (valxz>0.0) valxzi = 1.0/valxz
  if (valyz>0.0) valyzi = 1.0/valyz

  !  Form denominator

  den = dist(1)*dist(1)*valxi + &
        dist(2)*dist(2)*valyi + &
        dist(3)*dist(3)*valzi + &
        2*dist(1)*dist(2)*valxyi + &
        2*dist(1)*dist(3)*valxzi + &
        2*dist(2)*dist(3)*valyzi

  !  Form safe inverse denominator

  deni = 0.0
  if (den>0.0) deni=1.0/den

  !  Store final value

  FullTensorToScalarPotSafe = deni

end function FullTensorToScalarPotSafe

! ************************************************************************** !

function TortuosityTensorToScalar(material_auxvar,dist)
  !
  ! Calculates a scalar tortuosity from a tensor
  !
  ! Author: Jens Eckel
  ! Date: 12/21/23
  !
  implicit none

  type(material_auxvar_type) :: material_auxvar
  ! -1 = fraction upwind
  ! 0 = magnitude
  ! 1 = unit x-dir
  ! 2 = unit y-dir
  ! 3 = unit z-dir
  PetscReal, intent(in) :: dist(-1:3)

  PetscReal :: TortuosityTensorToScalar

  PetscReal :: tx, ty, tz

  tx = material_auxvar%tortuosity
  ty = material_auxvar%soil_properties(tortuosity_yy_index)
  tz = material_auxvar%soil_properties(tortuosity_zz_index)

  ! up to now only by structured/unstructured grid
  select case(tort_tensor_to_scalar_model)
    case(TENSOR_TO_SCALAR_LINEAR)
      TortuosityTensorToScalar = DiagTensorToScalar_Linear(tx,ty,tz,dist)
    case(TENSOR_TO_SCALAR_POTENTIAL)
      TortuosityTensorToScalar = DiagTensorToScalarPotSafe(tx,ty,tz,dist)
    case default
      TortuosityTensorToScalar = DiagTensorToScalar_Linear(tx,ty,tz,dist)
  end select

end function TortuosityTensorToScalar

! ************************************************************************** !

function MaterialAuxVarGetSoilPropIndex(ivar)
  !
  ! Returns the index in the soil properties array for the desire parameter
  !
  ! Author: Glenn Hammond
  ! Date: 12/21/23
  !
  use Variables_module, only : SOIL_COMPRESSIBILITY, &
                               SOIL_REFERENCE_PRESSURE, &
                               MATERIAL_ELECTRICAL_CONDUCTIVITY, &
                               ARCHIE_CEMENTATION_EXPONENT, &
                               ARCHIE_SATURATION_EXPONENT, &
                               ARCHIE_TORTUOSITY_CONSTANT, &
                               SURFACE_ELECTRICAL_CONDUCTIVITY, &
                               WAXMAN_SMITS_CLAY_CONDUCTIVITY, &
                               NUMBER_SECONDARY_CELLS, &
                               TORTUOSITY_Y, TORTUOSITY_Z

  implicit none

  PetscInt :: ivar

  PetscInt :: MaterialAuxVarGetSoilPropIndex

  PetscInt :: index_

  select case(ivar)
    case(SOIL_COMPRESSIBILITY)
      index_ = soil_compressibility_index
    case(SOIL_REFERENCE_PRESSURE)
      index_ = soil_reference_pressure_index
    case(MATERIAL_ELECTRICAL_CONDUCTIVITY)
      index_ = material_elec_conduct_index
    case(ARCHIE_CEMENTATION_EXPONENT)
      index_ = archie_cementation_exp_index
    case(ARCHIE_SATURATION_EXPONENT)
      index_ = archie_saturation_exp_index
    case(ARCHIE_TORTUOSITY_CONSTANT)
      index_ = archie_tortuosity_index
    case(SURFACE_ELECTRICAL_CONDUCTIVITY)
      index_ = surf_elec_conduct_index
    case(WAXMAN_SMITS_CLAY_CONDUCTIVITY)
      index_ = ws_clay_conduct_index
    case(TORTUOSITY_Y)
      index_ = tortuosity_yy_index
    case(TORTUOSITY_Z)
      index_ = tortuosity_zz_index
    ! ADD_SOIL_PROPERTY_INDEX_HERE
    case default
      print *, 'Unrecognized variable in MaterialAuxVarGetSoilPropIndex: ', &
               ivar
      stop
  end select
  MaterialAuxVarGetSoilPropIndex = index_

end function MaterialAuxVarGetSoilPropIndex

! ************************************************************************** !

function MaterialAuxVarGetValue(material_auxvar,ivar)
  !
  ! Returns the value of an entry in material_auxvar_type based on ivar.
  !
  ! Author: Glenn Hammond
  ! Date: 03/28/14
  !

  use Variables_module

  implicit none

  type(material_auxvar_type) :: material_auxvar
  PetscInt :: ivar

  PetscReal :: MaterialAuxVarGetValue

  PetscInt :: index_

  MaterialAuxVarGetValue = UNINITIALIZED_DOUBLE
  select case(ivar)
    case(VOLUME)
      MaterialAuxVarGetValue = material_auxvar%volume
    case(INITIAL_POROSITY)
      MaterialAuxVarGetValue = material_auxvar%porosity_0
    case(BASE_POROSITY)
      MaterialAuxVarGetValue = material_auxvar%porosity_base
    case(POROSITY)
      MaterialAuxVarGetValue = material_auxvar%porosity
    case(TORTUOSITY)
      MaterialAuxVarGetValue = material_auxvar%tortuosity
    case(PERMEABILITY_X)
      MaterialAuxVarGetValue = material_auxvar%permeability(perm_xx_index)
    case(PERMEABILITY_Y)
      MaterialAuxVarGetValue = material_auxvar%permeability(perm_yy_index)
    case(PERMEABILITY_Z)
      MaterialAuxVarGetValue = material_auxvar%permeability(perm_zz_index)
    case(PERMEABILITY_XY,PERMEABILITY_YZ,PERMEABILITY_XZ)
      if (size(material_auxvar%permeability) > 3) then
        select case(ivar)
          case(PERMEABILITY_XY)
            MaterialAuxVarGetValue = &
              material_auxvar%permeability(perm_xy_index)
          case(PERMEABILITY_YZ)
            MaterialAuxVarGetValue = &
              material_auxvar%permeability(perm_yz_index)
          case(PERMEABILITY_XZ)
            MaterialAuxVarGetValue = &
              material_auxvar%permeability(perm_xz_index)
        end select
      else
        MaterialAuxVarGetValue = 0.d0
      endif
    case(EPSILON)
      MaterialAuxVarGetValue = material_auxvar%secondary_prop%epsilon
    case(HALF_MATRIX_WIDTH)
      MaterialAuxVarGetValue = &
        material_auxvar%secondary_prop%half_matrix_width
    case default ! entries in material_auxvars%soil_properties
      index_ = MaterialAuxVarGetSoilPropIndex(ivar)
      MaterialAuxVarGetValue = material_auxvar%soil_properties(index_)
  end select

end function MaterialAuxVarGetValue

! ************************************************************************** !

subroutine MaterialAuxVarSetValue(material_auxvar,ivar,value)
  !
  ! Sets the value of an entry in material_auxvar_type based on ivar.
  !
  ! Author: Glenn Hammond
  ! Date: 03/28/14
  !

  use Variables_module

  implicit none

  type(material_auxvar_type) :: material_auxvar
  PetscInt :: ivar
  PetscReal :: value

  PetscInt :: index_

  select case(ivar)
    case(VOLUME)
      material_auxvar%volume = value
    case(INITIAL_POROSITY)
      material_auxvar%porosity_0 = value
    case(BASE_POROSITY)
      material_auxvar%porosity_base = value
    case(POROSITY)
      material_auxvar%porosity = value
    case(TORTUOSITY)
      material_auxvar%tortuosity = value
    case(PERMEABILITY_X)
      material_auxvar%permeability(perm_xx_index) = value
    case(PERMEABILITY_Y)
      material_auxvar%permeability(perm_yy_index) = value
    case(PERMEABILITY_Z)
      material_auxvar%permeability(perm_zz_index) = value
    case(PERMEABILITY_XY)
      material_auxvar%permeability(perm_xy_index) = value
    case(PERMEABILITY_YZ)
      material_auxvar%permeability(perm_yz_index) = value
    case(PERMEABILITY_XZ)
      material_auxvar%permeability(perm_xz_index) = value
    case(EPSILON)
      material_auxvar%secondary_prop%epsilon = value
    case(HALF_MATRIX_WIDTH)
      material_auxvar%secondary_prop%half_matrix_width = value
    case(NUMBER_SECONDARY_CELLS)
      material_auxvar%secondary_prop%ncells = int(value)
    case default ! entries in material_auxvars%soil_properties
      index_ = MaterialAuxVarGetSoilPropIndex(ivar)
      material_auxvar%soil_properties(index_) = value
  end select

end subroutine MaterialAuxVarSetValue

! ************************************************************************** !

subroutine MaterialAuxVarCompute(auxvar,pressure)
  !
  ! Updates secondary material properties that are a function of state
  ! variables
  !
  ! Author: Glenn Hammond
  ! Date: 08/21/19
  !

  implicit none

  type(material_auxvar_type), intent(inout) :: auxvar
  PetscReal, intent(in) :: pressure

  auxvar%porosity = auxvar%porosity_base
  auxvar%dporosity_dp = 0.d0
  if (soil_compressibility_index > 0) then
    call MaterialCompressSoil(auxvar,pressure,auxvar%porosity, &
                              auxvar%dporosity_dp)
  endif

end subroutine MaterialAuxVarCompute

! ************************************************************************** !

subroutine MaterialCompressSoilLeijnse(auxvar,pressure, &
                                       compressed_porosity, &
                                       dcompressed_porosity_dp)
  !
  ! Calculates soil matrix compression based on Leijnse, 1992.
  !
  ! Author: Glenn Hammond
  ! Date: 01/14/14
  !

  implicit none

  type(material_auxvar_type), intent(in) :: auxvar
  PetscReal, intent(in) :: pressure
  PetscReal, intent(out) :: compressed_porosity
  PetscReal, intent(out) :: dcompressed_porosity_dp

  PetscReal :: compressibility
  PetscReal :: compression
  PetscReal :: tempreal

  compressibility = auxvar%soil_properties(soil_compressibility_index)
  compression = &
    exp(-1.d0 * compressibility * &
        (pressure - auxvar%soil_properties(soil_reference_pressure_index)))
  tempreal = (1.d0 - auxvar%porosity_base) * compression
  compressed_porosity = 1.d0 - tempreal
  dcompressed_porosity_dp = tempreal * compressibility

end subroutine MaterialCompressSoilLeijnse

! ************************************************************************** !

subroutine MaterialCompressSoilBRAGFLO(auxvar,pressure, &
                                       compressed_porosity, &
                                       dcompressed_porosity_dp)
  !
  ! Calculates soil matrix compression based on Eq. 9.6.9 of BRAGFLO
  !
  ! Author: Glenn Hammond
  ! Date: 01/14/14
  !

  implicit none

  type(material_auxvar_type), intent(in) :: auxvar
  PetscReal, intent(in) :: pressure
  PetscReal, intent(out) :: compressed_porosity
  PetscReal, intent(out) :: dcompressed_porosity_dp

  PetscReal :: compressibility


  ! convert to pore compressiblity by dividing by base porosity
  compressibility = auxvar%soil_properties(soil_compressibility_index) / &
                    auxvar%porosity_base
  compressed_porosity = auxvar%porosity_base * &
    exp(compressibility * &
        (pressure - auxvar%soil_properties(soil_reference_pressure_index)))
  dcompressed_porosity_dp = compressibility * compressed_porosity

end subroutine MaterialCompressSoilBRAGFLO

! ************************************************************************** !

subroutine MaterialCompressSoilLinear(auxvar,pressure, &
                                      compressed_porosity, &
                                      dcompressed_porosity_dp)
  !
  ! Calculates soil matrix compression for standard constant
  ! aquifer compressibility
  !
  ! variable 'alpha' is Freeze and Cherry, 1982
  !
  ! Author: Danny Birdsell and Satish Karra
  ! Date: 07/26/2016
  !

  implicit none

  type(material_auxvar_type), intent(in) :: auxvar
  PetscReal, intent(in) :: pressure
  PetscReal, intent(out) :: compressed_porosity
  PetscReal, intent(out) :: dcompressed_porosity_dp

  PetscReal :: compressibility

  compressibility = auxvar%soil_properties(soil_compressibility_index)
  compressed_porosity = auxvar%porosity_base + compressibility * &
            (pressure - auxvar%soil_properties(soil_reference_pressure_index))
  dcompressed_porosity_dp = compressibility

end subroutine MaterialCompressSoilLinear

! ************************************************************************** !

subroutine MaterialCompressSoilPoroExp(auxvar,pressure, &
                                       compressed_porosity, &
                                       dcompressed_porosity_dp)
  !
  ! Calculates soil matrix compression based on Eq. 9.6.9 of BRAGFLO
  !
  ! Author: Glenn Hammond
  ! Date: 01/14/14
  !

  implicit none

  type(material_auxvar_type), intent(in) :: auxvar
  PetscReal, intent(in) :: pressure
  PetscReal, intent(out) :: compressed_porosity
  PetscReal, intent(out) :: dcompressed_porosity_dp

  PetscReal :: compressibility

  compressibility = auxvar%soil_properties(soil_compressibility_index)
  compressed_porosity = auxvar%porosity_base * &
    exp(compressibility * &
        (pressure - auxvar%soil_properties(soil_reference_pressure_index)))
  dcompressed_porosity_dp = compressibility * compressed_porosity

end subroutine MaterialCompressSoilPoroExp

! ************************************************************************** !

subroutine MaterialCompressSoilQuadratic(auxvar,pressure, &
                                         compressed_porosity, &
                                         dcompressed_porosity_dp)
  !
  ! Calculates soil matrix compression based on a quadratic model
  ! This is thedefaul model adopted in ECLIPSE
  !
  ! Author: Paolo Orsini
  ! Date: 02/27/17
  !

  implicit none

  type(material_auxvar_type), intent(in) :: auxvar
  PetscReal, intent(in) :: pressure
  PetscReal, intent(out) :: compressed_porosity
  PetscReal, intent(out) :: dcompressed_porosity_dp

  PetscReal :: compressibility
  PetscReal :: compress_factor

  compressibility = auxvar%soil_properties(soil_compressibility_index)

  compress_factor = compressibility * &
          (pressure - auxvar%soil_properties(soil_reference_pressure_index))

  compressed_porosity = auxvar%porosity_base * &
          ( 1.0 + compress_factor + (compress_factor**2)/2.0 )

  dcompressed_porosity_dp = auxvar%porosity_base * &
          ( 1.0 + compress_factor) * compressibility

end subroutine MaterialCompressSoilQuadratic

! ************************************************************************** !

subroutine MaterialAuxVarFractureStrip(fracture)
  !
  ! Deallocates a fracture auxiliary object
  !
  ! Author: Glenn Hammond
  ! Date: 06/14/17
  !
  use Utility_module, only : DeallocateArray

  implicit none

  type(fracture_auxvar_type), pointer :: fracture

  if (.not.associated(fracture)) return

  ! properties and vector are now static arrays.
  deallocate(fracture)
  nullify(fracture)

end subroutine MaterialAuxVarFractureStrip

! ************************************************************************** !

subroutine MaterialAuxVarStrip(auxvar)
  !
  ! Deallocates a material auxiliary object
  !
  ! Author: Glenn Hammond
  ! Date: 01/09/14
  !
  use Utility_module, only : DeallocateArray

  implicit none

  type(material_auxvar_type) :: auxvar

  call DeallocateArray(auxvar%permeability)
  call DeallocateArray(auxvar%sat_func_prop)
  call DeallocateArray(auxvar%soil_properties)
  call MaterialAuxVarFractureStrip(auxvar%fracture)
  if (associated(auxvar%secondary_prop)) then
    deallocate(auxvar%secondary_prop)
    nullify(auxvar%secondary_prop)
  endif
  if (associated(auxvar%geomechanics_subsurface_prop)) then
    call DeallocateArray(auxvar%geomechanics_subsurface_prop)
  endif

end subroutine MaterialAuxVarStrip

! ************************************************************************** !

subroutine MaterialAuxDestroy(aux)
  !
  ! Deallocates a material auxiliary object
  !
  ! Author: Glenn Hammond
  ! Date: 03/02/11
  !
  use Utility_module, only : DeallocateArray

  implicit none

  type(material_type), pointer :: aux

  PetscInt :: iaux

  if (.not.associated(aux)) return

  if (associated(aux%auxvars)) then
    do iaux = 1, aux%num_aux
      call MaterialAuxVarStrip(aux%auxvars(iaux))
    enddo
    deallocate(aux%auxvars)
  endif
  nullify(aux%auxvars)

  if (associated(aux%material_parameter)) then
    call DeallocateArray(aux%material_parameter%soil_heat_capacity)
    call DeallocateArray(aux%material_parameter%soil_thermal_conductivity)
  endif
  deallocate(aux%material_parameter)
  nullify(aux%material_parameter)
  call DeallocateArray(aux%soil_properties_ivar)
  call DeallocateArray(aux%soil_properties_name)

  deallocate(aux)
  nullify(aux)

end subroutine MaterialAuxDestroy

end module Material_Aux_module
