module Input_Aux_module

#include "petsc/finclude/petscsys.h"
  use petscsys
  use Option_module

  use PFLOTRAN_Constants_module

  implicit none

  private


  type, public :: input_type
    PetscInt :: fid
    PetscInt :: line_number
    PetscErrorCode :: ierr
    character(len=MAXSTRINGLENGTH) :: path
    character(len=MAXSTRINGLENGTH) :: filename
    character(len=MAXSTRINGLENGTH) :: buf
    character(len=MAXSTRINGLENGTH) :: err_buf
    character(len=MAXSTRINGLENGTH) :: err_buf2
    PetscBool :: broadcast_read
    PetscBool :: force_units ! force user to declare units on datasets
    type(input_type), pointer :: parent
  end type input_type

  type :: input_dbase_type
    character(len=MAXWORDLENGTH), pointer :: icard(:)
    character(len=MAXWORDLENGTH), pointer :: rcard(:)
    character(len=MAXWORDLENGTH), pointer :: ccard(:)
    PetscInt, pointer :: ivalue(:)
    PetscReal, pointer :: rvalue(:)
    character(len=MAXWORDLENGTH), pointer :: cvalue(:)
  end type input_dbase_type

  type(input_dbase_type), pointer, public :: dbase => null()

  interface InputCreate
    module procedure InputCreate1
    module procedure InputCreate2
    module procedure InputCreate3
  end interface

  interface InputReadWord
    module procedure InputReadWord1
    module procedure InputReadWord2
  end interface

  interface InputReadNChars
    module procedure InputReadNChars1
    module procedure InputReadNChars2
  end interface

  interface InputReadInt
    module procedure InputReadInt1
    module procedure InputReadInt2
#if defined(PETSC_USE_64BIT_INDICES) && (PETSC_SIZEOF_MPI_FINT * PETSC_BITS_PER_BYTE != 64)
    ! If PetscInt and PetscMPIInt have different sizes (occurs for some builds
    ! with 64 bit indices), then we need to have additional routines for the
    ! InputReadInt() generic subroutine.  (We use the above check instead of
    ! directly checking to see if PetscInt and PetscMPIInt have the same size
    ! because the size of PetscInt is not included in the
    ! $PETSC_DIR/$PETSC_ARCH/include/petscconf.h file.) If the two types have
    ! the same size, then these additional routines for type PetscMPIInt must
    ! *not* be defined, because then the interface becomes ambiguous, since
    ! Fortran doesn't know the difference between PetscInt and PetscMPIInt if
    ! they are identically sized integers.  --RTM
    module procedure InputReadInt3
    module procedure InputReadInt4
#endif
  end interface

  interface InputReadDouble
    module procedure InputReadDouble1
    module procedure InputReadDouble2
  end interface

  interface InputReadNDoubles
    module procedure InputReadNDoubles1
    module procedure InputReadNDoubles2
  end interface

  interface InputError
    module procedure InputError1
    module procedure InputError2
  end interface

  interface InputErrorMsg
    module procedure InputErrorMsg1
    module procedure InputErrorMsg2
  end interface

  interface InputDefaultMsg
    module procedure InputDefaultMsg1
    module procedure InputDefaultMsg2
  end interface

  interface InputReadStringErrorMsg
    module procedure InputReadStringErrorMsg1
    module procedure InputReadStringErrorMsg2
  end interface

  interface InputFindStringInFile
    module procedure InputFindStringInFile1
    module procedure InputFindStringInFile2
    module procedure InputFindStringInFile3
  end interface

  interface InputKeywordUnrecognized
    module procedure InputKeywordUnrecognized1
    module procedure InputKeywordUnrecognized2
  end interface

  interface InputCheckSupported
    module procedure InputCheckSupported1
    module procedure InputCheckSupported2
    module procedure InputCheckSupported3
  end interface

  interface InputPushBlock
    module procedure InputPushBlock1
    module procedure InputPushBlock2
  end interface

  public :: InputCreate, InputDestroy, InputReadPflotranString, &
            InputReadWord, InputReadDouble, InputReadInt, InputCheckExit, &
            InputReadNDoubles, &
            InputSkipToEND, InputFindStringInFile, InputErrorMsg, &
            InputDefaultMsg, InputReadStringErrorMsg, &
            InputFindStringErrorMsg, InputError, &
            InputReadNChars, InputReadQuotedWord, &
            InputGetCommandLineInt, &
            InputGetCommandLineReal, &
            InputGetCommandLineTruth, &
            InputGetCommandLineString, &
            InputGetLineCount, &
            InputReadToBuffer, &
            InputReadASCIIDbase, &
            InputKeywordUnrecognized, &
            InputCheckSupported, &
            InputCheckMandatoryUnits, &
            InputDbaseDestroy, &
            InputPushExternalFile, &
            InputReadCardDbaseCompatible, &
            InputReadAndConvertUnits, &
            InputRewind, &
            InputCloseNestedFiles, &
            InputReadFilename, &
            InputReadCard, &
            InputPushCard, &
            InputPushBlock, &
            InputPopBlock, &
            InputKeywordDeprecated, &
            InputCheckKeywordBlockCount, &
            InputCountWordsInBuffer

contains

! ************************************************************************** !

function InputCreate1(fid,path,filename,option)
  !
  ! Allocates and initializes a new Input object
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  use Option_module

  implicit none

  PetscInt :: fid
  character(len=*) :: path
  character(len=*) :: filename
  type(option_type) :: option

  type(input_type), pointer :: InputCreate1
  PetscInt :: istatus
  PetscInt :: islash
  character(len=MAXSTRINGLENGTH) :: full_path
  type(input_type), pointer :: input
  PetscBool, parameter :: back = PETSC_TRUE

  allocate(input)
  input%fid = fid
  input%line_number = 0
  input%path = ''
  input%filename = ''
  input%ierr = INPUT_ERROR_NONE
  input%buf = ''
  input%err_buf = ''
  input%err_buf2 = ''
  input%broadcast_read = PETSC_FALSE
  input%force_units = PETSC_FALSE
  nullify(input%parent)


  ! split the filename into a path and filename
                              ! backwards search
  islash = index(filename,'/',back)
  if (islash > 0) then
    input%path(1:islash) = filename(1:islash)
    input%filename(1:len_trim(filename)-islash) = &
      filename(islash+1:len_trim(filename))
  else
    input%filename = filename
  endif

  if (fid == MAX_IN_UNIT) then
    option%io_buffer = 'MAX_IN_UNIT in pflotran_constants.h must be &
      &increased to accommodate a larger number of embedded files.'
    call PrintErrMsg(option)
  endif

  full_path = trim(input%path) // trim(input%filename)
  open(unit=input%fid,file=full_path,status="old",iostat=istatus)
  !TODO(geh): update the error messaging
  if (istatus /= 0) then
    if (len_trim(full_path) == 0) full_path = '<blank>'
    option%io_buffer = 'File: "' // trim(full_path) // '" not found.'
    call PrintErrMsg(option)
    ! for non-blocking case, set error flag
    input%ierr = INPUT_ERROR_DEFAULT
  endif

  InputCreate1 => input

end function InputCreate1

! ************************************************************************** !

function InputCreate2(fid,filename,option)
  !
  ! Allocates and initializes a new Input object
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  use Option_module

  implicit none

  PetscInt :: fid
  character(len=*) :: filename
  type(option_type) :: option

  type(input_type), pointer :: InputCreate2
  character(len=MAXWORDLENGTH) :: word

  word = ''
  InputCreate2 => InputCreate1(fid,word,filename,option)

end function InputCreate2

! ************************************************************************** !

function InputCreate3(input,filename,option)
  !
  ! Allocates and initializes a new input object without a path
  !
  ! Author: Glenn Hammond
  ! Date: 09/28/17
  !

  use Option_module

  implicit none

  type(input_type), pointer :: input ! note that this is the old input object
  character(len=MAXSTRINGLENGTH) :: filename
  type(option_type) :: option

  type(input_type), pointer :: InputCreate3

  InputCreate3 => InputCreate1(input%fid + 1,input%path,filename,option)

end function InputCreate3

! ************************************************************************** !

subroutine InputDefaultMsg1(input,option,buffer)
  !
  ! Informs user that default value will be used.
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !
  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=*) :: buffer

  if (InputError(input)) then
    input%err_buf = buffer
    call InputDefaultMsg(input,option)
  endif

end subroutine InputDefaultMsg1

! ************************************************************************** !

subroutine InputDefaultMsg2(input,option)
  !
  ! Informs user that default value will be used.
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option

  if (InputError(input)) then
    option%io_buffer =  '"' // trim(input%err_buf) // &
                        '" set to default value.'
    call PrintMsg(option)
    input%ierr = INPUT_ERROR_NONE
  endif

end subroutine InputDefaultMsg2

! ************************************************************************** !

subroutine InputErrorMsg1(input,option,buffer1,buffer2)
  !
  ! Informs user of error and stops.
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=*) :: buffer1, buffer2

  if (InputError(input)) then
    input%err_buf = buffer1
    input%err_buf2 = buffer2
    call InputErrorMsg(input,option)
  endif

end subroutine InputErrorMsg1

! ************************************************************************** !

subroutine InputErrorMsg2(input,option)
  !
  ! Informs user of error and stops.
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !
  use String_module, only : StringWrite

  implicit none

  type(input_type) :: input
  type(option_type) :: option

  if (InputError(input)) then
    call InputPrintKeywordLog(input,option,PETSC_TRUE)
    option%io_buffer = 'While reading "' // trim(input%err_buf) // &
                       '" under keyword: ' // trim(input%err_buf2) // '.'
    select case(input%ierr)
      case(INPUT_ERROR_KEYWORD_LENGTH)
        option%io_buffer = trim(option%io_buffer) // &
          ' The length of the keyword may be too long (typically, up to ' // &
          StringWrite(MAXWORDLENGTH) // ' characters).'
      case default
    end select
    call PrintErrMsg(option)
  endif

end subroutine InputErrorMsg2

! ************************************************************************** !

subroutine InputReadStringErrorMsg1(input, option, buffer)
  !
  ! Informs user of error and stops.
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=*) :: buffer

  if (InputError(input)) then
    input%err_buf = buffer
    call InputReadStringErrorMsg(input, option)
  endif

end subroutine InputReadStringErrorMsg1

! ************************************************************************** !

subroutine InputReadStringErrorMsg2(input, option)
  !
  ! Informs user of error and stops.
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option

  if (InputError(input)) then
    call InputPrintKeywordLog(input,option,PETSC_TRUE)
    option%io_buffer = 'While reading in string in "' // &
                       trim(input%err_buf) // '".'
    call PrintErrMsg(option)
  endif

end subroutine InputReadStringErrorMsg2

! ************************************************************************** !

subroutine InputFindStringErrorMsg(input, option, string)
  !
  ! Informs user of error and stops.
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=*) :: string

  if (InputError(input)) then
    call InputPrintKeywordLog(input,option,PETSC_TRUE)
    option%io_buffer = 'Card (' // trim(string) // ') not &
                       &found in file.'
    call PrintErrMsg(option)
  endif

end subroutine InputFindStringErrorMsg

! ************************************************************************** !

subroutine InputReadInt1(input, option, int)
  !
  ! reads and removes an integer value from a string
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  PetscInt :: int

  character(len=MAXWORDLENGTH) :: word
  PetscBool :: found

  found = PETSC_FALSE
  if (associated(dbase)) then
    call InputParseDbaseForInt(input%buf,int,found,option,input%ierr)
  endif

  if (.not.found) then
    call InputReadWord(input%buf,word,PETSC_TRUE,input%ierr)

    if (.not.InputError(input)) then
      read(word,*,iostat=input%ierr) int
    endif
  endif

end subroutine InputReadInt1

! ************************************************************************** !

subroutine InputReadInt2(string, option, int, ierr)
  !
  ! reads and removes an integer value from a string
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  character(len=MAXSTRINGLENGTH) :: string
  type(option_type) :: option
  PetscInt :: int
  PetscErrorCode :: ierr

  character(len=MAXWORDLENGTH) :: word
  PetscBool :: found

  ierr = INPUT_ERROR_NONE

  found = PETSC_FALSE
  if (associated(dbase)) then
    call InputParseDbaseForInt(string,int,found,option,ierr)
  endif

  if (.not.found) then
    call InputReadWord(string,word,PETSC_TRUE,ierr)

    if (.not.InputError(ierr)) then
      read(word,*,iostat=ierr) int
    endif
  endif

end subroutine InputReadInt2

#if defined(PETSC_USE_64BIT_INDICES) && (PETSC_SIZEOF_MPI_FINT * PETSC_BITS_PER_BYTE != 64)

! ************************************************************************** !

subroutine InputReadInt3(input, option, int)
  !
  ! InputReadInt3() and InputReadInt4() must only be defined if PetscInt and
  ! PetscMPIInt differ in size.  See notes above in the interface definition.
  ! --RTM
  ! reads and removes an integer value from a string
  ! authors: Glenn Hammond, Richard Mills
  !
  ! Date: 2/3/2012
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  PetscMPIInt :: int

  character(len=MAXWORDLENGTH) :: word

  call InputReadWord(input%buf,word,PETSC_TRUE,input%ierr)

  if (.not.InputError(input)) then
    read(word,*,iostat=input%ierr) int
  endif

end subroutine InputReadInt3

! ************************************************************************** !

subroutine InputReadInt4(string, option, int, ierr)
  !
  ! reads and removes an integer value from a string
  ! authors: Glenn Hammond, Richard Mills
  !
  ! Date: 2/3/2012
  !

  implicit none

  character(len=MAXSTRINGLENGTH) :: string
  type(option_type) :: option
  PetscMPIInt :: int
  PetscErrorCode :: ierr

  character(len=MAXWORDLENGTH) :: word

  ierr = INPUT_ERROR_NONE
  call InputReadWord(string,word,PETSC_TRUE,ierr)

  if (.not.InputError(ierr)) then
    read(word,*,iostat=ierr) int
  endif

end subroutine InputReadInt4

#endif
! End of defined(PETSC_USE_64BIT_INDICES) &&
! (PETSC_SIZEOF_MPI_FINT * PETSC_BITS_PER_BYTE != 64) conditional

! ************************************************************************** !

subroutine InputReadDouble1(input, option, double)
  !
  ! reads and removes a real value from a string
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  PetscReal :: double

  character(len=MAXWORDLENGTH) :: word
  PetscBool :: found

  found = PETSC_FALSE
  if (associated(dbase)) then
    call InputParseDbaseForDouble(input%buf,double,found,option,input%ierr)
  endif

  if (.not.found) then
    call InputReadWord(input%buf,word,PETSC_TRUE,input%ierr)

    if (.not.InputError(input)) then
      read(word,*,iostat=input%ierr) double
      ! catch NaNs
      if (double /= double) input%ierr = INPUT_ERROR_DEFAULT
    endif
  endif

end subroutine InputReadDouble1

! ************************************************************************** !

subroutine InputReadDouble2(string, option, double, ierr)
  !
  ! reads and removes a real value from a string
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  character(len=MAXSTRINGLENGTH) :: string
  type(option_type) :: option
  PetscReal :: double
  PetscErrorCode :: ierr

  character(len=MAXWORDLENGTH) :: word
  PetscBool :: found

  ierr = INPUT_ERROR_NONE

  found = PETSC_FALSE
  if (associated(dbase)) then
    call InputParseDbaseForDouble(string,double,found,option,ierr)
  endif

  if (.not.found) then
    call InputReadWord(string,word,PETSC_TRUE,ierr)

    if (.not.InputError(ierr)) then
      read(word,*,iostat=ierr) double
      ! catch NaNs
      if (double /= double) ierr = INPUT_ERROR_DEFAULT
    endif
  endif

end subroutine InputReadDouble2

! ************************************************************************** !

subroutine InputReadNDoubles1(input, option, double, n)
  !
  ! reads and removes "n" real value from a string
  !
  ! Author: Glenn Hammond
  ! Date: 08/29/11
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  PetscInt :: n
  PetscReal :: double(n)

  PetscInt :: i

  do i = 1, n
    call InputReadDouble(input,option,double(i))
    if (InputError(input)) return
  enddo

end subroutine InputReadNDoubles1

! ************************************************************************** !

subroutine InputReadNDoubles2(string, option, double, n, ierr)
  !
  ! reads and removes "n" real values from a string
  !
  ! Author: Glenn Hammond
  ! Date: 08/29/11
  !

  implicit none

  character(len=MAXSTRINGLENGTH) :: string
  type(option_type) :: option
  PetscInt :: n
  PetscReal :: double(n)
  PetscErrorCode :: ierr

  PetscInt :: i

  do i = 1, n
    call InputReadDouble(string,option,double(i),ierr)
    if (InputError(ierr)) return
  enddo

end subroutine InputReadNDoubles2

! ************************************************************************** !

subroutine InputReadPflotranString(input, option)
  !
  ! Reads a string (strlen characters long) from a
  ! file while avoiding commented or skipped lines.
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  type(input_type), pointer :: input
  type(option_type) :: option

  PetscErrorCode :: ierr
  PetscInt :: flag

  if (input%broadcast_read) then
    if (OptionIsIORank(option)) then
      call InputReadPflotranStringSlave(input, option)
    endif
    flag = input%ierr
    call MPI_Bcast(flag,ONE_INTEGER_MPI,MPIU_INTEGER,option%comm%io_rank, &
                   option%mycomm,ierr);CHKERRQ(ierr)
    input%ierr = flag
    if (.not.InputError(input)) then
      call MPI_Bcast(input%buf,MAXSTRINGLENGTH,MPI_CHARACTER, &
                     option%comm%io_rank,option%mycomm,ierr);CHKERRQ(ierr)
    endif
  else
    call InputReadPflotranStringSlave(input, option)
  endif

end subroutine InputReadPflotranString

! ************************************************************************** !

subroutine InputReadPflotranStringSlave(input, option)
  !
  ! Reads a string (strlen characters long) from a
  ! file while avoiding commented or skipped lines.
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  use String_module

  implicit none

  type(input_type), pointer :: input
  type(option_type) :: option

  character(len=MAXSTRINGLENGTH) ::  tempstring
  PetscInt, parameter :: num_chars = 40
  character(len=num_chars) :: long_word
  PetscInt :: i
  PetscInt :: skip_count

  input%ierr = INPUT_ERROR_NONE

  long_word = ''

  do
    input%line_number = input%line_number + 1
    read(input%fid,'(a)',iostat=input%ierr) input%buf
    call StringAdjustl(input%buf)

    if (InputError(input)) then
      ! check to see if another file is on the stack
      if (InputPopExternalFile(input)) then
        cycle
      else
        exit
      endif
    endif

    if (input%buf(1:1) == '#' .or. input%buf(1:1) == '!') cycle

    tempstring = input%buf
    call InputReadNChars(tempstring,long_word,num_chars,PETSC_TRUE,input%ierr)
    call CatchLongKeywordError()
    call StringToUpper(long_word)

    if (long_word(1:13) == 'EXTERNAL_FILE') then
      call InputPushCard(input,long_word,option)
      ! have to strip the card 'EXTERNAL_FILE' from the buffer
      call InputReadNChars(input,option,long_word,num_chars,PETSC_FALSE)
      ! push a new input file to stack
      call InputPushExternalFile(input,option)
      cycle
    else if (long_word(1:4) == 'SKIP') then
      call InputPushCard(input,long_word,option)
      ! to avoid keywords that start with SKIP
      if (len_trim(long_word) > 4) then
        exit
      endif
      skip_count = 1
      do
        input%line_number = input%line_number + 1
        read(input%fid,'(a)',iostat=input%ierr) input%buf
        call StringAdjustl(input%buf)
        if (InputError(input)) then
          call InputPrintKeywordLog(input,option,PETSC_TRUE)
          option%io_buffer = 'End of file reached in ' // &
              'InputReadPflotranStringSlave.  SKIP encountered ' // &
              'without a matching NOSKIP.'
          call PrintErrMsg(option)
        endif
        tempstring = input%buf
        call InputReadNChars(tempstring,long_word,num_chars, &
                             PETSC_TRUE,input%ierr)
        call CatchLongKeywordError()
        call StringToUpper(long_word)
        if (long_word(1:4) == 'SKIP') then
          ! to avoid keywords that start with SKIP
          if (len_trim(long_word) == 4) then
            skip_count = skip_count + 1
            call InputPushCard(input,long_word,option)
          endif
        endif
        if (long_word(1:6) == 'NOSKIP') then
          call InputPushCard(input,long_word,option)
          skip_count = skip_count - 1
          if (skip_count == 0) exit
        endif
      enddo
      if (InputError(input)) exit
    else if (long_word(1:1) /= ' ' .and. long_word(1:6) /= 'NOSKIP') then
      exit
    endif
  enddo

  ! Check for comment midway along a string
  if (.not.InputError(input)) then
    tempstring = input%buf
    input%buf = repeat(' ',MAXSTRINGLENGTH)
    do i=1,len_trim(tempstring)
      if (tempstring(i:i) /= '#' .and. tempstring(i:i) /= '!') then
        input%buf(i:i) = tempstring(i:i)
      else
        exit
      endif
    enddo
  endif

contains

subroutine CatchLongKeywordError()

  if (.not.InputError(input)) return
  if (len_trim(input%buf) == 0) return

  option%io_buffer = 'Error reading keyword at beginning of input &
    &file string' // NL // NL // '"' // trim(input%buf) // '"' // &
    NL // NL // 'at line ' // StringWrite(input%line_number) // &
    '. Perhaps the keyword is greater than ' // StringWrite(num_chars) // &
    ' characters.'
  call PrintErrMsg(option)

end subroutine CatchLongKeywordError

end subroutine InputReadPflotranStringSlave

! ************************************************************************** !

subroutine InputReadCard(input, option, word, push_to_log)
  !
  ! Reads a keyword from the input deck, providing the option of logging
  !
  ! Author: Glenn Hammond
  ! Date: 09/20/19
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=MAXWORDLENGTH) :: word
  PetscBool, optional :: push_to_log

  if (InputError(input)) return

  call InputReadWord(input,option,word,PETSC_TRUE)

  if (present(push_to_log)) then
    call InputPushCard(input,word,option)
  else
    call InputPushCard(input,word,option)
  endif

end subroutine InputReadCard

! ************************************************************************** !

subroutine InputPrintKeywordLog(input,option,print_error)
  !
  ! Prints the current strings stored by keyword logging.
  !
  ! Author: Glenn Hammond
  ! Date: 09/25/19
  !
  type(input_type) :: input
  type(option_type) :: option
  PetscBool, optional :: print_error

  character(len=MAXWORDLENGTH) :: word

  if (option%keyword_logging) then
    if (present(print_error)) then
      if (print_error) then
        option%io_buffer = NL // &
                           ' ---------------------------------------&
                           &---------------------------------------' // &
                           NL
        call PrintMsg(option)
        option%io_buffer = &
           ' Helpful information for debugging the input deck:' // NL
        call PrintMsg(option)
        option%io_buffer = '     Filename : ' // trim(input%path) // &
                                                trim(input%filename)
        call PrintMsg(option)
        write(option%io_buffer,*) input%line_number
        option%io_buffer = '  Line Number : ' // trim(adjustl(option%io_buffer))
        call PrintMsg(option)
      endif
      word = '      Keyword :'
    else
      word = 'KEYWORD:'
    endif
    if (len_trim(option%keyword_log) > 0) then
      option%io_buffer = trim(word) // ' ' // trim(option%keyword_log) // &
                                       ',' // trim(option%keyword_buf)
    else
      option%io_buffer = trim(word) // ' ' // trim(option%keyword_buf)
    endif
    call PrintMsg(option)
  endif

  if (option%keyword_logging .and. present(print_error)) then
    if (print_error) then
      option%io_buffer = NL // &
                         ' ---------------------------------------&
                         &---------------------------------------'
      call PrintMsg(option)
   endif
 endif

end subroutine InputPrintKeywordLog

! ************************************************************************** !

subroutine InputPushCard(input,card,option)
  !
  ! Sometimes cards are optional and cannot be registered at the time of
  ! being read. This routines allows
  !
  ! Author: Glenn Hammond
  ! Date: 09/20/19
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=*) :: card

  character(len=MAXSTRINGLENGTH) :: string

  if (.not.option%keyword_logging) return
  if (InputError(input)) return

  string = ''
  if (len_trim(option%keyword_buf) > 0) then
    option%keyword_buf = trim(option%keyword_buf) // ',' // trim(card)
  else
    option%keyword_buf = trim(card)
  endif

  if (len_trim(option%keyword_buf) > 0 .and. &
      option%keyword_logging_screen_output) then
    call InputPrintKeywordLog(input,option)
  endif

  select case(card)
    case('SKIP')
      call InputPushBlock(input,option)
    case('NOSKIP')
      call InputPopBlock(input,option)
  end select

end subroutine InputPushCard

! ************************************************************************** !

subroutine InputPushBlock1(input,option)
  !
  ! Fill in
  !
  ! Author: Glenn Hammond
  ! Date: 09/23/19
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option

  call InputPushBlock(input,'',option)

end subroutine InputPushBlock1

! ************************************************************************** !

subroutine InputPushBlock2(input,block_name,option)
  !
  ! Fill in
  !
  ! Author: Glenn Hammond
  ! Date: 09/23/19
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=*) :: block_name

  character(len=MAXSTRINGLENGTH) :: string

  if (.not.option%keyword_logging) return

  if (len_trim(block_name) > 0) then
    string = block_name
  else
    string = option%keyword_buf
    option%keyword_buf = ''
  endif

  if (len_trim(option%keyword_log) > 0) then
    string = trim(option%keyword_log) // ',' // trim(string)
  endif
  option%keyword_log = trim(string)

  option%keyword_block_count = option%keyword_block_count + 1
  if (option%keyword_block_count > 0) then
    option%keyword_block_map(option%keyword_block_count) = &
      len_trim(option%keyword_log)
  endif

end subroutine InputPushBlock2

! ************************************************************************** !

subroutine InputPopBlock(input,option)
  !
  ! Fill in
  !
  ! Author: Glenn Hammond
  ! Date: 09/23/19
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option

  PetscInt :: i

  if (.not.option%keyword_logging) return

  option%keyword_buf = ''
  option%keyword_block_count = option%keyword_block_count - 1
  if (option%keyword_block_count > 0) then
    i = max(option%keyword_block_count,1)
    option%keyword_log = option%keyword_log(1:option%keyword_block_map(i))
  elseif (option%keyword_block_count < 0) then
    option%keyword_log = 'Negative Block Count'
  else
    option%keyword_log = ''
  endif

end subroutine InputPopBlock

! ************************************************************************** !

subroutine InputReadWord1(input, option, word, return_blank_error)
  !
  ! reads and removes a word (consecutive characters) from a string
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=MAXWORDLENGTH) :: word
  PetscBool :: return_blank_error

  if (InputError(input)) return

  call InputReadWord2(input%buf, word, return_blank_error, input%ierr)

end subroutine InputReadWord1

! ************************************************************************** !

subroutine InputReadWord2(string, word, return_blank_error, ierr)
  !
  ! reads and removes a word (consecutive characters) from a
  ! string
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08
  !

  implicit none

  character(len=*) :: string
  character(len=MAXWORDLENGTH) :: word
  PetscBool :: return_blank_error
  PetscErrorCode :: ierr

  call InputReadNChars2(string,word,MAXWORDLENGTH,return_blank_error,ierr)

end subroutine InputReadWord2

! ************************************************************************** !

subroutine InputReadCardDbaseCompatible(input, option, word)
  !
  ! reads a word and checks whether there is an entry in the Dbase with which
  ! to swap
  !
  ! Author: Glenn Hammond
  ! Date: 05/22/16
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=MAXWORDLENGTH) :: word

  PetscBool :: found

  if (InputError(input)) return

  found = PETSC_FALSE
  if (associated(dbase)) then
    call InputParseDbaseForWord(input%buf,word,found,option,input%ierr)
  endif

  if (.not.found) then
    call InputReadWord(input%buf,word,PETSC_TRUE,input%ierr)
  endif

  call InputPushCard(input,word,option)

end subroutine InputReadCardDbaseCompatible

! ************************************************************************** !

subroutine InputReadNChars1(input, option, chars, n, return_blank_error)
  !
  ! reads and removes a specified number of characters from a
  ! string
  !
  ! Author: Glenn Hammond
  ! Date: 11/02/00
  !

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  PetscBool :: return_blank_error ! Return an error for a blank line
                                   ! Therefore, a blank line is not acceptable.

  PetscInt :: n
  character(len=n) :: chars

  if (InputError(input)) return

  call InputReadNChars2(input%buf, chars, n, return_blank_error, input%ierr)

end subroutine InputReadNChars1

! ************************************************************************** !

subroutine InputReadNChars2(string, chars, num_chars, return_blank_error, ierr)
  !
  ! reads and removes a specified number of characters from a string
  !
  ! Author: Glenn Hammond
  ! Date: 11/02/09
  !
  implicit none

  character(len=*) :: string
  PetscInt :: num_chars
  character(len=num_chars) :: chars
  PetscBool :: return_blank_error ! Return an error for a blank line
                                  ! Therefore, a blank line is not acceptable.
  PetscErrorCode :: ierr

  PetscInt :: i
  PetscInt :: length
  PetscInt :: begins
  PetscInt :: ends
  character(len=1), parameter :: tab = achar(9), backslash = achar(92)

  if (InputError(ierr)) return

  ! Initialize character string to blank.
  chars(1:num_chars) = repeat(' ',num_chars)

  length = len_trim(string)
  if (length == 0) then
    if (return_blank_error) then
      ierr = INPUT_ERROR_DEFAULT
    else
      ierr = INPUT_ERROR_NONE
    endif
    return
  else
    ierr = INPUT_ERROR_NONE

    ! Remove leading blanks and tabs
    i=1
    do while((string(i:i) == ' ' .or. string(i:i) == ',' .or. &
             string(i:i) == tab) .and. i <= length)
      i=i+1
    enddo

    if (i > length) then
      if (return_blank_error) then
        ierr = INPUT_ERROR_KEYWORD_LENGTH
      else
        ierr = INPUT_ERROR_NONE
      endif
      return
    endif

    begins=i

    ! Count # of continuous characters (no blanks, commas, etc. in between)
    do while (string(i:i) /= ' ' .and. string(i:i) /= ',' .and. &
              string(i:i) /= tab  .and. &
              (i == begins .or. string(i:i) /= backslash))
      i=i+1
    enddo

    ends=i-1

    if (ends-begins+1 > num_chars) then ! string read is too large for 'chars'
      ierr = INPUT_ERROR_KEYWORD_LENGTH
      return
    endif

    ! Copy (ends-begins) characters to 'chars'
    chars = string(begins:ends)
    ! Remove chars from string
    string = string(ends+1:)

  endif

end subroutine InputReadNChars2

! ************************************************************************** !

subroutine InputReadFilename(input, option, filename)
  !
  ! Reads in a filename and prepends the input object path if applicable.
  !
  ! Author: Glenn Hammond
  ! Date: 01/31/18
  !
  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=MAXSTRINGLENGTH) :: filename

  PetscBool, parameter :: return_blank_error = PETSC_TRUE

  call InputReadNChars(input,option,filename, &
                       MAXSTRINGLENGTH,return_blank_error)

  ! only prepend a path if a path exist and the filename is not absolue (i.e.
  ! it does not start with a "/").
  if (len_trim(input%path) > 0 .and. index(filename,'/') /= 1) then
    filename = trim(input%path) // trim(filename)
  endif

end subroutine InputReadFilename

! ************************************************************************** !

subroutine InputReadQuotedWord(input, option, word, return_blank_error)
  !
  ! reads and removes a word from a string, that is
  ! delimited by "'".
  !
  ! Author: Glenn Hammond
  ! Date: 11/07/08
  !
  use String_module

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=*) :: word
  PetscBool :: return_blank_error ! Return an error for a blank line

  call StringReadQuotedWord(input%buf,word,return_blank_error,input%ierr)

end subroutine InputReadQuotedWord

! ************************************************************************** !

subroutine InputFindStringInFile1(input, option, string)
  !
  ! Rewinds file and finds the first occurrence of
  ! 'string'.  Note that the line must start with 'string'
  ! in order to match and that line is NOT returned
  !
  ! Author: Glenn Hammond
  ! Date: 03/07/07
  !

  use String_module

  implicit none

  type(input_type), pointer :: input
  type(option_type) :: option
  character(len=MAXSTRINGLENGTH) :: string

  call InputFindStringInFile2(input, option, string, PETSC_TRUE)

end subroutine InputFindStringInFile1

! ************************************************************************** !

subroutine InputFindStringInFile2(input, option, string, print_warning)
  !
  ! Rewinds file and finds the first occurrence of
  ! 'string'.  Note that the line must start with 'string'
  ! in order to match and that line is NOT returned
  ! This version of the overload can cope with a section that is not present
  !
  ! Author: Glenn Hammond
  ! Date: 03/07/07
  !

  use String_module

  implicit none

  type(input_type), pointer :: input
  type(option_type) :: option
  character(len=MAXSTRINGLENGTH) :: string
  PetscBool :: print_warning
  PetscBool :: found

  found = PETSC_FALSE

  call InputFindStringInFile3(input, option, string, print_warning,found)

end subroutine InputFindStringInFile2

! ************************************************************************** !

subroutine InputFindStringInFile3(input, option, string, print_warning,found)
  !
  ! Rewinds file and finds the first occurrence of
  ! 'string'.  Note that the line must start with 'string'
  ! in order to match and that line is NOT returned
  ! This routine differs from InputFindStringInFile2 only in that the
  ! result of the search is returned in the 'found' argument
  !
  ! Author: Glenn Hammond
  ! Date: 03/07/07
  !

  use String_module

  implicit none

  type(input_type), pointer :: input
  type(option_type) :: option
  character(len=MAXSTRINGLENGTH) :: string
  PetscBool :: print_warning
  PetscBool :: found

  character(len=MAXWORDLENGTH) :: word
  PetscInt :: length1, length2

  input%ierr = INPUT_ERROR_NONE
  found = PETSC_FALSE

  length1 = len_trim(string)

  do
    call InputReadPflotranString(input,option)
    if (InputError(input)) exit
    call InputReadWord(input,option,word,PETSC_TRUE)
    if (InputError(input)) exit
    length2 = len_trim(word)
    if (length1 == length2 .and. StringCompare(string,word,length1)) then
      found = PETSC_TRUE
      exit
    endif
  enddo

  ! if not found, rewind once and try again.  this approach avoids excessive
  ! reading if successive searches for strings are in descending order in
  ! the file.
  if (InputError(input)) then
    input%ierr = INPUT_ERROR_NONE
    call InputRewind(input)
    do
      call InputReadPflotranString(input,option)
      if (InputError(input)) exit
      call InputReadWord(input,option,word,PETSC_TRUE)
      if (InputError(input)) exit
      length2 = len_trim(word)
      if (length1 == length2 .and. StringCompare(string,word,length1)) then
        found = PETSC_TRUE
        exit
      endif
    enddo
  endif

  if (.not.found .and. print_warning) then
    option%io_buffer = 'Card (' // trim(string) // ') not found in input file.'
    call PrintWrnMsg(option)
    input%ierr = INPUT_ERROR_DEFAULT
  endif

end subroutine InputFindStringInFile3

! ************************************************************************** !

subroutine InputSkipToEND(input,option,string)
  !
  ! Skips to keyword END
  !
  ! Author: Glenn Hammond
  ! Date: 10/26/07
  !

  implicit none

  type(input_type), pointer :: input
  type(option_type) :: option
  character(len=*) :: string

  do
    call InputReadPflotranString(input,option)
    input%err_buf = 'End of file found before end of card ' // trim(string)
    call InputReadStringErrorMsg(input,option)
    if (InputCheckExit(input,option)) exit
  enddo

end subroutine InputSkipToEND

! ************************************************************************** !

function InputCheckExit(input,option)
  !
  ! Checks whether an end character (.,/,'END') has been found
  !
  ! Author: Glenn Hammond
  ! Date: 10/14/08
  !

  use String_module

  implicit none

  type(input_type) :: input
  type(option_type) :: option

  PetscInt :: i
  character(len=1) :: tab

  PetscBool :: InputCheckExit

  ! We must remove leading blanks and tabs. --RTM
  input%buf = adjustl(input%buf)
  tab = achar(9)
  i=1
  do while(input%buf(i:i) == tab .and. i < MAXSTRINGLENGTH)
    i=i+1
  enddo

  if (input%buf(i:i) == '/' .or. &
!geh: this fails when the keyword starts with END
!geh      StringCompare(input%buf(i:),'END',THREE_INTEGER)) then
      StringCompare(input%buf(i:),'END') .or. &
      ! to end a block, e.g. END_SUBSURFACE
      StringStartsWith(input%buf(i:),'END_')) then
    InputCheckExit = PETSC_TRUE
  else
    InputCheckExit = PETSC_FALSE
  endif

  option%keyword_buf = ''

end function InputCheckExit

! ************************************************************************** !

function InputError1(input)
  !
  ! Returns true if an error has occurred
  !
  ! Author: Glenn Hammond
  ! Date: 12/10/08
  !

  implicit none

  type(input_type) :: input

  PetscBool :: InputError1

  InputError1 = InputError2(input%ierr)

end function InputError1

! ************************************************************************** !

function InputError2(ierr)
  !
  ! Returns true if an error has occurred
  !
  ! Author: Glenn Hammond
  ! Date: 12/10/08
  !

  implicit none

  PetscErrorCode :: ierr

  PetscBool :: InputError2

  if (ierr == INPUT_ERROR_NONE) then
    InputError2 = PETSC_FALSE
  else
    InputError2 = PETSC_TRUE
  endif

end function InputError2

! ************************************************************************** !

subroutine InputGetCommandLineIndex(string,found,index)
  !
  ! Returns integer value associated with a command
  ! line argument
  !
  ! Author: Glenn Hammond
  ! Date: 02/05/09
  !

  use String_module
  use Option_module

  implicit none

  character(len=MAXSTRINGLENGTH) :: string
  PetscBool :: found
  PetscInt :: index

  PetscInt :: iarg, narg
  character(len=MAXSTRINGLENGTH) :: string2

  narg = getCommandLineArgumentCount()
  string = adjustl(string)
  found = PETSC_FALSE
  index = -1
  do iarg = 1, narg
    call getCommandLineArgument(iarg,string2)
    if (StringCompare(string,string2)) then
      found = PETSC_TRUE
      if (iarg+1 <= narg) then
        index = iarg+1
      endif
      exit
    endif
  enddo

end subroutine InputGetCommandLineIndex

! ************************************************************************** !

subroutine InputGetCommandLineInt(string,int_value,found,option)
  !
  ! Returns integer value associated with a command
  ! line argument
  !
  ! Author: Glenn Hammond
  ! Date: 02/05/09
  !

  use String_module
  use Option_module

  implicit none

  character(len=MAXSTRINGLENGTH) :: string
  type(option_type) :: option
  PetscBool :: found
  PetscInt :: int_value

  PetscInt :: index
  character(len=MAXSTRINGLENGTH) :: string2
  PetscErrorCode :: ierr

  ierr = INPUT_ERROR_NONE
  ! do not initialize int_value, as it may already have a value
  call InputGetCommandLineIndex(string,found,index)
  if (found) then
    if (index > 0) then
      call getCommandLineArgument(index,string2)
      call InputReadInt(string2,option,int_value,ierr)
    else
      ierr = INPUT_ERROR_DEFAULT
    endif
    if (InputError(ierr)) then
      option%io_buffer = 'Integer argument for command line argument "' // &
                         trim(adjustl(string)) // '" not found.'
      call PrintErrMsg(option)
    endif
  endif

end subroutine InputGetCommandLineInt

! ************************************************************************** !

subroutine InputGetCommandLineReal(string,double_value,found,option)
  !
  ! Returns real*8 value associated with a command
  ! line argument
  !
  ! Author: Glenn Hammond
  ! Date: 02/05/09
  !

  use String_module
  use Option_module

  implicit none

  character(len=MAXSTRINGLENGTH) :: string
  type(option_type) :: option
  PetscBool :: found
  PetscReal :: double_value

  PetscInt :: index
  character(len=MAXSTRINGLENGTH) :: string2
  PetscErrorCode :: ierr

  ierr = INPUT_ERROR_NONE
  ! do not initialize double_value, as it may already have a value
  call InputGetCommandLineIndex(string,found,index)
  if (found) then
    if (index > 0) then
      call getCommandLineArgument(index,string2)
      call InputReadDouble(string2,option,double_value,ierr)
    else
      ierr = INPUT_ERROR_DEFAULT
    endif
    if (InputError(ierr)) then
      option%io_buffer = 'Real argument for command line argument "' // &
                         trim(adjustl(string)) // '" not found.'
      call PrintErrMsg(option)
    endif
  endif

end subroutine InputGetCommandLineReal

! ************************************************************************** !

subroutine InputGetCommandLineString(string,string_value,found,option)
  !
  ! Returns a string associated with a command
  ! line argument
  !
  ! Author: Glenn Hammond
  ! Date: 02/05/09
  !

  use String_module
  use Option_module

  implicit none

  character(len=MAXSTRINGLENGTH) :: string
  type(option_type) :: option
  PetscBool :: found
  character(len=MAXSTRINGLENGTH) :: string_value

  PetscInt :: index
  character(len=MAXSTRINGLENGTH) :: string2
  PetscErrorCode :: ierr

  ierr = INPUT_ERROR_NONE
  ! do not initialize double_value, as it may already have a value
  call InputGetCommandLineIndex(string,found,index)
  if (found) then
    if (index > 0) then
      call getCommandLineArgument(index,string2)
      call InputReadNChars(string2,string_value,MAXSTRINGLENGTH, &
                           PETSC_TRUE,ierr)
      if (string_value(1:1) == '-') then
        ! no argument exists
        option%io_buffer = 'String argument (' // &
                            trim(adjustl(string_value)) // &
                            ') for command line argument "' // &
                            trim(adjustl(string)) // '" not recognized.'
        call PrintErrMsg(option)
      endif
    else
      ierr = INPUT_ERROR_DEFAULT
    endif
    if (InputError(ierr)) then
      option%io_buffer = 'String argument for command line argument "' // &
                         trim(adjustl(string)) // '" not found.'
      call PrintErrMsg(option)
    endif
  endif

end subroutine InputGetCommandLineString

! ************************************************************************** !

subroutine InputGetCommandLineTruth(string,truth_value,found,option)
  !
  ! Returns logical associated with a command
  ! line argument
  !
  ! Author: Glenn Hammond
  ! Date: 02/05/09
  !

  use String_module
  use Option_module

  implicit none

  character(len=MAXSTRINGLENGTH) :: string
  type(option_type) :: option
  PetscBool :: found
  PetscBool :: truth_value

  PetscInt :: index
  character(len=MAXSTRINGLENGTH) :: string2
  character(len=MAXWORDLENGTH) :: word
  PetscErrorCode :: ierr

  ierr = INPUT_ERROR_NONE
  ! do not initialize double_value, as it may already have a value
  call InputGetCommandLineIndex(string,found,index)
  if (found) then
    if (index > 0) then
      call getCommandLineArgument(index,string2)
      call InputReadWord(string2,word,PETSC_TRUE,ierr)
    else
      ! check if no argument exists, which is valid and means 'true'
      truth_value = PETSC_TRUE
      return
    endif
    if (word(1:1) == '-') then
      ! no argument exists, which is valid and means 'true'
      truth_value = PETSC_TRUE
      return
    endif
    call StringToUpper(word)
    select case(trim(word))
      case('YES','TRUE','1','ON')
        truth_value = PETSC_TRUE
      case('NO','FALSE','0','OFF')
        truth_value = PETSC_FALSE
      case default
        option%io_buffer = 'Truth argument for command line argument "' // &
                            trim(adjustl(string)) // '" not recognized.'
        call PrintErrMsg(option)
    end select
  endif

end subroutine InputGetCommandLineTruth

! ************************************************************************** !

function getCommandLineArgumentCount()
  !
  ! Returns the number of command line arguments
  !
  ! Author: Glenn Hammond
  ! Date: 02/05/10
  !

  implicit none

  PetscInt :: getCommandLineArgumentCount

  ! initialize to zero
  getCommandLineArgumentCount = 0

#if defined(PETSC_HAVE_FORTRAN_GET_COMMAND_ARGUMENT) || PETSC_VERSION_GE(3,21,0)
  getCommandLineArgumentCount = command_argument_count()
#elif defined(PETSC_HAVE_GETARG)
  getCommandLineArgumentCount = iargc()
#endif

end function getCommandLineArgumentCount

! ************************************************************************** !

subroutine getCommandLineArgument(i,arg)
  !
  ! Returns the ith command line argument
  !
  ! Author: Glenn Hammond
  ! Date: 02/05/10
  !

  implicit none

  PetscInt :: i
  character(len=*) :: arg

  integer*4 :: fortran_int

  fortran_int = i
#if defined(PETSC_HAVE_FORTRAN_GET_COMMAND_ARGUMENT) || PETSC_VERSION_GE(3,21,0)
  call get_command_argument(fortran_int,arg)
#elif defined(PETSC_HAVE_GETARG)
  call getarg(fortran_int,arg)
#endif

end subroutine getCommandLineArgument

! ************************************************************************** !

function InputGetLineCount(input,option)

  use String_module

  implicit none

  type(input_type), pointer :: input
  type(option_type) :: option

  PetscInt :: line_count
  PetscInt :: InputGetLineCount

  character(len=MAXSTRINGLENGTH) :: tempstring
  character(len=MAXWORDLENGTH) :: word

  call InputRewind(input)

  line_count = 0
  do
#if 1
    input%line_number = input%line_number + 1
    read(input%fid,'(a)',iostat=input%ierr) input%buf
    call StringAdjustl(input%buf)

    if (InputError(input)) then
      ! check to see if another file is on the stack
      if (InputPopExternalFile(input)) then
        cycle
      else
        exit
      endif
    endif

    tempstring = input%buf
    call InputReadWord(tempstring,word,PETSC_TRUE,input%ierr)
    call StringToUpper(word)

    if (word(1:13) == 'EXTERNAL_FILE') then
      ! have to strip the card 'EXTERNAL_FILE' from the buffer
      call InputReadWord(input,option,word,PETSC_TRUE)
      ! push a new input file to stack
      call InputPushExternalFile(input,option)
    endif
#else
    call InputReadPflotranString(input,option)
    if (InputError(input)) exit
#endif
    line_count = line_count + 1
  enddo

  InputGetLineCount = line_count

end function InputGetLineCount

! ************************************************************************** !

subroutine InputReadToBuffer(input, buffer, option)

  use String_module

  implicit none

  type(input_type), pointer :: input
  character(len=MAXSTRINGLENGTH) :: buffer(:)
  type(option_type) :: option

  character(len=MAXSTRINGLENGTH) :: tempstring
  character(len=MAXWORDLENGTH) :: word
  PetscInt :: line_count

  call InputRewind(input)

  line_count = 0
  do
#if 1
    input%line_number = input%line_number + 1
    read(input%fid,'(a)',iostat=input%ierr) input%buf
    call StringAdjustl(input%buf)

    if (InputError(input)) then
      ! check to see if another file is on the stack
      if (InputPopExternalFile(input)) then
        cycle
      else
        exit
      endif
    endif

    tempstring = input%buf
    call InputReadWord(tempstring,word,PETSC_TRUE,input%ierr)
    call StringToUpper(word)

    if (word(1:13) == 'EXTERNAL_FILE') then
      ! have to strip the card 'EXTERNAL_FILE' from the buffer
      call InputReadWord(input,option,word,PETSC_TRUE)
      ! push a new input file to stack
      call InputPushExternalFile(input,option)
    endif
#else
    call InputReadPflotranString(input,option)
    if (InputError(input)) exit
#endif
    line_count = line_count + 1
    buffer(line_count) = input%buf
  enddo

end subroutine InputReadToBuffer

! ************************************************************************** !

subroutine InputReadASCIIDbase(filename,option)
  !
  ! Read in an ASCII database
  !
  ! Author: Glenn Hammond
  ! Date: 08/19/14
  !
  use Option_module
  use String_module

  implicit none

  character(len=*) :: filename
  type(option_type) :: option

  character(len=MAXSTRINGLENGTH) :: string
  character(len=MAXWORDLENGTH) :: word
  character(len=MAXWORDLENGTH), allocatable :: words(:)
  character(len=MAXWORDLENGTH) :: object_name
  type(input_type), pointer :: input
  PetscInt :: icount
  PetscInt :: value_count
  PetscInt :: value_index
  PetscInt :: value_type
  PetscInt :: num_values_in_dataset
  PetscInt :: num_words, num_ints, num_reals

  input => InputCreate(IUNIT_TEMP,filename,option)

  icount = 0
  num_values_in_dataset = 0
  num_ints = 0
  num_reals = 0
  num_words = 0
  do
    call InputReadPflotranString(input,option)
    if (InputError(input)) exit
    call InputReadNChars(input,option,string,MAXSTRINGLENGTH,PETSC_FALSE)
    if (len_trim(string) > MAXWORDLENGTH) then
      call InputPrintKeywordLog(input,option,PETSC_TRUE)
      option%io_buffer = 'ASCII DBASE object names must be shorter than &
        &32 characters: ' // trim(string)
      call PrintErrMsg(option)
    endif
    word = trim(string)
    if (StringStartsWithAlpha(word)) then
      icount = icount + 1
      if (icount == 1) then
        string = input%buf
        do
          call InputReadWord(input,option,word,PETSC_TRUE)
          if (InputError(input)) exit
          num_values_in_dataset = num_values_in_dataset + 1
        enddo
        input%buf = string
      endif
      input%ierr = INPUT_ERROR_NONE
      call InputReadWord(input,option,word,PETSC_TRUE)
      call InputErrorMsg(input,option,'value','ASCII Dbase')
      select case(StringIntegerDoubleOrWord(word))
        case(STRING_IS_AN_INTEGER)
          num_ints = num_ints + 1
        case(STRING_IS_A_DOUBLE)
          num_reals = num_reals + 1
        case(STRING_IS_A_WORD)
          num_words = num_words + 1
      end select
    endif
  enddo

  value_index = 1
  if (option%id > 0) then
    if (option%id > num_values_in_dataset) then
      call InputPrintKeywordLog(input,option,PETSC_TRUE)
      write(word,*) num_values_in_dataset
        option%io_buffer = 'Data in DBASE_FILENAME "' // &
        trim(filename) // &
        '" is too small (' // trim(adjustl(word)) // &
        ') for number of realizations.'
      call PrintErrMsg(option)
    endif
    value_index = option%id
  endif
  allocate(words(num_values_in_dataset))
  words = ''

  call InputRewind(input)
  allocate(dbase)
  nullify(dbase%icard)
  nullify(dbase%rcard)
  nullify(dbase%ccard)
  nullify(dbase%ivalue)
  nullify(dbase%rvalue)
  nullify(dbase%cvalue)
  if (num_ints > 0) then
    allocate(dbase%icard(num_ints))
    dbase%icard = ''
    allocate(dbase%ivalue(num_ints))
    dbase%ivalue = UNINITIALIZED_INTEGER
  endif
  if (num_reals > 0) then
    allocate(dbase%rcard(num_reals))
    dbase%rcard = ''
    allocate(dbase%rvalue(num_reals))
    dbase%rvalue = UNINITIALIZED_DOUBLE
  endif
  if (num_words > 0) then
    allocate(dbase%ccard(num_words))
    dbase%ccard = ''
    allocate(dbase%cvalue(num_words))
    dbase%cvalue = '-999'
  endif
  num_ints = 0
  num_reals = 0
  num_words = 0
  do
    call InputReadPflotranString(input,option)
    if (InputError(input)) exit
    call InputReadWord(input,option,word,PETSC_FALSE)
    if (StringStartsWithAlpha(word)) then
      object_name = word
      words = ''
      value_count = 0
      do
        call InputReadWord(input,option,word,PETSC_TRUE)
        if (InputError(input)) exit
        value_count = value_count + 1
        if (value_count <= num_values_in_dataset) &
          words(value_count) = word
      enddo
      if (value_count /= num_values_in_dataset) then
        call InputPrintKeywordLog(input,option,PETSC_TRUE)
        write(word,*) value_count
        option%io_buffer = 'Data in DBASE_FILENAME "' // &
          trim(object_name) // &
          '" has an inconsistent number of values (' // &
          trim(adjustl(word)) // &
          ') for number of realizations ('
        write(word,*) num_values_in_dataset
        option%io_buffer = trim(option%io_buffer) // &
          trim(adjustl(word)) // ').'
        call PrintErrMsg(option)
      endif
      call StringToUpper(object_name)
      string = words(value_index)
      value_type = StringIntegerDoubleOrWord(string)
      string = words(value_index)
      select case(value_type)
        case(STRING_IS_AN_INTEGER)
          num_ints = num_ints + 1
          dbase%icard(num_ints) = adjustl(object_name)
          call InputReadInt(string,option,dbase%ivalue(num_ints),input%ierr)
          call InputErrorMsg(input,option,'ivalue','ASCII Dbase '//object_name)
        case(STRING_IS_A_DOUBLE)
          num_reals = num_reals + 1
          dbase%rcard(num_reals) = adjustl(object_name)
          call InputReadDouble(string,option,dbase%rvalue(num_reals),input%ierr)
          call InputErrorMsg(input,option,'rvalue','ASCII Dbase '//object_name)
        case(STRING_IS_A_WORD)
          num_words = num_words + 1
          dbase%ccard(num_words) = adjustl(object_name)
          dbase%cvalue(num_words) = words(value_index)
      end select
    endif
  enddo
  deallocate(words)

  call InputDestroy(input)

end subroutine InputReadASCIIDbase

! ************************************************************************** !

subroutine InputParseDbaseForInt(buffer,value,found,option,ierr)
  !
  ! Parses database for an integer value
  !
  ! Author: Glenn Hammond
  ! Date: 08/19/14
  !
  use String_module

  implicit none

  character(len=MAXSTRINGLENGTH) :: buffer
  PetscInt :: value
  PetscBool :: found
  type(option_type) :: option
  PetscErrorCode :: ierr

  character(len=MAXSTRINGLENGTH) :: buffer_save
  character(len=MAXWORDLENGTH) :: word
  character(len=MAXWORDLENGTH) :: dbase_keyword = 'DBASE_VALUE'

  buffer_save = buffer
  found = PETSC_FALSE
  call InputReadWord(buffer,word,PETSC_TRUE,ierr)
  if (StringCompareIgnoreCase(word,dbase_keyword)) then
    call InputReadWord(buffer,word,PETSC_TRUE,ierr)
    call DbaseLookupInt(word,value,option,ierr)
    if (.not.InputError(ierr)) then
      found = PETSC_TRUE
    endif
  else
    buffer = buffer_save
  endif

end subroutine InputParseDbaseForInt

! ************************************************************************** !

subroutine InputParseDbaseForDouble(buffer,value,found,option,ierr)
  !
  ! Parses database for an double precision value
  !
  ! Author: Glenn Hammond
  ! Date: 08/19/14
  !
  use String_module

  implicit none

  character(len=MAXSTRINGLENGTH) :: buffer
  PetscReal :: value
  PetscBool :: found
  type(option_type) :: option
  PetscErrorCode :: ierr

  character(len=MAXSTRINGLENGTH) :: buffer_save
  character(len=MAXWORDLENGTH) :: word
  character(len=MAXWORDLENGTH) :: dbase_keyword = 'DBASE_VALUE'

  buffer_save = buffer
  found = PETSC_FALSE
  call InputReadWord(buffer,word,PETSC_TRUE,ierr)
  if (StringCompareIgnoreCase(word,dbase_keyword)) then
    call InputReadWord(buffer,word,PETSC_TRUE,ierr)
    call DbaseLookupDouble(word,value,option,ierr)
    if (.not.InputError(ierr)) then
      found = PETSC_TRUE
    endif
  else
    buffer = buffer_save
  endif

end subroutine InputParseDbaseForDouble

! ************************************************************************** !

subroutine InputParseDbaseForWord(buffer,value,found,option,ierr)
  !
  ! Parses database for a word
  !
  ! Author: Glenn Hammond
  ! Date: 05/22/16
  !
  use String_module

  implicit none

  character(len=MAXSTRINGLENGTH) :: buffer
  character(len=MAXWORDLENGTH) :: value
  PetscBool :: found
  type(option_type) :: option
  PetscErrorCode :: ierr

  character(len=MAXSTRINGLENGTH) :: buffer_save
  character(len=MAXWORDLENGTH) :: word
  character(len=MAXWORDLENGTH) :: dbase_keyword = 'DBASE_VALUE'

  buffer_save = buffer
  found = PETSC_FALSE
  call InputReadWord(buffer,word,PETSC_TRUE,ierr)
  if (StringCompareIgnoreCase(word,dbase_keyword)) then
    call InputReadWord(buffer,word,PETSC_TRUE,ierr)
    call DbaseLookupWord(word,value,option,ierr)
    if (.not.InputError(ierr)) then
      found = PETSC_TRUE
    endif
  else
    buffer = buffer_save
  endif

end subroutine InputParseDbaseForWord

! ************************************************************************** !

subroutine DbaseLookupInt(keyword,value,option,ierr)
  !
  ! Looks up double precision value in database
  !
  ! Author: Glenn Hammond
  ! Date: 08/19/14
  !
  use String_module

  implicit none

  character(len=MAXWORDLENGTH) :: keyword
  PetscInt :: value
  type(option_type) :: option
  PetscErrorCode :: ierr

  PetscInt :: i
  PetscBool :: found

  ierr = INPUT_ERROR_NONE

  call StringToUpper(keyword)

  found = PETSC_FALSE
  if (associated(dbase%icard)) then
    do i = 1, size(dbase%icard)
      if (StringCompare(keyword,dbase%icard(i))) then
        found = PETSC_TRUE
        value = dbase%ivalue(i)
        exit
      endif
    enddo
  endif

  if (.not.found) then
    option%io_buffer = NL // 'ERROR: DBASE keyword "' // &
      trim(keyword) // &
      '" (for reading an "integer value") is not found in the database.'
    call PrintMsg(option)
    ierr = INPUT_ERROR_DEFAULT
  endif

end subroutine DbaseLookupInt

! ************************************************************************** !

subroutine DbaseLookupDouble(keyword,value,option,ierr)
  !
  ! Looks up double precision value in database
  !
  ! Author: Glenn Hammond
  ! Date: 08/19/14
  !
  use String_module

  implicit none

  character(len=MAXWORDLENGTH) :: keyword
  PetscReal :: value
  type(option_type) :: option
  PetscErrorCode :: ierr

  PetscInt :: i
  PetscBool :: found

  ierr = INPUT_ERROR_NONE

  call StringToUpper(keyword)

  found = PETSC_FALSE
  if (associated(dbase%rcard)) then
    do i = 1, size(dbase%rcard)
      if (StringCompare(keyword,dbase%rcard(i))) then
        found = PETSC_TRUE
        value = dbase%rvalue(i)
        exit
      endif
    enddo
  endif

  if (.not.found) then
    option%io_buffer = NL // 'ERROR: DBASE keyword "' // &
      trim(keyword) // &
      '" (for reading a "floating point value") is not found in the database.'
    call PrintMsg(option)
    ierr = INPUT_ERROR_DEFAULT
  endif

end subroutine DbaseLookupDouble

! ************************************************************************** !

subroutine DbaseLookupWord(keyword,value,option,ierr)
  !
  ! Looks up double precision value in database
  !
  ! Author: Glenn Hammond
  ! Date: 08/19/14
  !
  use String_module

  implicit none

  character(len=MAXWORDLENGTH) :: keyword
  character(len=MAXWORDLENGTH) :: value
  type(option_type) :: option
  PetscErrorCode :: ierr

  PetscInt :: i
  PetscBool :: found

  ierr = INPUT_ERROR_NONE

  call StringToUpper(keyword)

  found = PETSC_FALSE
  if (associated(dbase%ccard)) then
    do i = 1, size(dbase%ccard)
      if (StringCompare(keyword,dbase%ccard(i))) then
        found = PETSC_TRUE
        value = dbase%cvalue(i)
        exit
      endif
    enddo
  endif

  if (.not.found) then
    option%io_buffer = NL // 'ERROR: DBASE keyword "' // &
      trim(keyword) // &
      '" (for reading a "string value") is not found in the database.'
    call PrintMsg(option)
    ierr = INPUT_ERROR_DEFAULT
  endif

end subroutine DbaseLookupWord

! ************************************************************************** !

subroutine InputKeywordUnrecognized1(input,keyword,string,option)
  !
  ! Reports an unrecognized keyword in input deck
  !
  ! Author: Glenn Hammond
  ! Date: 08/19/14
  !
  use Option_module

  implicit none

  type(input_type) :: input
  character(len=*) :: keyword
  character(len=*) :: string
  type(option_type) :: option

  character(len=1) :: null_string

  null_string = ''
  call InputKeywordUnrecognized2(input,keyword,string,null_string,option)

end subroutine InputKeywordUnrecognized1

! ************************************************************************** !

subroutine InputKeywordUnrecognized2(input,keyword,string,string2,option)
  !
  ! Reports an unrecognized keyword in input deck
  !
  ! Author: Glenn Hammond
  ! Date: 08/19/14
  !
  use Option_module

  implicit none

  type(input_type) :: input
  character(len=*) :: keyword
  character(len=*) :: string
  character(len=*) :: string2
  type(option_type) :: option

  call InputPrintKeywordLog(input,option,PETSC_TRUE)
  option%io_buffer = 'Keyword "' // &
                     trim(keyword) // &
                     '" not recognized in ' // &
                     trim(string)
  if (len_trim(string2) > 0) then
    option%io_buffer = trim(option%io_buffer) // ' ' // &
                     trim(string2)
  endif
  option%io_buffer = trim(option%io_buffer) // '.'
  call PrintErrMsg(option)

end subroutine InputKeywordUnrecognized2

! ************************************************************************** !

subroutine InputCheckSupported1(input,option,keyword,error_string,pm_class)
  !
  ! Reports an unsupported keyword for set of process models in input deck
  !
  ! Author: Glenn Hammond
  ! Date: 03/04/24

  use Option_module

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=*) :: keyword
  character(len=*) :: error_string
  PetscInt :: pm_class

  call InputCheckSupported(input,option,keyword,error_string, &
                           pm_class,[pm_class])

end subroutine InputCheckSupported1

! ************************************************************************** !

subroutine InputCheckSupported2(input,option,keyword,error_string,pm_classes)
  !
  ! Reports an unsupported keyword for set of process models in input deck
  !
  ! Author: Glenn Hammond
  ! Date: 03/04/24

  use Option_module

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=*) :: keyword
  character(len=*) :: error_string
  PetscInt :: pm_classes(:)

  PetscInt :: i

  do i = 1, size(pm_classes)
    call InputCheckSupported(input,option,keyword,error_string, &
                             pm_classes(i),pm_classes)
  enddo

end subroutine InputCheckSupported2

! ************************************************************************** !

subroutine InputCheckSupported3(input,option,keyword,error_string, &
                                pm_class,required_classes)
  !
  ! Reports an unsupported keyword for set of process models in input deck
  !
  ! Author: Glenn Hammond
  ! Date: 03/04/24

  use Option_module

  implicit none

  type(input_type) :: input
  type(option_type) :: option
  character(len=*) :: keyword
  character(len=*) :: error_string
  PetscInt :: pm_class
  PetscInt :: required_classes(:)

  if (OptionCheckSupportedClass(pm_class,option)) return

  call InputPrintKeywordLog(input,option,PETSC_TRUE)
  option%io_buffer = 'Keyword "' // trim(keyword) // '" in ' // &
    trim(error_string) // &
    ' not supported for the current combination of process models.' // &
    NL // '  Process models in simulation: ' // &
    OptionGetEmployedClassesString(option) // &
    NL // '  Required process model(s): ' // &
    OptionListClassesString(required_classes,option)
  call PrintErrMsg(option)

end subroutine InputCheckSupported3

! ************************************************************************** !

subroutine InputKeywordDeprecated(old_keyword,new_keyword,option)
  !
  ! Prints an error message indicated that an keyword has been deprecated
  !
  ! Author: Glenn Hammond
  ! Date: 04/16/20
  !
  use Option_module

  implicit none

  character(len=*) :: old_keyword
  character(len=*) :: new_keyword
  type(option_type) :: option

  option%io_buffer = 'Keyword "' // trim(adjustl(old_keyword)) // &
    '" has been deprecated. Please use "' // trim(adjustl(new_keyword)) // &
    '" instead.'
  call PrintErrMsg(option)

end subroutine InputKeywordDeprecated

! ************************************************************************** !

subroutine InputCheckMandatoryUnits(input,option)
  !
  ! Looks up double precision value in database
  !
  ! Author: Glenn Hammond
  ! Date: 08/19/14
  !
  use Option_module

  implicit none

  type(input_type) :: input
  type(option_type) :: option

  if (input%force_units) then
    call InputPrintKeywordLog(input,option,PETSC_TRUE)
    option%io_buffer = 'Missing units'
    if (len_trim(input%err_buf) > 1) then
      option%io_buffer = trim(option%io_buffer) // ' in ' // &
                         trim(input%err_buf) // ',' // &
                         trim(input%err_buf2) // '.'
    endif
    call PrintErrMsg(option)
  endif

end subroutine InputCheckMandatoryUnits

! ************************************************************************** !

subroutine InputReadAndConvertUnits(input,double_value,internal_units, &
                                    keyword_string,option)
  !
  ! Reads units if they exist and returns the units conversion factor.
  !
  ! Author: Glenn Hammond
  ! Date: 07/26/16
  !
  use Option_module
  use Units_module

  implicit none

  type(input_type) :: input
  PetscReal :: double_value
  character(len=*) :: internal_units
  character(len=*) :: keyword_string
  type(option_type) :: option

  character(len=MAXWORDLENGTH) :: units
  character(len=MAXWORDLENGTH) :: internal_units_word
  character(len=MAXSTRINGLENGTH) :: string

  call InputReadWord(input,option,units,PETSC_TRUE)
  if (.not.InputError(input)) then
    if (len_trim(internal_units) < 1) then
      call InputPrintKeywordLog(input,option,PETSC_TRUE)
      option%io_buffer = 'No internal units provided in &
                         &InputReadAndConvertUnits()'
      call PrintErrMsg(option)
    endif
    internal_units_word = trim(internal_units)
    double_value = double_value * &
                   UnitsConvertToInternal(units,internal_units_word, &
                                          keyword_string,option)
  else
    string = trim(keyword_string) // ' units'
    call InputDefaultMsg(input,option,string)
  endif

end subroutine InputReadAndConvertUnits

! ************************************************************************** !

subroutine InputPushExternalFile(input,option)
  !
  ! Looks up double precision value in database
  !
  ! Author: Glenn Hammond
  ! Date: 08/19/14
  !
  use Option_module

  implicit none

  type(input_type), pointer :: input
  type(option_type) :: option

  character(len=MAXSTRINGLENGTH) :: string
  type(input_type), pointer :: input_child

  call InputReadFilename(input,option,string)
  call InputErrorMsg(input,option,'filename','EXTERNAL_FILE')
  input_child => InputCreate(input,string,option)
  input_child%parent => input
  input => input_child

end subroutine InputPushExternalFile

! ************************************************************************** !

function InputPopExternalFile(input)
  !
  ! Looks up double precision value in database
  !
  ! Author: Glenn Hammond
  ! Date: 08/19/14
  !

  implicit none

  type(input_type), pointer :: input

  PetscBool :: InputPopExternalFile
  type(input_type), pointer :: input_parent

  InputPopExternalFile = PETSC_FALSE
  if (associated(input%parent)) then
    input_parent => input%parent
    call InputDestroySingleLevel(input)
    input => input_parent
    nullify(input_parent)
    InputPopExternalFile = PETSC_TRUE
  endif

end function InputPopExternalFile

! ************************************************************************** !

subroutine InputCloseNestedFiles(input)
  !
  ! Closes all files opened through the EXTERNAL_FILE cards.
  !
  ! Author: Glenn Hammond
  ! Date: 09/27/17
  !
  implicit none

  type(input_type), pointer :: input

  ! With the EXTERNAL_FILE card, files may be nested.  The following loop
  ! un-nests the input deck back to the main input file.
  do
    if (InputPopExternalFile(input)) then
      cycle
    else
      exit
    endif
  enddo
  rewind(input%fid)

end subroutine InputCloseNestedFiles

! ************************************************************************** !

subroutine InputRewind(input)
  !
  ! Rewinds the input deck taking into count the EXTERNAL_FILE card
  ! capability.
  !
  ! Author: Glenn Hammond
  ! Date: 09/27/17
  !
  implicit none

  type(input_type), pointer :: input

  input%line_number = 0
  call InputCloseNestedFiles(input)
  rewind(input%fid)

end subroutine InputRewind

! ************************************************************************** !

subroutine InputCheckKeywordBlockCount(option)
  !
  ! Checks to ensure that the number of entered blocks due to nesting of
  ! keyword blocks in the input file is zero at the end of reading.
  !
  ! Author: Glenn Hammond
  ! Date: 02/17/21
  !
  use Option_module

  implicit none

  type(option_type) :: option

  if (option%keyword_block_count /= 0) then
    write(option%io_buffer,*) option%keyword_block_count
    option%io_buffer = 'Non-zero input block count (' // &
      trim(adjustl(option%io_buffer)) // '). Please email this message &
      &and your input deck to pflotran-dev@googlegroups.com'
    call PrintErrMsg(option)
  endif

end subroutine InputCheckKeywordBlockCount

! ************************************************************************** !

function InputCountWordsInBuffer(input,option)
  !
  ! Returns the number of words in the input buffer (e.g., for counting the
  ! number of integers on a line in the input file).
  !
  ! Author: Glenn Hammond
  ! Date: 05/19/23
  !
  use Option_module

  implicit none

  type(input_type), pointer :: input
  type(option_type) :: option

  PetscInt :: InputCountWordsInBuffer
  character(len=MAXWORDLENGTH) :: word

  InputCountWordsInBuffer = 0
  do
    call InputReadWord(input,option,word,PETSC_TRUE)
    if (InputError(input)) exit
    InputCountWordsInBuffer = InputCountWordsInBuffer + 1
  enddo

end function InputCountWordsInBuffer

! ************************************************************************** !

subroutine InputDbaseDestroy()
  !
  ! Destroys the input dbase and members
  !
  ! Author: Glenn Hammond
  ! Date: 08/20/14
  !

  implicit none

  if (associated(dbase)) then
    ! due to circular dependencies, cannot use Utilty_module::DeallocateArray
    if (associated(dbase%icard)) deallocate(dbase%icard)
    nullify(dbase%icard)
    if (associated(dbase%rcard)) deallocate(dbase%rcard)
    nullify(dbase%rcard)
    if (associated(dbase%ccard)) deallocate(dbase%ccard)
    nullify(dbase%ccard)
    if (associated(dbase%ivalue)) deallocate(dbase%ivalue)
    nullify(dbase%ivalue)
    if (associated(dbase%rvalue)) deallocate(dbase%rvalue)
    nullify(dbase%rvalue)
    if (associated(dbase%cvalue)) deallocate(dbase%cvalue)
    nullify(dbase%cvalue)
    deallocate(dbase)
    nullify(dbase)
  endif

end subroutine InputDbaseDestroy

! ************************************************************************** !

subroutine InputDestroySingleLevel(input)
  !
  ! Deallocates a single input object within a linked list of nested
  ! input objects.
  !
  ! Author: Glenn Hammond
  ! Date: 09/27/17
  !

  implicit none

  type(input_type), pointer :: input

  if (input%fid /= 0) close(input%fid)
  input%fid = 0
  input%line_number = 0
  deallocate(input)
  nullify(input)

end subroutine InputDestroySingleLevel

! ************************************************************************** !

recursive subroutine InputDestroy(input)
  !
  ! Deallocates all input objects, included those in a nestd linked list
  ! created due to the EXTERNAL_FILE capability.
  !
  ! Author: Glenn Hammond
  ! Date: 11/10/08, 09/27/17
  !

  implicit none

  type(input_type), pointer :: input

  ! destroy any parents first
  if (associated(input%parent)) then
    call InputDestroy(input%parent)
  endif

  if (input%fid /= 0) close(input%fid)
  input%fid = 0
  input%line_number = 0
  deallocate(input)
  nullify(input)

end subroutine InputDestroy

end module Input_Aux_module
