module Init_Subsurface_module

#include "petsc/finclude/petscvec.h"
  use petscvec
  !geh: there can be no dependencies on simulation object in this file
  use PFLOTRAN_Constants_module

  implicit none

  private


  public :: InitSubsurfAssignMatIDsToRegns, &
            InitSubsurfAssignMatProperties, &
            SubsurfInitMaterialProperties, &
            SubsurfAssignVolsToMatAuxVars, &
            SubsurfSandboxesSetup, &
            SubsurfReadDatasetToVecWithMask, &
            InitSubsurfProcessOutputVars

contains

! ************************************************************************** !

subroutine SubsurfInitMaterialProperties(realization)
  !
  ! Initializes material property data structres and assign them to the domain.
  !
  ! Author: Glenn Hammond
  ! Date: 10/07/14
  !
  use Realization_Subsurface_class

  implicit none

  class(realization_subsurface_type) :: realization

  call SubsurfAllocMatPropDataStructs(realization)
  call InitSubsurfAssignMatIDsToRegns(realization)
  call InitSubsurfAssignMatProperties(realization)

end subroutine SubsurfInitMaterialProperties

! ************************************************************************** !

subroutine SubsurfAllocMatPropDataStructs(realization)
  !
  ! Allocates data structures associated with storage of material properties
  !
  ! Author: Glenn Hammond
  ! Date: 10/07/14
  !
  use Realization_Subsurface_class
  use Material_module
  use Option_module
  use Discretization_module
  use Grid_module
  use Patch_module
  use Material_Aux_module
  use Fracture_module, only : FractureAuxVarInit

  implicit none

  class(realization_subsurface_type) :: realization

  PetscInt :: ghosted_id

  type(option_type), pointer :: option
  type(grid_type), pointer :: grid
  type(patch_type), pointer :: patch
  type(material_auxvar_type), pointer :: material_auxvars(:)

  option => realization%option
  patch => realization%patch
  grid => patch%grid

  ! initialize material auxiliary indices.  this does not have to be done
  ! for each patch, just once.
  call MaterialInitAuxIndices(realization%patch%material_property_array,option)

  ! create mappinging
  if (.not.associated(patch%imat)) then
    allocate(patch%imat(grid%ngmax))
    ! initialize to "unset"
    patch%imat = UNINITIALIZED_INTEGER
    select case(option%iflowmode)
      case(NULL_MODE,PNF_MODE)
      case(RICHARDS_MODE,WF_MODE,ZFLOW_MODE)
        allocate(patch%cc_id(grid%ngmax))
        patch%cc_id = UNINITIALIZED_INTEGER
      case default
        allocate(patch%cc_id(grid%ngmax))
        patch%cc_id = UNINITIALIZED_INTEGER
        allocate(patch%cct_id(grid%ngmax))
        patch%cct_id = UNINITIALIZED_INTEGER
    end select
  endif

  patch%aux%Material => MaterialAuxCreate(option)
  allocate(material_auxvars(grid%ngmax))
  do ghosted_id = 1, grid%ngmax
    call MaterialAuxVarInit(material_auxvars(ghosted_id),option)
    if (option%flow%fracture_on) then
      call FractureAuxVarInit(material_auxvars(ghosted_id))
    endif
  enddo
  patch%aux%Material%num_aux = grid%ngmax
  patch%aux%Material%auxvars => material_auxvars
  nullify(material_auxvars)

  ! Create Vec that holds compressibility
  if (soil_compressibility_index > 0) then
    call DiscretizationDuplicateVector(realization%discretization, &
                                       realization%field%work, &
                                       realization%field%compressibility0)
  endif

end subroutine SubsurfAllocMatPropDataStructs

! ************************************************************************** !

subroutine InitSubsurfAssignMatIDsToRegns(realization)
  !
  ! Assigns material properties to associated regions in the model
  !
  ! Author: Glenn Hammond
  ! Date: 11/02/07
  !
  use Dataset_Base_class
  use Dataset_Gridded_HDF5_class
  use Dataset_module
  use Realization_Subsurface_class
  use Strata_module
  use Region_module
  use Option_module
  use Grid_module
  use Patch_module
  use Field_module
  use Material_module
  use Material_Aux_module

  implicit none

  class(realization_subsurface_type) :: realization

  PetscInt :: icell, local_id, ghosted_id
  PetscInt :: istart, iend

  type(option_type), pointer :: option
  type(grid_type), pointer :: grid
  type(strata_type), pointer :: strata
  type(patch_type), pointer :: patch

  character(len=MAXSTRINGLENGTH) :: string
  type(material_property_type), pointer :: material_property
  type(region_type), pointer :: region
  type(material_auxvar_type), pointer :: material_auxvars(:)
  class(dataset_gridded_hdf5_type), pointer :: gridded_dataset
  PetscBool :: set_material_id
  PetscReal :: elevation

  nullify(gridded_dataset)
  option => realization%option
  patch => realization%patch
  grid => patch%grid

  ! set material ids to uninitialized
  material_auxvars => patch%aux%Material%auxvars
  patch%imat = UNINITIALIZED_INTEGER
  strata => patch%strata_list%first
  do
    if (.not.associated(strata)) exit
    ! if not within time period specified, skip the strata.
    ! use a one second tolerance on the start time and end time
    if (StrataWithinTimePeriod(strata,option%time) .and. &
        (.not. strata%well)) then
      if (len_trim(strata%dataset_name) > 0) then
        string = 'STRATA DATASET block'
        gridded_dataset => &
          DatasetGriddedHDF5Cast( &
            DatasetBaseGetPointer(realization%datasets, &
                                  strata%dataset_name, &
                                  string,option))
        string = 'Dataset "' // trim(strata%dataset_name) // &
          '" referenced in a STRATA block must be'
        if (.not.associated(gridded_dataset)) then
          option%io_buffer = trim(string) // ' a gridded dataset.'
          call PrintErrMsg(option)
        endif
        call DatasetGriddedHDF5Load(gridded_dataset,option)
        if (gridded_dataset%data_dim /= DIM_XY) then
          option%io_buffer = trim(string) // ' 2D in XY.'
          call PrintErrMsg(option)
        endif
      endif
      ! Read in cell by cell material ids if they exist
      if (.not.associated(strata%region) .and. &
          len_trim(strata%dataset_name) == 0 .and. &
          strata%active) then
        call SubsurfReadMaterialIDsFromFile(realization, &
                                            strata%realization_dependent, &
                                            strata%material_property_filename)
      ! Otherwise, set based on region
      else
        region => strata%region
        material_property => strata%material_property
        if (associated(region)) then
          istart = 1
          iend = region%num_cells
        else
          istart = 1
          iend = grid%nlmax
        endif
        do icell=istart, iend
          if (associated(region)) then
            local_id = region%cell_ids(icell)
          else
            local_id = icell
          endif
          ghosted_id = grid%nL2G(local_id)
          set_material_id = PETSC_TRUE
          if (associated(gridded_dataset)) then
            call DatasetGriddedHDF5InterpolateReal(gridded_dataset, &
                          grid%x(ghosted_id),grid%y(ghosted_id), &
                          UNINITIALIZED_DOUBLE,elevation,option)
            if (strata%dataset_mat_id_direction == &
                SET_MATERIAL_IDS_BELOW_SURFACE) then
              ! sets the material id if the cell center is below elevation
              if (grid%z(ghosted_id) > elevation) then
                set_material_id = PETSC_FALSE
              endif
            else ! SET_MATERIAL_IDS_ABOVE_SURFACE
              ! sets the material id if the cell center is above elevation
              if (grid%z(ghosted_id) < elevation) then
                set_material_id = PETSC_FALSE
              endif
            endif
          endif
          if (set_material_id) then
            if (strata%active) then
              patch%imat(ghosted_id) = material_property%internal_id
            else
              ! if not active, set material id to zero
              patch%imat(ghosted_id) = 0
            endif
          endif
        enddo
      endif
      if (associated(gridded_dataset)) then
        ! do not destroy; just strip as the dataset may be needed later
        call DatasetGriddedHDF5Strip(gridded_dataset)
        ! setting data_dim to NULL forces the dataset information to be
        ! read again on the next use, which is necessary
        gridded_dataset%data_dim = DIM_NULL
        nullify(gridded_dataset)
      endif
    endif
    strata => strata%next
  enddo

  ! ensure that ghosted values for material ids are up to date
  call RealLocalToLocalWithArray(realization,MATERIAL_ID_ARRAY)

  ! set material ids in material auxvar.  this must come after the update of
  ! ghost values.
  ! set material ids to uninitialized
  material_auxvars => patch%aux%Material%auxvars
  do ghosted_id = 1, grid%ngmax
    material_auxvars(ghosted_id)%id = patch%imat(ghosted_id)
  enddo

end subroutine InitSubsurfAssignMatIDsToRegns

! ************************************************************************** !

subroutine InitSubsurfAssignMatProperties(realization)
  !
  ! Assigns material properties based on material ids
  !
  ! Author: Glenn Hammond
  ! Date: 10/07/14
  !
  use Realization_Subsurface_class
  use Grid_module
  use Discretization_module
  use Field_module
  use Patch_module
  use Material_Aux_module
  use Material_module
  use Option_module
  use WIPP_module
  use Creep_Closure_module
  use Fracture_module
  use Geomechanics_Subsurface_Properties_module
  use String_module
  use Variables_module, only : PERMEABILITY_X, PERMEABILITY_Y, &
                               PERMEABILITY_Z, PERMEABILITY_XY, &
                               PERMEABILITY_YZ, PERMEABILITY_XZ, &
                               TORTUOSITY, POROSITY, SOIL_COMPRESSIBILITY, &
                               EPSILON, MATERIAL_ELECTRICAL_CONDUCTIVITY, &
                               ARCHIE_CEMENTATION_EXPONENT, &
                               ARCHIE_SATURATION_EXPONENT, &
                               ARCHIE_TORTUOSITY_CONSTANT, &
                               SURFACE_ELECTRICAL_CONDUCTIVITY, &
                               WAXMAN_SMITS_CLAY_CONDUCTIVITY, &
                               HALF_MATRIX_WIDTH, NUMBER_SECONDARY_CELLS, &
                               TORTUOSITY_Y,TORTUOSITY_Z

  use HDF5_module
  use Grid_Eclipse_module
  use Utility_module, only : DeallocateArray

  implicit none

  class(realization_subsurface_type) :: realization

  PetscReal, pointer :: por0_p(:)
  PetscReal, pointer :: tor0_p(:)
  PetscReal, pointer :: perm_xx_p(:)
  PetscReal, pointer :: perm_yy_p(:)
  PetscReal, pointer :: perm_zz_p(:)
  PetscReal, pointer :: perm_xy_p(:)
  PetscReal, pointer :: perm_xz_p(:)
  PetscReal, pointer :: perm_yz_p(:)
  PetscReal, pointer :: vec_p(:)
  PetscReal, pointer :: compress_p(:)
  PetscReal, pointer :: tort_yy_p(:)
  PetscReal, pointer :: tort_zz_p(:)

  character(len=MAXSTRINGLENGTH) :: string
  type(material_property_type), pointer :: material_property
  type(material_property_type), pointer :: null_material_property
  type(option_type), pointer :: option
  type(discretization_type), pointer :: discretization
  type(grid_type), pointer :: grid
  type(field_type), pointer :: field
  type(patch_type), pointer :: patch
  type(material_auxvar_type), pointer :: material_auxvars(:)
  PetscInt :: local_id, ghosted_id, material_id, i
  PetscInt :: natural_id, rprnc
  PetscReal :: poro, permx, permy, permz
  PetscInt, pointer :: inatsend(:)
  Vec :: tort_yy
  Vec :: tort_zz
  PetscErrorCode :: ierr

#ifdef GEOMECH_DEBUG
  PetscViewer :: viewer
#endif

  option => realization%option
  discretization => realization%discretization
  field => realization%field
  patch => realization%patch
  grid => patch%grid

  tort_yy = PETSC_NULL_VEC
  tort_zz = PETSC_NULL_VEC

  ! set cell by cell material properties
  ! create null material property for inactive cells
  null_material_property => MaterialPropertyCreate(option)
  if (option%nflowdof > 0) then
    call VecGetArrayF90(field%perm0_xx,perm_xx_p,ierr);CHKERRQ(ierr)
    call VecGetArrayF90(field%perm0_yy,perm_yy_p,ierr);CHKERRQ(ierr)
    call VecGetArrayF90(field%perm0_zz,perm_zz_p,ierr);CHKERRQ(ierr)
    if (option%flow%full_perm_tensor) then
      call VecGetArrayF90(field%perm0_xy,perm_xy_p,ierr);CHKERRQ(ierr)
      call VecGetArrayF90(field%perm0_xz,perm_xz_p,ierr);CHKERRQ(ierr)
      call VecGetArrayF90(field%perm0_yz,perm_yz_p,ierr);CHKERRQ(ierr)
    endif
    if (soil_compressibility_index > 0) then
      call VecGetArrayF90(field%compressibility0,compress_p, &
                          ierr);CHKERRQ(ierr)
    endif
  endif
  call VecGetArrayF90(field%porosity0,por0_p,ierr);CHKERRQ(ierr)
  if (option%transport%anisotropic_tortuosity) then
    call VecDuplicate(field%tortuosity0,tort_yy,ierr);CHKERRQ(ierr)
    call VecDuplicate(field%tortuosity0,tort_zz,ierr);CHKERRQ(ierr)
    call VecGetArrayF90(tort_yy,tort_yy_p,ierr);CHKERRQ(ierr)
    call VecGetArrayF90(tort_zz,tort_zz_p,ierr);CHKERRQ(ierr)
  endif
  call VecGetArrayF90(field%tortuosity0,tor0_p,ierr);CHKERRQ(ierr)

  ! have to use Material%auxvars() and not material_auxvars() due to memory
  ! errors in gfortran
  material_auxvars => patch%aux%Material%auxvars

  !if material is associated with fracture, then allocate memory.
  wipp => WIPPGetPtr()
  call CreepClosureConvertListToArray(wipp%creep_closure_tables, &
                                      wipp%creep_closure_tables_array, &
                                      option)

  do ghosted_id = 1, grid%ngmax
    material_id = patch%imat(ghosted_id)
    if (material_id > 0) then
      material_property => &
        patch%material_property_array(material_id)%ptr

      if (option%ngeomechdof > 0) then
        call GeomechanicsSubsurfacePropsAuxvarInit( &
              material_property%geomechanics_subsurface_properties, &
              material_auxvars(ghosted_id))
      endif

      ! lookup creep closure table id from creep closure table name
      if (option%flow%creep_closure_on) then
        material_property%creep_closure_id = &
          CreepClosureGetID(wipp%creep_closure_tables_array, &
                             material_property%creep_closure_name, &
                             material_property%name,option)
        ! copy creep closure table id from material to material_aux
        material_auxvars(ghosted_id)%creep_closure_id = &
          material_property%creep_closure_id
      endif
    endif
  enddo

  if (discretization%grid%itype == ECLIPSE_UNSTRUCTURED_GRID) then
    if (.not.OptionIsIORank(option)) then
      allocate(inatsend(grid%nlmax))
    endif
  endif

  do local_id = 1, grid%nlmax
    ghosted_id = grid%nL2G(local_id)
    material_id = patch%imat(ghosted_id)
    if (material_id == 0) then
      material_property => null_material_property
    else if (abs(material_id) <= &
             size(patch%material_property_array)) then
      if (material_id < 0) then
        material_property => null_material_property
      else
        material_property => &
          patch%material_property_array(material_id)%ptr
        if (.not.associated(material_property)) then
          write(string,*) &
            patch%imat_internal_to_external(material_id)
          option%io_buffer = 'No material property for material id ' // &
                              trim(adjustl(string)) &
                              //  ' defined in input file.'
          call PrintErrMsgByRank(option)
        endif
      endif
    else if (Uninitialized(material_id)) then
      write(string,*) grid%nG2A(ghosted_id)
      option%io_buffer = 'Uninitialized material id in patch at cell ' // &
                          trim(adjustl(string))
      call PrintErrMsgByRank(option)
    else if (material_id > size(patch%material_property_array)) then
      write(option%io_buffer,*) patch%imat_internal_to_external(material_id)
      option%io_buffer = 'Unmatched material id in patch: ' // &
        adjustl(trim(option%io_buffer))
      call PrintErrMsgByRank(option)
    else if (material_id == UNMAPPED_MATERIAL_ID) then
      option%io_buffer = 'A material ID has been assigned to a cell that &
        &does not match the IDs of any MATERIAL_PROPERTIES.'
      call PrintErrMsgByRank(option)
    else
      option%io_buffer = 'Something messed up with material ids. Possibly &
        &material ids not assigned to all grid cells. Contact Glenn!'
      call PrintErrMsgByRank(option)
    endif
    if (option%nflowdof > 0) then
      if (associated(patch%cc_id)) then
        patch%cc_id(ghosted_id) = &
          material_property%saturation_function_id
      endif
      if (associated(patch%cct_id)) then
        patch%cct_id(ghosted_id) = &
          material_property%thermal_conductivity_function_id
      endif
      perm_xx_p(local_id) = material_property%permeability(1,1)
      perm_yy_p(local_id) = material_property%permeability(2,2)
      perm_zz_p(local_id) = material_property%permeability(3,3)
      if (option%flow%full_perm_tensor) then
        perm_xy_p(local_id) = material_property%permeability(1,2)
        perm_xz_p(local_id) = material_property%permeability(1,3)
        perm_yz_p(local_id) = material_property%permeability(2,3)
      endif
      if (soil_compressibility_index > 0) then
        compress_p(local_id) = material_property%soil_compressibility
      endif
    endif
    por0_p(local_id) = material_property%porosity
    if (option%transport%anisotropic_tortuosity) then
      tor0_p(local_id) = material_property%tortuosity_anisotropic(1)
      tort_yy_p(local_id) = material_property%tortuosity_anisotropic(2)
      tort_zz_p(local_id) = material_property%tortuosity_anisotropic(3)
    else
      tor0_p(local_id) = material_property%tortuosity
    endif
    if (associated(material_auxvars)) then
      call MaterialAssignPropertyToAux(material_auxvars(ghosted_id), &
                                       material_property,option)
    endif

    !MAN: this needs to be refactored
    if (discretization%grid%itype == ECLIPSE_UNSTRUCTURED_GRID) then

      natural_id = grid%nG2A(ghosted_id)

      if (OptionIsIORank(option)) then
  !  Simply set up the values on the I/O proc
        call GridEclipseGetPorPerm(natural_id,poro,permx,permy,permz)
        por0_p(local_id)    = poro
        perm_xx_p(local_id) = permx
        perm_yy_p(local_id) = permy
        perm_zz_p(local_id) = permz
      else
  !  Add to the request list on other procs
        inatsend(local_id)=natural_id
      endif

    endif

  enddo

  !MAN: this needs to be refactored
  if (discretization%grid%itype == ECLIPSE_UNSTRUCTURED_GRID) then

    call GridEclipsePorPermExchangeAndSet(por0_p,perm_xx_p,perm_yy_p, &
                                          perm_zz_p,inatsend,grid%nlmax, &
                                          option)

    rprnc = GridEcilpseGetProcnumFlag()
    if( rprnc>0 ) then
      call GridEclipseProcnumExchangeAndSet(inatsend, grid%nlmax, option)
    endif
    if (.not.OptionIsIORank(option)) then
      call DeallocateArray(inatsend)
    endif

    call GridEcipseDeallocatePorPermArrays(PETSC_FALSE,option)
  endif

  call MaterialPropertyDestroy(null_material_property)

  if (option%nflowdof > 0) then
    call VecRestoreArrayF90(field%perm0_xx,perm_xx_p,ierr);CHKERRQ(ierr)
    call VecRestoreArrayF90(field%perm0_yy,perm_yy_p,ierr);CHKERRQ(ierr)
    call VecRestoreArrayF90(field%perm0_zz,perm_zz_p,ierr);CHKERRQ(ierr)
    if (option%flow%full_perm_tensor) then
      call VecRestoreArrayF90(field%perm0_xy,perm_xy_p,ierr);CHKERRQ(ierr)
      call VecRestoreArrayF90(field%perm0_xz,perm_xz_p,ierr);CHKERRQ(ierr)
      call VecRestoreArrayF90(field%perm0_yz,perm_yz_p,ierr);CHKERRQ(ierr)
    endif
    if (soil_compressibility_index > 0) then
      call VecRestoreArrayF90(field%compressibility0,compress_p, &
                              ierr);CHKERRQ(ierr)
    endif
  endif
  call VecRestoreArrayF90(field%porosity0,por0_p,ierr);CHKERRQ(ierr)
  call VecRestoreArrayF90(field%tortuosity0,tor0_p,ierr);CHKERRQ(ierr)
  if (option%transport%anisotropic_tortuosity) then
    call VecRestoreArrayF90(tort_yy,tort_yy_p,ierr);CHKERRQ(ierr)
    call VecRestoreArrayF90(tort_zz,tort_zz_p,ierr);CHKERRQ(ierr)
  endif

  ! read in any user-defined property fields
  do material_id = 1, size(patch%material_property_array)
    material_property => &
            patch%material_property_array(material_id)%ptr
    if (.not.associated(material_property)) cycle
    if (material_property%active) then
      if (associated(material_property%permeability_dataset)) then
        call SubsurfReadPermsFromFile(realization,material_property)
      endif
      call SubsurfMapDatasetToVec(realization,material_property, &
                                  material_property%compressibility_dataset, &
                                  field%compressibility0)
      if (associated(material_property%porosity_dataset)) then
        call SubsurfMapDatasetToVec(realization,material_property, &
                                    material_property%porosity_dataset, &
                                    field%porosity0)
        ! if tortuosity is a function of porosity, we must calculate the
        ! the tortuosity on a cell to cell basis.
        if (field%tortuosity0 /= PETSC_NULL_VEC .and. &
            material_property%tortuosity_function_of_porosity) then
          call VecGetArrayF90(field%porosity0,por0_p,ierr);CHKERRQ(ierr)
          call VecGetArrayF90(field%tortuosity0,tor0_p,ierr);CHKERRQ(ierr)
          do local_id = 1, grid%nlmax
            ghosted_id = grid%nL2G(local_id)
            if (patch%imat(ghosted_id) == material_property%internal_id) then
              tor0_p(local_id) = por0_p(local_id)** &
                material_property%tortuosity_func_porosity_pwr
            endif
          enddo
          call VecRestoreArrayF90(field%porosity0,por0_p,ierr);CHKERRQ(ierr)
          call VecRestoreArrayF90(field%tortuosity0,tor0_p, &
                                  ierr);CHKERRQ(ierr)
        endif
      endif
      call SubsurfMapDatasetToVec(realization,material_property, &
                                  material_property%tortuosity_dataset, &
                                  field%tortuosity0)
      call SubsurfMapDatasetToMatAuxVar(realization,material_property, &
                        material_property%material_elec_cond_dataset, &
                        MATERIAL_ELECTRICAL_CONDUCTIVITY)
      call SubsurfMapDatasetToMatAuxVar(realization,material_property, &
                        material_property%archie_cem_exp_dataset, &
                        ARCHIE_CEMENTATION_EXPONENT)
      call SubsurfMapDatasetToMatAuxVar(realization,material_property, &
                        material_property%archie_sat_exp_dataset, &
                        ARCHIE_SATURATION_EXPONENT)
      call SubsurfMapDatasetToMatAuxVar(realization,material_property, &
                        material_property%archie_tor_con_dataset, &
                        ARCHIE_TORTUOSITY_CONSTANT)
      call SubsurfMapDatasetToMatAuxVar(realization,material_property, &
                        material_property%surf_elec_cond_dataset, &
                        SURFACE_ELECTRICAL_CONDUCTIVITY)
      call SubsurfMapDatasetToMatAuxVar(realization,material_property, &
                        material_property%waxman_smits_clay_cond_dataset, &
                        WAXMAN_SMITS_CLAY_CONDUCTIVITY)
      if (associated(material_property%multicontinuum)) then
        call SubsurfMapDatasetToMatAuxVar(realization,material_property, &
                          material_property%multicontinuum%epsilon_dataset, &
                          EPSILON)
        call SubsurfMapDatasetToMatAuxVar(realization,material_property, &
                          material_property%multicontinuum% &
                            half_matrix_width_dataset, &
                          HALF_MATRIX_WIDTH)
        call SubsurfMapDatasetToMatAuxVar(realization,material_property, &
                          material_property%multicontinuum%ncells_dataset, &
                          NUMBER_SECONDARY_CELLS)
      endif
    endif
  enddo

  ! update ghosted values
  if (option%nflowdof > 0) then
    call DiscretizationGlobalToLocal(discretization,field%perm0_xx, &
                                     field%work_loc,ONEDOF)
    call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                                 PERMEABILITY_X,ZERO_INTEGER)
    call DiscretizationGlobalToLocal(discretization,field%perm0_yy, &
                                     field%work_loc,ONEDOF)
    call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                                 PERMEABILITY_Y,ZERO_INTEGER)
    call DiscretizationGlobalToLocal(discretization,field%perm0_zz, &
                                     field%work_loc,ONEDOF)
    call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                                 PERMEABILITY_Z,ZERO_INTEGER)
    if (option%flow%full_perm_tensor) then
      call DiscretizationGlobalToLocal(discretization,field%perm0_xy, &
                                       field%work_loc,ONEDOF)
      call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                                   PERMEABILITY_XY,ZERO_INTEGER)
      call DiscretizationGlobalToLocal(discretization,field%perm0_xz, &
                                       field%work_loc,ONEDOF)
      call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                                   PERMEABILITY_XZ,ZERO_INTEGER)
      call DiscretizationGlobalToLocal(discretization,field%perm0_yz, &
                                       field%work_loc,ONEDOF)
      call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                                   PERMEABILITY_YZ,ZERO_INTEGER)
    endif

    if (associated(patch%cc_id)) then
      call RealLocalToLocalWithArray(realization,CC_ID_ARRAY)
    endif
    if (associated(patch%cct_id)) then
      call RealLocalToLocalWithArray(realization,CCT_ID_ARRAY)
    endif

    if (soil_compressibility_index > 0) then
      call DiscretizationGlobalToLocal(discretization,field%compressibility0, &
                                       field%work_loc,ONEDOF)
      call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                                   SOIL_COMPRESSIBILITY,ZERO_INTEGER)
    endif
  endif

  call DiscretizationGlobalToLocal(discretization,field%porosity0, &
                                   field%work_loc,ONEDOF)
  call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                               POROSITY,POROSITY_INITIAL)
  call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                               POROSITY,POROSITY_BASE)
  call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                               POROSITY,POROSITY_CURRENT)
  call DiscretizationGlobalToLocal(discretization,field%tortuosity0, &
                                    field%work_loc,ONEDOF)
  call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                               TORTUOSITY,ZERO_INTEGER)
  if (option%transport%anisotropic_tortuosity) then
    call DiscretizationGlobalToLocal(discretization,tort_yy, &
                                     field%work_loc,ONEDOF)
    call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                                 TORTUOSITY_Y,ZERO_INTEGER)
    call DiscretizationGlobalToLocal(discretization,tort_zz, &
                                     field%work_loc,ONEDOF)
    call MaterialSetAuxVarVecLoc(patch%aux%Material,field%work_loc, &
                                 TORTUOSITY_Z,ZERO_INTEGER)
    call VecDestroy(tort_yy,ierr);CHKERRQ(ierr)
    call VecDestroy(tort_zz,ierr);CHKERRQ(ierr)
  endif

  ! copy rock properties to neighboring ghost cells
  do i = 1, max_material_index
    call VecGetArrayF90(field%work,vec_p,ierr);CHKERRQ(ierr)
    do local_id = 1, patch%grid%nlmax
      vec_p(local_id) = &
          material_auxvars(patch%grid%nL2G(local_id))%soil_properties(i)
    enddo
    call VecRestoreArrayF90(field%work,vec_p,ierr);CHKERRQ(ierr)
    call DiscretizationGlobalToLocal(discretization,field%work, &
                                     field%work_loc,ONEDOF)
    call VecGetArrayF90(field%work_loc,vec_p,ierr);CHKERRQ(ierr)
    do ghosted_id = 1, patch%grid%ngmax
      material_auxvars(ghosted_id)%soil_properties(i) = &
         vec_p(ghosted_id)
    enddo
    call VecRestoreArrayF90(field%work_loc,vec_p,ierr);CHKERRQ(ierr)
  enddo

  if (option%use_sc) then
    call VecGetArrayF90(field%work,vec_p,ierr);CHKERRQ(ierr)
    do local_id = 1, patch%grid%nlmax
      vec_p(local_id) = &
          material_auxvars(patch%grid%nL2G(local_id))%secondary_prop%epsilon
    enddo
    call VecRestoreArrayF90(field%work,vec_p,ierr);CHKERRQ(ierr)
    call DiscretizationGlobalToLocal(discretization,field%work, &
                                     field%work_loc,ONEDOF)
    call VecGetArrayF90(field%work_loc,vec_p,ierr);CHKERRQ(ierr)
    do ghosted_id = 1, patch%grid%ngmax
      material_auxvars(ghosted_id)%secondary_prop%epsilon = &
         vec_p(ghosted_id)
    enddo
    call VecRestoreArrayF90(field%work_loc,vec_p,ierr);CHKERRQ(ierr)

    call VecGetArrayF90(field%work,vec_p,ierr);CHKERRQ(ierr)
    do local_id = 1, patch%grid%nlmax
      vec_p(local_id) = &
          material_auxvars(patch%grid%nL2G(local_id))%secondary_prop%half_matrix_width
    enddo
    call VecRestoreArrayF90(field%work,vec_p,ierr);CHKERRQ(ierr)
    call DiscretizationGlobalToLocal(discretization,field%work, &
                                     field%work_loc,ONEDOF)
    call VecGetArrayF90(field%work_loc,vec_p,ierr);CHKERRQ(ierr)
    do ghosted_id = 1, patch%grid%ngmax
      material_auxvars(ghosted_id)%secondary_prop%half_matrix_width = &
         vec_p(ghosted_id)
    enddo
    call VecRestoreArrayF90(field%work_loc,vec_p,ierr);CHKERRQ(ierr)

    call VecGetArrayF90(field%work,vec_p,ierr);CHKERRQ(ierr)
    do local_id = 1, patch%grid%nlmax
      vec_p(local_id) = &
          material_auxvars(patch%grid%nL2G(local_id))%secondary_prop%ncells
    enddo
    call VecRestoreArrayF90(field%work,vec_p,ierr);CHKERRQ(ierr)
    call DiscretizationGlobalToLocal(discretization,field%work, &
                                     field%work_loc,ONEDOF)
    call VecGetArrayF90(field%work_loc,vec_p,ierr);CHKERRQ(ierr)
    do ghosted_id = 1, patch%grid%ngmax
      material_auxvars(ghosted_id)%secondary_prop%ncells = &
         int(vec_p(ghosted_id))
    enddo
    call VecRestoreArrayF90(field%work_loc,vec_p,ierr);CHKERRQ(ierr)
  endif

  if (option%ngeomechdof > 0) then
    call VecCopy(field%porosity0,field%porosity_geomech_store, &
                 ierr);CHKERRQ(ierr)
#ifdef GEOMECH_DEBUG
    print *, 'InitSubsurfAssignMatProperties'
    call PetscViewerASCIIOpen(realization%option%mycomm, &
                              'porosity_geomech_store_por0.out',viewer, &
                              ierr);CHKERRQ(ierr)
    call VecView(field%porosity_geomech_store,viewer,ierr);CHKERRQ(ierr)
    call PetscViewerDestroy(viewer,ierr);CHKERRQ(ierr)
#endif
  endif

  i = 0
  do local_id = 1, grid%nlmax
    if (patch%imat(grid%nL2G(local_id)) <= 0) i = i + 1
  enddo
  call MPI_Allreduce(MPI_IN_PLACE,i,ONE_INTEGER_MPI,MPIU_INTEGER, &
                     MPI_SUM,option%mycomm,ierr);CHKERRQ(ierr)
  option%io_buffer = NL // &
    'Number of active grid cells: '// StringWrite(grid%nmax-i) // &
    NL // 'Number of inactive grid cells: '// StringWrite(i)// NL
  call PrintMsg(option)

end subroutine InitSubsurfAssignMatProperties

! ************************************************************************** !

subroutine SubsurfMapDatasetToMatAuxVar(realization,material_property, &
                                        dataset,ivar)
  !
  ! Maps a dataset to a material auxvar variable
  !
  ! Author: Glenn Hammond
  ! Date: 05/16/24
  !
  use Dataset_Base_class
  use Material_module
  use Realization_Subsurface_class

  implicit none

  class(realization_subsurface_type) :: realization
  type(material_property_type) :: material_property
  class(dataset_base_type), pointer :: dataset
  PetscInt :: ivar

  if (associated(dataset)) then
    call SubsurfReadDatasetToVecWithMask(realization,dataset,&
                                         material_property%internal_id, &
                                         PETSC_FALSE,realization%field%work)
    call SubsurfMapVecToMatAuxByMaterial(realization,realization%field%work, &
                                         material_property%internal_id, &
                                         ivar)
  endif

end subroutine SubsurfMapDatasetToMatAuxVar

! ************************************************************************** !

subroutine SubsurfMapDatasetToVec(realization,material_property,dataset,vec)
  !
  ! Maps a dataset to a PETSc Vec
  !
  ! Author: Glenn Hammond
  ! Date: 05/16/24
  !
  use Dataset_Base_class
  use Material_module
  use Realization_Subsurface_class

  implicit none

  class(realization_subsurface_type) :: realization
  type(material_property_type) :: material_property
  class(dataset_base_type), pointer :: dataset
  Vec :: vec

  if (associated(dataset)) then
    call SubsurfReadDatasetToVecWithMask(realization,dataset,&
                                         material_property%internal_id, &
                                         PETSC_FALSE,vec)
  endif

end subroutine SubsurfMapDatasetToVec

! ************************************************************************** !

subroutine SubsurfReadMaterialIDsFromFile(realization,realization_dependent, &
                                          filename)
  !
  ! Reads in grid cell materials
  !
  ! Author: Glenn Hammond
  ! Date: 1/03/08
  !

  use Realization_Subsurface_class
  use Field_module
  use Grid_module
  use Option_module
  use Patch_module
  use Discretization_module
  use Logging_module
  use Input_Aux_module
  use Material_module
  use Petsc_Utility_module

  use HDF5_module

  implicit none

  class(realization_subsurface_type) :: realization
  PetscBool :: realization_dependent
  character(len=MAXSTRINGLENGTH) :: filename

  type(field_type), pointer :: field
  type(grid_type), pointer :: grid
  type(option_type), pointer :: option
  type(patch_type), pointer :: patch
  type(input_type), pointer :: input
  type(discretization_type), pointer :: discretization
  character(len=MAXSTRINGLENGTH) :: group_name
  character(len=MAXSTRINGLENGTH) :: dataset_name
  PetscInt :: ghosted_id, natural_id, material_id
  PetscInt, pointer :: external_to_internal_mapping(:)
  Vec :: global_vec
  Vec :: local_vec
  PetscErrorCode :: ierr

  field => realization%field
  patch => realization%patch
  grid => patch%grid
  option => realization%option
  discretization => realization%discretization

  if (index(filename,'.h5') > 0) then
    group_name = 'Materials'
    dataset_name = 'Material Ids'
    call DiscretizationCreateVector(discretization,ONEDOF,global_vec,GLOBAL, &
                                    option)
    call DiscretizationCreateVector(discretization,ONEDOF,local_vec,LOCAL, &
                                    option)
    call HDF5ReadCellIndexedIntegerArray(realization,global_vec, &
                                         filename,group_name, &
                                         dataset_name,realization_dependent)
    call DiscretizationGlobalToLocal(discretization,global_vec,local_vec,ONEDOF)
    call PetUtilUnloadVec(local_vec,patch%imat)
  else
    call PetscLogEventBegin(logging%event_hash_map,ierr);CHKERRQ(ierr)
    call GridCreateNaturalToGhostedHash(grid,option)
    input => InputCreate(IUNIT_TEMP,filename,option)
    do
      call InputReadPflotranString(input,option)
      if (InputError(input)) exit
      call InputReadInt(input,option,natural_id)
      call InputErrorMsg(input,option,'natural id','STRATA')
      ! natural ids in hash are zero-based
      ghosted_id = GridGetLocalGhostedIdFromHash(grid,natural_id)
      if (ghosted_id > 0) then
        call InputReadInt(input,option,material_id)
        call InputErrorMsg(input,option,'material id','STRATA')
        patch%imat(ghosted_id) = material_id
      endif
    enddo
    call InputDestroy(input)
    call GridDestroyHashTable(grid)
    call PetscLogEventEnd(logging%event_hash_map,ierr);CHKERRQ(ierr)
  endif

  call MaterialCreateExtToIntMapping(patch%material_property_array, &
                                     external_to_internal_mapping)
  call MaterialApplyMapping(external_to_internal_mapping,patch%imat)
  deallocate(external_to_internal_mapping)
  nullify(external_to_internal_mapping)

end subroutine SubsurfReadMaterialIDsFromFile

! ************************************************************************** !

subroutine SubsurfReadPermsFromFile(realization,material_property)
  !
  ! Reads in grid cell permeabilities
  !
  ! Author: Glenn Hammond
  ! Date: 01/19/09
  !

  use Realization_Subsurface_class
  use Field_module
  use Grid_module
  use Option_module
  use Patch_module
  use Discretization_module
  use Logging_module
  use Input_Aux_module
  use Material_module
  use HDF5_module
  use Dataset_Common_HDF5_class

  implicit none

  class(realization_subsurface_type) :: realization
  type(material_property_type) :: material_property

  type(field_type), pointer :: field
  type(patch_type), pointer :: patch
  type(grid_type), pointer :: grid
  type(option_type), pointer :: option
  type(discretization_type), pointer :: discretization
  class(dataset_common_hdf5_type), pointer :: dataset_common_hdf5_ptr
  PetscInt :: local_id
  PetscInt :: idirection, temp_int
  PetscReal :: ratio, scale
  Vec :: global_vec
  PetscErrorCode :: ierr

  PetscReal, pointer :: vec_p(:)
  PetscReal, pointer :: perm_xx_p(:)
  PetscReal, pointer :: perm_yy_p(:)
  PetscReal, pointer :: perm_zz_p(:)
  PetscReal, pointer :: perm_xy_p(:)
  PetscReal, pointer :: perm_yz_p(:)
  PetscReal, pointer :: perm_xz_p(:)
  PetscReal, pointer :: perm_ptr(:)

  field => realization%field
  patch => realization%patch
  grid => patch%grid
  option => realization%option
  discretization => realization%discretization

  call VecGetArrayF90(field%perm0_xx,perm_xx_p,ierr);CHKERRQ(ierr)
  call VecGetArrayF90(field%perm0_yy,perm_yy_p,ierr);CHKERRQ(ierr)
  call VecGetArrayF90(field%perm0_zz,perm_zz_p,ierr);CHKERRQ(ierr)
  if (option%flow%full_perm_tensor) then
    call VecGetArrayF90(field%perm0_xy,perm_xy_p,ierr);CHKERRQ(ierr)
    call VecGetArrayF90(field%perm0_xz,perm_xz_p,ierr);CHKERRQ(ierr)
    call VecGetArrayF90(field%perm0_yz,perm_yz_p,ierr);CHKERRQ(ierr)
  endif

  call DiscretizationCreateVector(discretization,ONEDOF,global_vec,GLOBAL, &
                                  option)
  if (material_property%isotropic_permeability .or. &
      (.not.material_property%isotropic_permeability .and. &
       Initialized(material_property%vertical_anisotropy_ratio))) then
    ! Although the mask of material ID is applied below, we must only read
    ! in the permeabilities that apply to this material so that small,
    ! localized gridded datasets (that only apply to a subset of the domain)
    ! can be used.
    call VecZeroEntries(global_vec,ierr);CHKERRQ(ierr)
    call SubsurfReadDatasetToVecWithMask(realization, &
                                    material_property%permeability_dataset, &
                                    material_property%internal_id, &
                                    PETSC_FALSE,global_vec)
    call VecGetArrayF90(global_vec,vec_p,ierr);CHKERRQ(ierr)
    ratio = 1.d0
    scale = 1.d0
    !TODO(geh): fix so that ratio and scale work for perms outside
    ! of dataset
    if (Initialized(material_property%vertical_anisotropy_ratio)) then
      ratio = material_property%vertical_anisotropy_ratio
    endif
    if (material_property%permeability_scaling_factor > 0.d0) then
      scale = material_property%permeability_scaling_factor
    endif
    do local_id = 1, grid%nlmax
      if (patch%imat(grid%nL2G(local_id)) == &
          material_property%internal_id) then
        perm_xx_p(local_id) = vec_p(local_id)*scale
        perm_yy_p(local_id) = vec_p(local_id)*scale
        perm_zz_p(local_id) = vec_p(local_id)*ratio*scale
        if (option%flow%full_perm_tensor) then
          perm_xy_p(local_id) = 0.d0
          perm_xz_p(local_id) = 0.d0
          perm_yz_p(local_id) = 0.d0
        endif
      endif
    enddo
    call VecRestoreArrayF90(global_vec,vec_p,ierr);CHKERRQ(ierr)
  else
    if (material_property%full_permeability_tensor) then
      temp_int = YZ_DIRECTION
    else
      temp_int = Z_DIRECTION
    endif
    do idirection = X_DIRECTION, temp_int
      select case(idirection)
        case(X_DIRECTION)
          dataset_common_hdf5_ptr => &
             DatasetCommonHDF5Cast(material_property%permeability_dataset)
          perm_ptr => perm_xx_p
        case(Y_DIRECTION)
          dataset_common_hdf5_ptr => &
             DatasetCommonHDF5Cast(material_property%permeability_dataset_y)
          perm_ptr => perm_yy_p
        case(Z_DIRECTION)
          dataset_common_hdf5_ptr => &
             DatasetCommonHDF5Cast(material_property%permeability_dataset_z)
          perm_ptr => perm_zz_p
        case(XY_DIRECTION)
          dataset_common_hdf5_ptr => &
             DatasetCommonHDF5Cast(material_property%permeability_dataset_xy)
          perm_ptr => perm_xy_p
        case(XZ_DIRECTION)
          dataset_common_hdf5_ptr => &
             DatasetCommonHDF5Cast(material_property%permeability_dataset_xz)
          perm_ptr => perm_xz_p
        case(YZ_DIRECTION)
          dataset_common_hdf5_ptr => &
             DatasetCommonHDF5Cast(material_property%permeability_dataset_yz)
          perm_ptr => perm_yz_p
      end select
      ! Although the mask of material ID is applied below, we must only read
      ! in the permeabilities that apply to this material so that small,
      ! localized gridded datasets (that only apply to a subset of the domain)
      ! can be used.
      call VecZeroEntries(global_vec,ierr);CHKERRQ(ierr)
      call SubsurfReadDatasetToVecWithMask(realization, &
                                           dataset_common_hdf5_ptr,&
                                           material_property%internal_id, &
                                           PETSC_FALSE,global_vec)
      call VecGetArrayF90(global_vec,vec_p,ierr);CHKERRQ(ierr)
      do local_id = 1, grid%nlmax
        if (patch%imat(grid%nL2G(local_id)) == &
            material_property%internal_id) then
          perm_ptr(local_id) = vec_p(local_id)
        endif
      enddo
      call VecRestoreArrayF90(global_vec,vec_p,ierr);CHKERRQ(ierr)
    enddo
  endif
  call VecDestroy(global_vec,ierr);CHKERRQ(ierr)

  call VecRestoreArrayF90(field%perm0_xx,perm_xx_p,ierr);CHKERRQ(ierr)
  call VecRestoreArrayF90(field%perm0_yy,perm_yy_p,ierr);CHKERRQ(ierr)
  call VecRestoreArrayF90(field%perm0_zz,perm_zz_p,ierr);CHKERRQ(ierr)
  if (option%flow%full_perm_tensor) then
    call VecRestoreArrayF90(field%perm0_xy,perm_xy_p,ierr);CHKERRQ(ierr)
    call VecRestoreArrayF90(field%perm0_xz,perm_xz_p,ierr);CHKERRQ(ierr)
    call VecRestoreArrayF90(field%perm0_yz,perm_yz_p,ierr);CHKERRQ(ierr)
  endif

end subroutine SubsurfReadPermsFromFile

! ************************************************************************** !

subroutine SubsurfReadDatasetToVecWithMask(realization,dataset, &
                                           material_id,read_all_values,vec)
  !
  ! Reads a dataset into a PETSc Vec using the material id as a mask
  !
  ! Author: Glenn Hammond
  ! Date: 01/19/2016
  !
  use Realization_Subsurface_class
  use Field_module
  use Grid_module
  use Option_module
  use Patch_module
  use Logging_module
  use Input_Aux_module
  use Material_module
  use HDF5_module
  use Dataset_Base_class
  use Dataset_Common_HDF5_class
  use Dataset_Gridded_HDF5_class

  implicit none

  class(realization_subsurface_type) :: realization
  class(dataset_base_type) :: dataset
  PetscInt :: material_id
  PetscBool :: read_all_values
  Vec :: vec

  type(field_type), pointer :: field
  type(patch_type), pointer :: patch
  type(grid_type), pointer :: grid
  type(option_type), pointer :: option
  type(input_type), pointer :: input
  character(len=MAXSTRINGLENGTH) :: group_name
  character(len=MAXSTRINGLENGTH) :: dataset_name
  character(len=MAXSTRINGLENGTH) :: filename
  PetscInt :: local_id, ghosted_id, natural_id
  PetscReal :: tempreal
  PetscErrorCode :: ierr

  PetscReal, pointer :: vec_p(:)
  PetscReal, pointer :: work_p(:)

  field => realization%field
  patch => realization%patch
  grid => patch%grid
  option => realization%option

  call VecGetArrayF90(vec,vec_p,ierr);CHKERRQ(ierr)

  if (index(dataset%filename,'.h5') > 0) then
    group_name = ''
    dataset_name = dataset%name
    select type(dataset)
      class is(dataset_gridded_hdf5_type)
        call DatasetGriddedHDF5Load(dataset,option)
        do local_id = 1, grid%nlmax
          ghosted_id = grid%nL2G(local_id)
          if (read_all_values .or. &
              patch%imat(ghosted_id) == material_id) then
            call DatasetGriddedHDF5InterpolateReal(dataset, &
                   grid%x(ghosted_id),grid%y(ghosted_id),grid%z(ghosted_id), &
                   vec_p(local_id),option)
          endif
        enddo
        ! now we strip the dataset to save storage, saving only the name
        ! and filename incase it must be read later
        filename = dataset%filename
        dataset_name = dataset%name
        call DatasetGriddedHDF5Strip(dataset)
        call DatasetGriddedHDF5Init(dataset)
        dataset%filename = filename
        dataset%name = trim(dataset_name)
      class is(dataset_common_hdf5_type)
        dataset_name = dataset%hdf5_dataset_name
        call HDF5ReadCellIndexedRealArray(realization,field%work, &
                                          dataset%filename, &
                                          group_name,dataset_name, &
                                          dataset%realization_dependent)
        call VecGetArrayF90(field%work,work_p,ierr);CHKERRQ(ierr)
        if (read_all_values) then
          vec_p(:) = work_p(:)
        else
          do local_id = 1, grid%nlmax
            if (patch%imat(grid%nL2G(local_id)) == material_id) then
              vec_p(local_id) = work_p(local_id)
            endif
          enddo
        endif
        call VecRestoreArrayF90(field%work,work_p,ierr);CHKERRQ(ierr)
      class default
        option%io_buffer = 'Dataset "' // trim(dataset%name) // '" is of the &
          &wrong type for SubsurfReadDatasetToVecWithMask()'
        call PrintErrMsg(option)
    end select
  else
    call PetscLogEventBegin(logging%event_hash_map,ierr);CHKERRQ(ierr)
    call GridCreateNaturalToGhostedHash(grid,option)
    input => InputCreate(IUNIT_TEMP,dataset%filename,option)
    do
      call InputReadPflotranString(input,option)
      if (InputError(input)) exit
      call InputReadInt(input,option,natural_id)
      call InputErrorMsg(input,option,'ASCII natural id', &
                         'SubsurfReadDatasetToVecWithMask')
      ghosted_id = GridGetLocalGhostedIdFromHash(grid,natural_id)
      if (ghosted_id > 0) then
        if (read_all_values .or. &
            patch%imat(ghosted_id) == material_id) then
          local_id = grid%nG2L(ghosted_id)
          if (local_id > 0) then
            call InputReadDouble(input,option,tempreal)
            call InputErrorMsg(input,option,'dataset value', &
                               'SubsurfReadDatasetToVecWithMask')
            vec_p(local_id) = tempreal
          endif
        endif
      endif
    enddo
    call InputDestroy(input)
    call GridDestroyHashTable(grid)
    call PetscLogEventEnd(logging%event_hash_map,ierr);CHKERRQ(ierr)
  endif

  call VecRestoreArrayF90(vec,vec_p,ierr);CHKERRQ(ierr)

end subroutine SubsurfReadDatasetToVecWithMask

! ************************************************************************** !

subroutine SubsurfMapVecToMatAuxByMaterial(realization,vec,material_id,ivar)
  !
  ! Maps values in a global vec to the material auxvar masked by material ID
  !
  ! Author: Glenn Hammond
  ! Date: 07/14/23
  !
  use Grid_module
  use Material_Aux_module
  use Option_module
  use Patch_module
  use Realization_Subsurface_class

  implicit none

  class(realization_subsurface_type) :: realization
  Vec :: vec  ! global vec
  PetscInt :: material_id
  PetscInt :: ivar

  type(patch_type), pointer :: patch
  type(grid_type), pointer :: grid
  type(option_type), pointer :: option
  type(material_auxvar_type), pointer :: material_auxvars(:)
  PetscInt :: local_id, ghosted_id
  PetscReal, pointer :: vec_p(:)
  PetscErrorCode :: ierr

  patch => realization%patch
  grid => patch%grid
  option => realization%option

  material_auxvars => patch%aux%Material%auxvars

  call VecGetArrayF90(vec,vec_p,ierr);CHKERRQ(ierr)
  do local_id = 1, grid%nlmax
    ghosted_id = grid%nL2G(local_id)
    if (patch%imat(ghosted_id) == material_id) then
      call MaterialAuxVarSetValue(material_auxvars(ghosted_id), &
                                  ivar,vec_p(local_id))
    endif
  enddo
  call VecRestoreArrayF90(vec,vec_p,ierr);CHKERRQ(ierr)

end subroutine SubsurfMapVecToMatAuxByMaterial

! ************************************************************************** !

subroutine SubsurfAssignVolsToMatAuxVars(realization)
  !
  ! Assigns the cell volumes currently stored in field%volume0 to the
  ! material auxiliary variable object
  !
  ! Author: Glenn Hammond
  ! Date: 01/13/14, 12/04/14
  !

  use Realization_Subsurface_class
  use Option_module
  use Material_module
  use Discretization_module
  use Field_module
  use Variables_module, only : VOLUME

  implicit none

  class(realization_subsurface_type) :: realization

  type(option_type), pointer :: option
  type(field_type), pointer :: field

  option => realization%option
  field => realization%field

  call DiscretizationGlobalToLocal(realization%discretization,field%volume0, &
                                   field%work_loc,ONEDOF)
  call MaterialSetAuxVarVecLoc(realization%patch%aux%Material, &
                               field%work_loc,VOLUME,ZERO_INTEGER)

end subroutine SubsurfAssignVolsToMatAuxVars

! ************************************************************************** !

subroutine SubsurfSandboxesSetup(realization)
  !
  ! Initializes sandbox objects.
  !
  ! Author: Glenn Hammond
  ! Date: 05/06/14, 12/04/14

  use Realization_Subsurface_class
  use SrcSink_Sandbox_module

  class(realization_subsurface_type) :: realization

  call SSSandboxSetup(realization%patch%grid, &
                      realization%patch%region_list, &
                      realization%patch%aux%Material%auxvars, &
                      realization%option,realization%output_option)

end subroutine SubsurfSandboxesSetup

! ************************************************************************** !

subroutine InitSubsurfProcessOutputVars(realization)
  !
  ! Maps PARAMETER output variables with named parameters to the actual
  ! parameters
  !
  ! Author: Glenn Hammond
  ! Date: 01/26/24

  use Realization_Subsurface_class
  use Output_Aux_module

  implicit none

  class(realization_subsurface_type) :: realization
  type(output_variable_list_type), pointer :: output_variable_list

  output_variable_list => realization%output_option%output_variable_list
  call RealizationProcessOutputVarList(output_variable_list,realization)
  output_variable_list => realization%output_option%output_snap_variable_list
  call RealizationProcessOutputVarList(output_variable_list,realization)
  output_variable_list => realization%output_option%output_obs_variable_list
  call RealizationProcessOutputVarList(output_variable_list,realization)
  output_variable_list => realization%output_option%aveg_output_variable_list
  call RealizationProcessOutputVarList(output_variable_list,realization)

end subroutine InitSubsurfProcessOutputVars

end module Init_Subsurface_module
