module Grid_Unstructured_Explicit_module

#include "petsc/finclude/petscvec.h"
  use petscvec
  use Geometry_module
  use Grid_Unstructured_Aux_module

  use PFLOTRAN_Constants_module

  implicit none

  private

  public :: UGridExplicitRead, &
            UGridExplicitReadHDF5, &
            UGridExplicitDecompose, &
            UGridExplicitSetInternConnect, &
            UGridExplicitSetCellCentroids, &
            UGridExplicitComputeVolumes, &
            UGridExplicitSetBoundaryConnect, &
            UGridExplicitSetConnections, &
            UGridExplicitGetClosestCellFromPoint, &
            UGridExplicitExpandGhostCells


contains

! ************************************************************************** !

subroutine UGridExplicitRead(unstructured_grid,filename,option)
  !
  ! Reads an explicit unstructured grid in parallel
  !
  ! Author: Glenn Hammond
  ! Date: 10/03/12
  !

  use Input_Aux_module
  use Option_module
  use String_module

  implicit none

  type(grid_unstructured_type) :: unstructured_grid
  type(unstructured_explicit_type), pointer :: explicit_grid
  character(len=MAXSTRINGLENGTH) :: filename
  type(option_type) :: option

  type(input_type), pointer :: input
  character(len=MAXSTRINGLENGTH) :: hint
  character(len=MAXWORDLENGTH) :: word, card
  PetscInt :: fileid, icell, iconn, irank, remainder, temp_int, num_to_read

  PetscInt :: num_cells, num_connections, num_elems
  PetscInt :: num_cells_local, num_cells_local_save
  PetscInt :: num_connections_local, num_connections_local_save
  PetscMPIInt :: status_mpi(MPI_STATUS_SIZE)
  PetscMPIInt :: int_mpi
  PetscErrorCode :: ierr
  PetscReal, allocatable :: temp_real_array(:,:)
  PetscInt :: ivertex, num_vertices, num_grid_vertices
#if UGRID_DEBUG
  character(len=MAXSTRINGLENGTH) :: string
#endif

  explicit_grid => unstructured_grid%explicit_grid
! Format of explicit unstructured grid file
! id_, id_up_, id_dn_ = integer
! x_, y_, z_, area_, volume_ = real
! definitions
! id_ = id of grid cell
! id_up_ = id of upwind grid cell in connection
! id_dn_ = id of downwind grid cell in connection
! x_ = x coordinate of cell center
! y_ = y coordinate of cell center
! z_ = z coordinate of cell center
! volume_ = volume of grid cell
! -----------------------------------------------------------------
! CELLS <integer>    integer = # cells (N)
! id_1 x_1 y_1 z_1 volume_1
! id_2 x_2 y_2 z_2 volume_2
! ...
! ...
! id_N x_N y_N z_N volume_N
! CONNECTIONS <integer>   integer = # connections (M)
! id_up_1 id_dn_1 x_1 y_1 z_1 area_1
! id_up_2 id_dn_2 x_2 y_2 z_2 area_2
! ...
! ...
! id_up_M id_dn_M x_M y_M z_M area_M
! -----------------------------------------------------------------

  call OptionSetBlocking(option,PETSC_FALSE)
  if (OptionIsIORank(option)) then

    fileid = 86
    input => InputCreate(fileid,filename,option)

    if (.not.InputError(input)) then ! non-blocking above will continue
      call InputReadPflotranString(input,option)
      ! read CELL card, though we already know the
      call InputReadCard(input,option,card,PETSC_FALSE)
      word = 'CELLS'
      call InputErrorMsg(input,option,word,card)
      if (.not.StringCompare(word,card)) then
        option%io_buffer = 'Unrecognized keyword "' // trim(card) // &
          '" in explicit grid file.'
        call PrintErrMsg(option)
      endif

      hint = 'Explicit Unstructured Grid CELLS'
      call InputReadInt(input,option,temp_int)
      call InputErrorMsg(input,option,'number of cells',hint)
    endif
  endif
  call OptionSetBlocking(option,PETSC_TRUE)
  call OptionCheckNonBlockingError(option)

  call MPI_Bcast(temp_int,ONE_INTEGER_MPI,MPI_INTEGER,option%comm%io_rank, &
                 option%mycomm,ierr);CHKERRQ(ierr)
  num_cells = temp_int
  explicit_grid%num_cells_global = num_cells

   ! divide cells across ranks
  num_cells_local = num_cells/option%comm%size
  num_cells_local_save = num_cells_local
  remainder = num_cells - &
              num_cells_local*option%comm%size
  if (option%myrank < remainder) num_cells_local = &
                                 num_cells_local + 1

  allocate(explicit_grid%cell_ids(num_cells_local))
  explicit_grid%cell_ids = 0
  allocate(explicit_grid%cell_volumes(num_cells_local))
  explicit_grid%cell_volumes = 0
  allocate(explicit_grid%cell_centroids(num_cells_local))
  do icell = 1, num_cells_local
    explicit_grid%cell_centroids(icell)%x = 0.d0
    explicit_grid%cell_centroids(icell)%y = 0.d0
    explicit_grid%cell_centroids(icell)%z = 0.d0
  enddo

  ! for now, read all cells from ASCII file through io_rank and communicate
  ! to other ranks
  call OptionSetBlocking(option,PETSC_FALSE)
  if (OptionIsIORank(option)) then
    allocate(temp_real_array(5,num_cells_local_save+1))
    ! read for other processors
    do irank = 0, option%comm%size-1
      temp_real_array = UNINITIALIZED_DOUBLE
      num_to_read = num_cells_local_save
      if (irank < remainder) num_to_read = num_to_read + 1
      do icell = 1, num_to_read
        call InputReadPflotranString(input,option)
        call InputReadStringErrorMsg(input,option,hint)
        call InputReadInt(input,option,temp_int)
        call InputErrorMsg(input,option,'cell id',hint)
        temp_real_array(1,icell) = dble(temp_int)
        call InputReadDouble(input,option,temp_real_array(2,icell))
        call InputErrorMsg(input,option,'cell x coordinate',hint)
        call InputReadDouble(input,option,temp_real_array(3,icell))
        call InputErrorMsg(input,option,'cell y coordinate',hint)
        call InputReadDouble(input,option,temp_real_array(4,icell))
        call InputErrorMsg(input,option,'cell z coordinate',hint)
        call InputReadDouble(input,option,temp_real_array(5,icell))
        call InputErrorMsg(input,option,'cell volume',hint)
      enddo

      ! if the cells reside on io_rank
      if (OptionIsIORank(option,irank)) then
#if UGRID_DEBUG
        write(string,*) num_cells_local
        string = trim(adjustl(string)) // ' cells stored on p0'
        print *, trim(string)
#endif
        do icell = 1, num_cells_local
          explicit_grid%cell_ids(icell) = int(temp_real_array(1,icell))
          explicit_grid%cell_centroids(icell)%x = temp_real_array(2,icell)
          explicit_grid%cell_centroids(icell)%y = temp_real_array(3,icell)
          explicit_grid%cell_centroids(icell)%z = temp_real_array(4,icell)
          explicit_grid%cell_volumes(icell) = temp_real_array(5,icell)
        enddo
      else
        ! otherwise communicate to other ranks
#if UGRID_DEBUG
        write(string,*) num_to_read
        write(word,*) irank
        string = trim(adjustl(string)) // ' cells sent from p0 to p' // &
                 trim(adjustl(word))
        print *, trim(string)
#endif
        int_mpi = num_to_read*5
        call MPI_Send(temp_real_array,int_mpi,MPI_DOUBLE_PRECISION,irank, &
                      num_to_read,option%mycomm,ierr);CHKERRQ(ierr)
      endif
    enddo
  else
    ! other ranks post the recv
#if UGRID_DEBUG
    write(string,*) num_cells_local
    write(word,*) option%myrank
    string = trim(adjustl(string)) // ' cells received from p0 at p' // &
              trim(adjustl(word))
    print *, trim(string)
#endif
    allocate(temp_real_array(5,num_cells_local))
    int_mpi = num_cells_local*5
    call MPI_Recv(temp_real_array,int_mpi,MPI_DOUBLE_PRECISION, &
                  option%comm%io_rank,MPI_ANY_TAG,option%mycomm,status_mpi, &
                  ierr);CHKERRQ(ierr)
    do icell = 1, num_cells_local
      explicit_grid%cell_ids(icell) = int(temp_real_array(1,icell))
      explicit_grid%cell_centroids(icell)%x = temp_real_array(2,icell)
      explicit_grid%cell_centroids(icell)%y = temp_real_array(3,icell)
      explicit_grid%cell_centroids(icell)%z = temp_real_array(4,icell)
      explicit_grid%cell_volumes(icell) = temp_real_array(5,icell)
    enddo

  endif
  call OptionSetBlocking(option,PETSC_TRUE)
  call OptionCheckNonBlockingError(option)
  deallocate(temp_real_array)

  call OptionSetBlocking(option,PETSC_FALSE)
  if (OptionIsIORank(option)) then
    call InputReadPflotranString(input,option)
    ! read CONNECTIONS card, though we already know the
    call InputReadCard(input,option,card,PETSC_FALSE)
    word = 'CONNECTIONS'
    call InputErrorMsg(input,option,word,card)
    if (.not.StringCompare(word,card)) then
      option%io_buffer = 'Unrecognized keyword "' // trim(card) // &
        '" in explicit grid file.'
      call PrintErrMsg(option)
    endif

    hint = 'Explicit Unstructured Grid CONNECTIONS'
    call InputReadInt(input,option,temp_int)
    call InputErrorMsg(input,option,'number of connections',hint)
  endif
  call OptionSetBlocking(option,PETSC_TRUE)
  call OptionCheckNonBlockingError(option)

  int_mpi = 1
  call MPI_Bcast(temp_int,ONE_INTEGER_MPI,MPI_INTEGER,option%comm%io_rank, &
                 option%mycomm,ierr);CHKERRQ(ierr)
  num_connections = temp_int

   ! divide cells across ranks
  num_connections_local = num_connections/option%comm%size
  num_connections_local_save = num_connections_local
  remainder = num_connections - &
              num_connections_local*option%comm%size
  if (option%myrank < remainder) num_connections_local = &
                                 num_connections_local + 1

  allocate(explicit_grid%connections(2,num_connections_local))
  explicit_grid%connections = 0
  allocate(explicit_grid%face_areas(num_connections_local))
  explicit_grid%face_areas = 0
  allocate(explicit_grid%face_centroids(num_connections_local))
  do iconn = 1, num_connections_local
    explicit_grid%face_centroids(iconn)%x = 0.d0
    explicit_grid%face_centroids(iconn)%y = 0.d0
    explicit_grid%face_centroids(iconn)%z = 0.d0
  enddo

  ! for now, read all cells from ASCII file through io_rank and communicate
  ! to other ranks
  call OptionSetBlocking(option,PETSC_FALSE)
  if (OptionIsIORank(option)) then
    allocate(temp_real_array(6,num_connections_local_save+1))
    ! read for other processors
    do irank = 0, option%comm%size-1
      temp_real_array = UNINITIALIZED_DOUBLE
      num_to_read = num_connections_local_save
      if (irank < remainder) num_to_read = num_to_read + 1
      do iconn = 1, num_to_read
        call InputReadPflotranString(input,option)
        call InputReadStringErrorMsg(input,option,hint)
        call InputReadInt(input,option,temp_int)
        call InputErrorMsg(input,option,'cell id upwind',hint)
        temp_real_array(1,iconn) = dble(temp_int)
        call InputReadInt(input,option,temp_int)
        call InputErrorMsg(input,option,'cell id downwind',hint)
        temp_real_array(2,iconn) = dble(temp_int)
        call InputReadDouble(input,option,temp_real_array(3,iconn))
        call InputErrorMsg(input,option,'face x coordinate',hint)
        call InputReadDouble(input,option,temp_real_array(4,iconn))
        call InputErrorMsg(input,option,'face y coordinate',hint)
        call InputReadDouble(input,option,temp_real_array(5,iconn))
        call InputErrorMsg(input,option,'face z coordinate',hint)
        call InputReadDouble(input,option,temp_real_array(6,iconn))
        call InputErrorMsg(input,option,'face area',hint)
      enddo

      ! if the cells reside on io_rank
      if (OptionIsIORank(option,irank)) then
#if UGRID_DEBUG
        write(string,*) num_connections_local
        string = trim(adjustl(string)) // ' connections stored on p0'
        print *, trim(string)
#endif
        do iconn = 1, num_connections_local
          explicit_grid%connections(1,iconn) = int(temp_real_array(1,iconn))
          explicit_grid%connections(2,iconn) = int(temp_real_array(2,iconn))
          explicit_grid%face_centroids(iconn)%x = temp_real_array(3,iconn)
          explicit_grid%face_centroids(iconn)%y = temp_real_array(4,iconn)
          explicit_grid%face_centroids(iconn)%z = temp_real_array(5,iconn)
          explicit_grid%face_areas(iconn) = temp_real_array(6,iconn)
        enddo
      else
        ! otherwise communicate to other ranks
#if UGRID_DEBUG
        write(string,*) num_to_read
        write(word,*) irank
        string = trim(adjustl(string)) // ' connections sent from p0 to p' // &
                 trim(adjustl(word))
        print *, trim(string)
#endif
        int_mpi = num_to_read*6
        call MPI_Send(temp_real_array,int_mpi,MPI_DOUBLE_PRECISION,irank, &
                      num_to_read,option%mycomm,ierr);CHKERRQ(ierr)
      endif
    enddo
  else
    ! other ranks post the recv
#if UGRID_DEBUG
    write(string,*) num_connections_local
    write(word,*) option%myrank
    string = trim(adjustl(string)) // ' connections received from p0 at p' // &
              trim(adjustl(word))
    print *, trim(string)
#endif
    allocate(temp_real_array(6,num_connections_local))
    int_mpi = num_connections_local*6
    call MPI_Recv(temp_real_array,int_mpi,MPI_DOUBLE_PRECISION, &
                  option%comm%io_rank,MPI_ANY_TAG,option%mycomm,status_mpi, &
                  ierr);CHKERRQ(ierr)
    do iconn = 1, num_connections_local
      explicit_grid%connections(1,iconn) = int(temp_real_array(1,iconn))
      explicit_grid%connections(2,iconn) = int(temp_real_array(2,iconn))
      explicit_grid%face_centroids(iconn)%x = temp_real_array(3,iconn)
      explicit_grid%face_centroids(iconn)%y = temp_real_array(4,iconn)
      explicit_grid%face_centroids(iconn)%z = temp_real_array(5,iconn)
      explicit_grid%face_areas(iconn) = temp_real_array(6,iconn)
    enddo

  endif
  call OptionSetBlocking(option,PETSC_TRUE)
  call OptionCheckNonBlockingError(option)
  deallocate(temp_real_array)

  call OptionSetBlocking(option,PETSC_FALSE)
  if (OptionIsIORank(option)) then
    call InputReadPflotranString(input,option)
    ! read ELEMENTS card, we only use this for tecplot output
    ! not used while solving the PDEs
    call InputReadCard(input,option,card,PETSC_FALSE)
    word = 'ELEMENTS'
    if (StringCompare(word,card)) then
      card = 'Explicit Unstruct. Grid ELEMENTS'
      call InputReadInt(input,option,num_elems)
      call InputErrorMsg(input,option,'number of elements',card)
          explicit_grid%num_elems = num_elems
      unstructured_grid%max_nvert_per_cell = 8 ! Initial guess
      allocate(explicit_grid%cell_vertices(0:unstructured_grid% &
                                    max_nvert_per_cell,num_elems))
      do iconn = 1, num_elems
        call InputReadPflotranString(input,option)
        call InputReadStringErrorMsg(input,option,card)
        call InputReadWord(input,option,word,PETSC_TRUE)
        call InputErrorMsg(input,option,'element_type',card)
        call StringtoUpper(word)
        select case (word)
          case('H')
            num_vertices = 8
          case('W')
            num_vertices = 6
          case('P')
            num_vertices = 5
          case('T')
            num_vertices = 4
          case('Q')
            num_vertices = 4
          case('TRI')
            num_vertices = 3
        end select
        explicit_grid%cell_vertices(0,iconn) = num_vertices
        do ivertex = 1, num_vertices
          call InputReadInt(input,option, &
                            explicit_grid%cell_vertices(ivertex,iconn))
          call InputErrorMsg(input,option,'vertex id',hint)
        enddo
      enddo
      call InputReadPflotranString(input,option)
      ! read VERTICES card, not used for calcuations, only tecplot output
      call InputReadCard(input,option,card,PETSC_FALSE)
      word = 'VERTICES'
      call InputErrorMsg(input,option,word,card)
      if (.not.StringCompare(word,card)) then
        option%io_buffer = 'Unrecognized keyword "' // trim(card) // &
          '" in explicit grid file.'
        call PrintErrMsg(option)
      endif

      ! at this point, as we read the grid, the output_mesh_type is
      ! not known yet
      call InputReadInt(input,option,num_grid_vertices)

      if (InputError(input)) then
        input%ierr = INPUT_ERROR_NONE
        ! if num_grid_vertices not entered assumes vertex_centered
        ! based - default
        explicit_grid%num_vertices = explicit_grid%num_cells_global
      else
        explicit_grid%num_vertices = num_grid_vertices
      end if

      allocate(explicit_grid%vertex_coordinates(explicit_grid%num_vertices))
      do icell = 1, explicit_grid%num_vertices
        explicit_grid%vertex_coordinates(icell)%x = 0.d0
        explicit_grid%vertex_coordinates(icell)%y = 0.d0
        explicit_grid%vertex_coordinates(icell)%z = 0.d0
      enddo
      do icell = 1, explicit_grid%num_vertices
        call InputReadPflotranString(input,option)
        call InputReadStringErrorMsg(input,option,card)
        call InputReadDouble(input,option, &
                             explicit_grid%vertex_coordinates(icell)%x)
        call InputErrorMsg(input,option,'vertex 1',card)
        call InputReadDouble(input,option, &
                             explicit_grid%vertex_coordinates(icell)%y)
        call InputErrorMsg(input,option,'vertex 2',card)
        call InputReadDouble(input,option, &
                             explicit_grid%vertex_coordinates(icell)%z)
        call InputErrorMsg(input,option,'vertex 3',card)
      enddo
    endif
  endif

  if (OptionIsIORank(option)) then
    call InputDestroy(input)
  endif
  call OptionSetBlocking(option,PETSC_TRUE)
  call OptionCheckNonBlockingError(option)

end subroutine UGridExplicitRead

! ************************************************************************** !

subroutine UGridExplicitReadHDF5(unstructured_grid,filename,option)
  !
  ! This routine reads an unstructured explicit grid from HDF5.
  !
  ! Author: Moise Rousseau
  ! Date: 17/01/21
  !
  use hdf5
  use Option_module
  use HDF5_Aux_module

! 64-bit stuff
#ifdef PETSC_USE_64BIT_INDICES
#define HDF_NATIVE_INTEGER H5T_NATIVE_INTEGER
#else
#define HDF_NATIVE_INTEGER H5T_NATIVE_INTEGER
#endif

  implicit none

  type(grid_unstructured_type) :: unstructured_grid
  character(len=MAXSTRINGLENGTH) :: filename
  type(option_type) :: option

  type(unstructured_explicit_type), pointer :: explicit_grid
  character(len=MAXSTRINGLENGTH) :: group_name
  PetscMPIInt :: hdf5_err
  PetscMPIInt :: rank_mpi
  PetscInt :: istart, iend, iconn, icell
  PetscInt :: num_cells, num_connections
  PetscInt :: num_cells_local, num_cells_local_save
  PetscInt :: num_connections_local, num_connections_local_save
  PetscInt :: remainder
  PetscReal, pointer :: double_buffer(:)
  PetscReal, pointer :: double_buffer_2d(:,:)
  PetscInt, pointer :: int_buffer_2d(:,:)
  PetscErrorCode :: ierr

  integer(HID_T) :: file_id
  integer(HID_T) :: prop_id
  integer(HID_T) :: data_set_id
  integer(HID_T) :: data_space_id
  integer(HID_T) :: memory_space_id
  integer(HSIZE_T), allocatable :: dims_h5(:), max_dims_h5(:)
  integer(HSIZE_T) :: offset(2), length(2)
  integer :: ndims_h5

  explicit_grid => unstructured_grid%explicit_grid

  !Structure of a HDF5 explicit grid
  ! All of the below under a group "Domain"
  ! - a group "Cells" with dataset "Centers" and "Volumes"
  ! - a group "Faces" with dataset "id_up", "id_dn", "Centers" and "Areas"

  ! Open the file collectively
  call HDF5FileOpenReadOnly(filename,file_id,PETSC_TRUE,'',option)

  ! Open group
  group_name = "Domain/Cells"
  option%io_buffer = 'Opening group: ' // trim(group_name)
  call PrintMsg(option)

  !
  ! Domain/Cells/Volumes
  !

  ! Open dataset
  call h5dopen_f(file_id, "Domain/Cells/Volumes", data_set_id, hdf5_err)

  ! Get dataset's dataspace
  call h5dget_space_f(data_set_id, data_space_id, hdf5_err)

  ! Get number of dimensions! Get number of dimensions and check
  ! Allocate memory
  ndims_h5 = 1
  allocate(dims_h5(ndims_h5))
  allocate(max_dims_h5(ndims_h5))
  ! Get dimensions of dataset
  call h5sget_simple_extent_dims_f(data_space_id, dims_h5, max_dims_h5, &
                                   hdf5_err)

  ! Determine the number of cells each that will be saved on each processor
  num_cells = INT(dims_h5(1))
  explicit_grid%num_cells_global = num_cells
  num_cells_local = num_cells/option%comm%size
  num_cells_local_save = num_cells_local
  remainder = num_cells - &
              num_cells_local*option%comm%size
  if (option%myrank < remainder) num_cells_local = &
                                  num_cells_local + 1

  ! Find istart and iend
  istart = 0
  iend   = 0
  call MPI_Exscan(num_cells_local,istart,ONE_INTEGER_MPI,MPIU_INTEGER,MPI_SUM, &
                  option%mycomm,ierr);CHKERRQ(ierr)
  call MPI_Scan(num_cells_local,iend,ONE_INTEGER_MPI,MPIU_INTEGER,MPI_SUM, &
                option%mycomm,ierr);CHKERRQ(ierr)
  ! Determine the length and offset of data to be read by each processor
  length(1) = iend-istart
  length(2) = 0
  offset(1) = istart
  offset(2) = 0

  ! Create data space for dataset
  rank_mpi = 1
  memory_space_id = -1
  call h5screate_simple_f(rank_mpi, length, memory_space_id, hdf5_err)

  ! Select hyperslab
  call h5dget_space_f(data_set_id, data_space_id, hdf5_err)
  call h5sselect_hyperslab_f(data_space_id, H5S_SELECT_SET_F, offset, length, &
                             hdf5_err)

  ! Initialize data buffer
  allocate(double_buffer(length(1)))
  double_buffer = UNINITIALIZED_DOUBLE

  ! Create property list
  call h5pcreate_f(H5P_DATASET_XFER_F, prop_id, hdf5_err)
#ifndef SERIAL_HDF5
  call h5pset_dxpl_mpio_f(prop_id, H5FD_MPIO_COLLECTIVE_F, hdf5_err)
#endif

  ! Read the dataset collectively
  call h5dread_f(data_set_id, H5T_NATIVE_DOUBLE, double_buffer, &
                 dims_h5, hdf5_err, memory_space_id, data_space_id)

  allocate(explicit_grid%cell_volumes(num_cells_local))
  allocate(explicit_grid%cell_ids(num_cells_local))
  explicit_grid%cell_volumes = UNINITIALIZED_DOUBLE
  explicit_grid%cell_ids = UNINITIALIZED_INTEGER
  do icell = 1, num_cells_local
    explicit_grid%cell_ids(icell) = icell + int(offset(1))
    explicit_grid%cell_volumes(icell) = double_buffer(icell)
  enddo

  call HDF5DatasetClose(data_set_id,option)
  deallocate(double_buffer)
  nullify(double_buffer)
  deallocate(dims_h5)
  deallocate(max_dims_h5)

  !
  ! Domain/Cells/Centers
  !

  ! Open dataset
  call h5dopen_f(file_id, "Domain/Cells/Centers", data_set_id, hdf5_err)

  ! Get dataset's dataspace
  call h5dget_space_f(data_set_id, data_space_id, hdf5_err)

  ! Get number of dimensions and check
  call h5sget_simple_extent_ndims_f(data_space_id, ndims_h5, hdf5_err)
  if (ndims_h5 /= 2) then
    option%io_buffer='Dimension of Domain/Cells/Center dataset in ' // &
                      trim(filename) // ' is not equal to 2.'
    call PrintErrMsg(option)
  endif
  ! Allocate memory
  allocate(dims_h5(ndims_h5))
  allocate(max_dims_h5(ndims_h5))
  ! Get dimensions of dataset
  call h5sget_simple_extent_dims_f(data_space_id, dims_h5, max_dims_h5, &
                                   hdf5_err)

  ! Determine the length and offset of data to be read by each processor
  length(1) = dims_h5(1)
  length(2) = iend-istart
  offset(1) = 0
  offset(2) = istart

  ! Create data space for dataset
  rank_mpi = 2
  memory_space_id = -1
  call h5screate_simple_f(rank_mpi, length, memory_space_id, hdf5_err)

  ! Select hyperslab
  call h5dget_space_f(data_set_id, data_space_id, hdf5_err)
  call h5sselect_hyperslab_f(data_space_id, H5S_SELECT_SET_F, offset, length, &
                             hdf5_err)

  ! Initialize data buffer
  allocate(double_buffer_2d(length(1),length(2)))
  double_buffer_2d = UNINITIALIZED_DOUBLE

  ! Create property list
  call h5pcreate_f(H5P_DATASET_XFER_F, prop_id, hdf5_err)
#ifndef SERIAL_HDF5
  call h5pset_dxpl_mpio_f(prop_id, H5FD_MPIO_COLLECTIVE_F, hdf5_err)
#endif

  ! Read the dataset collectively
  call h5dread_f(data_set_id, H5T_NATIVE_DOUBLE, double_buffer_2d, &
                 dims_h5, hdf5_err, memory_space_id, data_space_id)

  allocate(explicit_grid%cell_centroids(num_cells_local))
  do icell = 1, num_cells_local
    explicit_grid%cell_centroids(icell)%x = double_buffer_2d(1,icell)
    explicit_grid%cell_centroids(icell)%y = double_buffer_2d(2,icell)
    explicit_grid%cell_centroids(icell)%z = double_buffer_2d(3,icell)
  enddo

  call HDF5DatasetClose(data_set_id,option)
  deallocate(double_buffer_2d)
  nullify(double_buffer_2d)
  deallocate(dims_h5)
  deallocate(max_dims_h5)



  !
  ! Domain/Connections/Cell Ids
  !

  ! Open group
  group_name = "Domain/Connections"
  option%io_buffer = 'Opening group: ' // trim(group_name)
  call PrintMsg(option)

  ! Open dataset
  call h5dopen_f(file_id,"Domain/Connections/Cell Ids",data_set_id,hdf5_err)

  ! Get dataset's dataspace
  call h5dget_space_f(data_set_id, data_space_id, hdf5_err)

  ! Get number of dimensions and check
  call h5sget_simple_extent_ndims_f(data_space_id, ndims_h5, hdf5_err)
  if (ndims_h5 /= 2) then
    option%io_buffer='Dimension of Domain/Connections/Cell Ids dataset in ' &
                      // trim(filename) // ' is not equal to 2.'
    call PrintErrMsg(option)
  endif
  ! Allocate memory
  allocate(dims_h5(ndims_h5))
  allocate(max_dims_h5(ndims_h5))
  ! Get dimensions of dataset
  call h5sget_simple_extent_dims_f(data_space_id, dims_h5, max_dims_h5, &
                                   hdf5_err)

  ! Determine the number of cells each that will be saved on each processor
  num_connections = INT(dims_h5(2))
  num_connections_local = num_connections/option%comm%size
  num_connections_local_save = num_connections_local
  remainder = num_connections - &
              num_connections_local*option%comm%size
  if (option%myrank < remainder) num_connections_local = &
                                  num_connections_local + 1

  ! Find istart and iend
  istart = 0
  iend   = 0
  call MPI_Exscan(num_connections_local,istart,ONE_INTEGER_MPI,MPIU_INTEGER, &
                  MPI_SUM,option%mycomm,ierr);CHKERRQ(ierr)
  call MPI_Scan(num_connections_local,iend,ONE_INTEGER_MPI,MPIU_INTEGER, &
                MPI_SUM,option%mycomm,ierr);CHKERRQ(ierr)
  ! Determine the length and offset of data to be read by each processor
  length(1) = dims_h5(1)
  length(2) = iend-istart
  offset(1) = 0
  offset(2) = istart

  ! Create data space for dataset
  rank_mpi = 2
  memory_space_id = -1
  call h5screate_simple_f(rank_mpi, length, memory_space_id, hdf5_err)

  ! Select hyperslab
  call h5dget_space_f(data_set_id, data_space_id, hdf5_err)
  call h5sselect_hyperslab_f(data_space_id, H5S_SELECT_SET_F, offset, length, &
                             hdf5_err)

  ! Initialize data buffer
  allocate(int_buffer_2d(length(1),length(2)))
  int_buffer_2d = UNINITIALIZED_INTEGER

  ! Create property list
  call h5pcreate_f(H5P_DATASET_XFER_F, prop_id, hdf5_err)
#ifndef SERIAL_HDF5
  call h5pset_dxpl_mpio_f(prop_id, H5FD_MPIO_COLLECTIVE_F, hdf5_err)
#endif

  ! Read the dataset collectively
  call h5dread_f(data_set_id, H5T_NATIVE_INTEGER, int_buffer_2d, &
                 dims_h5, hdf5_err, memory_space_id, data_space_id)

  allocate(explicit_grid%connections(2,num_connections_local))
  do iconn = 1, num_connections_local
    explicit_grid%connections(1,iconn) = int_buffer_2d(1,iconn)
    explicit_grid%connections(2,iconn) = int_buffer_2d(2,iconn)
  enddo

  call HDF5DatasetClose(data_set_id,option)
  deallocate(int_buffer_2d)
  nullify(int_buffer_2d)
  deallocate(dims_h5)
  deallocate(max_dims_h5)

  !
  ! Domain/Connections/Centers
  !

  ! Open dataset
  call h5dopen_f(file_id,"Domain/Connections/Centers",data_set_id,hdf5_err)

  ! Get dataset's dataspace
  call h5dget_space_f(data_set_id, data_space_id, hdf5_err)

  ! Get number of dimensions and check
  call h5sget_simple_extent_ndims_f(data_space_id, ndims_h5, hdf5_err)
  if (ndims_h5 /= 2) then
    option%io_buffer='Dimension of Domain/Connections/Center dataset in ' // &
                      trim(filename) // ' is not equal to 2.'
    call PrintErrMsg(option)
  endif
  ! Allocate memory
  allocate(dims_h5(ndims_h5))
  allocate(max_dims_h5(ndims_h5))
  ! Get dimensions of dataset
  call h5sget_simple_extent_dims_f(data_space_id, dims_h5, max_dims_h5, &
                                   hdf5_err)

  ! Determine the length and offset of data to be read by each processor
  length(1) = dims_h5(1)

  ! Create data space for dataset
  rank_mpi = 2
  memory_space_id = -1
  call h5screate_simple_f(rank_mpi, length, memory_space_id, hdf5_err)

  ! Select hyperslab
  call h5dget_space_f(data_set_id, data_space_id, hdf5_err)
  call h5sselect_hyperslab_f(data_space_id, H5S_SELECT_SET_F, offset, length, &
                             hdf5_err)

  ! Initialize data buffer
  allocate(double_buffer_2d(length(1),length(2)))
  double_buffer_2d = UNINITIALIZED_DOUBLE

  ! Create property list
  call h5pcreate_f(H5P_DATASET_XFER_F, prop_id, hdf5_err)
#ifndef SERIAL_HDF5
  call h5pset_dxpl_mpio_f(prop_id, H5FD_MPIO_COLLECTIVE_F, hdf5_err)
#endif

  ! Read the dataset collectively
  call h5dread_f(data_set_id, H5T_NATIVE_DOUBLE, double_buffer_2d, &
                 dims_h5, hdf5_err, memory_space_id, data_space_id)

  allocate(explicit_grid%face_centroids(num_connections_local))
  do iconn = 1, num_connections_local
    explicit_grid%face_centroids(iconn)%x = double_buffer_2d(1,iconn)
    explicit_grid%face_centroids(iconn)%y = double_buffer_2d(2,iconn)
    explicit_grid%face_centroids(iconn)%z = double_buffer_2d(3,iconn)
  enddo

  call HDF5DatasetClose(data_set_id,option)
  deallocate(double_buffer_2d)
  nullify(double_buffer_2d)
  deallocate(dims_h5)
  deallocate(max_dims_h5)

  !
  ! Domain/Connections/Areas
  !

  ! Open dataset
  call h5dopen_f(file_id,"Domain/Connections/Areas",data_set_id,hdf5_err)

  ! Get dataset's dataspace
  call h5dget_space_f(data_set_id, data_space_id, hdf5_err)

  ! Get number of dimensions and check
  call h5sget_simple_extent_ndims_f(data_space_id, ndims_h5, hdf5_err)
  if (ndims_h5 /= 1) then
    option%io_buffer='Dimension of Domain/Connections/Areas dataset in ' // &
                      trim(filename) // ' is not equal to 1.'
    call PrintErrMsg(option)
  endif
  ! Allocate memory
  allocate(dims_h5(ndims_h5))
  allocate(max_dims_h5(ndims_h5))
  ! Get dimensions of dataset
  call h5sget_simple_extent_dims_f(data_space_id, dims_h5, max_dims_h5, &
                                   hdf5_err)

  ! Determine the length and offset of data to be read by each processor
  length(1) = iend-istart
  length(2) = 0
  offset(1) = istart
  offset(2) = 0

  ! Create data space for dataset
  rank_mpi = 1
  memory_space_id = -1
  call h5screate_simple_f(rank_mpi, length, memory_space_id, hdf5_err)

  ! Select hyperslab
  call h5dget_space_f(data_set_id, data_space_id, hdf5_err)
  call h5sselect_hyperslab_f(data_space_id, H5S_SELECT_SET_F, offset, length, &
                             hdf5_err)

  ! Initialize data buffer
  allocate(double_buffer(length(1)))
  double_buffer = UNINITIALIZED_DOUBLE

  ! Create property list
  call h5pcreate_f(H5P_DATASET_XFER_F, prop_id, hdf5_err)
#ifndef SERIAL_HDF5
  call h5pset_dxpl_mpio_f(prop_id, H5FD_MPIO_COLLECTIVE_F, hdf5_err)
#endif

  ! Read the dataset collectively
  call h5dread_f(data_set_id, H5T_NATIVE_DOUBLE, double_buffer, &
                 dims_h5, hdf5_err, memory_space_id, data_space_id)

  allocate(explicit_grid%face_areas(num_connections_local))
  do iconn = 1, num_connections_local
    explicit_grid%face_areas(iconn) = double_buffer(iconn)
  enddo

  call HDF5DatasetClose(data_set_id,option)
  call HDF5FileClose(file_id,option)
  deallocate(double_buffer)
  nullify(double_buffer)
  deallocate(dims_h5)
  deallocate(max_dims_h5)


end subroutine UGridExplicitReadHDF5

! ************************************************************************** !

subroutine UGridExplicitDecompose(ugrid,option)
  !
  ! Decomposes an explicit unstructured grid across
  ! ranks
  !
  ! Author: Glenn Hammond
  ! Date: 05/17/12
  !
#include "petsc/finclude/petscdm.h"
  use petscdm
  use Option_module
  use Utility_module, only: ReallocateArray, SearchOrderedArray
  use String_module

  implicit none

  type(grid_unstructured_type) :: ugrid
  type(option_type) :: option

  type(unstructured_explicit_type), pointer :: explicit_grid

  Mat :: M_mat,M_mat_loc
  Vec :: M_vec
  Mat :: Adj_mat
  Mat :: Dual_mat
  IS :: is_new
  IS :: is_scatter
  IS :: is_gather
  PetscInt :: num_cells_local_new, num_cells_local_old
  Vec :: cells_old, cells_local
  Vec :: connections_old, connections_local
  VecScatter :: vec_scatter

  PetscInt :: global_offset_old
  PetscInt :: ghosted_id
  PetscInt, allocatable :: local_connections(:), local_connection_offsets(:)
  PetscInt, allocatable :: local_connections2(:), local_connection_offsets2(:)
  PetscInt, allocatable :: int_array(:), int_array2(:), int_array3(:)
  PetscInt, allocatable :: int_array4(:)
  PetscInt, allocatable :: int_array2d(:,:)
  PetscInt :: num_connections_local_old, num_connections_local
  PetscInt :: num_connections_total
  PetscInt :: num_connections_global, global_connection_offset
  PetscInt :: id_up, id_dn, iconn, icell, count, offset
  PetscInt :: conn_id
  PetscBool :: found
  PetscInt :: i, temp_int
  PetscReal :: temp_real

  PetscBool :: success
  PetscInt, pointer :: ia_ptr(:), ja_ptr(:)
  PetscInt, pointer :: ia_ptr2(:), ja_ptr2(:)
  PetscReal, pointer :: vec_ptr(:)
  PetscInt :: num_rows, num_cols, istart, iend, icol
  PetscInt :: cell_stride, dual_offset, connection_offset, connection_stride
  PetscInt :: natural_id_offset
  PetscErrorCode :: ierr
  PetscInt :: icell_up,icell_dn

  character(len=MAXSTRINGLENGTH) :: string
#if UGRID_DEBUG
  PetscViewer :: viewer
  call PrintMsg(option,'Adjacency matrix')
#endif

  explicit_grid => ugrid%explicit_grid

  temp_int = minval(explicit_grid%cell_ids)
  call MPI_Allreduce(MPI_IN_PLACE,temp_int,ONE_INTEGER_MPI,MPIU_INTEGER, &
                     MPI_MIN,option%mycomm,ierr);CHKERRQ(ierr)
  if (temp_int < 1) then
    option%io_buffer = 'The minimum cell ID (' // &
      trim(StringWrite(temp_int)) // ') in the explicit unstructured &
      &grid used is less than 1. Grid cell IDs must be contiguous &
      &starting at 1.'
    call PrintErrMsg(option)
  endif

  num_cells_local_old = size(explicit_grid%cell_ids)

  call MPI_Allreduce(num_cells_local_old,ugrid%nmax,ONE_INTEGER_MPI, &
                     MPIU_INTEGER,MPI_SUM,option%mycomm,ierr);CHKERRQ(ierr)

  ! determine the global offset from 0 for cells on this rank
  global_offset_old = 0
  call MPI_Exscan(num_cells_local_old,global_offset_old,ONE_INTEGER_MPI, &
                  MPIU_INTEGER,MPI_SUM,option%mycomm,ierr);CHKERRQ(ierr)

  num_connections_local_old = size(explicit_grid%connections,2)

  call MPI_Allreduce(num_connections_local_old,num_connections_global, &
                     ONE_INTEGER_MPI,MPIU_INTEGER,MPI_SUM,option%mycomm, &
                     ierr);CHKERRQ(ierr)

  global_connection_offset = 0
  call MPI_Exscan(num_connections_local_old,global_connection_offset, &
                  ONE_INTEGER_MPI,MPIU_INTEGER,MPI_SUM,option%mycomm, &
                  ierr);CHKERRQ(ierr)

  call VecCreateMPI(option%mycomm,num_cells_local_old,ugrid%nmax,M_vec, &
                    ierr);CHKERRQ(ierr)
  call VecZeroEntries(M_vec,ierr);CHKERRQ(ierr)
  do iconn = 1, num_connections_local_old
    do i = 1, 2
      icell = explicit_grid%connections(i,iconn)-1
      call VecSetValue(M_vec,icell,1.d0,ADD_VALUES,ierr);CHKERRQ(ierr)
    enddo
  enddo
  call VecAssemblyBegin(M_vec,ierr);CHKERRQ(ierr)
  call VecAssemblyEnd(M_vec,ierr);CHKERRQ(ierr)

#if UGRID_DEBUG
  call PetscViewerASCIIOpen(option%mycomm,'M_vec.out',viewer, &
                            ierr);CHKERRQ(ierr)
  call VecView(M_vec,viewer,ierr);CHKERRQ(ierr)
  call PetscViewerDestroy(viewer,ierr);CHKERRQ(ierr)
#endif

  call VecMax(M_vec,PETSC_NULL_INTEGER,temp_real,ierr);CHKERRQ(ierr)
  call VecDestroy(M_vec,ierr);CHKERRQ(ierr)
  ugrid%max_ndual_per_cell = int(temp_real+0.1d0)
  call MatCreateAIJ(option%mycomm,num_cells_local_old,PETSC_DECIDE,ugrid%nmax, &
                    num_connections_global,ugrid%max_ndual_per_cell, &
                    PETSC_NULL_INTEGER,ugrid%max_ndual_per_cell, &
                    PETSC_NULL_INTEGER,M_mat,ierr);CHKERRQ(ierr)
  call MatZeroEntries(M_mat,ierr);CHKERRQ(ierr)
  do iconn = 1, num_connections_local_old
    temp_int = iconn+global_connection_offset-1
    do i = 1, 2
      icell = explicit_grid%connections(i,iconn)-1
      call MatSetValue(M_mat,icell,temp_int,1.d0,INSERT_VALUES, &
                       ierr);CHKERRQ(ierr)
    enddo
  enddo
  call MatAssemblyBegin(M_mat,MAT_FINAL_ASSEMBLY,ierr);CHKERRQ(ierr)
  call MatAssemblyEnd(M_mat,MAT_FINAL_ASSEMBLY,ierr);CHKERRQ(ierr)

#if UGRID_DEBUG
  call PetscViewerASCIIOpen(option%mycomm,'M_mat.out',viewer, &
                            ierr);CHKERRQ(ierr)
  call MatView(M_mat,viewer,ierr);CHKERRQ(ierr)
  call PetscViewerDestroy(viewer,ierr);CHKERRQ(ierr)
#endif

  ! GB: When MatConvert() is used, the diagonal entries are lost in Adj_mat
  !call MatConvert(M_mat,MATMPIADJ,MAT_INITIAL_MATRIX,Adj_mat,ierr)
  !call MatDestroy(M_mat,ierr)

  ! Alternate method of creating Adj_mat
  if (option%comm%size>1) then
    call MatMPIAIJGetLocalMat(M_mat,MAT_INITIAL_MATRIX,M_mat_loc, &
                              ierr);CHKERRQ(ierr)
    call MatGetRowIJF90(M_mat_loc,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE, &
                        num_rows,ia_ptr,ja_ptr,success,ierr);CHKERRQ(ierr)
  else
    call MatGetRowIJF90(M_mat,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE,num_rows, &
                        ia_ptr,ja_ptr,success,ierr);CHKERRQ(ierr)
  endif

  count=0
  do icell = 1,num_rows
    istart = ia_ptr(icell)
    iend = ia_ptr(icell+1)-1
    num_cols = iend-istart+1
    count = count+num_cols
  enddo
  allocate(local_connections(count))
  allocate(local_connection_offsets(num_rows+1))
  local_connection_offsets(1:num_rows+1) = ia_ptr(1:num_rows+1)
  local_connections(1:count)             = ja_ptr(1:count)

  call MatCreateMPIAdj(option%mycomm,num_cells_local_old, &
                       num_connections_global,local_connection_offsets, &
                       local_connections,PETSC_NULL_INTEGER,Adj_mat, &
                       ierr);CHKERRQ(ierr)

  if (option%comm%size>1) then
    call MatRestoreRowIJF90(M_mat_loc,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE, &
                            num_rows,ia_ptr,ja_ptr,success, &
                            ierr);CHKERRQ(ierr)
  else
    call MatRestoreRowIJF90(M_mat,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE, &
                            num_rows,ia_ptr,ja_ptr,success, &
                            ierr);CHKERRQ(ierr)
  endif
  call MatDestroy(M_mat,ierr);CHKERRQ(ierr)

#if UGRID_DEBUG
  call PetscViewerASCIIOpen(option%mycomm,'Adj.out',viewer, &
                            ierr);CHKERRQ(ierr)
  call MatView(Adj_mat,viewer,ierr);CHKERRQ(ierr)
  call PetscViewerDestroy(viewer,ierr);CHKERRQ(ierr)
#endif
!  call PrintErrMsg(option,'debugg')

  ! Create the Dual matrix.
  call MatCreateAIJ(option%mycomm,num_cells_local_old,PETSC_DECIDE,ugrid%nmax, &
                    ugrid%nmax,ugrid%max_ndual_per_cell,PETSC_NULL_INTEGER, &
                    ugrid%max_ndual_per_cell,PETSC_NULL_INTEGER,M_mat, &
                    ierr);CHKERRQ(ierr)
  do iconn = 1, num_connections_local_old
    icell_up = explicit_grid%connections(1,iconn)-1
    icell_dn = explicit_grid%connections(2,iconn)-1
    call MatSetValue(M_mat,icell_up,icell_dn,1.d0,INSERT_VALUES, &
                     ierr);CHKERRQ(ierr)
    call MatSetValue(M_mat,icell_dn,icell_up,1.d0,INSERT_VALUES, &
                     ierr);CHKERRQ(ierr)
  enddo

  call MatAssemblyBegin(M_mat,MAT_FINAL_ASSEMBLY,ierr);CHKERRQ(ierr)
  call MatAssemblyEnd(M_mat,MAT_FINAL_ASSEMBLY,ierr);CHKERRQ(ierr)

  !call MatConvert(M_mat,MATMPIADJ,MAT_INITIAL_MATRIX,Dual_mat,ierr)
  !call MatDestroy(M_mat,ierr)

  ! Alternate method of creating Dual_mat
  if (option%comm%size>1) then
    call MatMPIAIJGetLocalMat(M_mat,MAT_INITIAL_MATRIX,M_mat_loc, &
                              ierr);CHKERRQ(ierr)
    call MatGetRowIJF90(M_mat_loc,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE, &
                        num_rows,ia_ptr,ja_ptr,success,ierr);CHKERRQ(ierr)
  else
    call MatGetRowIJF90(M_mat,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE,num_rows, &
                        ia_ptr,ja_ptr,success,ierr);CHKERRQ(ierr)
  endif

  count=0
  do icell = 1,num_rows
    istart = ia_ptr(icell)
    iend = ia_ptr(icell+1)-1
    num_cols = iend-istart+1
    count = count+num_cols
  enddo
  allocate(local_connections2(count))
  allocate(local_connection_offsets2(num_rows+1))
  local_connection_offsets2(1:num_rows+1) = ia_ptr(1:num_rows+1)
  local_connections2(1:count)             = ja_ptr(1:count)

  call MatCreateMPIAdj(option%mycomm,num_cells_local_old,ugrid%nmax, &
                       local_connection_offsets2,local_connections2, &
                       PETSC_NULL_INTEGER,Dual_mat,ierr);CHKERRQ(ierr)

  if (option%comm%size>1) then
    call MatRestoreRowIJF90(M_mat_loc,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE, &
                            num_rows,ia_ptr,ja_ptr,success, &
                            ierr);CHKERRQ(ierr)
  else
    call MatRestoreRowIJF90(M_mat,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE, &
                            num_rows,ia_ptr,ja_ptr,success, &
                            ierr);CHKERRQ(ierr)
  endif
  call MatDestroy(M_mat,ierr);CHKERRQ(ierr)

#if UGRID_DEBUG
  call PetscViewerASCIIOpen(option%mycomm,'Dual_mat.out',viewer, &
                            ierr);CHKERRQ(ierr)
  call MatView(Dual_mat,viewer,ierr);CHKERRQ(ierr)
  call PetscViewerDestroy(viewer,ierr);CHKERRQ(ierr)
#endif

  call UGridPartition(ugrid,option,Dual_mat,is_new, &
                      num_cells_local_new)

  ! second argument of ZERO_INTEGER means to use 0-based indexing
  ! MagGetRowIJF90 returns row and column pointers for compressed matrix data
  call MatGetRowIJF90(Dual_mat,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE,num_rows, &
                      ia_ptr,ja_ptr,success,ierr);CHKERRQ(ierr)

  if (.not.success .or. num_rows /= num_cells_local_old) then
    print *, option%myrank, num_rows, success, num_cells_local_old
    option%io_buffer = 'Error getting IJ row indices from dual matrix'
    call PrintErrMsg(option)
  endif

  call MatRestoreRowIJF90(Dual_mat,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE, &
                          num_rows,ia_ptr,ja_ptr,success,ierr);CHKERRQ(ierr)

  ! in order to redistributed cell/connection data among ranks, I package it
  ! in a crude way within a strided petsc vec and pass it.  The stride
  ! determines the size of each cells "packaged" data
  connection_offset = 6 + 1 ! +1 for -777
  dual_offset = connection_offset + ugrid%max_ndual_per_cell + 1 ! +1 for -888
  cell_stride = dual_offset + ugrid%max_ndual_per_cell + 1 ! +1 for -999999
  natural_id_offset = 2

  ! Information for each cell is packed in a strided petsc vec
  ! The information is ordered within each stride as follows:
  ! -cell_N   ! global cell id (negative indicates 1-based)
  ! natural cell id
  ! cell x coordinate
  ! cell y coordinate
  ! cell z coordinate
  ! cell volume
  ! -777      ! separator between cell info and connection info
  ! conn1     ! connection ids between cell_N and others
  ! conn1
  ! ...
  ! connN
  ! -888      ! separator between connection info and dual ids
  ! dual1     ! dual ids between cell_N and others
  ! dual2
  ! ...
  ! dualN
  ! -999999   ! separator indicating end of information for cell_N

  ! the purpose of -888, and -999999 is to allow one to use cells of
  ! various geometry.

  call UGridCreateOldVec(ugrid,option,cells_old, &
                         num_cells_local_old, &
                         is_new,is_scatter,cell_stride)

  ! 0 = 0-based indexing
  ! MagGetRowIJF90 returns row and column pointers for compressed matrix data
  ! pointers to Dual mat
  call MatGetRowIJF90(Dual_mat,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE,num_rows, &
                      ia_ptr,ja_ptr,success,ierr);CHKERRQ(ierr)
  ! pointers to Adj mat
  call MatGetRowIJF90(Adj_mat,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE,temp_int, &
                      ia_ptr2,ja_ptr2,success,ierr);CHKERRQ(ierr)

  if (num_rows /= temp_int) then
    write(string,*) num_rows, temp_int
    option%io_buffer = 'Number of rows in Adj and Dual matrices inconsistent:'
    option%io_buffer = trim(option%io_buffer) // trim(adjustl(string))
    call PrintErrMsgByRank(option)
  endif

  call VecGetArrayF90(cells_old,vec_ptr,ierr);CHKERRQ(ierr)
  count = 0
  do icell = 1, num_cells_local_old
    count = count + 1
    ! set global cell id
    ! negate to indicate cell id with 1-based numbering (-0 = 0)
    vec_ptr(count) = -(global_offset_old+icell)
    count = count + 1
    vec_ptr(count) = explicit_grid%cell_ids(icell)
    count = count + 1
    vec_ptr(count) = explicit_grid%cell_centroids(icell)%x
    count = count + 1
    vec_ptr(count) = explicit_grid%cell_centroids(icell)%y
    count = count + 1
    vec_ptr(count) = explicit_grid%cell_centroids(icell)%z
    count = count + 1
    vec_ptr(count) = explicit_grid%cell_volumes(icell)
    ! add the separator
    count = count + 1
    vec_ptr(count) = -777  ! help differentiate

    ! add the connection ids
    istart = ia_ptr2(icell)
    iend = ia_ptr2(icell+1)-1
    num_cols = iend-istart+1
    if (num_cols > ugrid%max_ndual_per_cell) then
      option%io_buffer = &
        'Number of columns in Adj matrix is larger then max_ndual_per_cell.'
      call PrintErrMsgByRank(option)
    endif
    do icol = 1, ugrid%max_ndual_per_cell
      count = count + 1
      if (icol <= num_cols) then
        ! increment for 1-based ordering
        vec_ptr(count) = ja_ptr2(icol+istart) + 1
      else
        vec_ptr(count) = 0
      endif
    enddo

    ! add the separator
    count = count + 1
    vec_ptr(count) = -888  ! help differentiate

    ! add the dual ids
    istart = ia_ptr(icell)
    iend = ia_ptr(icell+1)-1
    num_cols = iend-istart+1
    if (num_cols > ugrid%max_ndual_per_cell) then
      option%io_buffer = &
        'Number of columns in Dual matrix is larger then max_ndual_per_cell.'
      call PrintErrMsgByRank(option)
    endif
    do icol = 1, ugrid%max_ndual_per_cell
      count = count + 1
      if (icol <= num_cols) then
        ! increment for 1-based ordering
        vec_ptr(count) = ja_ptr(icol+istart) + 1
      else
        vec_ptr(count) = 0
      endif
    enddo

    ! final separator
    count = count + 1
    vec_ptr(count) = -999999  ! help differentiate
  enddo
  call VecRestoreArrayF90(cells_old,vec_ptr,ierr);CHKERRQ(ierr)

  ! pointers to Dual mat
  call MatRestoreRowIJF90(Dual_mat,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE, &
                          num_rows,ia_ptr,ja_ptr,success,ierr);CHKERRQ(ierr)
  ! pointers to Adj mat
  call MatRestoreRowIJF90(Adj_mat,ZERO_INTEGER,PETSC_FALSE,PETSC_FALSE, &
                          temp_int,ia_ptr2,ja_ptr2,success, &
                          ierr);CHKERRQ(ierr)
  call MatDestroy(Dual_mat,ierr);CHKERRQ(ierr)
  call MatDestroy(Adj_mat,ierr);CHKERRQ(ierr)
  deallocate(local_connections)
  deallocate(local_connection_offsets)
  deallocate(local_connections2)
  deallocate(local_connection_offsets2)


  ! is_scatter is destroyed within UGridNaturalToPetsc
  call UGridNaturalToPetsc(ugrid,option, &
                           cells_old,cells_local, &
                           num_cells_local_new,cell_stride,dual_offset, &
                           natural_id_offset,is_scatter)

  call VecDestroy(cells_old,ierr);CHKERRQ(ierr)

  ! set up connections
  connection_stride = 8
  ! create strided vector with the old connection distribution
  call VecCreate(option%mycomm,connections_old,ierr);CHKERRQ(ierr)
  call VecSetSizes(connections_old, &
                   connection_stride*num_connections_local_old,PETSC_DECIDE, &
                   ierr);CHKERRQ(ierr)
  call VecSetFromOptions(connections_old,ierr);CHKERRQ(ierr)

  call VecGetArrayF90(connections_old,vec_ptr,ierr);CHKERRQ(ierr)
  do iconn = 1, num_connections_local_old
    offset = (iconn-1)*connection_stride
    vec_ptr(offset+1) = explicit_grid%connections(1,iconn)
    vec_ptr(offset+2) = explicit_grid%connections(2,iconn)
    vec_ptr(offset+3) = explicit_grid%face_centroids(iconn)%x
    vec_ptr(offset+4) = explicit_grid%face_centroids(iconn)%y
    vec_ptr(offset+5) = explicit_grid%face_centroids(iconn)%z
    vec_ptr(offset+6) = explicit_grid%face_areas(iconn)
    vec_ptr(offset+7) = 1.d0 ! flag for local connections
    vec_ptr(offset+8) = -888.d0
  enddo
  call VecRestoreArrayF90(connections_old,vec_ptr,ierr);CHKERRQ(ierr)


#if UGRID_DEBUG
  call PetscViewerASCIIOpen(option%mycomm,'connections_old.out',viewer, &
                            ierr);CHKERRQ(ierr)
  call VecView(connections_old,viewer,ierr);CHKERRQ(ierr)
  call PetscViewerDestroy(viewer,ierr);CHKERRQ(ierr)
#endif

  ! count the number of cells and their duals
  call VecGetArrayF90(cells_local,vec_ptr,ierr);CHKERRQ(ierr)
  count = 0
  do ghosted_id=1, ugrid%ngmax
    do iconn = 1, ugrid%max_ndual_per_cell
      conn_id = int(vec_ptr(iconn + connection_offset + (ghosted_id-1)*cell_stride))
      if (conn_id < 1) exit ! here we hit the 0 at the end of last connection
      ! yes, we will be counting them twice
      count = count + 1
    enddo
  enddo
  num_connections_total = count ! many of these are redundant and will be removed
  ! allocate and fill an array with the natural cell and dual ids
  allocate(int_array(num_connections_total))
  count = 0
  do ghosted_id=1, ugrid%ngmax
    do iconn = 1, ugrid%max_ndual_per_cell
      conn_id = int(vec_ptr(iconn + connection_offset + (ghosted_id-1)*cell_stride))
      if (conn_id < 1) exit ! again we hit the 0
      count = count + 1
      int_array(count) = conn_id
    enddo
  enddo
  call VecRestoreArrayF90(cells_local,vec_ptr,ierr);CHKERRQ(ierr)

  allocate(int_array2(num_connections_total))
  do iconn = 1, num_connections_total
    int_array2(iconn) = iconn
  enddo

  ! sort connections - int_array2 will return the reordering while int_array
  !                    remains the same.
  int_array2 = int_array2 - 1
  call PetscSortIntWithPermutation(num_connections_total,int_array,int_array2, &
                                   ierr);CHKERRQ(ierr)
  int_array2 = int_array2 + 1

  ! permute, remove duplicate connections, and renumber to local ordering
  allocate(int_array3(num_connections_total))
  allocate(int_array4(num_connections_total))
  int_array3 = UNINITIALIZED_INTEGER
  int_array4 = UNINITIALIZED_INTEGER
  int_array3(1) = int_array(int_array2(1))
  count = 1
  do iconn = 1, num_connections_total
    if (int_array(int_array2(iconn)) > int_array3(count)) then
      count = count + 1
      int_array3(count) = int_array(int_array2(iconn))
    endif
    int_array4(int_array2(iconn)) = count
  enddo
  deallocate(int_array)
  deallocate(int_array2)

  num_connections_local = count ! new compressed count
  allocate(int_array(num_connections_local))
  int_array = int_array3(1:num_connections_local)
  deallocate(int_array3)

  ! replace original connections ids (naturally numbered) with locally
  ! numbered connection ids (int_array4)
  call VecGetArrayF90(cells_local,vec_ptr,ierr);CHKERRQ(ierr)
  count = 0
  do ghosted_id=1, ugrid%ngmax
    do iconn = 1, ugrid%max_ndual_per_cell
      conn_id = int(vec_ptr(iconn + connection_offset + (ghosted_id-1)*cell_stride))
      if (conn_id < 1) exit ! again we hit the 0
      count = count + 1
      vec_ptr(iconn + connection_offset + (ghosted_id-1)*cell_stride) = &
        int_array4(count)
    enddo
  enddo
  deallocate(int_array4)
  call VecRestoreArrayF90(cells_local,vec_ptr,ierr);CHKERRQ(ierr)

  ! check to ensure that the number before/after are consistent
  if (count /= num_connections_total) then
    write(string,'(2i6)') count, num_connections_total
    option%io_buffer = 'Inconsistent values for num_connections_total: ' // &
      trim(adjustl(string))
    call PrintErrMsgByRank(option)
  endif
  num_connections_total = UNINITIALIZED_INTEGER ! set to uninitialized value to catch bugs

  call VecCreate(PETSC_COMM_SELF,connections_local,ierr);CHKERRQ(ierr)
  call VecSetSizes(connections_local,num_connections_local*connection_stride, &
                   PETSC_DECIDE,ierr);CHKERRQ(ierr)
  call VecSetBlockSize(connections_local,connection_stride, &
                       ierr);CHKERRQ(ierr)
  call VecSetFromOptions(connections_local,ierr);CHKERRQ(ierr)

  int_array = int_array-1
  call ISCreateBlock(option%mycomm,connection_stride,num_connections_local, &
                     int_array,PETSC_COPY_VALUES,is_scatter, &
                     ierr);CHKERRQ(ierr)
  do iconn = 1, num_connections_local
    int_array(iconn) = iconn-1
  enddo
  call ISCreateBlock(option%mycomm,connection_stride,num_connections_local, &
                     int_array,PETSC_COPY_VALUES,is_gather, &
                     ierr);CHKERRQ(ierr)
  deallocate(int_array)

#if UGRID_DEBUG
  call PetscViewerASCIIOpen(option%mycomm,'is_scatter_conn_old_to_local.out', &
                            viewer,ierr);CHKERRQ(ierr)
  call ISView(is_scatter,viewer,ierr);CHKERRQ(ierr)
  call PetscViewerDestroy(viewer,ierr);CHKERRQ(ierr)
  call PetscViewerASCIIOpen(option%mycomm,'is_gather_conn_old_to_local.out', &
                            viewer,ierr);CHKERRQ(ierr)
  call ISView(is_gather,viewer,ierr);CHKERRQ(ierr)
  call PetscViewerDestroy(viewer,ierr);CHKERRQ(ierr)
  call PrintMsg(option,'Scatter/gathering local connection info')
#endif

  ! scatter all the connection data from the old to local
  call VecScatterCreate(connections_old,is_scatter,connections_local, &
                        is_gather,vec_scatter,ierr);CHKERRQ(ierr)
  call ISDestroy(is_gather,ierr);CHKERRQ(ierr)
  call ISDestroy(is_scatter,ierr);CHKERRQ(ierr)
  call VecScatterBegin(vec_scatter,connections_old,connections_local, &
                       INSERT_VALUES,SCATTER_FORWARD,ierr);CHKERRQ(ierr)
  call VecScatterEnd(vec_scatter,connections_old,connections_local, &
                     INSERT_VALUES,SCATTER_FORWARD,ierr);CHKERRQ(ierr)
  call VecScatterDestroy(vec_scatter,ierr);CHKERRQ(ierr)


#if UGRID_DEBUG
  write(string,*) option%myrank
  string = 'connections_local_nat' // trim(adjustl(string)) // '.out'
  call PetscViewerASCIIOpen(PETSC_COMM_SELF,trim(string),viewer, &
                            ierr);CHKERRQ(ierr)
  call VecView(connections_local,viewer,ierr);CHKERRQ(ierr)
  call PetscViewerDestroy(viewer,ierr);CHKERRQ(ierr)
#endif

  ! loop over cells and change the natural ids in the duals to local ids
  allocate(int_array2d(2,num_connections_local))
  int_array2d = UNINITIALIZED_INTEGER
  call VecGetArrayF90(cells_local,vec_ptr,ierr);CHKERRQ(ierr)
  do ghosted_id=1, ugrid%ngmax
    do iconn = 1, ugrid%max_ndual_per_cell
      ! this connection id is now local
      conn_id = int(vec_ptr(iconn + connection_offset + (ghosted_id-1)*cell_stride))
      if (conn_id < 1) exit ! again we hit the 0
      do i = 1, 2
        if (int_array2d(i,conn_id) <= UNINITIALIZED_INTEGER) then
          int_array2d(i,conn_id) = ghosted_id
          exit
        endif
        if (i > 2) then
          write(string,'(2i5)') ghosted_id, conn_id
          option%io_buffer = 'Too many local cells match connection: ' // &
            trim(adjustl(string))
          call PrintErrMsgByRank(option)
        endif
      enddo
    enddo
  enddo
  call VecRestoreArrayF90(cells_local,vec_ptr,ierr);CHKERRQ(ierr)

  ! map natural ids in connections to local ids
  ! negate connection ids as a flag
  int_array2d = -1*int_array2d
  call VecGetArrayF90(connections_local,vec_ptr,ierr);CHKERRQ(ierr)
  do iconn = 1, num_connections_local
    offset = connection_stride*(iconn-1)
    ! all values should be negative at this point, unless uninitialized
    if (maxval(int_array2d(:,iconn)) >= 999) then
      ! connection is between two ghosted cells
      vec_ptr(offset+7) = 0.d0
      cycle
    endif
    id_up = int(vec_ptr(offset+1)) ! this is the natural id
    id_dn = int(vec_ptr(offset+2))
    count = 0
    found = PETSC_FALSE
    do i = 1, 2
      if (ugrid%cell_ids_natural(abs(int_array2d(i,iconn))) == id_up) then
        int_array2d(i,iconn) = abs(int_array2d(i,iconn))
        found = PETSC_TRUE
        count = count + 1
      endif
      if (ugrid%cell_ids_natural(abs(int_array2d(i,iconn))) == id_dn) then
        int_array2d(i,iconn) = abs(int_array2d(i,iconn))
        found = PETSC_TRUE
        count = count - 1
      endif
    enddo
    ! count should be zero, meaning it found the upwind and downwind cell
    ! ids
    if (count /= 0 .or. .not.found) then
      write(string,*) iconn, id_up, id_dn
      if (.not.found) then
        option%io_buffer = 'upwind/downwind cells not found: '
      else if (count < 0) then
        option%io_buffer = 'upwind cell not found: '
      else
        option%io_buffer = 'downwind cell not found: '
      endif
      option%io_buffer = trim(option%io_buffer) // trim(adjustl(string))
      call PrintErrMsgByRank(option)
    endif
    id_up = int_array2d(1,iconn)
    id_dn = int_array2d(2,iconn)
    if (id_up < id_dn) then
      vec_ptr(offset+1) = id_up  ! now local ids
      vec_ptr(offset+2) = id_dn
    else
      vec_ptr(offset+1) = id_dn
      vec_ptr(offset+2) = id_up
    endif
    if (id_up > ugrid%nlmax .and. id_dn > ugrid%nlmax) then
      ! connection is between two ghosted cells
      vec_ptr(offset+7) = 0.d0
    endif
  enddo
  call VecRestoreArrayF90(connections_local,vec_ptr,ierr);CHKERRQ(ierr)
  deallocate(int_array2d)

#if UGRID_DEBUG
  write(string,*) option%myrank
  string = 'connections_local_loc' // trim(adjustl(string)) // '.out'
  call PetscViewerASCIIOpen(PETSC_COMM_SELF,trim(string),viewer, &
                            ierr);CHKERRQ(ierr)
  call VecView(connections_local,viewer,ierr);CHKERRQ(ierr)
  call PetscViewerDestroy(viewer,ierr);CHKERRQ(ierr)
#endif

  ! deallocate/allocate grid cell info locally
  deallocate(explicit_grid%cell_ids)
  deallocate(explicit_grid%cell_volumes)
  deallocate(explicit_grid%cell_centroids)

  allocate(explicit_grid%cell_ids(ugrid%ngmax))
  explicit_grid%cell_ids = UNINITIALIZED_INTEGER
  allocate(explicit_grid%cell_volumes(ugrid%ngmax))
  explicit_grid%cell_volumes = UNINITIALIZED_DOUBLE
  allocate(explicit_grid%cell_centroids(ugrid%ngmax))
  do icell = 1, ugrid%ngmax
    explicit_grid%cell_centroids(icell)%x = UNINITIALIZED_DOUBLE
    explicit_grid%cell_centroids(icell)%y = UNINITIALIZED_DOUBLE
    explicit_grid%cell_centroids(icell)%z = UNINITIALIZED_DOUBLE
  enddo

  call VecGetArrayF90(cells_local,vec_ptr,ierr);CHKERRQ(ierr)
  do ghosted_id=1, ugrid%ngmax
    offset = cell_stride*(ghosted_id-1)
    explicit_grid%cell_ids(ghosted_id) = int(vec_ptr(offset + 2))
    explicit_grid%cell_centroids(ghosted_id)%x = vec_ptr(offset + 3)
    explicit_grid%cell_centroids(ghosted_id)%y = vec_ptr(offset + 4)
    explicit_grid%cell_centroids(ghosted_id)%z = vec_ptr(offset + 5)
    explicit_grid%cell_volumes(ghosted_id) = vec_ptr(offset + 6)
  enddo
  call VecRestoreArrayF90(cells_local,vec_ptr,ierr);CHKERRQ(ierr)

#if UGRID_DEBUG
  write(string,*) option%myrank
  string = 'cells_local_raw' // trim(adjustl(string)) // '.out'
  open(unit=86,file=trim(string))
  do ghosted_id = 1, ugrid%ngmax
    write(86,'(i5,4f10.3)') explicit_grid%cell_ids(ghosted_id), &
                explicit_grid%cell_centroids(ghosted_id)%x, &
                explicit_grid%cell_centroids(ghosted_id)%y, &
                explicit_grid%cell_centroids(ghosted_id)%z, &
                explicit_grid%cell_volumes(ghosted_id)
  enddo
  close(86)
#endif

  ! deallocate/allocate connection info locally
  deallocate(explicit_grid%connections)
  deallocate(explicit_grid%face_areas)
  deallocate(explicit_grid%face_centroids)

  count = 0
  call VecGetArrayF90(connections_local,vec_ptr,ierr);CHKERRQ(ierr)
  do iconn = 1, num_connections_local
    offset = connection_stride*(iconn-1)
    if (vec_ptr(offset+7) > 0.1d0) count = count + 1
  enddo
  call VecRestoreArrayF90(connections_local,vec_ptr,ierr);CHKERRQ(ierr)

  allocate(explicit_grid%connections(2,count))
  explicit_grid%connections = 0
  allocate(explicit_grid%face_areas(count))
  explicit_grid%face_areas = 0
  allocate(explicit_grid%face_centroids(count))
  do iconn = 1, count
    explicit_grid%face_centroids(iconn)%x = 0.d0
    explicit_grid%face_centroids(iconn)%y = 0.d0
    explicit_grid%face_centroids(iconn)%z = 0.d0
  enddo
  call VecGetArrayF90(connections_local,vec_ptr,ierr);CHKERRQ(ierr)
  count = 0
  do iconn = 1, num_connections_local
    offset = connection_stride*(iconn-1)
    if (vec_ptr(offset+7) > 0.1d0) then
      count = count + 1
      explicit_grid%connections(1,count) = int(vec_ptr(offset+1))
      explicit_grid%connections(2,count) = int(vec_ptr(offset+2))
      explicit_grid%face_centroids(count)%x = vec_ptr(offset+3)
      explicit_grid%face_centroids(count)%y = vec_ptr(offset+4)
      explicit_grid%face_centroids(count)%z = vec_ptr(offset+5)
      explicit_grid%face_areas(count) = vec_ptr(offset+6)
    endif
  enddo
  call VecRestoreArrayF90(connections_local,vec_ptr,ierr);CHKERRQ(ierr)
  num_connections_local = count

#if UGRID_DEBUG
  write(string,*) option%myrank
  string = 'connections_local_raw' // trim(adjustl(string)) // '.out'
  open(unit=86,file=trim(string))
  do iconn = 1, num_connections_local
    write(86,'(2i5,4f7.3)') explicit_grid%connections(1,iconn), &
                explicit_grid%connections(2,iconn), &
                explicit_grid%face_centroids(iconn)%x, &
                explicit_grid%face_centroids(iconn)%y, &
                explicit_grid%face_centroids(iconn)%z, &
                explicit_grid%face_areas(iconn)
  enddo
  close(86)
#endif

  call VecDestroy(connections_old,ierr);CHKERRQ(ierr)
  call VecDestroy(connections_local,ierr);CHKERRQ(ierr)
  call VecDestroy(cells_local,ierr);CHKERRQ(ierr)

end subroutine UGridExplicitDecompose

! ************************************************************************** !

subroutine UGridExplicitSetCellCentroids(explicit_grid,x,y,z, &
                                         x_min,x_max,y_min,y_max,z_min,z_max)
  !
  ! Sets the centroid of each grid cell
  !
  ! Author: Glenn Hammond
  ! Date: 05/17/12
  !

  use Option_module

  implicit none

  type(unstructured_explicit_type) :: explicit_grid
  PetscReal :: x(:), y(:), z(:)
  PetscReal :: x_min, x_max, y_min, y_max, z_min, z_max

  PetscInt :: icell

  do icell = 1, size(explicit_grid%cell_centroids)
    x(icell) = explicit_grid%cell_centroids(icell)%x
    y(icell) = explicit_grid%cell_centroids(icell)%y
    z(icell) = explicit_grid%cell_centroids(icell)%z
  enddo

  x_min = minval(x)
  x_max = maxval(x)
  y_min = minval(y)
  y_max = maxval(y)
  z_min = minval(z)
  z_max = maxval(z)

end subroutine UGridExplicitSetCellCentroids

! ************************************************************************** !

function UGridExplicitSetInternConnect(explicit_grid,upwind_fraction_method, &
                                       option)
  !
  ! Sets up the internal connectivity within
  ! the connectivity object
  !
  ! Author: Glenn Hammond
  ! Date: 05/17/12
  !

  use Utility_module
  use Connection_module
  use Option_module

  implicit none

  type(connection_set_type), pointer :: UGridExplicitSetInternConnect

  type(unstructured_explicit_type) :: explicit_grid
  PetscInt :: upwind_fraction_method
  type(option_type) :: option

  type(connection_set_type), pointer :: connections
  PetscInt :: num_connections
  PetscInt :: iconn
  PetscInt :: id_up, id_dn
  PetscReal :: pt_up(3), pt_dn(3), pt_center(3)
  PetscBool :: error

  num_connections = size(explicit_grid%connections,2)
  connections => &
    ConnectionCreate(num_connections,INTERNAL_FACE_CONNECTION_TYPE, &
                     EXPLICIT_UNSTRUCTURED_GRID)

  error = PETSC_FALSE
  do iconn = 1, num_connections
    id_up = explicit_grid%connections(1,iconn)
    id_dn = explicit_grid%connections(2,iconn)
    connections%id_up(iconn) = id_up
    connections%id_dn(iconn) = id_dn

    pt_up(1) = explicit_grid%cell_centroids(id_up)%x
    pt_up(2) = explicit_grid%cell_centroids(id_up)%y
    pt_up(3) = explicit_grid%cell_centroids(id_up)%z

    pt_dn(1) = explicit_grid%cell_centroids(id_dn)%x
    pt_dn(2) = explicit_grid%cell_centroids(id_dn)%y
    pt_dn(3) = explicit_grid%cell_centroids(id_dn)%z

    pt_center(1) = explicit_grid%face_centroids(iconn)%x
    pt_center(2) = explicit_grid%face_centroids(iconn)%y
    pt_center(3) = explicit_grid%face_centroids(iconn)%z
    call UGridCalculateDist(pt_up,pt_dn,pt_center, &
                            explicit_grid%cell_volumes(id_up), &
                            explicit_grid%cell_volumes(id_dn), &
                            upwind_fraction_method, &
                            connections%dist(:,iconn),error,option)
    connections%area(iconn) = explicit_grid%face_areas(iconn)
  enddo
  if (error) then
    option%io_buffer = 'Errors in UGridExplicitSetInternConnect(). &
      &See details above.'
    call PrintErrMsgByRank(option)
  endif

  UGridExplicitSetInternConnect => connections

end function UGridExplicitSetInternConnect

! ************************************************************************** !

subroutine UGridExplicitComputeVolumes(ugrid,option,volume)
  !
  ! Sets the volume of each grid cell
  !
  ! Author: Glenn Hammond
  ! Date: 05/17/12
  !

  use Option_module

  implicit none

  type(grid_unstructured_type) :: ugrid
  type(option_type) :: option
  Vec :: volume

  type(unstructured_explicit_type), pointer :: explicit_grid

  PetscInt :: icell
  PetscReal, pointer :: vec_ptr(:)
  PetscErrorCode :: ierr

  explicit_grid => ugrid%explicit_grid

  call VecGetArrayF90(volume,vec_ptr,ierr);CHKERRQ(ierr)
  do icell = 1, ugrid%nlmax
    vec_ptr(icell) = explicit_grid%cell_volumes(icell)
  enddo
  call VecRestoreArrayF90(volume,vec_ptr,ierr);CHKERRQ(ierr)

end subroutine UGridExplicitComputeVolumes

! ************************************************************************** !

function UGridExplicitSetBoundaryConnect(explicit_grid,cell_ids, &
                                         face_centroids,face_areas, &
                                         region_name,option)
  !
  ! Sets up the boundary connectivity within
  ! the connectivity object
  !
  ! Author: Glenn Hammond
  ! Date: 05/18/12
  !

  use Utility_module
  use Connection_module
  use Option_module

  implicit none

  type(connection_set_type), pointer :: UGridExplicitSetBoundaryConnect

  type(unstructured_explicit_type) :: explicit_grid
  PetscInt :: cell_ids(:)
  type(point3d_type) :: face_centroids(:)
  PetscReal :: face_areas(:)
  character(len=MAXWORDLENGTH) :: region_name
  type(option_type) :: option

  type(connection_set_type), pointer :: connections
  PetscInt :: num_connections
  PetscInt :: iconn
  PetscInt :: id
  PetscReal :: v(3)
  PetscReal :: distance
  character(len=MAXSTRINGLENGTH) :: string
  PetscBool :: error

  num_connections = size(cell_ids)
  connections => &
    ConnectionCreate(num_connections,BOUNDARY_FACE_CONNECTION_TYPE, &
                     EXPLICIT_UNSTRUCTURED_GRID)

  error = PETSC_FALSE
  do iconn = 1, num_connections
    id = cell_ids(iconn)
    connections%id_dn(iconn) = id

    v(1) = explicit_grid%cell_centroids(id)%x - &
           face_centroids(iconn)%x
    v(2) = explicit_grid%cell_centroids(id)%y - &
           face_centroids(iconn)%y
    v(3) = explicit_grid%cell_centroids(id)%z - &
           face_centroids(iconn)%z

    distance = sqrt(DotProduct(v,v))
    if (dabs(distance) < 1.d-40) then
      write(string,'(2(es16.9,","),es16.9)') &
        face_centroids(iconn)%x, face_centroids(iconn)%y, &
        face_centroids(iconn)%z
      error = PETSC_TRUE
      option%io_buffer = 'Coincident cell and face centroids found at (' // &
        trim(adjustl(string)) // ') '
      call PrintMsgByRank(option)
    endif
    connections%dist(-1,iconn) = 0.d0
    connections%dist(0,iconn) = distance
    connections%dist(1:3,iconn) = v/distance
    connections%area(iconn) = face_areas(iconn)
  enddo
  if (error) then
    option%io_buffer = 'Coincident cell and face centroids found in ' // &
      'UGridExplicitSetBoundaryConnect() for region "' // trim(region_name) // &
      '".  See details above.'
    call PrintErrMsgByRank(option)
  endif

  UGridExplicitSetBoundaryConnect => connections

end function UGridExplicitSetBoundaryConnect

! ************************************************************************** !

function UGridExplicitSetConnections(explicit_grid,cell_ids,connection_type, &
                                     option)
  !
  ! Sets up the connectivity for a region
  !
  ! Author: Glenn Hammond
  ! Date: 05/18/12
  !

  use Utility_module
  use Connection_module
  use Option_module

  implicit none

  type(connection_set_type), pointer :: UGridExplicitSetConnections

  type(unstructured_explicit_type) :: explicit_grid
  PetscInt, pointer :: cell_ids(:)
  PetscInt :: connection_type
  type(option_type) :: option

  type(connection_set_type), pointer :: connections
  PetscInt :: num_connections
  PetscInt :: iconn
  PetscInt :: id

  num_connections = 0
  if (associated(cell_ids)) then
    num_connections = size(cell_ids)
  endif
  connections => ConnectionCreate(num_connections,connection_type, &
                                  EXPLICIT_UNSTRUCTURED_GRID)

  do iconn = 1, num_connections
    id = cell_ids(iconn)
    connections%id_dn(iconn) = id
  enddo

  UGridExplicitSetConnections => connections

end function UGridExplicitSetConnections

! ************************************************************************** !

subroutine UGridExplicitGetClosestCellFromPoint(x,y,z,grid_explicit, &
                                                nG2L,option,icell, &
                                                cell_distance)

  !
  ! Returns the cell which its center is the closest for point x,y,z
  !
  ! Author: Moise Rousseau
  ! Date: 04/12/21
  !
  use Option_module
  use Geometry_module

  implicit none

  PetscReal :: x, y, z
  PetscInt :: icell
  PetscReal :: cell_distance
  type(unstructured_explicit_type) :: grid_explicit
  PetscInt, pointer :: nG2L(:)
  type(option_type) :: option

  type(point3d_type) :: pt_test
  type(point3d_type) :: pt_champion
  PetscReal :: min_distance, distance
  PetscInt :: ghosted_id, local_id, champion

  !initiate
  min_distance = MAX_DOUBLE
  !looking for champion
  do ghosted_id = 1, size(grid_explicit%cell_volumes)
    local_id = nG2L(ghosted_id)
    if (local_id <= 0) cycle
    pt_test = grid_explicit%cell_centroids(ghosted_id)
    distance = (pt_test%x-x)**2 + (pt_test%y-y)**2 + (pt_test%z-z)**2
    if (distance < min_distance) then
      champion = local_id
      min_distance = distance
      pt_champion = pt_test
      cycle
    endif
    if (distance == min_distance) then
      if ((pt_test%x < pt_champion%x) .or. &
         (pt_test%x == pt_champion%x .and. pt_test%y < pt_champion%y) .or. &
         (pt_test%x == pt_champion%x .and. pt_test%y == pt_champion%y .and. &
         pt_test%z < pt_champion%z)) then
        champion = local_id
        pt_champion = pt_test
      endif
    endif
  enddo

  icell = champion
  cell_distance = min_distance

end subroutine UGridExplicitGetClosestCellFromPoint

! ************************************************************************** !

subroutine UGridExplicitExpandGhostCells(ugrid,scatter_gtol,global_vec,local_vec, &
                                 option)
  !
  ! Expands arrays assocated with ghost cells due to a change in ghosting
  ! as prescribed by scatter_gtol
  !
  ! Author: Michael Nole
  ! Date: 01/15/25

  use Option_module
  use Petsc_Utility_module
  use Utility_module
  use Geometry_module

  implicit none

  type(grid_unstructured_type) :: ugrid
  VecScatter :: scatter_gtol ! global (non-ghosted) to local (ghosted
  Vec :: global_vec
  Vec :: local_vec
  type(option_type) :: option

  PetscReal, allocatable :: real_array(:)
  PetscInt, allocatable :: int_array(:)

  type(point3d_type), pointer :: cell_centroids_ghosted_new(:)

  PetscReal, pointer :: vec_loc_ptr(:)

  PetscInt :: ghosted_id
  type(point3d_type) :: centroid_local

  PetscInt :: icell
  PetscInt :: i
  PetscErrorCode :: ierr

  ! cell_ids
  allocate(int_array(ugrid%nlmax))
  int_array = ugrid%explicit_grid%cell_ids(1:ugrid%nlmax)
  call DeallocateArray(ugrid%explicit_grid%cell_ids)
  allocate(ugrid%explicit_grid%cell_ids(ugrid%ngmax))
  ugrid%explicit_grid%cell_ids = UNINITIALIZED_INTEGER
  call PetUtilLoadVec(global_vec,int_array)
  call VecScatterBegin(scatter_gtol,global_vec,local_vec,INSERT_VALUES, &
                       SCATTER_FORWARD,ierr);CHKERRQ(ierr)
  call VecScatterEnd(scatter_gtol,global_vec,local_vec,INSERT_VALUES, &
                     SCATTER_FORWARD,ierr);CHKERRQ(ierr)
  call PetUtilUnloadVec(local_vec,ugrid%explicit_grid%cell_ids)
  deallocate(int_array)

  ! cell_volumes
  allocate(real_array(ugrid%nlmax))
  real_array = ugrid%explicit_grid%cell_volumes(1:ugrid%nlmax)
  call DeallocateArray(ugrid%explicit_grid%cell_volumes)
  allocate(ugrid%explicit_grid%cell_volumes(ugrid%ngmax))
  ugrid%explicit_grid%cell_volumes = UNINITIALIZED_INTEGER
  call PetUtilLoadVec(global_vec,real_array)
  call VecScatterBegin(scatter_gtol,global_vec,local_vec,INSERT_VALUES, &
                       SCATTER_FORWARD,ierr);CHKERRQ(ierr)
  call VecScatterEnd(scatter_gtol,global_vec,local_vec,INSERT_VALUES, &
                     SCATTER_FORWARD,ierr);CHKERRQ(ierr)
  call PetUtilUnloadVec(local_vec,ugrid%explicit_grid%cell_volumes)
  deallocate(real_array)

  ! cell_centroids
  allocate(cell_centroids_ghosted_new(ugrid%ngmax))
  do i = 1, size(cell_centroids_ghosted_new)
    cell_centroids_ghosted_new(i)%id = 0
    cell_centroids_ghosted_new(i)%x = UNINITIALIZED_DOUBLE
    cell_centroids_ghosted_new(i)%y = UNINITIALIZED_DOUBLE
    cell_centroids_ghosted_new(i)%z = UNINITIALIZED_DOUBLE
  enddo
  allocate(real_array(ugrid%nlmax))
  do i = 1, 3
    do icell = 1, ugrid%nlmax
      ! ugrid%cell_vertices is ghosted, but the first nlmax values are local
      centroid_local = ugrid%explicit_grid%cell_centroids(icell)
      select case(i)
        case(1)
          real_array(icell) = centroid_local%x
        case(2)
          real_array(icell) = centroid_local%y
        case(3)
          real_array(icell) = centroid_local%z
      end select
    enddo
    call PetUtilLoadVec(global_vec,real_array)
    call VecScatterBegin(scatter_gtol,global_vec,local_vec,INSERT_VALUES, &
                        SCATTER_FORWARD,ierr);CHKERRQ(ierr)
    call VecScatterEnd(scatter_gtol,global_vec,local_vec,INSERT_VALUES, &
                      SCATTER_FORWARD,ierr);CHKERRQ(ierr)
    call VecGetArrayReadF90(local_vec,vec_loc_ptr,ierr);CHKERRQ(ierr)
    do ghosted_id = 1, ugrid%ngmax
      select case(i)
        case(1)
          cell_centroids_ghosted_new(ghosted_id)%x = vec_loc_ptr(ghosted_id)
        case(2)
          cell_centroids_ghosted_new(ghosted_id)%y = vec_loc_ptr(ghosted_id)
        case(3)
          cell_centroids_ghosted_new(ghosted_id)%z = vec_loc_ptr(ghosted_id)
      end select
    enddo
    call VecRestoreArrayReadF90(local_vec,vec_loc_ptr,ierr);CHKERRQ(ierr)
  enddo
  deallocate(real_array)

  ! connections

  deallocate(ugrid%explicit_grid%cell_centroids)
  ugrid%explicit_grid%cell_centroids => cell_centroids_ghosted_new
  nullify(cell_centroids_ghosted_new)

  ugrid%num_vertices_local = size(ugrid%vertex_ids_natural)

end subroutine UGridExplicitExpandGhostCells

end module Grid_Unstructured_Explicit_module
