module General_Common_module

#include "petsc/finclude/petscsys.h"

  use General_Aux_module
  use Global_Aux_module

  use PFLOTRAN_Constants_module
  use petscsys

  implicit none

  private

#define CONVECTION
#define LIQUID_DARCY_FLUX
#define GAS_DARCY_FLUX
#define DIFFUSION
#define LIQUID_DIFFUSION
#define GAS_DIFFUSION
#define CONDUCTION

#define WATER_SRCSINK
#define AIR_SRCSINK
#define ENERGY_SRCSINK
#define SALT_SRCSINK

!#define DEBUG_FLUXES

! Cutoff parameters
  PetscReal, parameter :: eps       = 1.d-8
  PetscReal, parameter :: floweps   = 1.d-24

  public :: GeneralAccumulation, &
            GeneralFlux, &
            GeneralBCFlux, &
            GeneralAuxVarComputeAndSrcSink, &
            GeneralAccumDerivative, &
            GeneralFluxDerivative, &
            GeneralBCFluxDerivative, &
            GeneralSrcSinkDerivative

  public :: GeneralDiffJacobian, &
            GeneralAuxVarDiff

contains

! ************************************************************************** !

subroutine GeneralAccumulation(gen_auxvar,global_auxvar,material_auxvar, &
                               soil_heat_capacity,option,Res,Jac, &
                               analytical_derivatives,soluble_matrix,debug_cell)
  !
  ! Computes the non-fixed portion of the accumulation
  ! term for the residual
  !
  ! Author: Glenn Hammond
  ! Date: 03/09/11
  !

  use Option_module
  use Material_Aux_module

  implicit none

  type(general_auxvar_type) :: gen_auxvar
  type(global_auxvar_type) :: global_auxvar
  type(material_auxvar_type) :: material_auxvar
  PetscReal :: soil_heat_capacity
  type(option_type) :: option
  PetscReal :: Res(option%nflowdof)
  PetscReal :: Jac(option%nflowdof,option%nflowdof)
  PetscBool :: analytical_derivatives
  PetscBool :: debug_cell

  PetscInt :: wat_comp_id, air_comp_id, energy_id, salt_id
  PetscInt :: icomp, iphase

  PetscReal :: porosity
  PetscReal :: volume_over_dt

  PetscBool :: soluble_matrix

  wat_comp_id = option%water_id
  air_comp_id = option%air_id
  energy_id = option%energy_id
  salt_id = option%salt_id

  ! v_over_t[m^3 bulk/sec] = vol[m^3 bulk] / dt[sec]
  volume_over_dt = material_auxvar%volume / option%flow_dt
  ! must use gen_auxvar%effective porosity here as it enables numerical
  ! derivatives to be employed
  porosity = gen_auxvar%effective_porosity

  ! accumulation term units = kmol/s
  Res = 0.d0
  do iphase = 1, option%nphase
    ! Res[kmol comp/m^3 void] = sat[m^3 phase/m^3 void] *
    !                           den[kmol phase/m^3 phase] *
    !                           xmol[kmol comp/kmol phase]
    do icomp = 1, option%nflowspec
#ifdef DEBUG_GENERAL
      ! for debug version, aux var entries are initialized to NaNs.  even if
      ! saturation is zero, density may be a NaN.  So the conditional prevents
      ! this calculation.  For non-debug, aux var entries are initialized to
      ! 0.d0
      if (gen_auxvar%sat(iphase) > 0.d0) then
#endif
      Res(icomp) = Res(icomp) + gen_auxvar%sat(iphase) * &
                                gen_auxvar%den(iphase) * &
                                gen_auxvar%xmol(icomp,iphase)
#ifdef DEBUG_GENERAL
      endif
#endif
    enddo
  enddo

  ! scale by porosity * volume / dt
  ! Res[kmol/sec] = Res[kmol/m^3 void] * por[m^3 void/m^3 bulk] *
  !                 vol[m^3 bulk] / dt[sec]
  Res(1:option%nflowspec) = Res(1:option%nflowspec) * &
                            porosity * volume_over_dt
  if (soluble_matrix) then
    ! Res[kmol/sec] = Res[kmol/sec] + (1-por)[m^3 solid/m^3 bulk] * den[kmol/m^3]
    !                 * vol[m^3 bulk] / dt[sec]
    Res(option%salt_id) = Res(option%salt_id) + (1.d0 - porosity) * &
                            material_auxvar%soil_particle_density / FMWNACL * &
                            volume_over_dt
    !
  endif

  do iphase = 1, option%nphase
    ! Res[MJ/m^3 void] = sat[m^3 phase/m^3 void] *
    !                    den[kmol phase/m^3 phase] * U[MJ/kmol phase]
#ifdef DEBUG_GENERAL
    ! for debug version, aux var entries are initialized to NaNs.  even if
    ! saturation is zero, density may be a NaN.  So the conditional prevents
    ! this calculation.  For non-debug, aux var entries are initialized to
    ! 0.d0
    if (gen_auxvar%sat(iphase) > 0.d0) then
#endif
    Res(energy_id) = Res(energy_id) + gen_auxvar%sat(iphase) * &
                                      gen_auxvar%den(iphase) * &
                                      gen_auxvar%U(iphase)
#ifdef DEBUG_GENERAL
    endif
#endif
  enddo
  ! Res[MJ/sec] = (Res[MJ/m^3 void] * por[m^3 void/m^3 bulk] +
  !                (1-por)[m^3 rock/m^3 bulk] *
  !                  dencpr[kg rock/m^3 rock * MJ/kg rock-K] * T[C]) &
  !               vol[m^3 bulk] / dt[sec]
  Res(energy_id) = (Res(energy_id) * porosity + &
                    (1.d0 - porosity) * &
                    material_auxvar%soil_particle_density * &
                    soil_heat_capacity * gen_auxvar%temp) * volume_over_dt

  if (analytical_derivatives) then
    Jac = 0.d0
    select case(global_auxvar%istate)
      case(LIQUID_STATE)
        ! satl = 1
        ! ----------
        ! Water Equation
        ! por * satl * denl * Xwl
        ! ---
        ! w/respect to liquid pressure
        ! dpor_dp * denl * Xwl +
        ! por * ddenl_dpl * Xwl
        Jac(1,1) = &
          gen_auxvar%d%por_p * gen_auxvar%den(1) * gen_auxvar%xmol(1,1) + &
          porosity * gen_auxvar%d%denl_pl * gen_auxvar%xmol(1,1)
        ! w/respect to air mole fraction
        ! liquid phase density is indepenent of air mole fraction
        ! por * denl * dXwl_dXal
        ! Xwl = 1. - Xal
        ! dXwl_dXal = -1.
        Jac(1,2) = porosity * gen_auxvar%den(1) * (-1.d0)
        ! w/repect to temperature
        ! por * ddenl_dT * Xwl
        Jac(1,3) = porosity * gen_auxvar%d%denl_T * gen_auxvar%xmol(1,1)
        ! ----------
        ! Air Equation
        ! por * satl * denl * Xal
        ! w/respect to liquid pressure
        ! dpor_dpl * denl * Xal +
        ! por * ddenl_dpl * Xal
        Jac(2,1) = &
          gen_auxvar%d%por_p * gen_auxvar%den(1) * gen_auxvar%xmol(2,1) + &
          porosity * gen_auxvar%d%denl_pl * gen_auxvar%xmol(2,1)
        ! w/respect to air mole fraction
        Jac(2,2) = porosity * gen_auxvar%den(1)
        ! w/repect to temperature
        ! por * ddenl_dT * Xwl
        Jac(2,3) = porosity * gen_auxvar%d%denl_T * gen_auxvar%xmol(2,1)
        ! ----------
        ! Energy Equation
        ! por * satl * denl * Ul + (1-por) * denr * Cp * T
        ! w/respect to liquid pressure
        ! dpor_dpl * denl * Ul +
        ! por * ddenl_dpl * Ul +
        ! por * denl * dUl_dpl +
        ! -dpor_dpl * dens * Cp * T
        Jac(3,1) = &
          gen_auxvar%d%por_p * gen_auxvar%den(1) * gen_auxvar%U(1) + &
          porosity * gen_auxvar%d%denl_pl * gen_auxvar%U(1) + &
          porosity * gen_auxvar%den(1) * gen_auxvar%d%Ul_pl + &
          (-1.d0) * gen_auxvar%d%por_p * &
            material_auxvar%soil_particle_density * &
            soil_heat_capacity * gen_auxvar%temp
        ! w/respect to air mole fraction
        Jac(3,2) = 0.d0
        ! w/respect to temperature
        ! por * ddenl_dT * Ul +
        ! por * denl * dUl_dT +
        ! (1-por) * dens * Cp
        Jac(3,3) = &
          porosity * gen_auxvar%d%denl_T * gen_auxvar%U(1) + &
          porosity * gen_auxvar%den(1) * gen_auxvar%d%Ul_T + &
          (1.d0 - porosity) * material_auxvar%soil_particle_density * &
            soil_heat_capacity
      case(GAS_STATE)
        ! satg = 1
        ! ----------
        ! Water Equation
        ! por * satg * deng * Xwg
        ! ---
        ! w/respect to gas pressure
        ! dpor_dp * deng * Xwg +
        ! por * ddeng_dpg * Xwg
        Jac(1,1) = &
          gen_auxvar%d%por_p * gen_auxvar%den(2) * gen_auxvar%xmol(1,2) + &
          porosity * gen_auxvar%d%deng_pg * gen_auxvar%xmol(1,2) + &
          ! xmol_p(1,2) = dXwg_dpg
          porosity * gen_auxvar%den(2) * gen_auxvar%d%xmol_p(1,2)
        ! w/respect to air pressure
        ! por * deng_pa * Xwg +
        ! por * deng * Xwg_pa
        Jac(1,2) = porosity * gen_auxvar%d%deng_pa * gen_auxvar%xmol(1,2) + &
          ! liquid phase index hijacked for air pressure derivative
          ! xmol_p(1,1) = dXwg_dpa
          porosity * gen_auxvar%den(2) * gen_auxvar%d%xmol_p(1,1)
        ! w/repect to temperature
        ! por * ddenl_dT * Xwl
        Jac(1,3) = porosity * gen_auxvar%d%deng_T * gen_auxvar%xmol(1,2)
        ! ----------
        ! Air Equation
        ! por * satg * deng * Xag
        ! w/respect to gas pressure
        ! dpor_dpl * denl * Xal +
        ! por * ddenl_dpl * Xal
        Jac(2,1) = &
          gen_auxvar%d%por_p * gen_auxvar%den(2) * gen_auxvar%xmol(2,2) + &
          porosity * gen_auxvar%d%deng_pg * gen_auxvar%xmol(2,2) + &
          ! por * ddeng_dpg * Xag
          ! xmol_p(2,2) = dXag_dpg
          porosity *  gen_auxvar%den(2) * gen_auxvar%d%xmol_p(2,2)
        ! w/respect to air pressure
        Jac(2,2) = porosity * gen_auxvar%d%deng_pa * gen_auxvar%xmol(2,2) + &
          ! liquid phase index hijacked for air pressure derivative
          ! por * ddeng_dpa * Xag
          ! xmol_p(2,1) = dXag_dpa
          porosity * gen_auxvar%den(2) * gen_auxvar%d%xmol_p(2,1)
        ! w/repect to temperature
        ! por * ddeng_dT * Xag
        Jac(2,3) = porosity * gen_auxvar%d%deng_T * gen_auxvar%xmol(2,2)
        ! ----------
        ! Energy Equation
        ! por * satg * deng * Ug + (1-por) * denr * Cp * T
        ! w/respect to gas pressure
        ! dpor_dpg * deng * Ug +
        ! por * ddeng_dpg * Ug +
        ! por * deng * dUg_dpg +
        ! -dpor_dpg * denr * Cp * T
        Jac(3,1) = &
          gen_auxvar%d%por_p * gen_auxvar%den(2) * gen_auxvar%U(2) + &
          porosity * gen_auxvar%d%deng_pg * gen_auxvar%U(2) + &
          porosity * gen_auxvar%den(2) * gen_auxvar%d%Ug_pg + &
          (-1.d0) * gen_auxvar%d%por_p * &
            material_auxvar%soil_particle_density * &
            soil_heat_capacity * gen_auxvar%temp
        ! w/respect to air pressure
        ! por * ddeng_dpa * Ug +
        ! por * deng * dUg_dpa
        Jac(3,2) = porosity * gen_auxvar%d%deng_pa * gen_auxvar%U(2) + &
          porosity * gen_auxvar%den(2) * gen_auxvar%d%Ug_pa
        ! w/respect to temperature
        ! por * ddeng_dT * Ug +
        ! por * deng * dUg_dT +
        ! (1-por) * dens * Cp
        Jac(3,3) = &
          porosity * gen_auxvar%d%denl_T * gen_auxvar%U(2) + &
          porosity * gen_auxvar%den(2) * gen_auxvar%d%Ug_T + &
          (1.d0 - porosity) * material_auxvar%soil_particle_density * &
            soil_heat_capacity
      case(TWO_PHASE_STATE)
        ! ----------
        ! Water Equation
        ! por * (satl * denl * Xwl + satg * deng * Xwg)
        ! ---
        ! w/respect to gas pressure
        ! dpor_dp * (satl * denl * Xwl + satg * deng * Xwg) +
        ! por * (satl * ddenl_dpg * Xwl + satg * ddeng_dpg * Xwg) +
        ! por * (satl * denl * dXwl_dpg + satg * deng * dXwg_dpg)
        Jac(1,1) = &
          gen_auxvar%d%por_p * &
            (gen_auxvar%sat(1) * gen_auxvar%den(1) * gen_auxvar%xmol(1,1) + &
             gen_auxvar%sat(2) * gen_auxvar%den(2) * gen_auxvar%xmol(1,2)) + &
          porosity * &
                                 ! denl_pl = denl_pg
            (gen_auxvar%sat(1) * gen_auxvar%d%denl_pl * gen_auxvar%xmol(1,1) + &
             gen_auxvar%sat(2) * gen_auxvar%d%deng_pg * gen_auxvar%xmol(1,2)) + &
          porosity * &
            (gen_auxvar%sat(1) * gen_auxvar%den(1) * gen_auxvar%d%xmol_p(1,1) + &
             gen_auxvar%sat(2) * gen_auxvar%den(2) * gen_auxvar%d%xmol_p(1,2))
        ! w/respect to gas saturation
        ! porosity, density, mole fraction are independent of gas saturation
        ! por * (dsatl_dsatg * denl * Xwl + dsatg_dsatg * deng * Xwg)
        ! dsatl_dsatg = -1.
        ! dsatg_dsatg = 1.
        Jac(1,2) = porosity * &
          (-1.d0 * gen_auxvar%den(1) * gen_auxvar%xmol(1,1) + &
            1.d0 * gen_auxvar%den(2) * gen_auxvar%xmol(1,2))
        ! w/repect to temperature
        ! porosity, saturation, mole fraction are independent of temperature
        ! por * (satl * ddenl_T * Xwl + satg * ddeng_T * Xwg) +
        ! por * (satl * denl * dXwl_dpg + satg * deng * dXwg_dpg)
        Jac(1,3) = porosity * &
          (gen_auxvar%sat(1) * gen_auxvar%d%denl_T * gen_auxvar%xmol(1,1) + &
           gen_auxvar%sat(2) * gen_auxvar%d%deng_T * gen_auxvar%xmol(1,2)) + &
          porosity * &
            (gen_auxvar%sat(1) * gen_auxvar%den(1) * gen_auxvar%d%xmol_T(1,1) + &
             gen_auxvar%sat(2) * gen_auxvar%den(2) * gen_auxvar%d%xmol_T(1,2))
        ! ----------
        ! Gas Equation
        ! por * (satl * denl * Xal + satg * deng * Xag)
        ! ---
        ! w/respect to gas pressure
        ! dpor_dp * (satl * denl * Xal + satg * deng * Xag) +
        ! por * (satl * ddenl_dpg * Xal + satg * ddeng_dpg * Xag) +
        ! por * (satl * denl * dXal_dpg + satg * deng * dXag_dpg)
        Jac(2,1) = &
          gen_auxvar%d%por_p * &
            (gen_auxvar%sat(1) * gen_auxvar%den(1) * gen_auxvar%xmol(2,1) + &
             gen_auxvar%sat(2) * gen_auxvar%den(2) * gen_auxvar%xmol(2,2)) + &
          porosity * &
            (gen_auxvar%sat(1) * gen_auxvar%d%denl_pl * gen_auxvar%xmol(2,1) + &
             gen_auxvar%sat(2) * gen_auxvar%d%deng_pg * gen_auxvar%xmol(2,2)) + &
          porosity * &
            (gen_auxvar%sat(1) * gen_auxvar%den(1) * gen_auxvar%d%xmol_p(2,1) + &
             gen_auxvar%sat(2) * gen_auxvar%den(2) * gen_auxvar%d%xmol_p(2,2))
        ! w/respect to gas saturation
        ! porosity, density, mole fraction are independent of gas saturation
        ! por * (dsatl_dsatg * denl * Xal + dsatg_dsatg * deng * Xag)
        ! dsatl_dsatg = -1.
        ! dsatg_dsatg = 1.
        Jac(2,2) = porosity * &
          (-1.d0 * gen_auxvar%den(1) * gen_auxvar%xmol(2,1) + &
            1.d0 * gen_auxvar%den(2) * gen_auxvar%xmol(2,2))
        ! w/repect to temperature
        ! porosity, saturation, mole fraction are independent of temperature
        ! por * (satl * ddenl_T * Xal + satg * ddeng_T * Xag)
        Jac(2,3) = porosity * &
          (gen_auxvar%sat(1) * gen_auxvar%d%denl_T * gen_auxvar%xmol(2,1) + &
           gen_auxvar%sat(2) * gen_auxvar%d%deng_T * gen_auxvar%xmol(2,2)) + &
          porosity * &
            (gen_auxvar%sat(1) * gen_auxvar%den(1) * gen_auxvar%d%xmol_T(2,1) + &
             gen_auxvar%sat(2) * gen_auxvar%den(2) * gen_auxvar%d%xmol_T(2,2))
        ! ----------
        ! Energy Equation
        ! por * (satl * denl * Ul + satg * deng * Ug) +
        ! (1-por) * denr * Cpr * T
        ! ---
        ! w/respect to gas pressure
        ! dpor_dp * (satl * denl * Ul + satg * deng * Ug) +
        ! por * (satl * denl_dp * Ul + satg * deng_dp * Ug) +
        ! por * (satl * denl * dUl_dp + satg * deng * dUg_dp) +
        ! -dpor_dp * denr * Cpr * T
        Jac(3,1) = &
          gen_auxvar%d%por_p * &
            (gen_auxvar%sat(1) * gen_auxvar%den(1) * gen_auxvar%U(1) + &
             gen_auxvar%sat(2) * gen_auxvar%den(2) * gen_auxvar%U(2)) + &
          porosity * &
            (gen_auxvar%sat(1) * gen_auxvar%d%denl_pl * gen_auxvar%U(1) + &
             gen_auxvar%sat(2) * gen_auxvar%d%deng_pg * gen_auxvar%U(2)) + &
          porosity * &
            (gen_auxvar%sat(1) * gen_auxvar%den(1) * gen_auxvar%d%Ul_pl + &
             gen_auxvar%sat(2) * gen_auxvar%den(2) * gen_auxvar%d%Ug_pg) + &
          (-1.d0) * gen_auxvar%d%por_p * &
            material_auxvar%soil_particle_density * &
            soil_heat_capacity * gen_auxvar%temp
        ! w/respect to gas saturation
        ! por * (dsatl_dsatg * denl * Ul + dsatg_dsatg * deng * Ug)
        Jac(3,2) = porosity * &
          (-1.d0 * gen_auxvar%den(1) * gen_auxvar%U(1) + &
            1.d0 * gen_auxvar%den(2) * gen_auxvar%U(2))
        ! w/respect to temperature
        Jac(3,3) = &
          porosity * &
            (gen_auxvar%sat(1) * gen_auxvar%d%denl_T * gen_auxvar%U(1) + &
             gen_auxvar%sat(2) * gen_auxvar%d%deng_T * gen_auxvar%U(2)) + &
          porosity * &
            (gen_auxvar%sat(1) * gen_auxvar%den(1) * gen_auxvar%d%Ul_T + &
             gen_auxvar%sat(2) * gen_auxvar%den(2) * gen_auxvar%d%Ug_T) + &
          (1.d0 - porosity) * &
            material_auxvar%soil_particle_density * &
            soil_heat_capacity
    end select
    Jac = Jac * volume_over_dt
  endif

end subroutine GeneralAccumulation

! ************************************************************************** !

subroutine GeneralFlux(gen_auxvar_up,global_auxvar_up, &
                       material_auxvar_up, &
                       thermal_cc_up, &
                       gen_auxvar_dn,global_auxvar_dn, &
                       material_auxvar_dn, &
                       thermal_cc_dn, &
                       area, dist, upwind_direction_, &
                       general_parameter, &
                       option,v_darcy,Res,Jup,Jdn, &
                       analytical_derivatives, &
                       update_upwind_direction_, &
                       count_upwind_direction_flip_, &
                       debug_connection)
  !
  ! Computes the internal flux terms for the residual
  !
  ! Author: Glenn Hammond
  ! Date: 03/09/11
  !
  use Option_module
  use Material_Aux_module
  use Connection_module
  use Fracture_module
  use Klinkenberg_module
  use Upwind_Direction_module
  use Characteristic_Curves_Thermal_module
  use Utility_module

  implicit none

  type(general_auxvar_type) :: gen_auxvar_up, gen_auxvar_dn
  type(global_auxvar_type) :: global_auxvar_up, global_auxvar_dn
  type(material_auxvar_type) :: material_auxvar_up, material_auxvar_dn
  type(option_type) :: option
  PetscReal :: v_darcy(option%nphase)
  PetscReal :: area
  PetscReal :: dist(-1:3)
  PetscInt :: upwind_direction_(option%nphase)
  type(general_parameter_type) :: general_parameter
  class(cc_thermal_type) :: thermal_cc_up, thermal_cc_dn
  PetscReal :: Res(option%nflowdof)
  PetscReal :: Jup(option%nflowdof,option%nflowdof)
  PetscReal :: Jdn(option%nflowdof,option%nflowdof)
  PetscBool :: analytical_derivatives
  PetscBool :: update_upwind_direction_
  PetscBool :: count_upwind_direction_flip_
  PetscBool :: debug_connection

  PetscReal :: dist_gravity  ! distance along gravity vector
  PetscReal :: dist_up, dist_dn
  PetscReal :: upweight
  PetscInt :: wat_comp_id, air_comp_id, energy_id, salt_comp_id
  PetscInt :: iphase

  PetscReal :: xmol(option%nflowspec)
  PetscReal :: density_ave, density_kg_ave
  PetscReal :: uH
  PetscReal :: perm_ave_over_dist(option%nphase)
  PetscReal :: perm_up, perm_dn
  PetscReal :: delta_pressure, delta_xmol, delta_temp
  PetscReal :: delta_xsmol
  PetscReal :: xmol_air_up, xmol_air_dn
  PetscReal :: xmass_air_up, xmass_air_dn, delta_xmass
  PetscReal :: delta_X_whatever
  PetscReal :: pressure_ave
  PetscReal :: gravity_term
  PetscReal :: mobility, q, kr
  PetscReal :: tot_mole_flux, wat_mole_flux, air_mole_flux
  PetscReal :: tot_mole_flux1, salt_mole_flux
  PetscReal :: stpd_up, stpd_dn
  PetscReal :: sat_up, sat_dn, den_up, den_dn
  PetscReal :: temp_ave, stpd_ave_over_dist, tempreal
  PetscReal :: k_eff_up, k_eff_dn, k_eff_ave, heat_flux

  PetscReal :: temp_perm_up, temp_perm_dn

  ! Darcy flux
  PetscReal :: ddelta_pressure_dpup, ddelta_pressure_dpdn
  PetscReal :: ddelta_pressure_dpaup, ddelta_pressure_dpadn
  PetscReal :: ddelta_pressure_dTup, ddelta_pressure_dTdn

  PetscReal :: up_scale, dn_scale
  PetscBool :: upwind
  PetscReal :: tot_mole_flux_ddel_pressure
  PetscReal :: ddensity_kg_ave_dden_kg_up, ddensity_kg_ave_dden_kg_dn
  PetscReal :: ddensity_ave_dden_up, ddensity_ave_dden_dn
  PetscReal :: dtot_mole_flux_dp, dtot_mole_flux_dT, dtot_mole_flux_dsatg
  PetscReal :: dpl_dsatg
  PetscReal :: ddelta_pressure_pl
  PetscReal :: perm3(3)

  ! Diffusion
  PetscReal :: dstpd_up_dporup, dstpd_dn_dpordn
  PetscReal :: dstpd_up_dsatup, dstpd_dn_dsatdn
  PetscReal :: dstpd_up_ddenup, dstpd_dn_ddendn
  PetscReal :: dsatup, dsatdn
  PetscReal :: delta_X_whatever_dxmolup, delta_X_whatever_dxmoldn
  PetscReal :: dxmass_air_up_dxmol_air_up, dxmass_air_dn_dxmol_air_dn
  PetscReal :: dtot_mole_flux_dstpd
  PetscReal :: dtot_mole_flux_ddeltaX, dtot_mole_flux_ddeltaS
  PetscReal :: dtot_mole_flux_dstpd1, dtot_mole_flux_ddenave1
  PetscReal :: dtot_mole_flux_ddenave
  PetscReal :: diffusion_scale
  PetscReal :: ddiffusion_coef_dTup, ddiffusion_coef_dTdn
  PetscReal :: ddiffusion_coef_dpup, ddiffusion_coef_dpdn
  PetscReal :: dtot_mole_flux_ddiffusion_coef
  PetscReal :: dstpd_ave_over_dist_dstpd_up, dstpd_ave_over_dist_dstpd_dn

  ! Conduction
  PetscReal :: dkeff_up_dsatlup, dkeff_dn_dsatldn
  PetscReal :: dkeff_up_dTup, dkeff_dn_dTdn
  PetscReal :: dkeff_ave_dkeffup, dkeff_ave_dkeffdn
  PetscReal :: dheat_flux_ddelta_temp, dheat_flux_dkeff_ave

  PetscReal :: Jlup(3,3), Jldn(3,3)
  PetscReal :: Jgup(3,3), Jgdn(3,3)
  PetscReal :: Jcup(3,3), Jcdn(3,3)

  wat_comp_id = option%water_id
  air_comp_id = option%air_id
  energy_id = option%energy_id
  salt_comp_id = option%salt_id

  call ConnectionCalculateDistances(dist,option%gravity,dist_up,dist_dn, &
                                    dist_gravity,upweight)
  call PermeabilityTensorToScalar(material_auxvar_up,dist,perm_up)
  call PermeabilityTensorToScalar(material_auxvar_dn,dist,perm_dn)

#if 0
!TODO(geh): remove for now
  ! Fracture permeability change only available for structured grid (Heeho)
  if (associated(material_auxvar_up%fracture)) then
    call FracturePermEvaluate(material_auxvar_up,perm_up,temp_perm_up, &
                              dummy_dperm_up,dist)
    perm_up = temp_perm_up
  endif
  if (associated(material_auxvar_dn%fracture)) then
    call FracturePermEvaluate(material_auxvar_dn,perm_dn,temp_perm_dn, &
                              dummy_dperm_dn,dist)
    perm_dn = temp_perm_dn
  endif
#endif

  if (associated(klinkenberg)) then
    perm_ave_over_dist(1) = (perm_up * perm_dn) / &
                            (dist_up*perm_dn + dist_dn*perm_up)
    perm3(:) = perm_up
    perm3 = klinkenberg%Evaluate(perm3, &
                                 gen_auxvar_up%pres(option%gas_phase))
    temp_perm_up = perm3(1)
    perm3(:) = perm_dn
    perm3 = klinkenberg%Evaluate(perm3, &
                                 gen_auxvar_dn%pres(option%gas_phase))
    temp_perm_dn = perm3(1)
    perm_ave_over_dist(2) = (temp_perm_up * temp_perm_dn) / &
                            (dist_up*temp_perm_dn + dist_dn*temp_perm_up)
  else
    perm_ave_over_dist(:) = (perm_up * perm_dn) / &
                            (dist_up*perm_dn + dist_dn*perm_up)
  endif

  Res = 0.d0
  Jup = 0.d0
  Jdn = 0.d0

  v_darcy = 0.d0

#ifdef CONVECTION
#ifdef LIQUID_DARCY_FLUX
  iphase = LIQUID_PHASE
  if (gen_auxvar_up%mobility(iphase) + &
      gen_auxvar_dn%mobility(iphase) > eps) then

    density_kg_ave = GeneralAverageDensity(iphase, &
                                           global_auxvar_up%istate, &
                                           global_auxvar_dn%istate, &
                                           gen_auxvar_up%den_kg, &
                                           gen_auxvar_dn%den_kg, &
                                           ddensity_kg_ave_dden_kg_up, &
                                           ddensity_kg_ave_dden_kg_dn)

    gravity_term = density_kg_ave * dist_gravity
    delta_pressure = gen_auxvar_up%pres(iphase) - &
                     gen_auxvar_dn%pres(iphase) + &
                     gravity_term
    if (analytical_derivatives) then
      ddelta_pressure_dpup = 1.d0 + dist_gravity * &
                             ddensity_kg_ave_dden_kg_up * &
                             gen_auxvar_up%d%denl_pl * fmw_comp(iphase)
      ddelta_pressure_dpdn = -1.d0 + dist_gravity * &
                             ddensity_kg_ave_dden_kg_dn * &
                             gen_auxvar_dn%d%denl_pl * fmw_comp(iphase)
      ddelta_pressure_dTup = dist_gravity * ddensity_kg_ave_dden_kg_up * &
                             gen_auxvar_up%d%denl_T * fmw_comp(iphase)
      ddelta_pressure_dTdn = dist_gravity * ddensity_kg_ave_dden_kg_dn * &
                             gen_auxvar_dn%d%denl_T * fmw_comp(iphase)
    endif
    up_scale = 0.d0
    dn_scale = 0.d0
    upwind = UpwindDirection(upwind_direction_(iphase),delta_pressure, &
                             .not.analytical_derivatives, &
                             count_upwind_direction_flip_, &
                             liq_upwind_flip_count_by_res, &
                             liq_upwind_flip_count_by_jac)
    if (upwind) then
      up_scale = 1.d0
      mobility = gen_auxvar_up%mobility(iphase)
      xmol(:) = gen_auxvar_up%xmol(:,iphase)
      uH = gen_auxvar_up%H(iphase)
      kr= gen_auxvar_up%kr(iphase)
    else
      dn_scale = 1.d0
      mobility = gen_auxvar_dn%mobility(iphase)
      xmol(:) = gen_auxvar_dn%xmol(:,iphase)
      uH = gen_auxvar_dn%H(iphase)
      kr=gen_auxvar_dn%kr(iphase)
    endif

    if (mobility > floweps ) then
       density_ave = GeneralAverageDensity(iphase, &
                                           global_auxvar_up%istate, &
                                           global_auxvar_dn%istate, &
                                           gen_auxvar_up%den, &
                                           gen_auxvar_dn%den, &
                                           ddensity_ave_dden_up, &
                                           ddensity_ave_dden_dn)
       if (.not. general_non_darcy_flow) then
          ! v_darcy[m/sec] = perm[m^2] / dist[m] * kr[-] / mu[Pa-sec]
          !                    dP[Pa]]
          v_darcy(iphase) = perm_ave_over_dist(iphase) * mobility * delta_pressure

          ! q[m^3 phase/sec] = v_darcy[m/sec] * area[m^2]
          q = v_darcy(iphase) * area
          ! mole_flux[kmol phase/sec] = q[m^3 phase/sec] *
          !                             density_ave[kmol phase/m^3 phase]
          tot_mole_flux = q*density_ave
          tot_mole_flux_ddel_pressure = perm_ave_over_dist(iphase) * &
                                           mobility * area * density_ave
       else
          !Taken from (Liu, 2014)
          call GeneralNonDarcyCorrection(delta_pressure,density_kg_ave,&
            perm_ave_over_dist(iphase),kr,mobility,density_ave, &
            area,v_darcy(iphase),tot_mole_flux_ddel_pressure,q,tot_mole_flux)
       endif

      ! comp_mole_flux[kmol comp/sec] = tot_mole_flux[kmol phase/sec] *
      !                                 xmol[kmol comp/kmol phase]
      wat_mole_flux = tot_mole_flux * xmol(wat_comp_id)
      air_mole_flux = tot_mole_flux * xmol(air_comp_id)
      Res(wat_comp_id) = Res(wat_comp_id) + wat_mole_flux
      Res(air_comp_id) = Res(air_comp_id) + air_mole_flux
      Res(energy_id) = Res(energy_id) + tot_mole_flux * uH
      if (general_salt) then
        salt_mole_flux = tot_mole_flux * xmol(salt_comp_id)
        Res(salt_comp_id) = Res(salt_comp_id) + salt_mole_flux
      endif

      if (analytical_derivatives) then
        Jlup = 0.d0
        Jldn = 0.d0
        select case(global_auxvar_up%istate)
          case(LIQUID_STATE)
            ! derivative wrt liquid pressure
            ! derivative total mole flux wrt liquid pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%denl_pl + &
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityl_pl + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpup
            ! derivative water wrt liquid pressure
            Jlup(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp
            ! derivative air wrt liquid pressure
            Jlup(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp
            ! derivative energy wrt liquid pressure
            Jlup(3,1) = uH * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hl_pl

            ! derivative wrt air mole fraction
            ! derivative water wrt air mole fraction
            Jlup(1,2) = -1.d0 * up_scale * tot_mole_flux
            ! derivative air wrt air mole fraction
            Jlup(2,2) = 1.d0 * up_scale * tot_mole_flux
            ! derivative energy wrt air mole fraction
            ! Jlup(3,2) = 0.d0

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%denl_T + &
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityl_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTup
            ! derivative water wrt temperature
            Jlup(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
            ! derivative air wrt temperature
            Jlup(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
            ! derivative energy wrt temperature
            Jlup(3,3) = uH * dtot_mole_flux_dT + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hl_T

          case(GAS_STATE)
            ! derivative wrt gas pressure
            ! pl = pg - pc and dpl_dpg = 1.  Therefore, we can use all the
            !   liquid pressure derivatives.
            ! derivative total mole flux wrt gas pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%denl_pl + &
              ! mole fraction has to be added in below since it differs for air
              ! and water
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityl_pl + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpup
            ! derivative water wrt gas pressure
            Jlup(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_p(wat_comp_id,iphase)
            ! derivative air wrt gas pressure
            Jlup(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_p(air_comp_id,iphase)
            ! derivative energy wrt gas pressure
            Jlup(3,1) = uH * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hl_pl

            ! derivative wrt air pressure
            ! derivative water wrt air saturation
            ! Jlup(1,2) = 0.d0
            ! derivative air wrt air saturation
            ! Jlup(2,2) = 0.d0
            ! derivative energy wrt air saturation
            ! Jlup(3,2) = 0.d0

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%denl_T + &
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityl_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTup
              ! there is no derivative of mole fraction wrt temperature in
              ! gas state
            ! derivative water wrt temperature
            Jlup(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
            ! derivative air wrt temperature
            Jlup(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
            ! derivative energy wrt temperature
            Jlup(3,3) = dtot_mole_flux_dT * uH + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hl_T

          case(TWO_PHASE_STATE)
            ! derivative wrt gas pressure
            ! pl = pg - pc and dpl_dpg = 1.  Therefore, we can use all the
            !   liquid pressure derivatives.
            ! derivative total mole flux wrt gas pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%denl_pl + &
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityl_pl + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpup
            ! derivative water wrt gas pressure
            Jlup(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_p(wat_comp_id,iphase)
            ! derivative air wrt gas pressure
            Jlup(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_p(air_comp_id,iphase)
            ! derivative energy wrt gas pressure
            Jlup(3,1) = uH * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hl_pl

            ! derivative wrt gas saturation
            ! pl = pg - pc(satg)
            dpl_dsatg = -1.d0 * gen_auxvar_up%d%pc_satg
            ! delta pressure = plup - pldn
            ddelta_pressure_pl = 1.d0
            ! derivative total mole flux wrt gas saturation
            dtot_mole_flux_dsatg = &
              ! liquid viscosity
              ! since liquid viscosity in a two phase state is a function
              ! of total pressure (gas pressure), there is not derivative
              ! wrt gas saturation
              !up_scale * &
              !tot_mole_flux / mobility * &
              !gen_auxvar_up%d%mobilityl_pl * dpl_dsatg + &
              ! relative permeability
              up_scale * &
              tot_mole_flux / mobility * &
              gen_auxvar_up%d%mobilityl_satg + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_pl * dpl_dsatg
            ! derivative water wrt gas saturation
            Jlup(1,2) = xmol(wat_comp_id) * dtot_mole_flux_dsatg
            ! derivative air wrt gas saturation
            Jlup(2,2) = xmol(air_comp_id) * dtot_mole_flux_dsatg
            ! derivative energy wrt gas saturation
            Jlup(3,2) = dtot_mole_flux_dsatg * uH

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%denl_T + &
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityl_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTup
            ! derivative water wrt temperature
            Jlup(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_T(wat_comp_id,iphase)
            ! derivative air wrt temperature
            Jlup(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_T(air_comp_id,iphase)
            ! derivative energy wrt temperature
            Jlup(3,3) = dtot_mole_flux_dT * uH + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hl_T
        end select
        select case(global_auxvar_dn%istate)
          case(LIQUID_STATE)
            ! derivative wrt liquid pressure
            ! derivative total mole flux wrt liquid pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn *gen_auxvar_dn%d%denl_pl + &
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityl_pl + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
            ! derivative water wrt liquid pressure
            Jldn(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp
            ! derivative air wrt liquid pressure
            Jldn(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp
            ! derivative energy wrt liquid pressure
            Jldn(3,1) = uH * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hl_pl

            ! derivative wrt air mole fraction
            ! derivative water wrt air mole fraction
            Jldn(1,2) = -1.d0 * dn_scale * tot_mole_flux
            ! derivative air wrt air mole fraction
            Jldn(2,2) = 1.d0 * dn_scale * tot_mole_flux
            ! derivative energy wrt air mole fraction
            ! Jldn(3,2) = 0.d0

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn * gen_auxvar_dn%d%denl_T + &
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityl_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
            ! derivative water wrt temperature
            Jldn(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
            ! derivative air wrt temperature
            Jldn(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
            ! derivative energy wrt temperature
            Jldn(3,3) = uH * dtot_mole_flux_dT + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hl_T

          case(GAS_STATE)
            ! derivative wrt gas pressure
            ! pl = pg - pc and dpl_dpg = 1.  Therefore, we can use all the
            !   liquid pressure derivatives.
            ! derivative total mole flux wrt gas pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn * gen_auxvar_dn%d%denl_pl + &
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityl_pl + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
            ! derivative water wrt gas pressure
            Jldn(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_p(wat_comp_id,iphase)
            ! derivative air wrt gas pressure
            Jldn(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_p(air_comp_id,iphase)
            ! derivative energy wrt gas pressure
            Jldn(3,1) = uH * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hl_pl

            ! derivative wrt air pressure
            ! derivative water wrt air saturation
            ! Jldn(1,2) = 0.d0
            ! derivative air wrt air saturation
            ! Jldn(2,2) = 0.d0
            ! derivative energy wrt air saturation
            ! Jldn(3,2) = 0.d0

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn * gen_auxvar_dn%d%denl_T + &
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityl_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
              ! there is no derivative of mole fraction wrt temperature in
              ! gas state
            ! derivative water wrt temperature
            Jldn(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
            ! derivative air wrt temperature
            Jldn(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
            ! derivative energy wrt temperature
            Jldn(3,3) = dtot_mole_flux_dT * uH + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hl_T

          case(TWO_PHASE_STATE)
            ! derivative wrt gas pressure
            ! pl = pg - pc and dpl_dpg = 1.  Therefore, we can use all the
            !   liquid pressure derivatives.
            ! derivative total mole flux wrt gas pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn *gen_auxvar_dn%d%denl_pl + &
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityl_pl + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
            ! derivative water wrt gas pressure
            Jldn(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_p(wat_comp_id,iphase)
            ! derivative air wrt gas pressure
            Jldn(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_p(air_comp_id,iphase)
            ! derivative energy wrt gas pressure
            Jldn(3,1) = uH * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hl_pl

            ! derivative wrt gas saturation
            ! pl = pg - pc(satg)
            dpl_dsatg = -1.d0 * gen_auxvar_dn%d%pc_satg
            ! delta pressure = plup - pldn
            ddelta_pressure_pl = -1.d0
            ! derivative total mole flux wrt gas saturation
            dtot_mole_flux_dsatg = &
              ! liquid viscosity
              ! since liquid viscosity in a two phase state is a function
              ! of total pressure (gas pressure), there is not derivative
              ! wrt gas saturation
              !dn_scale * &
              !tot_mole_flux / mobility * &
              !gen_auxvar_dn%d%mobilityl_pl * dpl_dsatg + &
              ! relative permeability
              dn_scale * &
              tot_mole_flux / mobility * &
              gen_auxvar_dn%d%mobilityl_satg + &
              !pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_pl * dpl_dsatg
            ! derivative water wrt gas saturation
            Jldn(1,2) = xmol(wat_comp_id) * dtot_mole_flux_dsatg
            ! derivative air wrt gas saturation
            Jldn(2,2) = xmol(air_comp_id) * dtot_mole_flux_dsatg
            ! derivative energy wrt gas saturation
            Jldn(3,2) = dtot_mole_flux_dsatg * uH

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn * gen_auxvar_dn%d%denl_T + &
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityl_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
            ! derivative water wrt temperature
            Jldn(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_T(wat_comp_id,iphase)
            ! derivative air wrt temperature
            Jldn(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_T(air_comp_id,iphase)
            ! derivative energy wrt temperature
            Jldn(3,3) = dtot_mole_flux_dT * uH + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hl_T
        end select
        Jup = Jup + Jlup
        Jdn = Jdn + Jldn
      endif
    endif
  endif
#endif
#ifdef GAS_DARCY_FLUX
  iphase = GAS_PHASE
  if (gen_auxvar_up%mobility(iphase) + &
      gen_auxvar_dn%mobility(iphase) > eps) then

    density_kg_ave = GeneralAverageDensity(iphase, &
                                           global_auxvar_up%istate, &
                                           global_auxvar_dn%istate, &
                                           gen_auxvar_up%den_kg, &
                                           gen_auxvar_dn%den_kg, &
                                           ddensity_kg_ave_dden_kg_up, &
                                           ddensity_kg_ave_dden_kg_dn)

    gravity_term = density_kg_ave * dist_gravity
    delta_pressure = gen_auxvar_up%pres(iphase) - &
                     gen_auxvar_dn%pres(iphase) + &
                     gravity_term
    ! if a gas phase does not exist on either side of the connection, the gas
    ! phase properties from the opposite side are used.
    if (analytical_derivatives) then
      ddelta_pressure_dpup = 1.d0 + dist_gravity * &
                             ddensity_kg_ave_dden_kg_up * &
                             gen_auxvar_up%d%deng_pg * fmw_comp(iphase)
      ddelta_pressure_dpdn = -1.d0 + dist_gravity * &
                             ddensity_kg_ave_dden_kg_dn * &
                             gen_auxvar_dn%d%deng_pg * fmw_comp(iphase)
      ddelta_pressure_dpaup = dist_gravity * ddensity_kg_ave_dden_kg_up * &
                              gen_auxvar_up%d%deng_pa * fmw_comp(iphase)
      ddelta_pressure_dpadn = dist_gravity * ddensity_kg_ave_dden_kg_dn * &
                              gen_auxvar_dn%d%deng_pa * fmw_comp(iphase)
      ddelta_pressure_dTup = dist_gravity * ddensity_kg_ave_dden_kg_up * &
                             gen_auxvar_up%d%deng_T * fmw_comp(iphase)
      ddelta_pressure_dTdn = dist_gravity * ddensity_kg_ave_dden_kg_dn * &
                             gen_auxvar_dn%d%deng_T * fmw_comp(iphase)
    endif
    up_scale = 0.d0
    dn_scale = 0.d0
    upwind = UpwindDirection(upwind_direction_(iphase),delta_pressure, &
                             .not.analytical_derivatives, &
                             count_upwind_direction_flip_, &
                             gas_upwind_flip_count_by_res, &
                             gas_upwind_flip_count_by_jac)
    if (upwind) then
      up_scale = 1.d0
      mobility = gen_auxvar_up%mobility(iphase)
      xmol(:) = gen_auxvar_up%xmol(:,iphase)
      uH = gen_auxvar_up%H(iphase)
      kr=gen_auxvar_up%kr(iphase)
    else
      dn_scale = 1.d0
      mobility = gen_auxvar_dn%mobility(iphase)
      xmol(:) = gen_auxvar_dn%xmol(:,iphase)
      uH = gen_auxvar_dn%H(iphase)
      kr=gen_auxvar_dn%kr(iphase)
    endif

    if (mobility > floweps) then
      density_ave = GeneralAverageDensity(iphase, &
                                          global_auxvar_up%istate, &
                                          global_auxvar_dn%istate, &
                                          gen_auxvar_up%den, &
                                          gen_auxvar_dn%den, &
                                          ddensity_ave_dden_up, &
                                          ddensity_ave_dden_dn)
      if (.not. general_non_darcy_flow) then
         ! v_darcy[m/sec] = perm[m^2] / dist[m] * kr[-] / mu[Pa-sec]
         !                    dP[Pa]]
         v_darcy(iphase) = perm_ave_over_dist(iphase) * mobility * delta_pressure

         ! q[m^3 phase/sec] = v_darcy[m/sec] * area[m^2]
         q = v_darcy(iphase) * area
         ! mole_flux[kmol phase/sec] = q[m^3 phase/sec] *
         !                             density_ave[kmol phase/m^3 phase]
         tot_mole_flux = q*density_ave
         tot_mole_flux_ddel_pressure = perm_ave_over_dist(iphase) * &
              mobility * area * density_ave

      else
         !Taken from (Liu, 2014)
         call GeneralNonDarcyCorrection(delta_pressure,density_kg_ave, &
              perm_ave_over_dist(iphase),kr,mobility,density_ave,area, &
              v_darcy(iphase),tot_mole_flux_ddel_pressure,q,tot_mole_flux)
      endif

      ! comp_mole_flux[kmol comp/sec] = tot_mole_flux[kmol phase/sec] *
      !                                 xmol[kmol comp/kmol phase]
      wat_mole_flux = tot_mole_flux * xmol(wat_comp_id)
      air_mole_flux = tot_mole_flux * xmol(air_comp_id)
      Res(wat_comp_id) = Res(wat_comp_id) + wat_mole_flux
      Res(air_comp_id) = Res(air_comp_id) + air_mole_flux
      Res(energy_id) = Res(energy_id) + tot_mole_flux * uH

      if (analytical_derivatives) then

        Jgup = 0.d0
        Jgdn = 0.d0
        select case(global_auxvar_up%istate)
          case(LIQUID_STATE)
            ! derivative wrt liquid pressure
            ! derivative total mole flux wrt liquid pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%deng_pg + &
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityg_pg + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpup
            ! derivative water wrt liquid pressure
            Jgup(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp
            ! derivative air wrt liquid pressure
            Jgup(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp
            ! derivative energy wrt liquid pressure
            Jgup(3,1) = uH * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hg_pg

            ! derivative wrt air mole fraction
            ! derivative water wrt air mole fraction
            Jgup(1,2) = -1.d0 * up_scale * tot_mole_flux
            ! derivative air wrt air mole fraction
            Jgup(2,2) = 1.d0 * up_scale * tot_mole_flux
            ! derivative energy wrt air mole fraction
            ! Jgup(3,2) = 0.d0

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%deng_T + &
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityg_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTup
            ! derivative water wrt temperature
            Jgup(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
            ! derivative air wrt temperature
            Jgup(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
            ! derivative energy wrt temperature
            Jgup(3,3) = uH * dtot_mole_flux_dT + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hg_T

          case(GAS_STATE)
            ! derivative wrt gas pressure
            ! derivative total mole flux wrt gas pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%deng_pg + &
              ! mole fraction has to be added in below since it differs for air
              ! and water
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityg_pg + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpup
            ! derivative water wrt gas pressure
            Jgup(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_p(wat_comp_id,iphase)
            ! derivative air wrt gas pressure
            Jgup(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_p(air_comp_id,iphase)
            ! derivative energy wrt gas pressure
            Jgup(3,1) = uH * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hg_pg

            ! derivative wrt air pressure
            ! derivative water wrt air saturation
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%deng_pa + &
              ! mole fraction has to be added in below since it differs for air
              ! and water
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityg_pa + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpaup
            Jgup(1,2) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                       up_scale * &
                       ! dXwg_pa for gas phase is stored in liquid phase of xmol_p
                       tot_mole_flux * gen_auxvar_up%d%xmol_p(wat_comp_id,LIQUID_PHASE)
            ! derivative air wrt air saturation
            Jgup(2,2) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                       up_scale * &
                       ! dXag_pa for gas phase is stored in liquid phase of xmol_p
                       tot_mole_flux * gen_auxvar_up%d%xmol_p(air_comp_id,LIQUID_PHASE)
            ! derivative energy wrt air saturation
            Jgup(3,2) = uH * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hg_pa

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%deng_T + &
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityg_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTup
              ! there is no derivative of mole fraction wrt temperature in
              ! gas state
            ! derivative water wrt temperature
            Jgup(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
            ! derivative air wrt temperature
            Jgup(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
            ! derivative energy wrt temperature
            Jgup(3,3) = dtot_mole_flux_dT * uH + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hg_T

          case(TWO_PHASE_STATE)
            ! derivative wrt gas pressure
            ! pl = pg - pc and dpl_dpg = 1.  Therefore, we can use all the
            !   liquid pressure derivatives.
            ! derivative total mole flux wrt gas pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%deng_pg + &
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityg_pg + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpup
            ! derivative water wrt gas pressure
            Jgup(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_p(wat_comp_id,iphase)
            ! derivative air wrt gas pressure
            Jgup(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_p(air_comp_id,iphase)
            ! derivative energy wrt gas pressure
            Jgup(3,1) = uH * dtot_mole_flux_dp + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hg_pg

            ! derivative wrt gas saturation
            ! derivative total mole flux wrt gas saturation
            dtot_mole_flux_dsatg = &
              ! relative permeability
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityg_satg
            ! derivative water wrt gas saturation
            Jgup(1,2) = xmol(wat_comp_id) * dtot_mole_flux_dsatg
            ! derivative air wrt gas saturation
            Jgup(2,2) = xmol(air_comp_id) * dtot_mole_flux_dsatg
            ! derivative energy wrt gas saturation
            Jgup(3,2) = dtot_mole_flux_dsatg * uH

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_up * gen_auxvar_up%d%deng_T + &
              ! liquid mobility
              up_scale * &
              tot_mole_flux / mobility * gen_auxvar_up%d%mobilityg_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTup
            ! derivative water wrt temperature
            Jgup(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_T(wat_comp_id,iphase)
            ! derivative air wrt temperature
            Jgup(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%xmol_T(air_comp_id,iphase)
            ! derivative energy wrt temperature
            Jgup(3,3) = dtot_mole_flux_dT * uH + &
                       up_scale * &
                       tot_mole_flux * gen_auxvar_up%d%Hg_T
        end select
        select case(global_auxvar_dn%istate)
          case(LIQUID_STATE)
            ! derivative wrt liquid pressure
            ! derivative total mole flux wrt liquid pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_pg + &
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityg_pg + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
            ! derivative water wrt liquid pressure
            Jgdn(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp
            ! derivative air wrt liquid pressure
            Jgdn(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp
            ! derivative energy wrt liquid pressure
            Jgdn(3,1) = uH * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hg_pg

            ! derivative wrt air mole fraction
            ! derivative water wrt air mole fraction
            Jgdn(1,2) = -1.d0 * dn_scale * tot_mole_flux
            ! derivative air wrt air mole fraction
            Jgdn(2,2) = 1.d0 * dn_scale * tot_mole_flux
            ! derivative energy wrt air mole fraction
            ! Jgdn(3,2) = 0.d0

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_T + &
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityg_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
            ! derivative water wrt temperature
            Jgdn(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
            ! derivative air wrt temperature
            Jgdn(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
            ! derivative energy wrt temperature
            Jgdn(3,3) = uH * dtot_mole_flux_dT + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hg_T

          case(GAS_STATE)
            ! derivative wrt gas pressure
            ! derivative total mole flux wrt gas pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_pg + &
              ! mole fraction has to be added in below since it differs for air
              ! and water
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityg_pg + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
            ! derivative water wrt gas pressure
            Jgdn(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_p(wat_comp_id,iphase)
            ! derivative air wrt gas pressure
            Jgdn(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_p(air_comp_id,iphase)
            ! derivative energy wrt gas pressure
            Jgdn(3,1) = uH * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hg_pg

            ! derivative wrt air pressure
            ! derivative water wrt air saturation
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_pa + &
              ! mole fraction has to be added in below since it differs for air
              ! and water
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityg_pa + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpadn
            Jgdn(1,2) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                       dn_scale * &
                       ! dXwg_pa for gas phase is stored in liquid phase of xmol_p
                       tot_mole_flux * gen_auxvar_dn%d%xmol_p(wat_comp_id,LIQUID_PHASE)
            ! derivative air wrt air saturation
            Jgdn(2,2) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                       dn_scale * &
                       ! dXag_pa for gas phase is stored in liquid phase of xmol_p
                       tot_mole_flux * gen_auxvar_dn%d%xmol_p(air_comp_id,LIQUID_PHASE)
            ! derivative energy wrt air saturation
            Jgdn(3,2) = uH * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hg_pa

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_T + &
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityg_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
              ! there is no derivative of mole fraction wrt temperature in
              ! gas state
            ! derivative water wrt temperature
            Jgdn(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
            ! derivative air wrt temperature
            Jgdn(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
            ! derivative energy wrt temperature
            Jgdn(3,3) = dtot_mole_flux_dT * uH + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hg_T

          case(TWO_PHASE_STATE)
            ! derivative wrt gas pressure
            ! pl = pg - pc and dpl_dpg = 1.  Therefore, we can use all the
            !   liquid pressure derivatives.
            ! derivative total mole flux wrt gas pressure
            dtot_mole_flux_dp = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_pg + &
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityg_pg + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
            ! derivative water wrt gas pressure
            Jgdn(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_p(wat_comp_id,iphase)
            ! derivative air wrt gas pressure
            Jgdn(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_p(air_comp_id,iphase)
            ! derivative energy wrt gas pressure
            Jgdn(3,1) = uH * dtot_mole_flux_dp + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hg_pg

            ! derivative wrt gas saturation
            ! derivative total mole flux wrt gas saturation
            dtot_mole_flux_dsatg = &
              ! relative permeability
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityg_satg
            ! derivative water wrt gas saturation
            Jgdn(1,2) = xmol(wat_comp_id) * dtot_mole_flux_dsatg
            ! derivative air wrt gas saturation
            Jgdn(2,2) = xmol(air_comp_id) * dtot_mole_flux_dsatg
            ! derivative energy wrt gas saturation
            Jgdn(3,2) = dtot_mole_flux_dsatg * uH

            ! derivative wrt temperature
            ! derivative total mole flux wrt temperature
            dtot_mole_flux_dT = &
              ! ave. liquid density
              q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_T + &
              ! liquid mobility
              dn_scale * &
              tot_mole_flux / mobility * gen_auxvar_dn%d%mobilityg_T + &
              ! pressure gradient
              tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
            ! derivative water wrt temperature
            Jgdn(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_T(wat_comp_id,iphase)
            ! derivative air wrt temperature
            Jgdn(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%xmol_T(air_comp_id,iphase)
            ! derivative energy wrt temperature
            Jgdn(3,3) = dtot_mole_flux_dT * uH + &
                       dn_scale * &
                       tot_mole_flux * gen_auxvar_dn%d%Hg_T
        end select
        Jup = Jup + Jgup
        Jdn = Jdn + Jgdn
      endif
    endif
  endif
#endif
  ! CONVECTION
#endif

#ifdef DIFFUSION
  if (.not.general_immiscible) then
  ! add in gas component diffusion in gas and liquid phases
!#if 0
#ifdef LIQUID_DIFFUSION
  iphase = LIQUID_PHASE
  sat_up = gen_auxvar_up%sat(iphase)
  sat_dn = gen_auxvar_dn%sat(iphase)
  dsatup = 1.d0
  dsatdn = 1.d0
  ! by changing #if 1 -> 0, gas component is allowed to diffuse in liquid
  ! phase even if the phase does not exist.
#if 1
  if (sqrt(sat_up*sat_dn) > eps) then
#else
  if (sat_up > eps .or. sat_dn > eps) then
    ! for now, if liquid state neighboring gas, we allow for minute
    ! diffusion in liquid phase.
    if (iphase == option%liquid_phase) then
      if ((sat_up > eps .or. sat_dn > eps)) then
        ! sat_up = max(sat_up,eps)
        if (sat_up < eps) then
          sat_up = eps
          dsatup = 0.d0
        endif
        ! sat_dn = max(sat_dn,eps)
        if (sat_dn < eps) then
          sat_dn = eps
          dsatdn = 0.d0
        endif
      endif
    endif
#endif
    if (general_harmonic_diff_density) then
      ! density_ave in this case is not used.
      density_ave = 1.d0
      den_up = gen_auxvar_up%den(iphase)
      den_dn = gen_auxvar_dn%den(iphase)
      ddensity_ave_dden_up = 0.d0
      ddensity_ave_dden_dn = 0.d0
      tempreal = 1.d0
    else
      ! den_up and den_dn are not used in this case
      den_up = 1.d0
      den_dn = 1.d0
      ! we use upstream weighting when iphase is not equal, otherwise
      ! arithmetic with 50/50 weighting
      density_ave = GeneralAverageDensity(iphase, &
                                      global_auxvar_up%istate, &
                                      global_auxvar_dn%istate, &
                                      gen_auxvar_up%den, &
                                      gen_auxvar_dn%den, &
                                      ddensity_ave_dden_up, &
                                      ddensity_ave_dden_dn)
      ! used to zero out derivative below
      tempreal = 0.d0
    endif
    stpd_up = sat_up*material_auxvar_up%tortuosity* &
              gen_auxvar_up%effective_porosity*den_up
    stpd_dn = sat_dn*material_auxvar_dn%tortuosity* &
              gen_auxvar_dn%effective_porosity*den_dn

    dstpd_up_dporup = stpd_up / gen_auxvar_up%effective_porosity
    dstpd_dn_dpordn = stpd_dn / gen_auxvar_dn%effective_porosity
    dstpd_up_dsatup = stpd_up / sat_up
    dstpd_dn_dsatdn = stpd_dn / sat_dn
    dstpd_up_ddenup = tempreal * stpd_up / den_up
    dstpd_dn_ddendn = tempreal * stpd_dn / den_dn
    ! units = [mole/m^4 bulk]
    tempreal = stpd_up*dist_dn+stpd_dn*dist_up
    stpd_ave_over_dist = stpd_up*stpd_dn/tempreal
    dstpd_ave_over_dist_dstpd_up = (stpd_dn-stpd_ave_over_dist*dist_dn)/tempreal
    dstpd_ave_over_dist_dstpd_dn = (stpd_up-stpd_ave_over_dist*dist_up)/tempreal

    if (general_diffuse_xmol) then ! delta of mole fraction
      delta_xmol = gen_auxvar_up%xmol(air_comp_id,iphase) - &
                   gen_auxvar_dn%xmol(air_comp_id,iphase)
      delta_X_whatever = delta_xmol
      delta_X_whatever_dxmolup = 1.d0
      delta_X_whatever_dxmoldn = -1.d0
      if (general_salt) then
        delta_xsmol = gen_auxvar_up%xmol(salt_comp_id,iphase) - &
                      gen_auxvar_dn%xmol(salt_comp_id,iphase)
      endif
    else ! delta of mass fraction
      xmol_air_up = gen_auxvar_up%xmol(air_comp_id,iphase)
      xmol_air_dn = gen_auxvar_dn%xmol(air_comp_id,iphase)
      tempreal = (xmol_air_up*fmw_comp(2) + (1.d0-xmol_air_up)*fmw_comp(1))
      xmass_air_up = xmol_air_up*fmw_comp(2) / tempreal
      dxmass_air_up_dxmol_air_up = (fmw_comp(2) - xmass_air_up * (fmw_comp(2) - fmw_comp(1))) / tempreal
      tempreal = (xmol_air_dn*fmw_comp(2) + (1.d0-xmol_air_dn)*fmw_comp(1))
      xmass_air_dn = xmol_air_dn*fmw_comp(2) / tempreal
      dxmass_air_dn_dxmol_air_dn = (fmw_comp(2) - xmass_air_dn * (fmw_comp(2) - fmw_comp(1))) / tempreal
      delta_xmass = xmass_air_up - xmass_air_dn
      delta_X_whatever = delta_xmass
      delta_X_whatever_dxmolup = 1.d0 * dxmass_air_up_dxmol_air_up
      delta_X_whatever_dxmoldn = -1.d0 * dxmass_air_dn_dxmol_air_dn
    endif

    ! units = mole/sec
    dtot_mole_flux_ddeltaX = density_ave * stpd_ave_over_dist * &
                             general_parameter%diffusion_coefficient(iphase) * &
                             area
    if (general_salt) then
      dtot_mole_flux_ddeltaS = density_ave * stpd_ave_over_dist * &
                               general_parameter%diffusion_coefficient(PRECIPITATE_PHASE) * &
                               area
    endif

    tot_mole_flux = dtot_mole_flux_ddeltaX * delta_X_whatever
    dtot_mole_flux_dstpd = tot_mole_flux / stpd_ave_over_dist
    dtot_mole_flux_ddenave = tot_mole_flux / density_ave
    Res(wat_comp_id) = Res(wat_comp_id) - tot_mole_flux
    Res(air_comp_id) = Res(air_comp_id) + tot_mole_flux
    if (general_salt) then
      tot_mole_flux1 = dtot_mole_flux_ddeltaS * delta_xsmol
      dtot_mole_flux_dstpd1 = tot_mole_flux1 / stpd_ave_over_dist
      dtot_mole_flux_ddenave1 = tot_mole_flux1 / density_ave
      Res(wat_comp_id) = Res(wat_comp_id) - tot_mole_flux1
      Res(salt_comp_id) = Res(salt_comp_id) + tot_mole_flux1
    endif

    if (analytical_derivatives) then

      Jlup = 0.d0
      Jldn = 0.d0
      select case(global_auxvar_up%istate)
        case(LIQUID_STATE)
          ! derivative wrt liquid pressure
          ! derivative total mole flux wrt liquid pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            (dstpd_up_dporup * gen_auxvar_up%d%por_p + &
            ! if density harmonic averaged
             dstpd_up_ddenup * gen_auxvar_up%d%denl_pl) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%denl_pl
          ! derivative water wrt liquid pressure
          Jlup(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt liquid pressure
          Jlup(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt liquid pressure
          Jlup(3,1) = 0.d0

          ! derivative wrt air mole fraction
          ! derivative water wrt air mole fraction
          Jlup(1,2) = -1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup
          ! derivative air wrt air mole fraction
          Jlup(2,2) = 1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup
          ! derivative energy wrt air mole fraction
          ! Jlup(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            dstpd_up_ddenup * gen_auxvar_up%d%denl_T + &
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%denl_T
            ! diffusion coefficient derivative wrt temperature
          ! derivative water wrt temperature
          Jlup(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jlup(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jlup(3,3) = 0.d0

        case(GAS_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            (dstpd_up_dporup * gen_auxvar_up%d%por_p + &
             dstpd_up_ddenup * gen_auxvar_up%d%denl_pl) + &
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%denl_pl
          ! derivative water wrt gas pressure
          Jlup(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jlup(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jlup(3,1) = 0.d0

          ! derivative wrt air pressure
          dtot_mole_flux_dp = &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup / &
            gen_auxvar_up%d%Hc
          ! derivative water wrt air saturation
           Jlup(1,2) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt air saturation
           Jlup(2,2) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt air saturation
           Jlup(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            dstpd_up_ddenup * gen_auxvar_up%d%denl_T + &
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%denl_T + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup  * &
            (-1.d0) * gen_auxvar_up%xmol(air_comp_id,LIQUID_PHASE) / &
            gen_auxvar_up%d%Hc * gen_auxvar_up%d%Hc_T
          ! diffusion coefficient derivative wrt temperature
          ! derivative water wrt temperature
          Jlup(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jlup(2,3) = dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jlup(3,3) = 0.d0

        case(TWO_PHASE_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            (dstpd_up_dporup * gen_auxvar_up%d%por_p + &
             dstpd_up_ddenup * gen_auxvar_up%d%denl_pl) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%denl_pl + &
            ! air mole fraction
            1.d0 * & ! xmolup - xmoldn, not -1 in docs
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup * &
            gen_auxvar_up%d%xmol_p(air_comp_id,LIQUID_PHASE)
          ! derivative water wrt gas pressure
          Jlup(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jlup(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jlup(3,1) = 0.d0

          ! derivative wrt gas saturation
          ! derivative total mole flux wrt gas saturation
          dtot_mole_flux_dsatg = &
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            dstpd_up_dsatup * dsatup * (-1.d0) ! satl -> satg
          ! derivative water wrt gas saturation
          Jlup(1,2) = -1.d0 * dtot_mole_flux_dsatg
          ! derivative air wrt gas saturation
          Jlup(2,2) = 1.d0 * dtot_mole_flux_dsatg
          ! derivative energy wrt gas saturation
          Jlup(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            dstpd_up_ddenup * gen_auxvar_up%d%denl_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%denl_T + &
            ! dispersion coefficient
            ! air mole fraction
            1.d0 * & ! xmolup - xmoldn, not -1 in docs
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup * &
            gen_auxvar_up%d%xmol_T(air_comp_id,LIQUID_PHASE)
          ! derivative water wrt temperature
          Jlup(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jlup(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jlup(3,3) = 0.d0
      end select
      select case(global_auxvar_dn%istate)
        case(LIQUID_STATE)
          ! derivative wrt liquid pressure
          ! derivative total mole flux wrt liquid pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%denl_pl) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_pl
          ! derivative water wrt liquid pressure
          Jldn(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt liquid pressure
          Jldn(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt liquid pressure
          Jldn(3,1) = 0.d0

          ! derivative wrt air mole fraction
          ! derivative water wrt air mole fraction
          Jldn(1,2) = -1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn
          ! derivative air wrt air mole fraction
          Jldn(2,2) = 1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn
          ! derivative energy wrt air mole fraction
          ! Jldn(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%denl_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_T
            ! diffusion coefficient derivative wrt temperature
          ! derivative water wrt temperature
          Jldn(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jldn(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jldn(3,3) = 0.d0

        case(GAS_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%denl_pl) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_pl
          ! derivative water wrt gas pressure
          Jldn(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jldn(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jldn(3,1) = 0.d0

          ! derivative wrt air pressure
          dtot_mole_flux_dp = &
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn / &
            gen_auxvar_dn%d%Hc
          ! derivative water wrt air saturation
           Jldn(1,2) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt air saturation
           Jldn(2,2) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt air saturation
           Jldn(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%denl_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_T + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn  * &
            (-1.d0) * gen_auxvar_dn%xmol(air_comp_id,LIQUID_PHASE) / &
            gen_auxvar_dn%d%Hc * gen_auxvar_dn%d%Hc_T
          ! diffusion coefficient derivative wrt temperature
          ! derivative water wrt temperature
          Jldn(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jldn(2,3) = dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jldn(3,3) = 0.d0

        case(TWO_PHASE_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%denl_pl) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_pl + &
            ! air mole fraction
            1.d0 * & ! xmoldn - xmoldn, not -1 in docs
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_p(air_comp_id,LIQUID_PHASE)
          ! derivative water wrt gas pressure
          Jldn(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jldn(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jldn(3,1) = 0.d0

          ! derivative wrt gas saturation
          ! derivative total mole flux wrt gas saturation
          dtot_mole_flux_dsatg = &
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_dsatdn * dsatdn * (-1.d0) ! satl -> satg
          ! derivative water wrt gas saturation
          Jldn(1,2) = -1.d0 * dtot_mole_flux_dsatg
          ! derivative air wrt gas saturation
          Jldn(2,2) = 1.d0 * dtot_mole_flux_dsatg
          ! derivative energy wrt gas saturation
          Jldn(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%denl_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_T + &
            ! dispersion coefficient
            ! air mole fraction
            1.d0 * & ! xmoldn - xmoldn, not -1 in docs
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_T(air_comp_id,LIQUID_PHASE)
          ! derivative water wrt temperature
          Jldn(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jldn(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jldn(3,3) = 0.d0
      end select
      Jup = Jup + Jlup
      Jdn = Jdn + Jldn
    endif
  endif
#endif
!#if 0
#ifdef GAS_DIFFUSION
  iphase = GAS_PHASE
  sat_up = gen_auxvar_up%sat(iphase)
  sat_dn = gen_auxvar_dn%sat(iphase)
  !geh: i am not sure why both of these conditionals were included.  seems
  !     like the latter would never be false.
  if (sqrt(sat_up*sat_dn) > eps) then
    dsatup = 1.d0
    dsatdn = 1.d0
    if (general_harmonic_diff_density) then
      ! density_ave in this case is not used.
      density_ave = 1.d0
      den_up = gen_auxvar_up%den(iphase)
      den_dn = gen_auxvar_dn%den(iphase)
      ddensity_ave_dden_up = 0.d0
      ddensity_ave_dden_dn = 0.d0
      tempreal = 1.d0
    else
      ! den_up and den_dn are not used in this case
      den_up = 1.d0
      den_dn = 1.d0
      ! we use upstream weighting when iphase is not equal, otherwise
      ! arithmetic with 50/50 weighting
      density_ave = GeneralAverageDensity(iphase, &
                                      global_auxvar_up%istate, &
                                      global_auxvar_dn%istate, &
                                      gen_auxvar_up%den, &
                                      gen_auxvar_dn%den, &
                                      ddensity_ave_dden_up, &
                                      ddensity_ave_dden_dn)
      ! used to zero out derivative below
      tempreal = 0.d0
    endif
    stpd_up = sat_up*material_auxvar_up%tortuosity* &
              gen_auxvar_up%effective_porosity*den_up
    stpd_dn = sat_dn*material_auxvar_dn%tortuosity* &
              gen_auxvar_dn%effective_porosity*den_dn

    dstpd_up_dporup = stpd_up / gen_auxvar_up%effective_porosity
    dstpd_dn_dpordn = stpd_dn / gen_auxvar_dn%effective_porosity
    dstpd_up_dsatup = stpd_up / sat_up
    dstpd_dn_dsatdn = stpd_dn / sat_dn
    dstpd_up_ddenup = tempreal * stpd_up / den_up
    dstpd_dn_ddendn = tempreal * stpd_dn / den_dn
    ! units = [mole/m^4 bulk]
    tempreal = stpd_up*dist_dn+stpd_dn*dist_up
    stpd_ave_over_dist = stpd_up*stpd_dn/tempreal
    dstpd_ave_over_dist_dstpd_up = (stpd_dn-stpd_ave_over_dist*dist_dn)/tempreal
    dstpd_ave_over_dist_dstpd_dn = (stpd_up-stpd_ave_over_dist*dist_up)/tempreal

    if (general_diffuse_xmol) then ! delta of mole fraction
      delta_xmol = gen_auxvar_up%xmol(air_comp_id,iphase) - &
                   gen_auxvar_dn%xmol(air_comp_id,iphase)
      delta_X_whatever = delta_xmol
      delta_X_whatever_dxmolup = 1.d0
      delta_X_whatever_dxmoldn = -1.d0
    else ! delta of mass fraction
      xmol_air_up = gen_auxvar_up%xmol(air_comp_id,iphase)
      xmol_air_dn = gen_auxvar_dn%xmol(air_comp_id,iphase)
      tempreal = (xmol_air_up*fmw_comp(2) + (1.d0-xmol_air_up)*fmw_comp(1))
      xmass_air_up = xmol_air_up*fmw_comp(2) / tempreal
      dxmass_air_up_dxmol_air_up = (fmw_comp(2) - xmass_air_up * (fmw_comp(2) - fmw_comp(1))) / tempreal
      tempreal = (xmol_air_dn*fmw_comp(2) + (1.d0-xmol_air_dn)*fmw_comp(1))
      xmass_air_dn = xmol_air_dn*fmw_comp(2) / tempreal
      dxmass_air_dn_dxmol_air_dn = (fmw_comp(2) - xmass_air_dn * (fmw_comp(2) - fmw_comp(1))) / tempreal
      delta_xmass = xmass_air_up - xmass_air_dn
      delta_X_whatever = delta_xmass
      delta_X_whatever_dxmolup = 1.d0 * dxmass_air_up_dxmol_air_up
      delta_X_whatever_dxmoldn = -1.d0 * dxmass_air_dn_dxmol_air_dn
    endif
    ! need to account for multiple phases
    ! Eq. 1.9b.  The gas density is added below
    if (general_temp_dep_gas_air_diff) then
      temp_ave = 0.5d0*(gen_auxvar_up%temp+gen_auxvar_dn%temp)
      pressure_ave = 0.5d0*(gen_auxvar_up%pres(iphase)+ &
                            gen_auxvar_dn%pres(iphase))
      tempreal = (temp_ave+T273K)/T273K
      diffusion_scale = tempreal**1.8d0 * 101325.d0 / pressure_ave
                             ! 0.9d0 = 0.5 * 1.8
      ddiffusion_coef_dTup = 0.9d0 * diffusion_scale / (tempreal * T273K)
      ddiffusion_coef_dTdn = ddiffusion_coef_dTup
      ddiffusion_coef_dpup = -1.d0 * diffusion_scale / pressure_ave * 0.5d0
      ddiffusion_coef_dpdn = ddiffusion_coef_dpup
    else
      diffusion_scale = 1.d0
      ddiffusion_coef_dTup = 0.d0
      ddiffusion_coef_dTdn = 0.d0
      ddiffusion_coef_dpup = 0.d0
      ddiffusion_coef_dpdn = 0.d0
    endif
    ! units = mole/sec
    dtot_mole_flux_ddeltaX = density_ave * stpd_ave_over_dist * &
                             diffusion_scale * &
                             general_parameter%diffusion_coefficient(iphase) * &
                             area
    tot_mole_flux = dtot_mole_flux_ddeltaX * delta_X_whatever
    dtot_mole_flux_dstpd = tot_mole_flux / stpd_ave_over_dist
    dtot_mole_flux_ddiffusion_coef = tot_mole_flux / diffusion_scale
    dtot_mole_flux_ddenave = tot_mole_flux / density_ave
    Res(wat_comp_id) = Res(wat_comp_id) - tot_mole_flux
    Res(air_comp_id) = Res(air_comp_id) + tot_mole_flux

    if (analytical_derivatives) then

      Jgup = 0.d0
      Jgdn = 0.d0
      select case(global_auxvar_up%istate)
        case(LIQUID_STATE)
          ! derivative wrt liquid pressure
          ! derivative total mole flux wrt liquid pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            (dstpd_up_dporup * gen_auxvar_up%d%por_p + &
            ! if density harmonic averaged
             dstpd_up_ddenup * gen_auxvar_up%d%deng_pg) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%deng_pg + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dpup
          ! derivative water wrt liquid pressure
          Jgup(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt liquid pressure
          Jgup(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt liquid pressure
          Jgup(3,1) = 0.d0

          ! derivative wrt air mole fraction
          ! derivative water wrt air mole fraction
          Jgup(1,2) = -1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup
          ! derivative air wrt air mole fraction
          Jgup(2,2) = 1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup
          ! derivative energy wrt air mole fraction
          ! Jgup(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            dstpd_up_ddenup * gen_auxvar_up%d%deng_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%deng_T + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dTup
          ! derivative water wrt temperature
          Jgup(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jgup(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jgup(3,3) = 0.d0

        case(GAS_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            (dstpd_up_dporup * gen_auxvar_up%d%por_p + &
             dstpd_up_ddenup * gen_auxvar_up%d%deng_pg) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%deng_pg + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dpup + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup * &
            gen_auxvar_up%d%xmol_p(air_comp_id,GAS_PHASE)
          ! derivative water wrt gas pressure
          Jgup(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jgup(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jgup(3,1) = 0.d0

          ! derivative wrt air pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
             dstpd_up_ddenup * gen_auxvar_up%d%deng_pa + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%deng_pa + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup * &
                              ! liquid phase is hijacked to store \dpa
            gen_auxvar_up%d%xmol_p(air_comp_id,LIQUID_PHASE)
          ! derivative water wrt air saturation
           Jgup(1,2) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt air saturation
           Jgup(2,2) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt air saturation
           Jgup(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            dstpd_up_ddenup * gen_auxvar_up%d%deng_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%deng_T + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dTup + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup * &
            gen_auxvar_up%d%xmol_T(air_comp_id,GAS_PHASE)
          ! derivative water wrt temperature
          Jgup(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jgup(2,3) = dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jgup(3,3) = 0.d0

        case(TWO_PHASE_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            (dstpd_up_dporup * gen_auxvar_up%d%por_p + &
             dstpd_up_ddenup * gen_auxvar_up%d%deng_pg) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%deng_pg + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dpup + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup * &
            gen_auxvar_up%d%xmol_p(air_comp_id,GAS_PHASE)
          ! derivative water wrt gas pressure
          Jgup(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jgup(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jgup(3,1) = 0.d0

          ! derivative wrt gas saturation
          ! derivative total mole flux wrt gas saturation
          dtot_mole_flux_dsatg = &
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            dstpd_up_dsatup * dsatup ! satg
          ! derivative water wrt gas saturation
          Jgup(1,2) = -1.d0 * dtot_mole_flux_dsatg
          ! derivative air wrt gas saturation
          Jgup(2,2) = 1.d0 * dtot_mole_flux_dsatg
          ! derivative energy wrt gas saturation
          Jgup(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_up * &
            dstpd_up_ddenup * gen_auxvar_up%d%deng_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_up * &
            gen_auxvar_up%d%deng_T + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dTup  + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmolup * &
            gen_auxvar_up%d%xmol_T(air_comp_id,GAS_PHASE)
          ! derivative water wrt temperature
          Jgup(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jgup(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jgup(3,3) = 0.d0
      end select
      select case(global_auxvar_dn%istate)
        case(LIQUID_STATE)
          ! derivative wrt liquid pressure
          ! derivative total mole flux wrt liquid pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%deng_pg) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_pg + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dpdn
          ! derivative water wrt liquid pressure
          Jgdn(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt liquid pressure
          Jgdn(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt liquid pressure
          Jgdn(3,1) = 0.d0

          ! derivative wrt air mole fraction
          ! derivative water wrt air mole fraction
          Jgdn(1,2) = -1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn
          ! derivative air wrt air mole fraction
          Jgdn(2,2) = 1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn
          ! derivative energy wrt air mole fraction
          ! Jgdn(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%deng_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_T + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dTdn
          ! derivative water wrt temperature
          Jgdn(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jgdn(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jgdn(3,3) = 0.d0

        case(GAS_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%deng_pg) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_pg + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dpdn + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_p(air_comp_id,GAS_PHASE)
          ! derivative water wrt gas pressure
          Jgdn(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jgdn(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jgdn(3,1) = 0.d0

          ! derivative wrt air pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
             dstpd_dn_ddendn * gen_auxvar_dn%d%deng_pa + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_pa + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
                              ! liquid phase is hijacked to store \dpa
            gen_auxvar_dn%d%xmol_p(air_comp_id,LIQUID_PHASE)
          ! derivative water wrt air saturation
           Jgdn(1,2) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt air saturation
           Jgdn(2,2) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt air saturation
           Jgdn(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%deng_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_T + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dTdn + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_T(air_comp_id,GAS_PHASE)
          ! derivative water wrt temperature
          Jgdn(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jgdn(2,3) = dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jgdn(3,3) = 0.d0

        case(TWO_PHASE_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%deng_pg) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_pg + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dpdn + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_p(air_comp_id,GAS_PHASE)
          ! derivative water wrt gas pressure
          Jgdn(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jgdn(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jgdn(3,1) = 0.d0

          ! derivative wrt gas saturation
          ! derivative total mole flux wrt gas saturation
          dtot_mole_flux_dsatg = &
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_dsatdn * dsatdn ! satg
          ! derivative water wrt gas saturation
          Jgdn(1,2) = -1.d0 * dtot_mole_flux_dsatg
          ! derivative air wrt gas saturation
          Jgdn(2,2) = 1.d0 * dtot_mole_flux_dsatg
          ! derivative energy wrt gas saturation
          Jgdn(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%deng_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_T + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dTdn  + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_T(air_comp_id,GAS_PHASE)
          ! derivative water wrt temperature
          Jgdn(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jgdn(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jgdn(3,3) = 0.d0
      end select
      Jup = Jup + Jgup
      Jdn = Jdn + Jgdn
    endif
  endif
#endif
! DIFFUSION
  endif ! if (.not.general_immiscible)
#endif

#ifdef CONDUCTION
  ! add heat conduction flux
  sat_up = gen_auxvar_up%sat(option%liquid_phase)
  sat_dn = gen_auxvar_dn%sat(option%liquid_phase)

  ! derive wet and dry conductivities with anisotropy tensor and direction
  call thermal_cc_up%thermal_conductivity_function% &
       TCondTensorToScalar(dist,option)

  call thermal_cc_dn%thermal_conductivity_function% &
       TCondTensorToScalar(dist,option)

  ! thermal conductivity a function of temperature and liquid saturation
  call thermal_cc_up%thermal_conductivity_function%CalculateTCond(sat_up, &
       gen_auxvar_up%temp,gen_auxvar_up%effective_porosity, &
       k_eff_up,dkeff_up_dsatlup,dkeff_up_dTup,option)

  call thermal_cc_dn%thermal_conductivity_function%CalculateTCond(sat_dn, &
       gen_auxvar_dn%temp,gen_auxvar_dn%effective_porosity, &
       k_eff_dn,dkeff_dn_dsatldn,dkeff_dn_dTdn,option)

  if (k_eff_up > 0.d0 .or. k_eff_dn > 0.d0) then
    tempreal = k_eff_up*dist_dn + k_eff_dn*dist_up
    k_eff_ave = k_eff_up*k_eff_dn/tempreal
    dkeff_ave_dkeffup = (k_eff_dn-k_eff_ave*dist_dn)/tempreal
    dkeff_ave_dkeffdn = (k_eff_up-k_eff_ave*dist_up)/tempreal
  else
    k_eff_ave = 0.d0
    dkeff_ave_dkeffup = 0.d0
    dkeff_ave_dkeffdn = 0.d0
  endif
  ! units:
  ! k_eff = W/K-m = J/s/K-m
  ! delta_temp = K
  ! area = m^2
  ! heat_flux = k_eff * delta_temp * area = J/s
  ! 1.0E-6 term accounts for change in units: J/s -> MJ/s

  delta_temp = gen_auxvar_up%temp - gen_auxvar_dn%temp
  dheat_flux_ddelta_temp = k_eff_ave * area * 1.d-6 ! J/s -> MJ/s
  heat_flux = dheat_flux_ddelta_temp * delta_temp
  dheat_flux_dkeff_ave = area * 1.d-6 * delta_temp

  ! MJ/s or MW
  Res(energy_id) = Res(energy_id) + heat_flux

  if (analytical_derivatives) then
    Jcup = 0.d0
    Jcdn = 0.d0
    select case(global_auxvar_up%istate)
      case(LIQUID_STATE,GAS_STATE)
        ! only derivative is energy wrt temperature
        ! derivative energy wrt temperature
        ! positive for upwind
        Jcup(3,3) = 1.d0 * dheat_flux_ddelta_temp

      case(TWO_PHASE_STATE)
        ! only derivatives are energy wrt saturation and temperature
        ! derivative energy wrt gas saturation
        Jcup(3,2) = dheat_flux_dkeff_ave * dkeff_ave_dkeffup * &
                    dkeff_up_dsatlup * (-1.d0) ! satl -> satg
        ! derivative energy wrt temperature
        ! positive for upwind


        Jcup(3,3) = 1.d0 * dheat_flux_ddelta_temp

    end select
    select case(global_auxvar_dn%istate)
      case(LIQUID_STATE,GAS_STATE)
        ! only derivative is energy wrt temperature
        ! derivative energy wrt temperature
        ! positive for upwind


        Jcdn(3,3) = -1.d0 * dheat_flux_ddelta_temp

      case(TWO_PHASE_STATE)
        ! only derivatives are energy wrt saturation and temperature
        ! derivative energy wrt gas saturation
        Jcdn(3,2) = dheat_flux_dkeff_ave * dkeff_ave_dkeffdn * &
                    dkeff_dn_dsatldn * (-1.d0) ! satl -> satg
        ! derivative energy wrt temperature
        ! positive for upwind


        Jcdn(3,3) = -1.d0 * dheat_flux_ddelta_temp

    end select
    Jup = Jup + Jcup
    Jdn = Jdn + Jcdn
  endif
! CONDUCTION
#endif

end subroutine GeneralFlux

! ************************************************************************** !

subroutine GeneralBCFlux(ibndtype,auxvar_mapping,auxvars, &
                         gen_auxvar_up,global_auxvar_up, &
                         gen_auxvar_dn,global_auxvar_dn, &
                         material_auxvar_dn, &
                         thermal_cc_dn, &
                         area,dist,upwind_direction_, &
                         general_parameter, &
                         option,v_darcy,Res,J, &
                         analytical_derivatives, &
                         update_upwind_direction_, &
                         count_upwind_direction_flip_, &
                         debug_connection)
  !
  ! Computes the boundary flux terms for the residual
  !
  ! Author: Glenn Hammond
  ! Date: 03/09/11
  !
  use Option_module
  use Material_Aux_module
  use Fracture_module
  use Klinkenberg_module
  use Upwind_Direction_module
  use Characteristic_Curves_Thermal_module
  use Utility_module

  implicit none

  type(option_type) :: option
  PetscInt :: ibndtype(1:option%nflowdof)
  PetscInt :: auxvar_mapping(GENERAL_MAX_INDEX)
  PetscReal :: auxvars(:) ! from aux_real_var array
  type(general_auxvar_type) :: gen_auxvar_up, gen_auxvar_dn
  type(global_auxvar_type) :: global_auxvar_up, global_auxvar_dn
  type(material_auxvar_type) :: material_auxvar_dn
  PetscReal :: area
  PetscReal :: dist(-1:3)
  PetscInt :: upwind_direction_(option%nphase)
  type(general_parameter_type) :: general_parameter
  PetscReal :: v_darcy(option%nphase)
  PetscReal :: Res(1:option%nflowdof)
  class(cc_thermal_type) :: thermal_cc_dn
  PetscReal :: J(option%nflowdof,option%nflowdof)
  PetscBool :: analytical_derivatives
  PetscBool :: update_upwind_direction_
  PetscBool :: count_upwind_direction_flip_
  PetscBool :: debug_connection
  PetscBool :: dirichlet_solute = PETSC_FALSE


  PetscInt :: wat_comp_id, air_comp_id, energy_id, salt_comp_id
  PetscInt :: iphase
  PetscInt :: bc_type
  PetscReal :: xmol(option%nflowspec)
  PetscReal :: density_ave, density_kg_ave
  PetscReal :: uH
  PetscReal :: perm_dn_adj(option%nphase)
  PetscReal :: perm_ave_over_dist
  PetscReal :: dist_gravity
  PetscReal :: delta_pressure, delta_xmol, delta_temp
  PetscReal :: delta_xsmol
  PetscReal :: gravity_term
  PetscReal :: mobility, q, kr
  PetscReal :: tot_mole_flux, tot_mole_flux1
  PetscReal :: sat_dn, perm_dn, den_dn
  PetscReal :: temp_ave, stpd_ave_over_dist
  PetscReal :: k_eff_dn, k_eff_ave, heat_flux
  PetscReal :: boundary_pressure
  PetscReal :: xmass_air_up, xmass_air_dn, delta_xmass
  PetscReal :: xmol_air_up, xmol_air_dn
  PetscReal :: tempreal
  PetscReal :: delta_X_whatever
  PetscReal :: wat_mole_flux, air_mole_flux, salt_mole_flux
  PetscBool :: upwind
  PetscBool :: water_cannot_flow_in

  ! Darcy flux
  PetscReal :: ddelta_pressure_dpdn
  PetscReal :: ddelta_pressure_dpadn
  PetscReal :: ddelta_pressure_dTdn
  PetscReal :: dv_darcy_ddelta_pressure
  PetscReal :: dv_darcy_dmobility

  PetscReal :: dn_scale
  PetscReal :: ddensity_kg_ave_dden_kg_up, ddensity_kg_ave_dden_kg_dn
  PetscReal :: ddensity_ave_dden_up, ddensity_ave_dden_dn
  PetscReal :: dtot_mole_flux_dp, dtot_mole_flux_dT, dtot_mole_flux_dsatg
  PetscReal :: dpl_dsatg
  PetscReal :: ddelta_pressure_pl
  PetscReal :: tot_mole_flux_ddel_pressure, tot_mole_flux_dmobility
  PetscReal :: xmol_bool

  ! Diffusion
  PetscReal :: stpd_dn
  PetscReal :: dstpd_dn_dpordn
  PetscReal :: dstpd_dn_dsatdn
  PetscReal :: dstpd_dn_ddendn
  PetscReal :: dsatdn
  PetscReal :: delta_X_whatever_dxmoldn
  PetscReal :: dxmass_air_dn_dxmol_air_dn
  PetscReal :: dtot_mole_flux_dstpd, dtot_mole_flux_ddeltaX
  PetscReal :: dtot_mole_flux_ddeltaS
  PetscReal :: dtot_mole_flux_dstpd1, dtot_mole_flux_ddenave1
  PetscReal :: dtot_mole_flux_ddenave
  PetscReal :: diffusion_scale
  PetscReal :: ddiffusion_coef_dTdn
  PetscReal :: ddiffusion_coef_dpdn
  PetscReal :: dtot_mole_flux_ddiffusion_coef
  PetscReal :: dstpd_ave_over_dist_dstpd_dn
  PetscReal :: pressure_ave
  PetscReal :: perm3(3)

  ! Conduction
  PetscReal :: dkeff_dn_dsatldn, dkeff_dn_dTdn
  PetscReal :: dkeff_ave_dkeffdn
  PetscReal :: dheat_flux_ddelta_temp, dheat_flux_dkeff_ave

  ! DELETE

  PetscReal :: Jl(3,3)
  PetscReal :: Jg(3,3)
  PetscReal :: Jc(3,3)

  !Non darcy
  PetscInt :: idof

  wat_comp_id = option%water_id
  air_comp_id = option%air_id
  energy_id = option%energy_id
  salt_comp_id = option%salt_id

  Res = 0.d0
  J = 0.d0
  v_darcy = 0.d0

  call PermeabilityTensorToScalar(material_auxvar_dn,dist,perm_dn)

#if 0
  ! Fracture permeability change only available for structured grid (Heeho)
  if (associated(material_auxvar_dn%fracture)) then
    call FracturePermEvaluate(material_auxvar_dn,perm_dn,temp_perm_dn, &
                              dummy_dperm_dn,dist)
    perm_dn = temp_perm_dn
  endif
#endif

  if (associated(klinkenberg)) then
    perm_dn_adj(1) = perm_dn
    perm3(:) = perm_dn
    perm3 = klinkenberg%Evaluate(perm3, &
                                 gen_auxvar_dn%pres(option%gas_phase))
    perm_dn_adj(2) = perm3(1)
  else
    perm_dn_adj(:) = perm_dn
  endif

#ifdef CONVECTION
#ifdef LIQUID_DARCY_FLUX
  iphase = LIQUID_PHASE
  mobility = 0.d0
  xmol_bool = 1.d0
  bc_type = ibndtype(iphase)
  select case(bc_type)
    ! figure out the direction of flow
    case(DIRICHLET_BC,HYDROSTATIC_BC,HYDROSTATIC_SEEPAGE_BC, &
         HYDROSTATIC_CONDUCTANCE_BC,DIRICHLET_SEEPAGE_BC)
      if (gen_auxvar_up%mobility(iphase) + &
          gen_auxvar_dn%mobility(iphase) > eps) then

        ! dist(0) = scalar - magnitude of distance
        ! gravity = vector(3)
        ! dist(1:3) = vector(3) - unit vector
        dist_gravity = dist(0) * dot_product(option%gravity,dist(1:3))

        if (bc_type == HYDROSTATIC_CONDUCTANCE_BC) then
          select case(iphase)
            case(LIQUID_PHASE)
              idof = auxvar_mapping(GENERAL_LIQUID_CONDUCTANCE_INDEX)
            case(GAS_PHASE)
              idof = auxvar_mapping(GENERAL_GAS_CONDUCTANCE_INDEX)
          end select
          perm_ave_over_dist = auxvars(idof)
        else
          perm_ave_over_dist = perm_dn_adj(iphase) / dist(0)
        endif

        boundary_pressure = gen_auxvar_up%pres(iphase)
        water_cannot_flow_in = PETSC_FALSE
        select case(bc_type)
          case(DIRICHLET_BC,HYDROSTATIC_BC)
          case default
             if (gen_auxvar_up%pres(option%capillary_pressure_id) > 0.d0) then
               water_cannot_flow_in = PETSC_TRUE
               boundary_pressure = gen_auxvar_up%pres(option%gas_phase)
             endif
        end select
        if (iphase == LIQUID_PHASE .and. &
            global_auxvar_up%istate == GAS_STATE) then
          ! the idea here is to accommodate a free surface boundary
          ! face.  this will not work for an interior grid cell as
          ! there should be capillary pressure in force.
          boundary_pressure = gen_auxvar_up%pres(option%gas_phase)
        endif
        density_kg_ave = GeneralAverageDensity(iphase, &
                                                global_auxvar_up%istate, &
                                                global_auxvar_dn%istate, &
                                                gen_auxvar_up%den_kg, &
                                                gen_auxvar_dn%den_kg, &
                                                ddensity_kg_ave_dden_kg_up, &
                                                ddensity_kg_ave_dden_kg_dn)
        ddensity_kg_ave_dden_kg_up = 0.d0 ! always
        gravity_term = density_kg_ave * dist_gravity
        delta_pressure = boundary_pressure - &
                          gen_auxvar_dn%pres(iphase) + &
                          gravity_term
        if (analytical_derivatives) then
          ddelta_pressure_dpdn = -1.d0 + dist_gravity * &
                                 ddensity_kg_ave_dden_kg_dn * &
                                 gen_auxvar_dn%d%denl_pl * fmw_comp(iphase)
          ddelta_pressure_dTdn = dist_gravity * ddensity_kg_ave_dden_kg_dn * &
                                 gen_auxvar_dn%d%denl_T * fmw_comp(iphase)
        endif
        if (bc_type == HYDROSTATIC_SEEPAGE_BC .or. &
            bc_type == HYDROSTATIC_CONDUCTANCE_BC) then
              ! flow in         ! boundary cell is <= pref
          if (delta_pressure > 0.d0 .and. water_cannot_flow_in) then
            delta_pressure = 0.d0
            if (analytical_derivatives) then
              option%io_buffer = 'HYDROSTATIC_CONDUCTANCE_BC and &
                &HYDROSTATIC_SEEPAGE_BC need to be verified in GeneralBCFlux().'
              call PrintErrMsg(option)
              ddelta_pressure_dpdn = 0.d0
              ddelta_pressure_dTdn = 0.d0
            endif
          endif
        endif

        if (bc_type == DIRICHLET_SEEPAGE_BC) then
          if (delta_pressure > 0.d0 .and. water_cannot_flow_in) then
            delta_pressure = 0.d0
            if (analytical_derivatives) then
              option%io_buffer = 'DIRCHLET_SEEPAGE_BC &
                &needs to be verified in GeneralBCFlux().'
              call PrintErrMsg(option)
            endif
          endif
        endif

        dn_scale = 0.d0
        upwind = UpwindDirection(upwind_direction_(iphase),delta_pressure, &
                                 .not.analytical_derivatives, &
                                 count_upwind_direction_flip_, &
                                 liq_bc_upwind_flip_count_by_res, &
                                 liq_bc_upwind_flip_count_by_jac)
        if (upwind) then
          mobility = gen_auxvar_up%mobility(iphase)
          xmol(:) = gen_auxvar_up%xmol(:,iphase)
          uH = gen_auxvar_up%H(iphase)
          kr=gen_auxvar_up%kr(iphase)
        else
          dn_scale = 1.d0
          mobility = gen_auxvar_dn%mobility(iphase)
          xmol(:) = gen_auxvar_dn%xmol(:,iphase)
          uH = gen_auxvar_dn%H(iphase)
          kr=gen_auxvar_dn%kr(iphase)
        endif

        ! only need average density if velocity > 0.
        density_ave = GeneralAverageDensity(iphase, &
                                            global_auxvar_up%istate, &
                                            global_auxvar_dn%istate, &
                                            gen_auxvar_up%den, &
                                            gen_auxvar_dn%den, &
                                            ddensity_ave_dden_up, &
                                            ddensity_ave_dden_dn)
        ddensity_ave_dden_up = 0.d0 ! always

        if (.not. general_non_darcy_flow) then
           ! v_darcy[m/sec] = perm[m^2] / dist[m] * kr[-] / mu[Pa-sec]
           !                    dP[Pa]]
           dv_darcy_ddelta_pressure = perm_ave_over_dist * mobility
           v_darcy(iphase) = dv_darcy_ddelta_pressure * delta_pressure
           dv_darcy_dmobility = perm_ave_over_dist * delta_pressure

        else
           !Taken from (Liu, 2014)
           call GeneralNonDarcyCorrectionBC(delta_pressure,density_kg_ave, &
                perm_ave_over_dist,kr,mobility,v_darcy(iphase), &
                dv_darcy_ddelta_pressure,dv_darcy_dmobility)
        endif
      endif
    case(NEUMANN_BC)
      xmol_bool = 0.d0
      dv_darcy_ddelta_pressure = 0.d0
      dv_darcy_dmobility = 0.d0
      ddensity_ave_dden_up = 0.d0
      ddensity_ave_dden_dn = 0.d0
      ddelta_pressure_dpdn = 0.d0
      ddelta_pressure_dTdn = 0.d0
      dn_scale = 0.d0
      select case(iphase)
        case(LIQUID_PHASE)
          idof = auxvar_mapping(GENERAL_LIQUID_FLUX_INDEX)
        case(GAS_PHASE)
          idof = auxvar_mapping(GENERAL_GAS_FLUX_INDEX)
      end select
      xmol = 0.d0
      !geh: we should read in the mole fraction for both phases as the
      !     enthalpy, etc. applies to phase, not pure component.
      xmol(iphase) = 1.d0
      !     if liquid flux is set to 0, SALT_MOLE_FRACTION becomes a dirichlet BC
      !     if liquid flux is non-zero, SALT_MOLE_FRACTION becomes mole fraction of flux fluid
      if (dabs(auxvars(idof)) > floweps) then
        v_darcy(iphase) = auxvars(idof)
        if (general_salt) then
          xmol(option%salt_id) = auxvars(GENERAL_LIQUID_STATE_S_MOLE_DOF)
          xmol(iphase) = 1.d0 - xmol(option%salt_id)
        endif
        if (v_darcy(iphase) > 0.d0) then
          density_ave = gen_auxvar_up%den(iphase)
          uH = gen_auxvar_up%H(iphase)
        else
          dn_scale = 1.d0
          density_ave = gen_auxvar_dn%den(iphase)
          uH = gen_auxvar_dn%H(iphase)
          ddensity_ave_dden_dn = 1.d0
        endif
      endif
    case default
      option%io_buffer = &
        'Boundary condition type not recognized in GeneralBCFlux phase loop.'
      call PrintErrMsg(option)
  end select
  if (dabs(v_darcy(iphase)) > 0.d0 .or. mobility > 0.d0) then
    ! q[m^3 phase/sec] = v_darcy[m/sec] * area[m^2]
    q = v_darcy(iphase) * area
    ! mole_flux[kmol phase/sec] = q[m^3 phase/sec] *
    !                             density_ave[kmol phase/m^3 phase]
    tot_mole_flux = q*density_ave
    tot_mole_flux_ddel_pressure = dv_darcy_ddelta_pressure * area * &
                                  density_ave
    tot_mole_flux_dmobility = dv_darcy_dmobility * area * density_ave
    ! comp_mole_flux[kmol comp/sec] = tot_mole_flux[kmol phase/sec] *
    !                                 xmol[kmol comp/kmol phase]
    wat_mole_flux = tot_mole_flux * xmol(wat_comp_id)
    air_mole_flux = tot_mole_flux * xmol(air_comp_id)
    Res(wat_comp_id) = Res(wat_comp_id) + wat_mole_flux
    Res(air_comp_id) = Res(air_comp_id) + air_mole_flux
    Res(energy_id) = Res(energy_id) + tot_mole_flux * uH
    if (general_salt) then
      salt_mole_flux = tot_mole_flux * xmol(salt_comp_id)
      Res(salt_comp_id) = Res(salt_comp_id) + salt_mole_flux
    endif
    if (analytical_derivatives) then
      Jl = 0.d0
      select case(global_auxvar_dn%istate)
        case(LIQUID_STATE)
          ! derivative wrt liquid pressure
          ! derivative total mole flux wrt liquid pressure
          dtot_mole_flux_dp = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn *gen_auxvar_dn%d%denl_pl + &
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityl_pl + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
          ! derivative water wrt liquid pressure
          Jl(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp
          ! derivative air wrt liquid pressure
          Jl(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp
          ! derivative energy wrt liquid pressure
          Jl(3,1) = uH * dtot_mole_flux_dp + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hl_pl

          ! derivative wrt air mole fraction
          ! derivative water wrt air mole fraction
          Jl(1,2) = -1.d0 * dn_scale * tot_mole_flux * xmol_bool
          ! derivative air wrt air mole fraction
          Jl(2,2) = 1.d0 * dn_scale * tot_mole_flux * xmol_bool
          ! derivative energy wrt air mole fraction
          ! Jl(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn * gen_auxvar_dn%d%denl_T + &
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityl_T + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
          ! derivative water wrt temperature
          Jl(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jl(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jl(3,3) = uH * dtot_mole_flux_dT + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hl_T

        case(GAS_STATE)
          ! derivative wrt gas pressure
          ! pl = pg - pc and dpl_dpg = 1.  Therefore, we can use all the
          !   liquid pressure derivatives.
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn * gen_auxvar_dn%d%denl_pl + &
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityl_pl + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
          ! derivative water wrt gas pressure
          Jl(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_p(wat_comp_id,iphase)
          ! derivative air wrt gas pressure
          Jl(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_p(air_comp_id,iphase)
          ! derivative energy wrt gas pressure
          Jl(3,1) = uH * dtot_mole_flux_dp + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hl_pl

          ! derivative wrt air pressure
          ! derivative water wrt air saturation
          ! Jl(1,2) = 0.d0
          ! derivative air wrt air saturation
          ! Jl(2,2) = 0.d0
          ! derivative energy wrt air saturation
          ! Jl(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn * gen_auxvar_dn%d%denl_T + &
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityl_T + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
            ! there is no derivative of mole fraction wrt temperature in
            ! gas state
          ! derivative water wrt temperature
          Jl(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jl(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jl(3,3) = dtot_mole_flux_dT * uH + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hl_T

        case(TWO_PHASE_STATE)
          ! derivative wrt gas pressure
          ! pl = pg - pc and dpl_dpg = 1.  Therefore, we can use all the
          !   liquid pressure derivatives.
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn *gen_auxvar_dn%d%denl_pl + &
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityl_pl + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
          ! derivative water wrt gas pressure
          Jl(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_p(wat_comp_id,iphase)
          ! derivative air wrt gas pressure
          Jl(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_p(air_comp_id,iphase)
          ! derivative energy wrt gas pressure
          Jl(3,1) = uH * dtot_mole_flux_dp + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hl_pl

          ! derivative wrt gas saturation
          ! pl = pg - pc(satg)
          dpl_dsatg = -1.d0 * gen_auxvar_dn%d%pc_satg
          ! delta pressure = plup - pldn
          ddelta_pressure_pl = -1.d0
          ! derivative total mole flux wrt gas saturation
          dtot_mole_flux_dsatg = &
            ! liquid viscosity
            ! since liquid viscosity in a two phase state is a function
            ! of total pressure (gas pressure), there is no derivative
            ! wrt gas saturation
            !dn_scale * &
            !tot_mole_flux_dmobility * &
            !gen_auxvar_dn%d%mobilityl_pl * dpl_dsatg + &
            ! relative permeability
            dn_scale * &
            tot_mole_flux_dmobility * &
            gen_auxvar_dn%d%mobilityl_satg + &
            !pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_pl * dpl_dsatg
          ! derivative water wrt gas saturation
          Jl(1,2) = xmol(wat_comp_id) * dtot_mole_flux_dsatg
          ! derivative air wrt gas saturation
          Jl(2,2) = xmol(air_comp_id) * dtot_mole_flux_dsatg
          ! derivative energy wrt gas saturation
          Jl(3,2) = dtot_mole_flux_dsatg * uH

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn * gen_auxvar_dn%d%denl_T + &
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityl_T + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
          ! derivative water wrt temperature
          Jl(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_T(wat_comp_id,iphase)
          ! derivative air wrt temperature
          Jl(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_T(air_comp_id,iphase)
          ! derivative energy wrt temperature
          Jl(3,3) = dtot_mole_flux_dT * uH + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hl_T
      end select
      J = J + Jl
    endif
  endif
#endif
#ifdef GAS_DARCY_FLUX
  iphase = GAS_PHASE
  mobility = 0.d0
  xmol_bool = 1.d0
  bc_type = ibndtype(iphase)
  select case(bc_type)
    case(DIRICHLET_BC,HYDROSTATIC_BC,HYDROSTATIC_SEEPAGE_BC, &
         HYDROSTATIC_CONDUCTANCE_BC,DIRICHLET_SEEPAGE_BC)
      if (gen_auxvar_up%mobility(iphase) + &
          gen_auxvar_dn%mobility(iphase) > eps) then

        ! dist(0) = scalar - magnitude of distance
        ! gravity = vector(3)
        ! dist(1:3) = vector(3) - unit vector
        dist_gravity = dist(0) * dot_product(option%gravity,dist(1:3))

        if (bc_type == HYDROSTATIC_CONDUCTANCE_BC) then
          select case(iphase)
            case(LIQUID_PHASE)
              idof = auxvar_mapping(GENERAL_LIQUID_CONDUCTANCE_INDEX)
            case(GAS_PHASE)
              idof = auxvar_mapping(GENERAL_GAS_CONDUCTANCE_INDEX)
          end select
          perm_ave_over_dist = auxvars(idof)
        else
          perm_ave_over_dist = perm_dn_adj(iphase) / dist(0)
        endif

        boundary_pressure = gen_auxvar_up%pres(iphase)
        density_kg_ave = GeneralAverageDensity(iphase, &
                                                global_auxvar_up%istate, &
                                                global_auxvar_dn%istate, &
                                                gen_auxvar_up%den_kg, &
                                                gen_auxvar_dn%den_kg, &
                                                ddensity_kg_ave_dden_kg_up, &
                                                ddensity_kg_ave_dden_kg_dn)
        ddensity_kg_ave_dden_kg_up = 0.d0 ! always
        gravity_term = density_kg_ave * dist_gravity
        delta_pressure = boundary_pressure - &
                          gen_auxvar_dn%pres(iphase) + &
                          gravity_term
        if (analytical_derivatives) then
          ddelta_pressure_dpadn = dist_gravity * ddensity_kg_ave_dden_kg_dn * &
                                  gen_auxvar_dn%d%deng_pa * fmw_comp(iphase)
          ddelta_pressure_dpdn = -1.d0 + dist_gravity * &
                                 ddensity_kg_ave_dden_kg_dn * &
                                 gen_auxvar_dn%d%deng_pg * fmw_comp(iphase)
          ddelta_pressure_dTdn = dist_gravity * ddensity_kg_ave_dden_kg_dn * &
                                 gen_auxvar_dn%d%deng_T * fmw_comp(iphase)
        endif
        dn_scale = 0.d0
        ! don't expect the derivative to match precisely at delta_pressure = 0
        ! due to potential switch in direction for numerically perturbed
        ! residual
        upwind = UpwindDirection(upwind_direction_(iphase),delta_pressure, &
                                 .not.analytical_derivatives, &
                                 count_upwind_direction_flip_, &
                                 gas_bc_upwind_flip_count_by_res, &
                                 gas_bc_upwind_flip_count_by_jac)
        if (upwind) then
          mobility = gen_auxvar_up%mobility(iphase)
          xmol(:) = gen_auxvar_up%xmol(:,iphase)
          uH = gen_auxvar_up%H(iphase)
          kr=gen_auxvar_up%kr(iphase)
        else
          dn_scale = 1.d0
          mobility = gen_auxvar_dn%mobility(iphase)
          xmol(:) = gen_auxvar_dn%xmol(:,iphase)
          uH = gen_auxvar_dn%H(iphase)
          kr=gen_auxvar_dn%kr(iphase)
        endif

        ! only need average density if velocity > 0.
        density_ave = GeneralAverageDensity(iphase, &
                                            global_auxvar_up%istate, &
                                            global_auxvar_dn%istate, &
                                            gen_auxvar_up%den, &
                                            gen_auxvar_dn%den, &
                                            ddensity_ave_dden_up, &
                                            ddensity_ave_dden_dn)
        ddensity_ave_dden_up = 0.d0 ! always

        if (.not. general_non_darcy_flow) then
           ! v_darcy[m/sec] = perm[m^2] / dist[m] * kr[-] / mu[Pa-sec]
           !                    dP[Pa]]
           dv_darcy_ddelta_pressure = perm_ave_over_dist * mobility
           v_darcy(iphase) = dv_darcy_ddelta_pressure * delta_pressure

           dv_darcy_dmobility = perm_ave_over_dist * delta_pressure

        else
           !Taken from (Liu, 2014)
          call GeneralNonDarcyCorrectionBC(delta_pressure,density_kg_ave, &
               perm_ave_over_dist,kr,mobility,v_darcy(iphase), &
               dv_darcy_ddelta_pressure,dv_darcy_dmobility)
        endif
      endif
    case(NEUMANN_BC)
      xmol_bool = 0.d0
      dv_darcy_ddelta_pressure = 0.d0
      dv_darcy_dmobility = 0.d0
      ddensity_ave_dden_up = 0.d0 ! always
      ddensity_ave_dden_dn = 0.d0
      ddelta_pressure_dpdn = 0.d0
      ddelta_pressure_dpadn = 0.d0
      ddelta_pressure_dTdn = 0.d0
      dn_scale = 0.d0
      select case(iphase)
        case(LIQUID_PHASE)
          idof = auxvar_mapping(GENERAL_LIQUID_FLUX_INDEX)
        case(GAS_PHASE)
          idof = auxvar_mapping(GENERAL_GAS_FLUX_INDEX)
      end select
      xmol = 0.d0
      !geh: we should read in the mole fraction for both phases as the
      !     enthalpy, etc. applies to phase, not pure component.
      xmol(iphase) = 1.d0
      if (dabs(auxvars(idof)) > floweps) then
        v_darcy(iphase) = auxvars(idof)
        if (v_darcy(iphase) > 0.d0) then
          density_ave = gen_auxvar_up%den(iphase)
          uH = gen_auxvar_up%H(iphase)
        else
          dn_scale = 1.d0
          density_ave = gen_auxvar_dn%den(iphase)
          uH = gen_auxvar_dn%H(iphase)
          ddensity_ave_dden_dn = 1.d0
        endif
      endif
    case default
      option%io_buffer = &
        'Boundary condition type not recognized in GeneralBCFlux phase loop.'
      call PrintErrMsg(option)
  end select

  if (dabs(v_darcy(iphase)) > 0.d0 .or. mobility > 0.d0) then
    ! q[m^3 phase/sec] = v_darcy[m/sec] * area[m^2]
    q = v_darcy(iphase) * area
    ! mole_flux[kmol phase/sec] = q[m^3 phase/sec] *
    !                             density_ave[kmol phase/m^3 phase]
    tot_mole_flux = q*density_ave
    tot_mole_flux_ddel_pressure = dv_darcy_ddelta_pressure * area * &
                                  density_ave
    tot_mole_flux_dmobility = dv_darcy_dmobility * area * density_ave
    ! comp_mole_flux[kmol comp/sec] = tot_mole_flux[kmol phase/sec] *
    !                                 xmol[kmol comp/kmol phase]
    wat_mole_flux = tot_mole_flux * xmol(wat_comp_id)
    air_mole_flux = tot_mole_flux * xmol(air_comp_id)
    Res(wat_comp_id) = Res(wat_comp_id) + wat_mole_flux
    Res(air_comp_id) = Res(air_comp_id) + air_mole_flux
    Res(energy_id) = Res(energy_id) + tot_mole_flux * uH

    if (analytical_derivatives) then
      Jg = 0.d0
      select case(global_auxvar_dn%istate)
        case(LIQUID_STATE)
          ! derivative wrt liquid pressure
          ! derivative total mole flux wrt liquid pressure
          dtot_mole_flux_dp = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_pg + &
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityg_pg + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
          ! derivative water wrt liquid pressure
          Jg(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp
          ! derivative air wrt liquid pressure
          Jg(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp
          ! derivative energy wrt liquid pressure
          Jg(3,1) = uH * dtot_mole_flux_dp + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hg_pg

          ! derivative wrt air mole fraction
          ! derivative water wrt air mole fraction
          Jg(1,2) = -1.d0 * dn_scale * tot_mole_flux * xmol_bool
          ! derivative air wrt air mole fraction
          Jg(2,2) = 1.d0 * dn_scale * tot_mole_flux * xmol_bool
          ! derivative energy wrt air mole fraction
          ! Jg(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_T + &
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityg_T + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
          ! derivative water wrt temperature
          Jg(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jg(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jg(3,3) = uH * dtot_mole_flux_dT + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hg_T

        case(GAS_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_pg + &
            ! mole fraction has to be added in below since it differs for air
            ! and water
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityg_pg + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
          ! derivative water wrt gas pressure
          Jg(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_p(wat_comp_id,iphase)
          ! derivative air wrt gas pressure
          Jg(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_p(air_comp_id,iphase)
          ! derivative energy wrt gas pressure
          Jg(3,1) = uH * dtot_mole_flux_dp + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hg_pg

          ! derivative wrt air pressure
          ! derivative water wrt air saturation
          dtot_mole_flux_dp = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_pa + &
            ! mole fraction has to be added in below since it differs for air
            ! and water
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityg_pa + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dpadn
          Jg(1,2) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                      dn_scale * xmol_bool * &
                      ! dXwg_pa for gas phase is stored in liquid phase of xmol_p
                      tot_mole_flux * gen_auxvar_dn%d%xmol_p(wat_comp_id,LIQUID_PHASE)
          ! derivative air wrt air saturation
          Jg(2,2) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                      dn_scale * xmol_bool * &
                      ! dXag_pa for gas phase is stored in liquid phase of xmol_p
                      tot_mole_flux * gen_auxvar_dn%d%xmol_p(air_comp_id,LIQUID_PHASE)
          ! derivative energy wrt air saturation
          Jg(3,2) = uH * dtot_mole_flux_dp + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hg_pa

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_T + &
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityg_T + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
            ! there is no derivative of mole fraction wrt temperature in
            ! gas state
          ! derivative water wrt temperature
          Jg(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jg(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jg(3,3) = dtot_mole_flux_dT * uH + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hg_T

        case(TWO_PHASE_STATE)
          ! derivative wrt gas pressure
          ! pl = pg - pc and dpl_dpg = 1.  Therefore, we can use all the
          !   liquid pressure derivatives.
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_pg + &
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityg_pg + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dpdn
          ! derivative water wrt gas pressure
          Jg(1,1) = xmol(wat_comp_id) * dtot_mole_flux_dp + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_p(wat_comp_id,iphase)
          ! derivative air wrt gas pressure
          Jg(2,1) = xmol(air_comp_id) * dtot_mole_flux_dp + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_p(air_comp_id,iphase)
          ! derivative energy wrt gas pressure
          Jg(3,1) = uH * dtot_mole_flux_dp + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hg_pg

          ! derivative wrt gas saturation
          ! derivative total mole flux wrt gas saturation
          dtot_mole_flux_dsatg = &
            ! relative permeability
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityg_satg
          ! derivative water wrt gas saturation
          Jg(1,2) = xmol(wat_comp_id) * dtot_mole_flux_dsatg
          ! derivative air wrt gas saturation
          Jg(2,2) = xmol(air_comp_id) * dtot_mole_flux_dsatg
          ! derivative energy wrt gas saturation
          Jg(3,2) = dtot_mole_flux_dsatg * uH

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! ave. liquid density
            q * ddensity_ave_dden_dn * gen_auxvar_dn%d%deng_T + &
            ! liquid mobility
            dn_scale * &
            tot_mole_flux_dmobility * gen_auxvar_dn%d%mobilityg_T + &
            ! pressure gradient
            tot_mole_flux_ddel_pressure * ddelta_pressure_dTdn
          ! derivative water wrt temperature
          Jg(1,3) = xmol(wat_comp_id) * dtot_mole_flux_dT + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_T(wat_comp_id,iphase)
          ! derivative air wrt temperature
          Jg(2,3) = xmol(air_comp_id) * dtot_mole_flux_dT + &
                      dn_scale * xmol_bool * &
                      tot_mole_flux * gen_auxvar_dn%d%xmol_T(air_comp_id,iphase)
          ! derivative energy wrt temperature
          Jg(3,3) = dtot_mole_flux_dT * uH + &
                      dn_scale * &
                      tot_mole_flux * gen_auxvar_dn%d%Hg_T
      end select
      J = J + Jg
    endif
  endif
#endif
! CONVECTION
#endif

#ifdef DIFFUSION
  if (.not.general_immiscible) then
#ifdef LIQUID_DIFFUSION
  iphase = LIQUID_PHASE
  dsatdn = 1.d0
  ! diffusion all depends upon the downwind cell.  phase diffusion only
  ! occurs if a phase exists in both auxvars (boundary and internal) or
  ! a liquid phase exists in the internal cell. so, one could say that
  ! liquid diffusion always exists as the internal cell has a liquid phase,
  ! but gas phase diffusion only occurs if the internal cell has a gas
  ! phase.

  ! This checks for a dirichlet condition on either solute
  if (general_salt) then
    if (ibndtype(GENERAL_LIQUID_STATE_S_MOLE_DOF)==DIRICHLET_BC .or. &
        ibndtype(GENERAL_LIQUID_STATE_S_MOLE_DOF)==AT_SOLUBILITY_BC .or. &
        ibndtype(GENERAL_LIQUID_STATE_X_MOLE_DOF)==DIRICHLET_BC) then
      dirichlet_solute = PETSC_TRUE
    endif
  else
    if (ibndtype(GENERAL_LIQUID_STATE_X_MOLE_DOF)==DIRICHLET_BC) then
      dirichlet_solute = PETSC_TRUE
    endif
  endif
  sat_dn = gen_auxvar_dn%sat(iphase)
  if ((sat_dn > eps .and. ibndtype(iphase) /= NEUMANN_BC) &
      .or. (ibndtype(iphase)==NEUMANN_BC .and. dirichlet_solute)) then
    if (general_harmonic_diff_density) then
      ! density_ave in this case is not used.
      density_ave = 1.d0
      den_dn = gen_auxvar_dn%den(iphase)
      ddensity_ave_dden_dn = 0.d0
      tempreal = 1.d0
    else
      ! den_up and den_dn are not used in this case
      den_dn = 1.d0
      ! we use upstream weighting when iphase is not equal, otherwise
      ! arithmetic with 50/50 weighting
      density_ave = GeneralAverageDensity(iphase, &
                                      global_auxvar_up%istate, &
                                      global_auxvar_dn%istate, &
                                      gen_auxvar_up%den, &
                                      gen_auxvar_dn%den, &
                                      ddensity_ave_dden_up, &
                                      ddensity_ave_dden_dn)
      ddensity_ave_dden_up = 0.d0
      ! used to zero out derivative below
      tempreal = 0.d0
    endif
    stpd_dn = sat_dn*material_auxvar_dn%tortuosity* &
              gen_auxvar_dn%effective_porosity*den_dn

    dstpd_dn_dpordn = stpd_dn / gen_auxvar_dn%effective_porosity
    dstpd_dn_dsatdn = stpd_dn / sat_dn
    dstpd_dn_ddendn = tempreal * stpd_dn / den_dn
    ! units = [mole/m^4 bulk]
    dstpd_ave_over_dist_dstpd_dn = 1.d0 / dist(0)
    stpd_ave_over_dist = stpd_dn * dstpd_ave_over_dist_dstpd_dn

    if (general_diffuse_xmol) then ! delta of mole fraction
      delta_xmol = gen_auxvar_up%xmol(air_comp_id,iphase) - &
                   gen_auxvar_dn%xmol(air_comp_id,iphase)
      if (general_salt) then
        delta_xsmol = gen_auxvar_up%xmol(salt_comp_id,iphase) - &
                      gen_auxvar_dn%xmol(salt_comp_id,iphase)
      endif
      delta_X_whatever = delta_xmol
      delta_X_whatever_dxmoldn = -1.d0
    else ! delta of mass fraction
      ! 4th dof not implemented here.
      if (general_salt) then
        option%io_buffer = 'Mass-based flux not implented for general mode &
                            &with fully coupled salt transport.'
        call PrintErrMsg(option)
      endif
      xmol_air_up = gen_auxvar_up%xmol(air_comp_id,iphase)
      xmol_air_dn = gen_auxvar_dn%xmol(air_comp_id,iphase)
      tempreal = (xmol_air_up*fmw_comp(2) + (1.d0-xmol_air_up)*fmw_comp(1))
      xmass_air_up = xmol_air_up*fmw_comp(2) / tempreal
      tempreal = (xmol_air_dn*fmw_comp(2) + (1.d0-xmol_air_dn)*fmw_comp(1))
      xmass_air_dn = xmol_air_dn*fmw_comp(2) / tempreal
      dxmass_air_dn_dxmol_air_dn = (fmw_comp(2) - xmass_air_dn * (fmw_comp(2) &
                                    - fmw_comp(1))) / tempreal
      delta_xmass = xmass_air_up - xmass_air_dn
      delta_X_whatever = delta_xmass
      delta_X_whatever_dxmoldn = -1.d0 * dxmass_air_dn_dxmol_air_dn
    endif

    ! units = mole/sec
    dtot_mole_flux_ddeltaX = density_ave * stpd_ave_over_dist * &
                             general_parameter%diffusion_coefficient(iphase) * &
                             area
    if (general_salt) then
      dtot_mole_flux_ddeltaS = density_ave * stpd_ave_over_dist * &
                               general_parameter% &
                                 diffusion_coefficient(PRECIPITATE_PHASE) * &
                               area
    endif

    tot_mole_flux = dtot_mole_flux_ddeltaX * delta_X_whatever
    dtot_mole_flux_dstpd = tot_mole_flux / stpd_ave_over_dist
    dtot_mole_flux_ddenave = tot_mole_flux / density_ave
    Res(wat_comp_id) = Res(wat_comp_id) - tot_mole_flux
    Res(air_comp_id) = Res(air_comp_id) + tot_mole_flux
    if (general_salt) then
      tot_mole_flux1 = dtot_mole_flux_ddeltaS * delta_xsmol
      dtot_mole_flux_dstpd1 = tot_mole_flux1 / stpd_ave_over_dist
      dtot_mole_flux_ddenave1 = tot_mole_flux1 / density_ave
      !DF: The diffusion of dissolved air (above) is equal and opposite to water.
      !    This is not currently being applied to salt.
      !Res(wat_comp_id) = Res(wat_comp_id) - tot_mole_flux1
      Res(salt_comp_id) = Res(salt_comp_id) + tot_mole_flux1
    endif

    if (analytical_derivatives) then
      Jl = 0.d0
      select case(global_auxvar_dn%istate)
        case(LIQUID_STATE)
          ! derivative wrt liquid pressure
          ! derivative total mole flux wrt liquid pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%denl_pl) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_pl
          ! derivative water wrt liquid pressure
          Jl(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt liquid pressure
          Jl(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt liquid pressure
          Jl(3,1) = 0.d0

          ! derivative wrt air mole fraction
          ! derivative water wrt air mole fraction
          Jl(1,2) = -1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn
          ! derivative air wrt air mole fraction
          Jl(2,2) = 1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn
          ! derivative energy wrt air mole fraction
          ! Jl(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%denl_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_T
            ! diffusion coefficient derivative wrt temperature
          ! derivative water wrt temperature
          Jl(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jl(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jl(3,3) = 0.d0

        case(GAS_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%denl_pl) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_pl
          ! derivative water wrt gas pressure
          Jl(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jl(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jl(3,1) = 0.d0

          ! derivative wrt air pressure
          dtot_mole_flux_dp = &
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn / &
            gen_auxvar_dn%d%Hc
          ! derivative water wrt air saturation
           Jl(1,2) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt air saturation
           Jl(2,2) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt air saturation
           Jl(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%denl_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_T + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn  * &
            (-1.d0) * gen_auxvar_dn%xmol(air_comp_id,LIQUID_PHASE) / &
            gen_auxvar_dn%d%Hc * gen_auxvar_dn%d%Hc_T
          ! diffusion coefficient derivative wrt temperature
          ! derivative water wrt temperature
          Jl(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jl(2,3) = dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jl(3,3) = 0.d0

        case(TWO_PHASE_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%denl_pl) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_pl + &
            ! air mole fraction
            1.d0 * & ! xmoldn - xmoldn, not -1 in docs
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_p(air_comp_id,LIQUID_PHASE)
          ! derivative water wrt gas pressure
          Jl(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jl(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jl(3,1) = 0.d0

          ! derivative wrt gas saturation
          ! derivative total mole flux wrt gas saturation
          dtot_mole_flux_dsatg = &
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_dsatdn * dsatdn * (-1.d0) ! satl -> satg
          ! derivative water wrt gas saturation
          Jl(1,2) = -1.d0 * dtot_mole_flux_dsatg
          ! derivative air wrt gas saturation
          Jl(2,2) = 1.d0 * dtot_mole_flux_dsatg
          ! derivative energy wrt gas saturation
          Jl(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%denl_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%denl_T + &
            ! dispersion coefficient
            ! air mole fraction
            1.d0 * & ! xmoldn - xmoldn, not -1 in docs
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_T(air_comp_id,LIQUID_PHASE)
          ! derivative water wrt temperature
          Jl(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jl(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jl(3,3) = 0.d0
      end select
      J = J + Jl
    endif
  endif
#endif
#ifdef GAS_DIFFUSION
  iphase = GAS_PHASE
  sat_dn = gen_auxvar_dn%sat(iphase)
  !geh: i am not sure why both of these conditionals were included.  seems
  !     like the latter would never be false.
  if (sat_dn > eps .and. ibndtype(iphase) /= NEUMANN_BC) then
    dsatdn = 1.d0
    if (general_harmonic_diff_density) then
      ! density_ave in this case is not used.
      density_ave = 1.d0
      den_dn = gen_auxvar_dn%den(iphase)
      ddensity_ave_dden_dn = 0.d0
      tempreal = 1.d0
    else
      ! den_up and den_dn are not used in this case
      den_dn = 1.d0
      ! we use upstream weighting when iphase is not equal, otherwise
      ! arithmetic with 50/50 weighting
      !TODO(geh): why are we averaging density here?
      density_ave = GeneralAverageDensity(iphase, &
                                      global_auxvar_up%istate, &
                                      global_auxvar_dn%istate, &
                                      gen_auxvar_up%den, &
                                      gen_auxvar_dn%den, &
                                      ddensity_ave_dden_up, &
                                      ddensity_ave_dden_dn)
      ddensity_ave_dden_up = 0.d0
      ! used to zero out derivative below
      tempreal = 0.d0
    endif
    stpd_dn = sat_dn*material_auxvar_dn%tortuosity* &
              gen_auxvar_dn%effective_porosity*den_dn

    dstpd_dn_dpordn = stpd_dn / gen_auxvar_dn%effective_porosity
    dstpd_dn_dsatdn = stpd_dn / sat_dn
    dstpd_dn_ddendn = tempreal * stpd_dn / den_dn
    ! units = [mole/m^4 bulk]
    dstpd_ave_over_dist_dstpd_dn = 1.d0 / dist(0)
    stpd_ave_over_dist = stpd_dn * dstpd_ave_over_dist_dstpd_dn

    if (general_diffuse_xmol) then ! delta of mole fraction
      delta_xmol = gen_auxvar_up%xmol(air_comp_id,iphase) - &
                   gen_auxvar_dn%xmol(air_comp_id,iphase)
      delta_X_whatever = delta_xmol
      delta_X_whatever_dxmoldn = -1.d0
    else ! delta of mass fraction
      xmol_air_up = gen_auxvar_up%xmol(air_comp_id,iphase)
      xmol_air_dn = gen_auxvar_dn%xmol(air_comp_id,iphase)
      tempreal = (xmol_air_up*fmw_comp(2) + (1.d0-xmol_air_up)*fmw_comp(1))
      xmass_air_up = xmol_air_up*fmw_comp(2) / tempreal
      tempreal = (xmol_air_dn*fmw_comp(2) + (1.d0-xmol_air_dn)*fmw_comp(1))
      xmass_air_dn = xmol_air_dn*fmw_comp(2) / tempreal
      delta_xmass = xmass_air_up - xmass_air_dn
      delta_X_whatever = delta_xmass
      delta_X_whatever_dxmoldn = -1.d0 * dxmass_air_dn_dxmol_air_dn
    endif
    ! need to account for multiple phases
    ! Eq. 1.9b.  The gas density is added below
    if (general_temp_dep_gas_air_diff) then
      temp_ave = 0.5d0*(gen_auxvar_up%temp+gen_auxvar_dn%temp)
      pressure_ave = 0.5d0*(gen_auxvar_up%pres(iphase)+ &
                            gen_auxvar_dn%pres(iphase))
      tempreal = (temp_ave+T273K)/T273K
      diffusion_scale = tempreal**1.8d0 * 101325.d0 / pressure_ave
                             ! 0.9d0 = 0.5 * 1.8
      ddiffusion_coef_dTdn = 0.9d0 * diffusion_scale / (tempreal * T273K)
      ddiffusion_coef_dpdn = -1.d0 * diffusion_scale / pressure_ave * 0.5d0
    else
      diffusion_scale = 1.d0
      ddiffusion_coef_dTdn = 0.d0
      ddiffusion_coef_dpdn = 0.d0
    endif
    ! units = mole/sec
    dtot_mole_flux_ddeltaX = density_ave * stpd_ave_over_dist * &
                             diffusion_scale * &
                             general_parameter%diffusion_coefficient(iphase) * &
                             area
    tot_mole_flux = dtot_mole_flux_ddeltaX * delta_X_whatever
    dtot_mole_flux_dstpd = tot_mole_flux / stpd_ave_over_dist
    dtot_mole_flux_ddiffusion_coef = tot_mole_flux / diffusion_scale
    dtot_mole_flux_ddenave = tot_mole_flux / density_ave
    Res(wat_comp_id) = Res(wat_comp_id) - tot_mole_flux
    Res(air_comp_id) = Res(air_comp_id) + tot_mole_flux

    if (analytical_derivatives) then

      Jg = 0.d0
      select case(global_auxvar_dn%istate)
        case(LIQUID_STATE)
          ! derivative wrt liquid pressure
          ! derivative total mole flux wrt liquid pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%deng_pg) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_pg + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dpdn
          ! derivative water wrt liquid pressure
          Jg(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt liquid pressure
          Jg(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt liquid pressure
          Jg(3,1) = 0.d0

          ! derivative wrt air mole fraction
          ! derivative water wrt air mole fraction
          Jg(1,2) = -1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn
          ! derivative air wrt air mole fraction
          Jg(2,2) = 1.d0 * dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn
          ! derivative energy wrt air mole fraction
          ! Jg(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%deng_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_T + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dTdn
          ! derivative water wrt temperature
          Jg(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jg(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jg(3,3) = 0.d0

        case(GAS_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%deng_pg) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_pg + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dpdn + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_p(air_comp_id,GAS_PHASE)
          ! derivative water wrt gas pressure
          Jg(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jg(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jg(3,1) = 0.d0

          ! derivative wrt air pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
             dstpd_dn_ddendn * gen_auxvar_dn%d%deng_pa + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_pa + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
                              ! liquid phase is hijacked to store \dpa
            gen_auxvar_dn%d%xmol_p(air_comp_id,LIQUID_PHASE)
          ! derivative water wrt air saturation
           Jg(1,2) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt air saturation
           Jg(2,2) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt air saturation
           Jg(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%deng_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_T + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dTdn + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_T(air_comp_id,GAS_PHASE)
          ! derivative water wrt temperature
          Jg(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jg(2,3) = dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jg(3,3) = 0.d0

        case(TWO_PHASE_STATE)
          ! derivative wrt gas pressure
          ! derivative total mole flux wrt gas pressure
          dtot_mole_flux_dp = &
            ! liquid density and porosity
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            (dstpd_dn_dpordn * gen_auxvar_dn%d%por_p + &
             dstpd_dn_ddendn * gen_auxvar_dn%d%deng_pg) + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_pg + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dpdn + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_p(air_comp_id,GAS_PHASE)
          ! derivative water wrt gas pressure
          Jg(1,1) = -1.d0 * dtot_mole_flux_dp
          ! derivative air wrt gas pressure
          Jg(2,1) = 1.d0 * dtot_mole_flux_dp
          ! derivative energy wrt gas pressure
          Jg(3,1) = 0.d0

          ! derivative wrt gas saturation
          ! derivative total mole flux wrt gas saturation
          dtot_mole_flux_dsatg = &
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_dsatdn * dsatdn ! satg
          ! derivative water wrt gas saturation
          Jg(1,2) = -1.d0 * dtot_mole_flux_dsatg
          ! derivative air wrt gas saturation
          Jg(2,2) = 1.d0 * dtot_mole_flux_dsatg
          ! derivative energy wrt gas saturation
          Jg(3,2) = 0.d0

          ! derivative wrt temperature
          ! derivative total mole flux wrt temperature
          dtot_mole_flux_dT = &
            ! liquid density
            dtot_mole_flux_dstpd * dstpd_ave_over_dist_dstpd_dn * &
            dstpd_dn_ddendn * gen_auxvar_dn%d%deng_T + &
            ! if density arithmetically averaged
            dtot_mole_flux_ddenave * ddensity_ave_dden_dn * &
            gen_auxvar_dn%d%deng_T + &
            ! diffusion coefficient
            dtot_mole_flux_ddiffusion_coef * ddiffusion_coef_dTdn  + &
            ! air mole fraction
            dtot_mole_flux_ddeltaX * delta_X_whatever_dxmoldn * &
            gen_auxvar_dn%d%xmol_T(air_comp_id,GAS_PHASE)
          ! derivative water wrt temperature
          Jg(1,3) = -1.d0 * dtot_mole_flux_dT
          ! derivative air wrt temperature
          Jg(2,3) = 1.d0 * dtot_mole_flux_dT
          ! derivative energy wrt temperature
          Jg(3,3) = 0.d0
      end select
      J = J + Jg
    endif
  endif
#endif
! DIFFUSION
  endif ! if (.not.general_immiscible)
#endif

#ifdef CONDUCTION
  ! add heat conduction flux
  heat_flux = 0.d0
  select case (ibndtype(GENERAL_ENERGY_EQUATION_INDEX))
    case (DIRICHLET_BC)
      sat_dn = gen_auxvar_dn%sat(option%liquid_phase)
      call thermal_cc_dn%thermal_conductivity_function% &
           TCondTensorToScalar(dist,option)
      call thermal_cc_dn%thermal_conductivity_function%CalculateTCond(sat_dn, &
           gen_auxvar_dn%temp,gen_auxvar_dn%effective_porosity, &
           k_eff_dn,dkeff_dn_dsatldn,dkeff_dn_dTdn,option)

      dkeff_ave_dkeffdn = 1.d0 / dist(0)
      k_eff_ave = k_eff_dn * dkeff_ave_dkeffdn
      ! units:
      ! k_eff = W/K-m = J/s/K-m
      ! delta_temp = K
      ! area = m^2
      ! heat_flux = k_eff * delta_temp * area = J/s
      delta_temp = gen_auxvar_up%temp - gen_auxvar_dn%temp
      dheat_flux_ddelta_temp = k_eff_ave * area * 1.d-6 ! J/s -> MJ/s
      heat_flux = dheat_flux_ddelta_temp * delta_temp
      dheat_flux_dkeff_ave = area * 1.d-6 * delta_temp
    case(NEUMANN_BC)
                  ! flux prescribed as MW/m^2
      heat_flux = auxvars(auxvar_mapping(GENERAL_ENERGY_FLUX_INDEX)) * area
      dheat_flux_ddelta_temp = 0.d0
      dkeff_dn_dsatldn = 0.d0
      dkeff_ave_dkeffdn = 0.d0
      dheat_flux_dkeff_ave = 0.d0
    case default
      option%io_buffer = 'Boundary condition type not recognized in ' // &
        'GeneralBCFlux heat conduction loop.'
      call PrintErrMsg(option)
  end select
  ! MJ/s
  Res(energy_id) = Res(energy_id) + heat_flux

  if (analytical_derivatives) then
    Jc = 0.d0
    select case(global_auxvar_dn%istate)
      case(LIQUID_STATE,GAS_STATE)
        ! only derivative is energy wrt temperature
        ! derivative energy wrt temperature
        ! positive for upwind
        Jc(3,3) = -1.d0 * dheat_flux_ddelta_temp
      case(TWO_PHASE_STATE)
        ! only derivatives are energy wrt saturation and temperature
        ! derivative energy wrt gas saturation
        Jc(3,2) = dheat_flux_dkeff_ave * dkeff_ave_dkeffdn * &
                  dkeff_dn_dsatldn * (-1.d0) ! satl -> satg
        ! derivative energy wrt temperature
        ! positive for upwind
        Jc(3,3) = -1.d0 * dheat_flux_ddelta_temp
    end select
    J = J + Jc
  endif
! CONDUCTION
#endif

end subroutine GeneralBCFlux

! ************************************************************************** !

subroutine GeneralAuxVarComputeAndSrcSink(option,qsrc,flow_src_sink_type, &
                          gen_auxvar_ss, &
                          gen_auxvar,global_auxvar, global_auxvar_ss, &
                          material_auxvar,ss_flow_vol_flux, &
                          characteristic_curves, natural_id, scale,Res,J, &
                          analytical_derivatives,aux_var_compute_only, &
                          soluble_matrix,soil_heat_capacity,debug_cell)
  !
  ! Computes the source/sink terms for the residual
  !
  ! Author: Glenn Hammond
  ! Date: 03/09/11
  !

  use Option_module
  use EOS_Water_module
  use EOS_Gas_module
  use Material_Aux_module
  use Characteristic_Curves_module

  implicit none

  type(option_type) :: option
  type(general_auxvar_type) :: gen_auxvar,gen_auxvar_ss
  type(global_auxvar_type) :: global_auxvar,global_auxvar_ss
  type(material_auxvar_type) :: material_auxvar
  PetscReal :: ss_flow_vol_flux(option%nphase)
  class(characteristic_curves_type) :: characteristic_curves
  PetscInt :: natural_id
  PetscReal :: scale
  PetscReal :: Res(option%nflowdof)
  PetscReal :: J(option%nflowdof,option%nflowdof)
  PetscBool :: analytical_derivatives
  PetscBool :: aux_var_compute_only
  PetscBool :: debug_cell

  PetscReal :: qsrc(option%nflowdof)
  PetscInt :: flow_src_sink_type
  PetscReal :: qsrc_mol
  PetscReal :: enthalpy, internal_energy
  PetscInt :: wat_comp_id, air_comp_id, energy_id, salt_comp_id
  PetscReal :: Jl(option%nflowdof,option%nflowdof)
  PetscReal :: Jg(option%nflowdof,option%nflowdof)
  PetscReal :: Je(option%nflowdof,option%nflowdof)
  PetscReal :: dden_bool
  PetscReal :: hw_dp, hw_dT, ha_dp, ha_dT
  PetscReal :: mob_tot
  PetscReal :: cell_pressure
  PetscReal :: xxss(option%nflowdof)
  PetscInt :: lid, gid, pid, apid
  PetscReal :: soil_heat_capacity

  PetscBool :: soluble_matrix

  lid = option%liquid_phase
  gid = option%gas_phase
  pid = option%precipitate_phase
  apid = option%air_pressure_id
  wat_comp_id = option%water_id
  air_comp_id = option%air_id
  energy_id = option%energy_id
  salt_comp_id = option%salt_id

  Res = 0.d0
  J = 0.d0

  ! Index 0 contains user-defined source/sink characteristics.
  ! If certain variables are not defined in the source sink,
  ! those are set to internal values.
  xxss(1) = maxval(gen_auxvar_ss%pres(option% &
                     liquid_phase:option%gas_phase))
  xxss(2) = gen_auxvar_ss%sat(gid)
  xxss(3) = gen_auxvar_ss%temp
  if (general_salt) then
    if (soluble_matrix) then
      xxss(4) = gen_auxvar_ss%effective_porosity
    else
      xxss(4) = gen_auxvar_ss%sat(pid)
    endif
  endif

  cell_pressure = maxval(gen_auxvar%pres(option%liquid_phase: &
                         option%gas_phase))

  ! If extracting
  if (qsrc(wat_comp_id)<0.d0 .or. qsrc(air_comp_id)<0.d0) then
    ! Use cell primary variables for state variable calculations
    xxss(ONE_INTEGER) = cell_pressure
    select case(global_auxvar%istate)
      case(LIQUID_STATE)
        xxss(TWO_INTEGER) = gen_auxvar%xmol(air_comp_id,lid)
        xxss(THREE_INTEGER) = gen_auxvar%temp
        if (general_salt) then
          xxss(FOUR_INTEGER) = gen_auxvar%xmol(salt_comp_id,lid)
        endif
      case(GAS_STATE)
        if (general_gas_air_mass_dof == GENERAL_AIR_PRESSURE_INDEX) then
          xxss(TWO_INTEGER) = gen_auxvar%pres(apid)
        else
          !MAN: I think this should be air partial pressure?
          xxss(TWO_INTEGER) = gen_auxvar%xmol(wat_comp_id,air_comp_id)
        endif
        xxss(THREE_INTEGER) = gen_auxvar%temp
      case(TWO_PHASE_STATE)
        xxss(TWO_INTEGER) = gen_auxvar%sat(gid)
        if (general_2ph_energy_dof == GENERAL_TEMPERATURE_INDEX) then
          xxss(THREE_INTEGER) = gen_auxvar%temp
        else
          xxss(THREE_INTEGER) = gen_auxvar%pres(apid)
        endif
      case(LP_STATE)
        xxss(TWO_INTEGER) = gen_auxvar_ss%xmol(air_comp_id, wat_comp_id)
        xxss(THREE_INTEGER) = gen_auxvar_ss%temp
        if (general_salt) then
          if (soluble_matrix) then
            xxss(FOUR_INTEGER) = gen_auxvar_ss%effective_porosity
          else
            xxss(FOUR_INTEGER) = gen_auxvar_ss%sat(pid)
          endif
        endif
      case(LGP_STATE)
        xxss(TWO_INTEGER) = gen_auxvar_ss%sat(gid)
        xxss(THREE_INTEGER) = gen_auxvar_ss%temp
        if (general_salt) then
          if (soluble_matrix) then
            xxss(FOUR_INTEGER) = gen_auxvar_ss%effective_porosity
          else
            xxss(FOUR_INTEGER) = gen_auxvar_ss%sat(pid)
          endif
        endif
      case(GP_STATE)
        if (general_gas_air_mass_dof == GENERAL_AIR_PRESSURE_INDEX) then
          xxss(TWO_INTEGER) = gen_auxvar_ss%pres(apid)
        else
          xxss(TWO_INTEGER) = gen_auxvar_ss%xmol(wat_comp_id,air_comp_id)
        endif
        xxss(THREE_INTEGER) = gen_auxvar_ss%temp
        if (general_salt) then
          if (soluble_matrix) then
            xxss(FOUR_INTEGER) = gen_auxvar_ss%effective_porosity
          else
            xxss(FOUR_INTEGER) = gen_auxvar_ss%sat(pid)
          endif
        endif
    end select
    global_auxvar_ss%istate = global_auxvar%istate
  else ! If injecting, use primary variables from user-supplied conditions
    select case(global_auxvar_ss%istate)
      case(LIQUID_STATE)
        xxss(TWO_INTEGER) = gen_auxvar_ss%xmol(air_comp_id, wat_comp_id)
        xxss(THREE_INTEGER) = gen_auxvar_ss%temp
        if (general_salt) then
          xxss(FOUR_INTEGER) = gen_auxvar_ss%xmol(salt_comp_id, wat_comp_id)
        endif
      case(GAS_STATE)
        if (general_gas_air_mass_dof == GENERAL_AIR_PRESSURE_INDEX) then
          xxss(TWO_INTEGER) = gen_auxvar_ss%pres(apid)
        else
          !MAN: I think this should be air partial pressure?
          xxss(TWO_INTEGER) = gen_auxvar_ss%xmol(wat_comp_id,air_comp_id)
        endif
        xxss(THREE_INTEGER) = gen_auxvar_ss%temp
      case(TWO_PHASE_STATE)
        xxss(TWO_INTEGER) = gen_auxvar_ss%sat(gid)
        if (general_2ph_energy_dof == GENERAL_TEMPERATURE_INDEX) then
          xxss(THREE_INTEGER) = gen_auxvar_ss%temp
        else
          xxss(THREE_INTEGER) = gen_auxvar_ss%pres(apid)
        endif
        if (general_salt) then
          xxss(FOUR_INTEGER) = gen_auxvar_ss%xmol(salt_comp_id,wat_comp_id)
        endif
      case(LP_STATE)
        xxss(TWO_INTEGER) = gen_auxvar_ss%xmol(air_comp_id, wat_comp_id)
        xxss(THREE_INTEGER) = gen_auxvar_ss%temp
        if (general_salt) then
          if (soluble_matrix) then
            xxss(FOUR_INTEGER) = gen_auxvar_ss%effective_porosity
          else
            xxss(FOUR_INTEGER) = gen_auxvar_ss%sat(pid)
          endif
        endif
      case(LGP_STATE)
        xxss(TWO_INTEGER) = gen_auxvar_ss%sat(gid)
        xxss(THREE_INTEGER) = gen_auxvar_ss%temp
        if (general_salt) then
          if (soluble_matrix) then
            xxss(FOUR_INTEGER) = gen_auxvar_ss%effective_porosity
          else
            xxss(FOUR_INTEGER) = gen_auxvar_ss%sat(pid)
          endif
        endif
      case(GP_STATE)
        if (general_gas_air_mass_dof == GENERAL_AIR_PRESSURE_INDEX) then
          xxss(TWO_INTEGER) = gen_auxvar_ss%pres(apid)
        else
          xxss(TWO_INTEGER) = gen_auxvar_ss%xmol(wat_comp_id,air_comp_id)
        endif
        xxss(THREE_INTEGER) = gen_auxvar_ss%temp
        if (general_salt) then
          if (soluble_matrix) then
            xxss(FOUR_INTEGER) = gen_auxvar_ss%effective_porosity
          else
            xxss(FOUR_INTEGER) = gen_auxvar_ss%sat(pid)
          endif
        endif
    end select
    if (soluble_matrix) then
      global_auxvar_ss%istate = global_auxvar%istate
    endif
  endif

  option%iflag = GENERAL_UPDATE_FOR_SS

  ! Compute state variables
  if (.not. general_salt) then
    call GeneralAuxVarCompute(xxss,gen_auxvar_ss, global_auxvar_ss, &
                              material_auxvar, characteristic_curves, &
                              natural_id,option)
  elseif (general_salt) then
    call GeneralAuxVarCompute4(xxss,gen_auxvar_ss, global_auxvar_ss, &
                              material_auxvar, characteristic_curves, &
                              natural_id,soluble_matrix,option)
  endif
  if (aux_var_compute_only) return

  qsrc_mol = 0.d0
  if (flow_src_sink_type == TOTAL_MASS_RATE_SS) then
    !Scales the mass of water and gas extracted by the mobility ratio.
    mob_tot = gen_auxvar_ss%mobility(lid) + &
              gen_auxvar_ss%mobility(gid)
      if (gen_auxvar_ss%sat(gid) <= 0.d0) then
        ! Water component, liquid phase
        ! kg/s phase to kmol/sec phase
        qsrc_mol = qsrc(wat_comp_id) * gen_auxvar_ss%den(lid) / &
                   gen_auxvar_ss%den_kg(lid)
        ! kmol/sec phase to kmol/sec component
        qsrc_mol = qsrc_mol * gen_auxvar_ss%xmol(lid,lid)
      elseif (gen_auxvar_ss%sat(lid) <= 0.d0) then
        ! Water component, gas phase
        ! kg/s phase to kmol/sec phase
        qsrc_mol = qsrc(wat_comp_id) * gen_auxvar_ss%den(gid) / &
                   gen_auxvar_ss%den_kg(gid)
        ! kmol/sec phase to kmol/sec component
        qsrc_mol = qsrc_mol * gen_auxvar_ss%xmol(lid,gid)
      else
        ! Water component, liquid phase
        qsrc_mol = qsrc(wat_comp_id) * &
                   gen_auxvar_ss%den(lid) / &
                   gen_auxvar_ss%den_kg(lid) * &
                   gen_auxvar_ss%mobility(lid)/mob_tot * &
                   gen_auxvar_ss%xmol(lid,lid)
        ! Water component, gas phase
        qsrc_mol = qsrc_mol + qsrc(wat_comp_id) * &
                   gen_auxvar_ss%den(gid) / &
                   gen_auxvar_ss%den_kg(gid) * &
                   gen_auxvar_ss%mobility(gid)/mob_tot * &
                   gen_auxvar_ss%xmol(lid,gid)
      endif

      ss_flow_vol_flux(wat_comp_id) = qsrc_mol / gen_auxvar_ss% &
                                      den(wat_comp_id)
      Res(wat_comp_id) = qsrc_mol

      if (gen_auxvar_ss%sat(gid) <= 0.d0) then
        ! Air component, liquid phase
        ! kg/s phase to kmol/sec phase
        qsrc_mol = qsrc(wat_comp_id) * gen_auxvar_ss%den(lid) / &
                   gen_auxvar_ss%den_kg(lid)
        ! kmol/sec phase to kmol/sec component
        qsrc_mol = qsrc_mol * gen_auxvar_ss%xmol(gid,lid)
      elseif (gen_auxvar_ss%sat(lid) <= 0.d0) then
        ! Air component, gas phase
        ! kg/s phase to kmol/sec phase
        qsrc_mol = qsrc(wat_comp_id) * gen_auxvar_ss%den(gid) / &
                   gen_auxvar_ss%den_kg(gid)
        ! kmol/sec phase to kmol/sec component
        qsrc_mol = qsrc_mol * gen_auxvar_ss%xmol(gid,gid)
      else
      ! Air component, liquid phase
        qsrc_mol = qsrc(wat_comp_id) * gen_auxvar_ss%den(lid) / &
                   gen_auxvar_ss%den_kg(lid)* &
                   gen_auxvar_ss%mobility(lid)/mob_tot * &
                   gen_auxvar_ss%xmol(gid,lid)
      ! Air component, gas phase
        qsrc_mol = qsrc_mol + qsrc(wat_comp_id) * gen_auxvar_ss% &
                   den(gid) / gen_auxvar_ss%den_kg(gid)* &
                   gen_auxvar_ss%mobility(gid)/mob_tot * &
                   gen_auxvar_ss%xmol(gid,gid)
      endif

      ss_flow_vol_flux(air_comp_id) = qsrc_mol / gen_auxvar_ss% &
                                      den(air_comp_id)
      Res(air_comp_id) = qsrc_mol

      if (gen_auxvar_ss%sat(gid) <= 0.d0) then
        Res(energy_id) = qsrc(wat_comp_id) * gen_auxvar_ss% &
                         den(lid) / gen_auxvar_ss%den_kg(lid) * &
                         gen_auxvar_ss%h(lid)
      elseif (gen_auxvar_ss%sat(lid) <= 0.d0) then
        Res(energy_id) = qsrc(wat_comp_id) * gen_auxvar_ss% &
                         den(gid) / gen_auxvar_ss%den_kg(gid) * &
                         gen_auxvar_ss%h(gid)
      else
        Res(energy_id) = qsrc(wat_comp_id) * gen_auxvar_ss% &
                         mobility(lid) / mob_tot * gen_auxvar_ss%&
                         den(lid) / gen_auxvar_ss%den_kg(lid) * &
                         gen_auxvar_ss%h(lid)
        Res(energy_id) = Res(energy_id) + qsrc(wat_comp_id) * &
                         gen_auxvar_ss%mobility(gid) / mob_tot * &
                         gen_auxvar_ss%den(gid) / &
                         gen_auxvar_ss%den_kg(gid) * &
                         gen_auxvar_ss%h(gid)
      endif

      if (general_salt) then
        option%io_buffer = 'TOTAL_MASS_RATE not set up &
            &for salt in GeneralAuxVarComputeAndSrcSink.'
        call PrintErrMsg(option)
      endif
  else

#ifdef WATER_SRCSINK
  qsrc_mol = 0.d0
  dden_bool = 0.d0
  select case(flow_src_sink_type)
    case(MASS_RATE_SS)
      qsrc_mol = qsrc(wat_comp_id)/fmw_comp(wat_comp_id) ! kg/sec -> kmol/sec
    case(SCALED_MASS_RATE_SS)                       ! kg/sec -> kmol/sec
      qsrc_mol = qsrc(wat_comp_id)/fmw_comp(wat_comp_id)*scale
    case(VOLUMETRIC_RATE_SS)  ! assume local density for now
      ! qsrc1 = m^3/sec
      ! den = kmol/m^3
      qsrc_mol = qsrc(wat_comp_id)*gen_auxvar_ss%den(wat_comp_id)
      dden_bool = 1.d0
    case(SCALED_VOLUMETRIC_RATE_SS)  ! assume local density for now
      ! qsrc1 = m^3/sec             ! den = kmol/m^3
      qsrc_mol = qsrc(wat_comp_id)*gen_auxvar_ss% &
                 den(wat_comp_id)*scale
      dden_bool = 1.d0
  end select
  ss_flow_vol_flux(wat_comp_id) = qsrc_mol/gen_auxvar_ss% &
                                  den(wat_comp_id)
  Res(wat_comp_id) = qsrc_mol
  if (analytical_derivatives) then
    Jl = 0.d0
    select case(global_auxvar%istate)
      case(LIQUID_STATE)
        ! derivative wrt liquid pressure
        Jl(1,1) = dden_bool * qsrc(wat_comp_id) * &
                  gen_auxvar_ss%d%denl_pl
        ! derivative wrt air mole fraction
        ! derivative wrt temperature
        Jl(1,3) = dden_bool * qsrc(wat_comp_id) * &
                  gen_auxvar_ss%d%denl_T
      case(GAS_STATE)
        if (dabs(Res(wat_comp_id)) > 1.d-40 .and. dden_bool > 0.d0) then
          option%io_buffer = 'Volumetric water injection not set up &
            &for gas state in GeneralAuxVarComputeAndSrcSink.'
          call PrintErrMsg(option)
        endif
        ! derivative wrt gas pressure
        ! derivative wrt air pressure
        ! derivative wrt temperature
      case(TWO_PHASE_STATE)
        ! derivative wrt gas pressure
        Jl(1,1) = dden_bool * qsrc(wat_comp_id) * &
                  gen_auxvar_ss%d%denl_pl
        ! derivative wrt gas saturation
        ! derivative wrt temperature
        Jl(1,3) = dden_bool * qsrc(wat_comp_id) * &
                  gen_auxvar_ss%d%denl_T
    end select
    J = J + Jl
  endif
#endif

#ifdef AIR_SRCSINK
  qsrc_mol = 0.d0
  dden_bool = 0.d0
  select case(flow_src_sink_type)
    case(MASS_RATE_SS)
      qsrc_mol = qsrc(air_comp_id)/fmw_comp(air_comp_id) ! kg/sec -> kmol/sec
    case(SCALED_MASS_RATE_SS)                       ! kg/sec -> kmol/sec
      qsrc_mol = qsrc(air_comp_id)/fmw_comp(air_comp_id)*scale
    case(VOLUMETRIC_RATE_SS)  ! assume local density for now
      ! qsrc1 = m^3/sec
      ! den = kmol/m^3
      qsrc_mol = qsrc(air_comp_id)*gen_auxvar_ss%den(air_comp_id)
      dden_bool = 1.d0
    case(SCALED_VOLUMETRIC_RATE_SS)  ! assume local density for now
      ! qsrc1 = m^3/sec             ! den = kmol/m^3
      qsrc_mol = qsrc(air_comp_id)* &
                 gen_auxvar_ss%den(air_comp_id)*scale
      dden_bool = 1.d0
  end select
  ss_flow_vol_flux(air_comp_id) = qsrc_mol / &
                                  gen_auxvar_ss%den(air_comp_id)
  Res(air_comp_id) = qsrc_mol
  if (analytical_derivatives) then
    Jg = 0.d0
    select case(global_auxvar%istate)
      case(LIQUID_STATE)
        if (dabs(Res(air_comp_id)) > 1.d-40 .and. dden_bool > 0.d0) then
          option%io_buffer = 'Volumetric air injection not set up for &
            &liquid state in GeneralAuxVarComputeAndSrcSink as there is &
            &no air density.'
          call PrintErrMsg(option)
        endif
        ! derivative wrt liquid pressure
        ! derivative wrt air mole fraction
        ! derivative wrt temperature
      case(GAS_STATE)
        ! derivative wrt gas pressure
        Jg(2,1) = dden_bool * qsrc(air_comp_id) * &
                  gen_auxvar_ss%d%deng_pg
        ! derivative wrt air pressure
        Jg(2,2) = dden_bool * qsrc(air_comp_id) * &
                  gen_auxvar_ss%d%deng_pa
        ! derivative wrt temperature
        Jg(2,3) = dden_bool * qsrc(air_comp_id) * &
                  gen_auxvar_ss%d%deng_T
      case(TWO_PHASE_STATE)
        ! derivative wrt gas pressure
        Jg(2,1) = dden_bool * qsrc(air_comp_id) * &
                  gen_auxvar_ss%d%deng_pg
        ! derivative wrt gas saturation
        ! derivative wrt temperature
        Jg(2,3) = dden_bool * qsrc(air_comp_id) * &
                  gen_auxvar_ss%d%deng_T
    end select
    J = J + Jg
  endif
#endif
  endif

  if (dabs(qsrc(air_comp_id)) < 1.d-40 .and. flow_src_sink_type /= &
          TOTAL_MASS_RATE_SS .and. qsrc(wat_comp_id) < 0.d0) then ! extraction only
    ! Res(1) holds qsrc_mol for water.  If the src/sink value for air is zero,
    ! remove/add the equivalent mole fraction of air in the liquid phase.
    qsrc_mol = Res(wat_comp_id)*gen_auxvar_ss% &
               xmol(air_comp_id,wat_comp_id)
    Res(air_comp_id) = qsrc_mol
    ss_flow_vol_flux(air_comp_id) = qsrc_mol / &
                                    gen_auxvar_ss%den(air_comp_id)
    if (analytical_derivatives) then
      !Jg = 0.d0
      select case(global_auxvar%istate)
        case(LIQUID_STATE)
          ! derivative wrt liquid pressure
          ! derivative wrt air mole fraction
          Jg(2,2) = Jg(2,2) + Res(wat_comp_id)
          ! derivative wrt temperature
        case(GAS_STATE)
          ! derivative wrt gas pressure
          ! derivative wrt air pressure
          ! derivative wrt temperature
        case(TWO_PHASE_STATE)
          ! derivative wrt gas pressure
          Jg(2,1) = Jg(2,1) + &
                    dden_bool * qsrc(wat_comp_id) * &
                    gen_auxvar_ss%d%denl_pl * &
                    gen_auxvar_ss%xmol(air_comp_id,wat_comp_id)+ &
                    Res(wat_comp_id) * gen_auxvar_ss%d%xmol_p(2,1)
          ! derivative wrt gas saturation
          ! derivative wrt temperature
          Jg(2,3) = Jg(2,3) + &
                    dden_bool * qsrc(wat_comp_id) * &
                    gen_auxvar_ss%d%denl_T * &
                    gen_auxvar_ss%xmol(air_comp_id,wat_comp_id)+ &
                    Res(wat_comp_id) * gen_auxvar_ss%d%xmol_T(2,1)
      end select
      J = J + Jg
    endif
  endif

  if (general_salt) then
#ifdef SALT_SRCSINK
    qsrc_mol = 0.d0
    dden_bool = 0.d0
    select case(flow_src_sink_type)
      case(MASS_RATE_SS)
        qsrc_mol = qsrc(salt_comp_id)/fmw_comp(salt_comp_id) ! kg/sec -> kmol/sec
      case(SCALED_MASS_RATE_SS)                       ! kg/sec -> kmol/sec
        qsrc_mol = qsrc(salt_comp_id)/fmw_comp(salt_comp_id)*scale
      case(VOLUMETRIC_RATE_SS)  ! assume local density for now
        ! qsrc1 = m^3/sec
        ! den = kmol/m^3
        qsrc_mol = qsrc(salt_comp_id)*gen_auxvar_ss%den(salt_comp_id)
        dden_bool = 1.d0
      case(SCALED_VOLUMETRIC_RATE_SS)  ! assume local density for now
        ! qsrc1 = m^3/sec             ! den = kmol/m^3
        qsrc_mol = qsrc(salt_comp_id)* &
                   gen_auxvar_ss%den(salt_comp_id)*scale
        dden_bool = 1.d0
    end select
    ss_flow_vol_flux(salt_comp_id) = qsrc_mol / &
                                    gen_auxvar_ss%den(salt_comp_id)
    if (general_salt_src_flag) then
      if (gen_auxvar_ss%effective_porosity <= general_min_por_srcsink) then
        Res(salt_comp_id) = 0.d0
      elseif (maxval(gen_auxvar_ss%pres(:)) > general_max_pres_srcsink) then
        Res(salt_comp_id) = 0.d0
      else
        Res(salt_comp_id) = qsrc_mol
      endif
    else
      Res(salt_comp_id) = qsrc_mol
    endif
    ! No analytical derivative for the salt phase
    if (analytical_derivatives) then
      option%io_buffer = 'Analytical derivatives not implemented in general mode with 4 dof.'
      call PrintErrMsg(option)
    endif
#endif
    !endif

    if (dabs(qsrc(salt_comp_id)) < 1.d-40 .and. flow_src_sink_type /= &
            TOTAL_MASS_RATE_SS .and. qsrc(wat_comp_id) < 0.d0) then ! extraction only
      ! Res(1) holds qsrc_mol for water.  If the src/sink value for air is zero,
      ! remove/add the equivalent mole fraction of air in the liquid phase.
      qsrc_mol = Res(wat_comp_id)*gen_auxvar_ss% &
                 xmol(salt_comp_id,wat_comp_id)
      Res(salt_comp_id) = qsrc_mol
      ss_flow_vol_flux(salt_comp_id) = qsrc_mol / &
                                      gen_auxvar_ss%den(salt_comp_id)
    endif
endif

  ! energy units: MJ/sec
  if ((.not. general_salt .and. size(qsrc) == THREE_INTEGER).or. &
      (general_salt .and. size(qsrc) == FOUR_INTEGER)) then
    if (flow_src_sink_type /= TOTAL_MASS_RATE_SS) then
      if (dabs(qsrc(wat_comp_id)) > 1.d-40) then
        if (associated(gen_auxvar_ss%d)) then
          hw_dp = gen_auxvar_ss%d%Hl_pl
          hw_dT = gen_auxvar_ss%d%Hl_T
        endif
        enthalpy = gen_auxvar_ss%h(wat_comp_id)
        ! enthalpy units: MJ/kmol                       ! water component mass
        Res(energy_id) = Res(energy_id) + Res(wat_comp_id) * enthalpy
        if (analytical_derivatives) then
          Je = 0.d0
          Je(3,1) = Jl(1,1) * enthalpy + Res(wat_comp_id) * hw_dp
          Je(3,3) = Jl(1,3) * enthalpy + Res(wat_comp_id) * hw_dT
        endif
        J = J + Je
      endif

      if (dabs(qsrc(air_comp_id)) > 1.d-40) then
        ! this is pure air, we use the enthalpy of air, NOT the air/water
        ! mixture in gas
        ! air enthalpy is only a function of temperature
        if (associated(gen_auxvar_ss%d)) then
          ha_dp = gen_auxvar_ss%d%Ha_pg
          ha_dT = gen_auxvar_ss%d%Ha_T
        endif

        internal_energy = gen_auxvar_ss%u(air_comp_id)
        enthalpy = gen_auxvar_ss%h(air_comp_id)
        ! enthalpy units: MJ/kmol                       ! air component mass
        Res(energy_id) = Res(energy_id) + Res(air_comp_id) * enthalpy
        if (analytical_derivatives) then
          Je = 0.d0
          Je(3,1) = Jg(2,1) * enthalpy + Res(air_comp_id) * ha_dp
          Je(3,2) = Jg(2,2) * enthalpy
          Je(3,3) = Jg(2,3) * enthalpy + Res(air_comp_id) * ha_dT
        endif
        J = J + Je
      endif
      if (general_salt) then
        if (dabs(qsrc(salt_comp_id)) > 1.d-40) then
          if (soluble_matrix) then
                             !kmol/sec * kg/kmol * MJ/kg-K * T[C] = MJ/sec
            Res(energy_id) = Res(energy_id) + Res(salt_comp_id) * FMWNACL * &
              soil_heat_capacity * gen_auxvar_ss%temp
          else
            internal_energy = gen_auxvar_ss%u(salt_comp_id)
            enthalpy = gen_auxvar_ss%h(salt_comp_id)
            Res(energy_id) = Res(energy_id) + Res(salt_comp_id) * enthalpy
          endif
        endif
      endif
    endif
    Res(energy_id) = Res(energy_id) + qsrc(energy_id)*scale ! MJ/s
    ! no derivative
  endif

end subroutine GeneralAuxVarComputeAndSrcSink

! ************************************************************************** !

subroutine GeneralAccumDerivative(gen_auxvar,global_auxvar,material_auxvar, &
                                  soil_heat_capacity,option,soluble_matrix,J)
  !
  ! Computes derivatives of the accumulation
  ! term for the Jacobian
  !
  ! Author: Glenn Hammond
  ! Date: 03/09/11
  !

  use Option_module
  use Material_Aux_module
  use Material_module

  implicit none

  type(general_auxvar_type) :: gen_auxvar(0:)
  type(global_auxvar_type) :: global_auxvar
  type(material_auxvar_type) :: material_auxvar
  type(option_type) :: option
  PetscReal :: soil_heat_capacity
  PetscReal :: J(option%nflowdof,option%nflowdof)

  PetscReal :: res(option%nflowdof), res_pert(option%nflowdof)
  PetscReal :: res_pert_minus(option%nflowdof)
  PetscReal :: jac(option%nflowdof,option%nflowdof)
  PetscReal :: jac_pert(option%nflowdof,option%nflowdof)
  PetscInt :: idof, irow
  PetscBool :: soluble_matrix

!geh:print *, 'GeneralAccumDerivative'
  if (.not. general_central_diff_jacobian) then
    call GeneralAccumulation(gen_auxvar(ZERO_INTEGER),global_auxvar, &
                             material_auxvar,soil_heat_capacity,option, &
                             res,jac,general_analytical_derivatives, &
                             soluble_matrix,PETSC_FALSE)
  endif

  if (general_analytical_derivatives) then
    J = jac
  else
    if (general_central_diff_jacobian) then
      do idof = 1, option%nflowdof
        call GeneralAccumulation(gen_auxvar(idof),global_auxvar, &
                                 material_auxvar,soil_heat_capacity,option, &
                                 res_pert,jac_pert,PETSC_FALSE,soluble_matrix,&
                                 PETSC_FALSE)

        call GeneralAccumulation(gen_auxvar(idof+option%nflowdof),global_auxvar, &
                                 material_auxvar,soil_heat_capacity,option, &
                                 res_pert_minus,jac_pert,PETSC_FALSE,soluble_matrix,&
                                 PETSC_FALSE)
        do irow = 1, option%nflowdof
          J(irow,idof) = (res_pert(irow)-res_pert_minus(irow))/ (2.d0 * &
                          gen_auxvar(idof)%pert)
        enddo !irow
      enddo ! idof
    else
      do idof = 1, option%nflowdof
        call GeneralAccumulation(gen_auxvar(idof),global_auxvar, &
                                 material_auxvar,soil_heat_capacity,option, &
                                 res_pert,jac_pert,PETSC_FALSE,soluble_matrix,&
                                 PETSC_FALSE)

        do irow = 1, option%nflowdof
          J(irow,idof) = (res_pert(irow)-res(irow))/gen_auxvar(idof)%pert
        enddo !irow
      enddo ! idof
    endif
  endif

  if (option%flow%isothermal) then
    J(GENERAL_ENERGY_EQUATION_INDEX,:) = 0.d0
    J(:,GENERAL_ENERGY_EQUATION_INDEX) = 0.d0
  endif

  if (general_no_air) then
    J(GENERAL_GAS_EQUATION_INDEX,:) = 0.d0
    J(:,GENERAL_GAS_EQUATION_INDEX) = 0.d0
  endif

end subroutine GeneralAccumDerivative

! ************************************************************************** !

subroutine GeneralFluxDerivative(gen_auxvar_up,global_auxvar_up, &
                                 material_auxvar_up, &
                                 thermal_cc_up, &
                                 gen_auxvar_dn,global_auxvar_dn, &
                                 material_auxvar_dn, &
                                 thermal_cc_dn, &
                                 area, dist, upwind_direction_, &
                                 general_parameter, &
                                 option,Jup,Jdn)
  !
  ! Computes the derivatives of the internal flux terms
  ! for the Jacobian
  !
  ! Author: Glenn Hammond
  ! Date: 03/09/11
  !
  use Option_module
  use Material_Aux_module
  use Upwind_Direction_module, only : count_upwind_direction_flip
  use Characteristic_Curves_Thermal_module

  implicit none

  type(general_auxvar_type) :: gen_auxvar_up(0:), gen_auxvar_dn(0:)
  type(global_auxvar_type) :: global_auxvar_up, global_auxvar_dn
  type(material_auxvar_type) :: material_auxvar_up, material_auxvar_dn
  type(option_type) :: option
  PetscReal :: area
  PetscReal :: dist(-1:3)
  PetscInt :: upwind_direction_(option%nphase)
  type(general_parameter_type) :: general_parameter
  class(cc_thermal_type) :: thermal_cc_up, thermal_cc_dn
  PetscReal :: Jup(option%nflowdof,option%nflowdof)
  PetscReal :: Jdn(option%nflowdof,option%nflowdof)
  PetscReal :: Janal_up(option%nflowdof,option%nflowdof)
  PetscReal :: Janal_dn(option%nflowdof,option%nflowdof)
  PetscReal :: Jdummy(option%nflowdof,option%nflowdof)

  PetscReal :: v_darcy(option%nphase)
  PetscReal :: res(option%nflowdof), res_pert(option%nflowdof)
  PetscReal :: res_pert_minus(option%nflowdof)
  PetscInt :: idof, irow

  Jup = 0.d0
  Jdn = 0.d0

!geh:print *, 'GeneralFluxDerivative'
  option%iflag = -2

  if (.not. general_central_diff_jacobian) then
    call GeneralFlux(gen_auxvar_up(ZERO_INTEGER),global_auxvar_up, &
                     material_auxvar_up, &
                     thermal_cc_up, &
                     gen_auxvar_dn(ZERO_INTEGER),global_auxvar_dn, &
                     material_auxvar_dn, &
                     thermal_cc_dn, &
                     area,dist,upwind_direction_, &
                     general_parameter, &
                     option,v_darcy,res,Janal_up,Janal_dn,&
                     general_analytical_derivatives, &
                     PETSC_FALSE, & ! update the upwind direction
                     ! avoid double counting upwind direction flip
                     PETSC_FALSE, & ! count upwind direction flip
                     PETSC_FALSE)
  endif

  if (general_analytical_derivatives) then
    Jup = Janal_up
    Jdn = Janal_dn
  else
    ! upgradient derivatives
    if (general_central_diff_jacobian) then
       do idof = 1, option%nflowdof
         call GeneralFlux(gen_auxvar_up(idof),global_auxvar_up, &
                          material_auxvar_up, &
                          thermal_cc_up, &
                          gen_auxvar_dn(ZERO_INTEGER),global_auxvar_dn, &
                          material_auxvar_dn, &
                          thermal_cc_dn, &
                          area,dist,upwind_direction_, &
                          general_parameter, &
                          option,v_darcy,res_pert,Jdummy,Jdummy, &
                          PETSC_FALSE, & ! analytical derivatives
                          PETSC_FALSE, & ! update the upwind direction
                          count_upwind_direction_flip, &
                          PETSC_FALSE)

         call GeneralFlux(gen_auxvar_up(idof+option%nflowdof),&
                          global_auxvar_up, &
                          material_auxvar_up, &
                          thermal_cc_up, &
                          gen_auxvar_dn(ZERO_INTEGER),global_auxvar_dn, &
                          material_auxvar_dn, &
                          thermal_cc_dn, &
                          area,dist,upwind_direction_, &
                          general_parameter, &
                          option,v_darcy,res_pert_minus,Jdummy,Jdummy, &
                          PETSC_FALSE, & ! analytical derivatives
                          PETSC_FALSE, & ! update the upwind direction
                          count_upwind_direction_flip, &
                          PETSC_FALSE)
         do irow = 1, option%nflowdof
           Jup(irow,idof) = (res_pert(irow)-res_pert_minus(irow))/(2.d0 * &
                            gen_auxvar_up(idof)%pert)
         enddo !irow
      enddo !idof
    else
      do idof = 1, option%nflowdof
        call GeneralFlux(gen_auxvar_up(idof),global_auxvar_up, &
                         material_auxvar_up, &
                         thermal_cc_up, &
                         gen_auxvar_dn(ZERO_INTEGER),global_auxvar_dn, &
                         material_auxvar_dn, &
                         thermal_cc_dn, &
                         area,dist,upwind_direction_, &
                         general_parameter, &
                         option,v_darcy,res_pert,Jdummy,Jdummy, &
                         PETSC_FALSE, & ! analytical derivatives
                         PETSC_FALSE, & ! update the upwind direction
                         count_upwind_direction_flip, &
                         PETSC_FALSE)

        do irow = 1, option%nflowdof
          Jup(irow,idof) = (res_pert(irow)-res(irow))/gen_auxvar_up(idof)%pert

    !geh:print *, 'up: ', irow, idof, Jup(irow,idof), gen_auxvar_up(idof)%pert
        enddo !irow
      enddo ! idof
    endif

    ! downgradient derivatives
    if (general_central_diff_jacobian) then
       do idof = 1, option%nflowdof
         call GeneralFlux(gen_auxvar_up(ZERO_INTEGER),global_auxvar_up, &
                          material_auxvar_up, &
                          thermal_cc_up, &
                          gen_auxvar_dn(idof),global_auxvar_dn, &
                          material_auxvar_dn, &
                          thermal_cc_dn, &
                          area,dist,upwind_direction_, &
                          general_parameter, &
                          option,v_darcy,res_pert,Jdummy,Jdummy, &
                          PETSC_FALSE, & ! analytical derivatives
                          PETSC_FALSE, & ! update the upwind direction
                          count_upwind_direction_flip, &
                          PETSC_FALSE)

         call GeneralFlux(gen_auxvar_up(ZERO_INTEGER),&
                          global_auxvar_up, &
                          material_auxvar_up, &
                          thermal_cc_up, &
                          gen_auxvar_dn(idof+option%nflowdof),global_auxvar_dn, &
                          material_auxvar_dn, &
                          thermal_cc_dn, &
                          area,dist,upwind_direction_, &
                          general_parameter, &
                          option,v_darcy,res_pert_minus,Jdummy,Jdummy, &
                          PETSC_FALSE, & ! analytical derivatives
                          PETSC_FALSE, & ! update the upwind direction
                          count_upwind_direction_flip, &
                          PETSC_FALSE)
         do irow = 1, option%nflowdof
           Jdn(irow,idof) = (res_pert(irow)-res_pert_minus(irow))/(2.d0 * &
                            gen_auxvar_up(idof)%pert)
         enddo !irow
       enddo ! idof
    else
      do idof = 1, option%nflowdof
        call GeneralFlux(gen_auxvar_up(ZERO_INTEGER),global_auxvar_up, &
                         material_auxvar_up, &
                         thermal_cc_up, &
                         gen_auxvar_dn(idof),global_auxvar_dn, &
                         material_auxvar_dn, &
                         thermal_cc_dn, &
                         area,dist,upwind_direction_, &
                         general_parameter, &
                         option,v_darcy,res_pert,Jdummy,Jdummy, &
                         PETSC_FALSE, & ! analytical derivatives
                         PETSC_FALSE, & ! update the upwind direction
                         count_upwind_direction_flip, &
                         PETSC_FALSE)


        do irow = 1, option%nflowdof
          Jdn(irow,idof) = (res_pert(irow)-res(irow))/gen_auxvar_dn(idof)%pert
    !geh:print *, 'dn: ', irow, idof, Jdn(irow,idof), gen_auxvar_dn(idof)%pert
        enddo !irow
      enddo ! idof
    endif ! forward diff
  endif

  if (option%flow%isothermal) then
    Jup(GENERAL_ENERGY_EQUATION_INDEX,:) = 0.d0
    Jup(:,GENERAL_ENERGY_EQUATION_INDEX) = 0.d0
    Jdn(GENERAL_ENERGY_EQUATION_INDEX,:) = 0.d0
    Jdn(:,GENERAL_ENERGY_EQUATION_INDEX) = 0.d0
  endif

  if (general_no_air) then
    Jup(GENERAL_GAS_EQUATION_INDEX,:) = 0.d0
    Jup(:,GENERAL_GAS_EQUATION_INDEX) = 0.d0
    Jdn(GENERAL_GAS_EQUATION_INDEX,:) = 0.d0
    Jdn(:,GENERAL_GAS_EQUATION_INDEX) = 0.d0
  endif

end subroutine GeneralFluxDerivative

! ************************************************************************** !

subroutine GeneralBCFluxDerivative(ibndtype,auxvar_mapping,auxvars, &
                                   gen_auxvar_up, &
                                   global_auxvar_up, &
                                   gen_auxvar_dn,global_auxvar_dn, &
                                   material_auxvar_dn, &
                                   thermal_cc_dn, &
                                   area,dist,upwind_direction_, &
                                   general_parameter, &
                                   option,Jdn)
  !
  ! Computes the derivatives of the boundary flux terms
  ! for the Jacobian
  !
  ! Author: Glenn Hammond
  ! Date: 03/09/11
  !

  use Option_module
  use Material_Aux_module
  use Upwind_Direction_module, only : count_upwind_direction_flip
  use Characteristic_Curves_Thermal_module

  implicit none

  type(option_type) :: option
  PetscInt :: ibndtype(1:option%nflowdof)
  PetscInt :: auxvar_mapping(GENERAL_MAX_INDEX)
  PetscReal :: auxvars(:) ! from aux_real_var array
  type(general_auxvar_type) :: gen_auxvar_up, gen_auxvar_dn(0:)
  type(global_auxvar_type) :: global_auxvar_up, global_auxvar_dn
  type(material_auxvar_type) :: material_auxvar_dn
  class(cc_thermal_type) :: thermal_cc_dn
  PetscReal :: area
  PetscReal :: dist(-1:3)
  PetscInt :: upwind_direction_(option%nphase)
  type(general_parameter_type) :: general_parameter
  PetscReal :: Jdn(option%nflowdof,option%nflowdof)

  PetscReal :: v_darcy(option%nphase)
  PetscReal :: res(option%nflowdof), res_pert(option%nflowdof)
  PetscReal :: res_pert_minus(option%nflowdof)
  PetscInt :: idof, irow
  PetscReal :: Jdum(option%nflowdof,option%nflowdof)

  Jdn = 0.d0
!geh:print *, 'GeneralBCFluxDerivative'

  option%iflag = -2

  if (.not. general_central_diff_jacobian) then
    call GeneralBCFlux(ibndtype,auxvar_mapping,auxvars, &
                       gen_auxvar_up,global_auxvar_up, &
                       gen_auxvar_dn(ZERO_INTEGER),global_auxvar_dn, &
                       material_auxvar_dn, &
                       thermal_cc_dn, &
                       area,dist,upwind_direction_, &
                       general_parameter, &
                       option,v_darcy,res,Jdum, &
                       general_analytical_derivatives, &
                       PETSC_FALSE, & ! update the upwind direction
                       ! avoid double counting upwind direction flip
                       PETSC_FALSE, & ! count upwind direction flip
                       PETSC_FALSE)
  endif

  if (general_analytical_derivatives) then
    Jdn = Jdum
  else
    ! downgradient derivatives
    if (general_central_diff_jacobian) then
      do idof = 1, option%nflowdof
        call GeneralBCFlux(ibndtype,auxvar_mapping,auxvars, &
                           gen_auxvar_up,global_auxvar_up, &
                           gen_auxvar_dn(idof),global_auxvar_dn, &
                           material_auxvar_dn, &
                           thermal_cc_dn, &
                           area,dist,upwind_direction_, &
                           general_parameter, &
                           option,v_darcy,res_pert,Jdum, &
                           PETSC_FALSE, & ! analytical derivatives
                           PETSC_FALSE, & ! update the upwind direction
                           count_upwind_direction_flip, &
                           PETSC_FALSE)

        call GeneralBCFlux(ibndtype,auxvar_mapping,auxvars, &
                           gen_auxvar_up,global_auxvar_up, &
                           gen_auxvar_dn(idof+option%nflowdof),global_auxvar_dn, &
                           material_auxvar_dn, &
                           thermal_cc_dn, &
                           area,dist,upwind_direction_, &
                           general_parameter, &
                           option,v_darcy,res_pert_minus,Jdum, &
                           PETSC_FALSE, & ! analytical derivatives
                           PETSC_FALSE, & ! update the upwind direction
                           count_upwind_direction_flip, &
                           PETSC_FALSE)


        do irow = 1, option%nflowdof
          Jdn(irow,idof) = (res_pert(irow)-res_pert_minus(irow))/ (2.d0 * &
                            gen_auxvar_dn(idof)%pert)
        enddo !irow
      enddo ! idof
    else
      do idof = 1, option%nflowdof
        call GeneralBCFlux(ibndtype,auxvar_mapping,auxvars, &
                           gen_auxvar_up,global_auxvar_up, &
                           gen_auxvar_dn(idof),global_auxvar_dn, &
                           material_auxvar_dn, &
                           thermal_cc_dn, &
                           area,dist,upwind_direction_, &
                           general_parameter, &
                           option,v_darcy,res_pert,Jdum, &
                           PETSC_FALSE, & ! analytical derivatives
                           PETSC_FALSE, & ! update the upwind direction
                           count_upwind_direction_flip, &
                           PETSC_FALSE)


        do irow = 1, option%nflowdof
          Jdn(irow,idof) = (res_pert(irow)-res(irow))/gen_auxvar_dn(idof)%pert
        enddo !irow
      enddo ! idof
    endif !forward difference
  endif

  if (option%flow%isothermal) then
    Jdn(GENERAL_ENERGY_EQUATION_INDEX,:) = 0.d0
    Jdn(:,GENERAL_ENERGY_EQUATION_INDEX) = 0.d0
  endif

  if (general_no_air) then
    Jdn(GENERAL_GAS_EQUATION_INDEX,:) = 0.d0
    Jdn(:,GENERAL_GAS_EQUATION_INDEX) = 0.d0
  endif

end subroutine GeneralBCFluxDerivative

! ************************************************************************** !

subroutine GeneralSrcSinkDerivative(option,source_sink,gen_auxvar_ss, &
                                    gen_auxvar,global_auxvar, &
                                    global_auxvar_ss, &
                                    characteristic_curves,natural_id, &
                                    material_auxvar,scale,soluble_matrix, &
                                    soil_heat_capacity, Jac)
  !
  ! Computes the source/sink terms for the residual
  !
  ! Author: Glenn Hammond
  ! Date: 03/09/11
  !

  use Option_module
  use Coupler_module
  use Material_Aux_module
  use Characteristic_Curves_module

  implicit none

  type(option_type) :: option
  type(coupler_type), pointer :: source_sink
  type(general_auxvar_type) :: gen_auxvar(0:), gen_auxvar_ss(0:1)
  type(global_auxvar_type) :: global_auxvar, global_auxvar_ss
  class(characteristic_curves_type) :: characteristic_curves
  PetscInt :: natural_id
  type(material_auxvar_type) :: material_auxvar
  PetscReal :: scale
  PetscReal :: Jac(option%nflowdof,option%nflowdof)

  PetscReal :: qsrc(option%nflowdof)
  PetscInt :: flow_src_sink_type
  PetscReal :: res(option%nflowdof), res_pert(option%nflowdof)
  PetscReal :: res_pert_minus(option%nflowdof)
  PetscReal :: dummy_real(option%nphase)
  PetscInt :: idof, irow
  PetscReal :: Jdum(option%nflowdof,option%nflowdof)
  PetscBool :: soluble_matrix
  PetscReal :: soil_heat_capacity

  qsrc = source_sink%flow_condition%general%rate%dataset%rarray(:)
  flow_src_sink_type = source_sink%flow_condition%general%rate%itype

  option%iflag = -3

  ! Index 0 contains user-specified conditions
  ! Index 1 contains auxvars to be used in src/sink calculations
  if (.not. general_central_diff_jacobian) then
    call GeneralAuxVarComputeAndSrcSink(option,qsrc,flow_src_sink_type, &
                        gen_auxvar_ss(ZERO_INTEGER), gen_auxvar(ZERO_INTEGER),&
                        global_auxvar, global_auxvar_ss, &
                        material_auxvar, dummy_real, characteristic_curves, &
                        natural_id, scale,res,Jdum, &
                        general_analytical_derivatives, PETSC_FALSE, soluble_matrix, &
                        soil_heat_capacity, PETSC_FALSE)
  endif

  if (general_analytical_derivatives) then
    Jac = Jdum
  else
    ! downgradient derivatives
    if (general_central_diff_jacobian) then
      do idof = 1, option%nflowdof
        call GeneralAuxVarCopy(gen_auxvar_ss(ZERO_INTEGER), &
                               gen_auxvar_ss(ONE_INTEGER), option)
        call GeneralAuxVarComputeAndSrcSink(option,qsrc,flow_src_sink_type, &
                            gen_auxvar_ss(ONE_INTEGER), gen_auxvar(idof), &
                            global_auxvar,global_auxvar_ss, &
                            material_auxvar, dummy_real, characteristic_curves, &
                            natural_id, scale, res_pert,Jdum,PETSC_FALSE, &
                            PETSC_FALSE, soluble_matrix, soil_heat_capacity, PETSC_FALSE)
        call GeneralAuxVarComputeAndSrcSink(option,qsrc,flow_src_sink_type, &
                            gen_auxvar_ss(ONE_INTEGER), gen_auxvar(idof+option%nflowdof), &
                            global_auxvar,global_auxvar_ss, &
                            material_auxvar, dummy_real, characteristic_curves, &
                            natural_id, scale, res_pert_minus,Jdum,PETSC_FALSE, &
                            PETSC_FALSE, soluble_matrix, soil_heat_capacity, PETSC_FALSE)

        do irow = 1, option%nflowdof
          Jac(irow,idof) = (res_pert(irow)-res_pert_minus(irow))/ (2.d0 * &
                            gen_auxvar(idof)%pert)
        enddo !irow
      enddo ! idof
    else
      do idof = 1, option%nflowdof
        call GeneralAuxVarCopy(gen_auxvar_ss(ZERO_INTEGER), &
                               gen_auxvar_ss(ONE_INTEGER), option)
        call GeneralAuxVarComputeAndSrcSink(option,qsrc,flow_src_sink_type, &
                            gen_auxvar_ss(ONE_INTEGER), gen_auxvar(idof), &
                            global_auxvar,global_auxvar_ss, &
                            material_auxvar, dummy_real,characteristic_curves, &
                            natural_id, scale, res_pert,Jdum,PETSC_FALSE, &
                            PETSC_FALSE, soluble_matrix, soil_heat_capacity, PETSC_FALSE)

        do irow = 1, option%nflowdof
          Jac(irow,idof) = (res_pert(irow)-res(irow))/gen_auxvar(idof)%pert
        enddo !irow
      enddo ! idof
    endif !forward diff
  endif

  if (option%flow%isothermal) then
    Jac(GENERAL_ENERGY_EQUATION_INDEX,:) = 0.d0
    Jac(:,GENERAL_ENERGY_EQUATION_INDEX) = 0.d0
  endif

  if (general_no_air) then
    Jac(GENERAL_GAS_EQUATION_INDEX,:) = 0.d0
    Jac(:,GENERAL_GAS_EQUATION_INDEX) = 0.d0
  endif

end subroutine GeneralSrcSinkDerivative

! ************************************************************************** !

function GeneralAverageDensity(iphase,istate_up,istate_dn, &
                               density_up,density_dn,dden_up,dden_dn)
  !
  ! Averages density, using opposite cell density if phase non-existent
  !
  ! Author: Glenn Hammond
  ! Date: 03/07/14
  !

  implicit none

  PetscInt :: iphase
  PetscInt :: istate_up, istate_dn
  PetscReal :: density_up(:), density_dn(:)
  PetscReal :: dden_up, dden_dn

  PetscReal :: GeneralAverageDensity

  dden_up = 0.d0
  dden_dn = 0.d0
  if (iphase == LIQUID_PHASE) then
    if (istate_up == GAS_STATE) then
      GeneralAverageDensity = density_dn(iphase)
      dden_dn = 1.d0
    else if (istate_dn == GAS_STATE) then
      GeneralAverageDensity = density_up(iphase)
      dden_up = 1.d0
    else
      GeneralAverageDensity = 0.5d0*(density_up(iphase)+density_dn(iphase))
      dden_up = 0.5d0
      dden_dn = 0.5d0
    endif
  else if (iphase == GAS_PHASE) then
    if (istate_up == LIQUID_STATE) then
      GeneralAverageDensity = density_dn(iphase)
      dden_dn = 1.d0
    else if (istate_dn == LIQUID_STATE) then
      GeneralAverageDensity = density_up(iphase)
      dden_up = 1.d0
    else
      GeneralAverageDensity = 0.5d0*(density_up(iphase)+density_dn(iphase))
      dden_up = 0.5d0
      dden_dn = 0.5d0
    endif
  endif

end function GeneralAverageDensity

! ************************************************************************** !

subroutine GeneralAuxVarDiff(idof,general_auxvar,global_auxvar, &
                             material_auxvar, &
                             general_auxvar_pert,global_auxvar_pert, &
                             material_auxvar_pert, &
                             pert,string,compare_analytical_derivative, &
                             option)

  use Option_module
  use General_Aux_module
  use Global_Aux_module
  use Material_Aux_module

  implicit none

  type(option_type) :: option
  PetscInt :: idof
  type(general_auxvar_type) :: general_auxvar, general_auxvar_pert
  type(global_auxvar_type) :: global_auxvar, global_auxvar_pert
  type(material_auxvar_type) :: material_auxvar, material_auxvar_pert
  character(len=MAXSTRINGLENGTH) :: string
  PetscBool :: compare_analytical_derivative
  PetscReal :: pert


  PetscInt :: apid, cpid, vpid, spid
  PetscInt :: gid, lid, acid, wid, eid
  PetscReal :: liquid_mass, gas_mass
  PetscReal :: liquid_density, gas_density
  PetscReal :: liquid_energy, gas_energy
  PetscReal :: liquid_saturation, gas_saturation
  PetscReal :: liquid_mass_pert, gas_mass_pert
  PetscReal :: liquid_density_pert, gas_density_pert
  PetscReal :: liquid_energy_pert, gas_energy_pert
  PetscReal :: liquid_saturation_pert, gas_saturation_pert

  PetscReal :: dpl
  PetscReal :: dpg
  PetscReal :: dpa
  PetscReal :: dpc
  PetscReal :: dpv
  PetscReal :: dps
  PetscReal :: dsatl
  PetscReal :: dsatg
  PetscReal :: ddenl
  PetscReal :: ddeng
  PetscReal :: ddenlkg
  PetscReal :: ddengkg
  PetscReal :: dUl
  PetscReal :: dHl
  PetscReal :: dUg
  PetscReal :: dHg
  PetscReal :: dUv
  PetscReal :: dHv
  PetscReal :: dUa
  PetscReal :: dHa
  PetscReal :: dpsat
  PetscReal :: dmobilityl
  PetscReal :: dmobilityg
  PetscReal :: dxmolwl
  PetscReal :: dxmolal
  PetscReal :: dxmolwg
  PetscReal :: dxmolag
  PetscReal :: denv
  PetscReal :: dena
  PetscReal :: dHc
  PetscReal :: dmug

  PetscReal, parameter :: uninitialized_value = -999.d0

  dpl = uninitialized_value
  dpg = uninitialized_value
  dpa = uninitialized_value
  dpc = uninitialized_value
  dpv = uninitialized_value
  dps = uninitialized_value
  dsatl = uninitialized_value
  dsatg = uninitialized_value
  ddenl = uninitialized_value
  ddeng = uninitialized_value
  ddenlkg = uninitialized_value
  ddengkg = uninitialized_value
  dUl = uninitialized_value
  dHl = uninitialized_value
  dUg = uninitialized_value
  dHg = uninitialized_value
  dUv = uninitialized_value
  dHv = uninitialized_value
  dUa = uninitialized_value
  dHa = uninitialized_value
  dpsat = uninitialized_value
  dmobilityl = uninitialized_value
  dmobilityg = uninitialized_value
  dxmolwl = uninitialized_value
  dxmolal = uninitialized_value
  dxmolwg = uninitialized_value
  dxmolag = uninitialized_value
  denv = uninitialized_value
  dena = uninitialized_value
  dHc = uninitialized_value
  dmug = uninitialized_value

  lid = option%liquid_phase
  gid = option%gas_phase
  apid = option%air_pressure_id
  cpid = option%capillary_pressure_id
  vpid = option%vapor_pressure_id
  spid = option%saturation_pressure_id

  acid = option%air_id ! air component id
  wid = option%water_id
  eid = option%energy_id

  liquid_density = 0.d0
  gas_density = 0.d0
  liquid_energy = 0.d0
  gas_energy = 0.d0
  liquid_saturation = 0.d0
  gas_saturation = 0.d0

  if (compare_analytical_derivative) then
    select case(global_auxvar%istate)
      case(LIQUID_STATE)
        select case(idof)
          case(1) ! liquid pressure pl
            dpl = 1.d0
            dpv = 0.d0
            dpa = general_auxvar%d%Hc_p*general_auxvar%xmol(acid,lid)
            dps = 0.d0
            dpv = general_auxvar%d%pv_p
            ddenl = general_auxvar%d%denl_pl
            dUl = general_auxvar%d%Ul_pl
            dHl = general_auxvar%d%Hl_pl
            dmobilityl = general_auxvar%d%mobilityl_pl
          case(2) ! xmole air in liquid
            dxmolwl = -1.d0
            dxmolal = 1.d0
          case(3) ! temperature
            dpl = 0.d0 ! pl = pg - pc
            dpg = 0.d0 ! pg = pg
            dpa = general_auxvar%d%Hc_T*general_auxvar%xmol(acid,lid)
            dps = general_auxvar%d%psat_T
            dHc = general_auxvar%d%Hc_T
            dsatl = 0.d0
            dsatg = 0.d0
            ddenl = general_auxvar%d%denl_T
            ddeng = general_auxvar%d%deng_T
            ddenlkg = ddenl*fmw_comp(1)
            ddengkg = general_auxvar%d%dengkg_T
            dUl = general_auxvar%d%Ul_T
            dHl = general_auxvar%d%Hl_T

            dpv = general_auxvar%d%pv_T
            dmobilityl = general_auxvar%d%mobilityl_T
            dxmolwl = general_auxvar%d%xmol_T(wid,lid)
            dxmolal = general_auxvar%d%xmol_T(acid,lid)
        end select
      case(GAS_STATE)
        select case(idof)
          case(1)
            dpg = 1.d0 ! pg = pg
            dpv = general_auxvar%d%pv_p
            dps = 0.d0
            dHc = general_auxvar%d%Hc_p
            ddeng = general_auxvar%d%deng_pg
            ddengkg = general_auxvar%d%dengkg_pg
            dUg = general_auxvar%d%Ug_pg
            dHg = general_auxvar%d%Hg_pg

            dHv = general_auxvar%d%Hv_pg
            dUv = general_auxvar%d%Uv_pg
            dHa = general_auxvar%d%Ha_pg
            dUa = general_auxvar%d%Ua_pg

            dmug = general_auxvar%d%mug_pg
            dmobilityg = general_auxvar%d%mobilityg_pg
            dxmolwg = general_auxvar%d%xmol_p(wid,gid)
            dxmolag = general_auxvar%d%xmol_p(acid,gid)
          case(2)
            dpg = 0.d0
            dpa = 1.d0
            ddeng = general_auxvar%d%deng_pa
            dpv = general_auxvar%d%pv_pa
            dUg = general_auxvar%d%Ug_pa
            dHg = general_auxvar%d%Hg_pa
            dHv = general_auxvar%d%Hv_pa
            dUv = general_auxvar%d%Uv_pa
            dHa = general_auxvar%d%Ha_pa
            dUa = general_auxvar%d%Ua_pa
            ! for gas state, derivative wrt air pressure is under lid
            dxmolwg = general_auxvar%d%xmol_p(wid,lid)
            dxmolag = general_auxvar%d%xmol_p(acid,lid)
            dmobilityg = general_auxvar%d%mobilityg_pa
          case(3)
            dpg = 0.d0
            dpa = 0.d0
            dpv = 0.d0
            dps = general_auxvar%d%psat_T
            dHc = general_auxvar%d%Hc_T
            ddeng = general_auxvar%d%deng_T
            ddengkg = general_auxvar%d%dengkg_T
            dUg = general_auxvar%d%Ug_T
            dHg = general_auxvar%d%Hg_T

            dHv = general_auxvar%d%Hv_T
            dUv = general_auxvar%d%Uv_T
            dHa = general_auxvar%d%Ha_T
            dUa = general_auxvar%d%Ua_T
            denv = general_auxvar%d%denv_T
            dena = general_auxvar%d%dena_T

            dmug = general_auxvar%d%mug_T
            dmobilityg = general_auxvar%d%mobilityg_T
            dxmolwg = general_auxvar%d%xmol_T(wid,gid)
            dxmolag = general_auxvar%d%xmol_T(acid,gid)
        end select
      case(TWO_PHASE_STATE)
        select case(idof)
          case(1) ! gas pressure pg
            dpl = 1.d0 ! pl = pg - pc
            dpg = 1.d0 ! pg = pg
            dpa = 1.d0 ! pa = pg - pv
            dpv = 0.d0
            dps = 0.d0
            dsatl = 0.d0
            dsatg = 0.d0
            dHc = general_auxvar%d%Hc_p
            ddenl = general_auxvar%d%denl_pl*dpl
            ddeng = general_auxvar%d%deng_pg
            ddenlkg = ddenl*fmw_comp(1)
            ddengkg = general_auxvar%d%dengkg_pg
            dUl = general_auxvar%d%Ul_pl
            dHl = general_auxvar%d%Hl_pl
            dUg = general_auxvar%d%Ug_pg
            dHg = general_auxvar%d%Hg_pg

            denv = general_auxvar%d%denv_pg
            dena = general_auxvar%d%dena_pg
            dHv = general_auxvar%d%Hv_pg
            dUv = general_auxvar%d%Uv_pg
            dHa = general_auxvar%d%Ha_pg
            dUa = general_auxvar%d%Ua_pg

            dmug = general_auxvar%d%mug_pg
            dmobilityl = general_auxvar%d%mobilityl_pl
            dmobilityg = general_auxvar%d%mobilityg_pg
            dxmolwl = general_auxvar%d%xmol_p(wid,lid)
            dxmolal = general_auxvar%d%xmol_p(acid,lid)
            dxmolwg = general_auxvar%d%xmol_p(wid,gid)
            dxmolag = general_auxvar%d%xmol_p(acid,gid)
          case(2) ! gas saturation
            dpl = -1.d0*general_auxvar%d%pc_satg ! pl = pg - pc
            dpg = 0.d0
            dpa = 0.d0
            dpc = general_auxvar%d%pc_satg
            dpv = 0.d0
            dps = 0.d0
            dsatl = -1.d0
            dsatg = 1.d0
            ddenl = 0.d0
            dmobilityl = general_auxvar%d%mobilityl_satg
            dmobilityg = general_auxvar%d%mobilityg_satg
          case(3) ! temperature
            dpl = 0.d0 ! pl = pg - pc
            dpg = 0.d0 ! pg = pg
            dpa = -1.d0*general_auxvar%d%psat_T ! pa = pg - pv
            dpv = general_auxvar%d%psat_T
            dps = general_auxvar%d%psat_T
            dHc = general_auxvar%d%Hc_T
            dsatl = 0.d0
            dsatg = 0.d0
            ddenl = general_auxvar%d%denl_T
            ddeng = general_auxvar%d%deng_T
            ddenlkg = ddenl*fmw_comp(1)
            ddengkg = general_auxvar%d%dengkg_T
            dUl = general_auxvar%d%Ul_T
            dHl = general_auxvar%d%Hl_T
            dUg = general_auxvar%d%Ug_T
            dHg = general_auxvar%d%Hg_T

            dHv = general_auxvar%d%Hv_T
            dUv = general_auxvar%d%Uv_T
            dHa = general_auxvar%d%Ha_T
            dUa = general_auxvar%d%Ua_T
            denv = general_auxvar%d%denv_T
            dena = general_auxvar%d%dena_T

            dmug = general_auxvar%d%mug_T
            dmobilityl = general_auxvar%d%mobilityl_T
            dmobilityg = general_auxvar%d%mobilityg_T
            dxmolwl = general_auxvar%d%xmol_T(wid,lid)
            dxmolal = general_auxvar%d%xmol_T(acid,lid)
            dxmolwg = general_auxvar%d%xmol_T(wid,gid)
            dxmolag = general_auxvar%d%xmol_T(acid,gid)
        end select
      end select
    endif

  print *, '--------------------------------------------------------'
  print *, 'Derivative with respect to ' // trim(string)
  select case(global_auxvar%istate)
    case(LIQUID_STATE)
      print *, '     Thermodynamic state: Liquid phase'
      liquid_density = general_auxvar%den(lid)
      liquid_energy = general_auxvar%U(lid)
      liquid_saturation = general_auxvar%sat(lid)
    case(GAS_STATE)
      print *, '     Thermodynamic state: Gas phase'
      gas_density = general_auxvar%den(gid)
      gas_energy = general_auxvar%U(gid)
      gas_saturation = general_auxvar%sat(gid)
    case(TWO_PHASE_STATE)
      print *, '     Thermodynamic state: Two phase'
      liquid_density = general_auxvar%den(lid)
      gas_density = general_auxvar%den(gid)
      liquid_energy = general_auxvar%U(lid)
      gas_energy = general_auxvar%U(gid)
      liquid_saturation = general_auxvar%sat(lid)
      gas_saturation = general_auxvar%sat(gid)
  end select
  liquid_mass = (liquid_density*general_auxvar%xmol(lid,lid)* &
                 liquid_saturation+ &
                 gas_density*general_auxvar%xmol(lid,gid)* &
                 gas_saturation)* &
                 general_auxvar%effective_porosity*material_auxvar%volume
  gas_mass = (liquid_density*general_auxvar%xmol(gid,lid)* &
              liquid_saturation+ &
              gas_density*general_auxvar%xmol(gid,gid)* &
              gas_saturation)* &
              general_auxvar%effective_porosity*material_auxvar%volume
  select case(global_auxvar_pert%istate)
    case(LIQUID_STATE)
      print *, '     Thermodynamic state (pert): Liquid phase'
      liquid_density_pert = general_auxvar_pert%den(lid)
      liquid_energy_pert = general_auxvar_pert%U(lid)
      liquid_saturation_pert = general_auxvar_pert%sat(lid)
      gas_density_pert = 0.d0
      gas_energy_pert = 0.d0
      gas_saturation_pert = 0.d0
    case(GAS_STATE)
      print *, '     Thermodynamic state (pert): Gas phase'
      liquid_density_pert = 0.d0
      liquid_energy_pert = 0.d0
      liquid_saturation_pert = 0.d0
      gas_density_pert = general_auxvar_pert%den(gid)
      gas_energy_pert = general_auxvar_pert%U(gid)
      gas_saturation_pert = general_auxvar_pert%sat(gid)
    case(TWO_PHASE_STATE)
      print *, '     Thermodynamic state (pert): Two phase'
      liquid_density_pert = general_auxvar_pert%den(lid)
      gas_density_pert = general_auxvar_pert%den(gid)
      liquid_energy_pert = general_auxvar_pert%U(lid)
      gas_energy_pert = general_auxvar_pert%U(gid)
      liquid_saturation_pert = general_auxvar_pert%sat(lid)
      gas_saturation_pert = general_auxvar_pert%sat(gid)
  end select
  liquid_mass_pert = (liquid_density_pert*general_auxvar_pert%xmol(lid,lid)* &
                 liquid_saturation_pert+ &
                 gas_density_pert*general_auxvar_pert%xmol(lid,gid)* &
                 gas_saturation_pert)* &
                 general_auxvar_pert%effective_porosity*material_auxvar_pert%volume
  gas_mass_pert = (liquid_density_pert*general_auxvar_pert%xmol(gid,lid)* &
              liquid_saturation_pert+ &
              gas_density_pert*general_auxvar_pert%xmol(gid,gid)* &
              gas_saturation_pert)* &
              general_auxvar_pert%effective_porosity*material_auxvar_pert%volume


  call GeneralAuxVarPrintResult('tot liq comp mass [kmol]', &
                                (liquid_mass_pert-liquid_mass)/pert, &
                                uninitialized_value,uninitialized_value,option)
  call GeneralAuxVarPrintResult('tot gas comp mass [kmol]', &
                                (gas_mass_pert-gas_mass)/pert, &
                                uninitialized_value,uninitialized_value,option)
  call GeneralAuxVarPrintResult('             energy [MJ]', &
                                ((liquid_mass_pert*liquid_energy_pert + &
                                  gas_mass_pert*gas_energy_pert)- &
                                 (liquid_mass*liquid_energy + &
                                  gas_mass*gas_energy))/pert, &
                                uninitialized_value,uninitialized_value,option)
  call GeneralAuxVarPrintResult('         liquid pressure', &
                                (general_auxvar_pert%pres(lid)-general_auxvar%pres(lid))/pert, &
                                dpl,uninitialized_value,option)
  call GeneralAuxVarPrintResult('            gas pressure', &
                                (general_auxvar_pert%pres(gid)-general_auxvar%pres(gid))/pert, &
                                dpg,uninitialized_value,option)
  call GeneralAuxVarPrintResult('            air pressure', &
                                (general_auxvar_pert%pres(apid)-general_auxvar%pres(apid))/pert, &
                                dpa,uninitialized_value,option)
  call GeneralAuxVarPrintResult('      capillary pressure', &
                                (general_auxvar_pert%pres(cpid)-general_auxvar%pres(cpid))/pert, &
                                dpc,uninitialized_value,option)
  call GeneralAuxVarPrintResult('          vapor pressure', &
                                (general_auxvar_pert%pres(vpid)-general_auxvar%pres(vpid))/pert, &
                                dpv,uninitialized_value,option)
  call GeneralAuxVarPrintResult("        Henry's constant", &
                                (general_auxvar_pert%d%Hc-general_auxvar%d%Hc)/pert, &
                                dHc,uninitialized_value,option)
  call GeneralAuxVarPrintResult('     saturation pressure', &
                                (general_auxvar_pert%pres(spid)-general_auxvar%pres(spid))/pert, &
                                dps,uninitialized_value,option)
  call GeneralAuxVarPrintResult('       liquid saturation', &
                                (general_auxvar_pert%sat(lid)-general_auxvar%sat(lid))/pert, &
                                dsatl,uninitialized_value,option)
  call GeneralAuxVarPrintResult('          gas saturation', &
                                (general_auxvar_pert%sat(gid)-general_auxvar%sat(gid))/pert, &
                                dsatg,uninitialized_value,option)
  call GeneralAuxVarPrintResult('   liquid density [kmol]', &
                                (general_auxvar_pert%den(lid)-general_auxvar%den(lid))/pert, &
                                ddenl,uninitialized_value,option)
  call GeneralAuxVarPrintResult('      gas density [kmol]', &
                                (general_auxvar_pert%den(gid)-general_auxvar%den(gid))/pert, &
                                ddeng,uninitialized_value,option)
  call GeneralAuxVarPrintResult('     liquid density [kg]', &
                                (general_auxvar_pert%den_kg(lid)-general_auxvar%den_kg(lid))/pert, &
                                ddenlkg,uninitialized_value,option)
  call GeneralAuxVarPrintResult('        gas density [kg]', &
                                (general_auxvar_pert%den_kg(gid)-general_auxvar%den_kg(gid))/pert, &
                                ddengkg,uninitialized_value,option)
  call GeneralAuxVarPrintResult('         temperature [C]', &
                                (general_auxvar_pert%temp-general_auxvar%temp)/pert, &
                                uninitialized_value,uninitialized_value,option)
  call GeneralAuxVarPrintResult('      liquid H [MJ/kmol]', &
                                (general_auxvar_pert%H(lid)-general_auxvar%H(lid))/pert, &
                                dHl,uninitialized_value,option)
  call GeneralAuxVarPrintResult('         gas H [MJ/kmol]', &
                                (general_auxvar_pert%H(gid)-general_auxvar%H(gid))/pert, &
                                dHg,uninitialized_value,option)
  call GeneralAuxVarPrintResult('      liquid U [MJ/kmol]', &
                                (general_auxvar_pert%U(lid)-general_auxvar%U(lid))/pert, &
                                dUl,uninitialized_value,option)
  call GeneralAuxVarPrintResult('         gas U [MJ/kmol]', &
                                (general_auxvar_pert%U(gid)-general_auxvar%U(gid))/pert, &
                                dUg,uninitialized_value,option)
  !------------------------------
  call GeneralAuxVarPrintResult('       vapor H [MJ/kmol]', &
                                (general_auxvar_pert%d%Hv-general_auxvar%d%Hv)/pert, &
                                dHv,uninitialized_value,option)
  call GeneralAuxVarPrintResult('         air H [MJ/kmol]', &
                                (general_auxvar_pert%d%Ha-general_auxvar%d%Ha)/pert, &
                                dHa,uninitialized_value,option)
  call GeneralAuxVarPrintResult('       vapor U [MJ/kmol]', &
                                (general_auxvar_pert%d%Uv-general_auxvar%d%Uv)/pert, &
                                dUv,uninitialized_value,option)
  call GeneralAuxVarPrintResult('         air U [MJ/kmol]', &
                                (general_auxvar_pert%d%Ua-general_auxvar%d%Ua)/pert, &
                                dUa,uninitialized_value,option)
  call GeneralAuxVarPrintResult('    vapor density [kmol]', &
                                (general_auxvar_pert%d%denv-general_auxvar%d%denv)/pert, &
                                denv,uninitialized_value,option)
  call GeneralAuxVarPrintResult('      air density [kmol]', &
                                (general_auxvar_pert%d%dena-general_auxvar%d%dena)/pert, &
                                dena,uninitialized_value,option)
  !------------------------------
  call GeneralAuxVarPrintResult('     X (water in liquid)', &
                                (general_auxvar_pert%xmol(wid,lid)-general_auxvar%xmol(wid,lid))/pert, &
                                dxmolwl,uninitialized_value,option)
  call GeneralAuxVarPrintResult('       X (air in liquid)', &
                                (general_auxvar_pert%xmol(acid,lid)-general_auxvar%xmol(acid,lid))/pert, &
                                dxmolal,uninitialized_value,option)
  call GeneralAuxVarPrintResult('        X (water in gas)', &
                                (general_auxvar_pert%xmol(wid,gid)-general_auxvar%xmol(wid,gid))/pert, &
                                dxmolwg,uninitialized_value,option)
  call GeneralAuxVarPrintResult('          X (air in gas)', &
                                (general_auxvar_pert%xmol(acid,gid)-general_auxvar%xmol(acid,gid))/pert, &
                                dxmolag,uninitialized_value,option)
  call GeneralAuxVarPrintResult('         liquid mobility', &
                                (general_auxvar_pert%mobility(lid)-general_auxvar%mobility(lid))/pert, &
                                dmobilityl,uninitialized_value,option)
  call GeneralAuxVarPrintResult('            gas mobility', &
                                (general_auxvar_pert%mobility(gid)-general_auxvar%mobility(gid))/pert, &
                                dmobilityg,uninitialized_value,option)
  call GeneralAuxVarPrintResult('           gas viscosity', &
                                (general_auxvar_pert%d%mug-general_auxvar%d%mug)/pert, &
                                dmug,uninitialized_value,option)
  call GeneralAuxVarPrintResult('      effective porosity', &
                                (general_auxvar_pert%effective_porosity-general_auxvar%effective_porosity)/pert, &
                                uninitialized_value,uninitialized_value,option)
#if 0
100 format(a,2(es13.5),es16.8)
  write(*,100) 'tot liq comp mass [kmol]: ', (liquid_mass_pert-liquid_mass)/pert
  write(*,100) 'tot gas comp mass [kmol]: ', (gas_mass_pert-gas_mass)/pert
  write(*,100) '             energy [MJ]: ', ((liquid_mass_pert*liquid_energy_pert + &
                                           gas_mass_pert*gas_energy_pert)- &
                                          (liquid_mass*liquid_energy + &
                                           gas_mass*gas_energy))/pert
  write(*,100) '         liquid pressure: ', (general_auxvar_pert%pres(lid)-general_auxvar%pres(lid))/pert,dpl
  write(*,100) '            gas pressure: ', (general_auxvar_pert%pres(gid)-general_auxvar%pres(gid))/pert,dpg
  write(*,100) '            air pressure: ', (general_auxvar_pert%pres(apid)-general_auxvar%pres(apid))/pert,dpa !,general_auxvar_pert%pres(apid)-general_auxvar%pres(apid)
  write(*,100) '      capillary pressure: ', (general_auxvar_pert%pres(cpid)-general_auxvar%pres(cpid))/pert,dpc
  write(*,100) '          vapor pressure: ', (general_auxvar_pert%pres(vpid)-general_auxvar%pres(vpid))/pert,dpv !,general_auxvar_pert%pres(vpid)-general_auxvar%pres(vpid)
  write(*,100) "        Henry's constant: ", (general_auxvar_pert%d%Hc-general_auxvar%d%Hc)/pert,dHc
  write(*,100) '     saturation pressure: ', (general_auxvar_pert%pres(spid)-general_auxvar%pres(spid))/pert,dps
  write(*,100) '       liquid saturation: ', (general_auxvar_pert%sat(lid)-general_auxvar%sat(lid))/pert,dsatl
  write(*,100) '          gas saturation: ', (general_auxvar_pert%sat(gid)-general_auxvar%sat(gid))/pert,dsatg
  write(*,100) '   liquid density [kmol]: ', (general_auxvar_pert%den(lid)-general_auxvar%den(lid))/pert,ddenl
  write(*,100) '      gas density [kmol]: ', (general_auxvar_pert%den(gid)-general_auxvar%den(gid))/pert,ddeng
  write(*,100) '     liquid density [kg]: ', (general_auxvar_pert%den_kg(lid)-general_auxvar%den_kg(lid))/pert,ddenl*fmw_comp(1)
  write(*,100) '        gas density [kg]: ', (general_auxvar_pert%den_kg(gid)-general_auxvar%den_kg(gid))/pert,ddengkg
  write(*,100) '         temperature [C]: ', (general_auxvar_pert%temp-general_auxvar%temp)/pert
  write(*,100) '      liquid H [MJ/kmol]: ', (general_auxvar_pert%H(lid)-general_auxvar%H(lid))/pert,dHl
  write(*,100) '         gas H [MJ/kmol]: ', (general_auxvar_pert%H(gid)-general_auxvar%H(gid))/pert,dHg
  write(*,100) '      liquid U [MJ/kmol]: ', (general_auxvar_pert%U(lid)-general_auxvar%U(lid))/pert,dUl
  write(*,100) '         gas U [MJ/kmol]: ', (general_auxvar_pert%U(gid)-general_auxvar%U(gid))/pert,dUg

  write(*,100) '       vapor H [MJ/kmol]: ', (general_auxvar_pert%d%Hv-general_auxvar%d%Hv)/pert,dHv
  write(*,100) '         air H [MJ/kmol]: ', (general_auxvar_pert%d%Ha-general_auxvar%d%Ha)/pert,dHa
  write(*,100) '       vapor U [MJ/kmol]: ', (general_auxvar_pert%d%Uv-general_auxvar%d%Uv)/pert,dUv

  write(*,100) '         air U [MJ/kmol]: ', (general_auxvar_pert%d%Ua-general_auxvar%d%Ua)/pert,dUa
  write(*,100) '    vapor density [kmol]: ', (general_auxvar_pert%d%denv-general_auxvar%d%denv)/pert,denv
  write(*,100) '      air density [kmol]: ', (general_auxvar_pert%d%dena-general_auxvar%d%dena)/pert,dena

  write(*,100) '     X (water in liquid): ', (general_auxvar_pert%xmol(wid,lid)-general_auxvar%xmol(wid,lid))/pert,dxmolwl
  write(*,100) '       X (air in liquid): ', (general_auxvar_pert%xmol(acid,lid)-general_auxvar%xmol(acid,lid))/pert,dxmolal
  write(*,100) '        X (water in gas): ', (general_auxvar_pert%xmol(wid,gid)-general_auxvar%xmol(wid,gid))/pert,dxmolwg
  write(*,100) '          X (air in gas): ', (general_auxvar_pert%xmol(acid,gid)-general_auxvar%xmol(acid,gid))/pert,dxmolag
  write(*,100) '         liquid mobility: ', (general_auxvar_pert%mobility(lid)-general_auxvar%mobility(lid))/pert,dmobilityl
  write(*,100) '            gas mobility: ', (general_auxvar_pert%mobility(gid)-general_auxvar%mobility(gid))/pert,dmobilityg
  write(*,100) '      effective porosity: ', (general_auxvar_pert%effective_porosity-general_auxvar%effective_porosity)/pert
#endif
  write(*,*) '--------------------------------------------------------'

end subroutine GeneralAuxVarDiff

! ************************************************************************** !

subroutine GeneralAuxVarPrintResult(string,numerical,analytical, &
                                    uninitialized_value,option)

  use Option_module
  use Utility_module

  implicit none

  character(len=*) :: string
  PetscReal :: numerical
  PetscReal :: analytical
  PetscReal :: uninitialized_value
  type(option_type) :: option

  character(len=8) :: word
  character(len=2) :: precision
  PetscReal :: tempreal
  PetscReal, parameter :: tol = 1.d-5

100 format(a24,': ',2(es13.5),2x,a2,x,a8,x,es16.8)

  precision = DigitsOfAccuracy(numerical,analytical)
  word = ''
  if (dabs(analytical-uninitialized_value) > 1.d-20) then
    if (dabs(analytical) > 0.d0) then
      tempreal = dabs((numerical-analytical)/analytical)
      if (tempreal < tol) then
        word = ' PASS'
      else
        word = '-FAIL-'
        if (tempreal < 1.d1*tol) word = trim(word) // ' *'
      endif
    else
      if (dabs(numerical) > 1.d-20) then
        word = '-FAIL-'
      else
        word = ' PASS'
      endif
    endif
    write(*,100) trim(string), numerical, analytical, precision, word
  else
    write(*,100) trim(string), numerical
  endif


end subroutine GeneralAuxVarPrintResult

! ************************************************************************** !

subroutine GeneralDiffJacobian(string,numerical_jacobian,analytical_jacobian, &
                               residual,residual_pert,perturbation, &
                               perturbation_tolerance,general_auxvar,option)

  use Option_module
  use Utility_module

  implicit none

  character(len=*) :: string
  type(general_auxvar_type) :: general_auxvar(0:)
  type(option_type) :: option
  PetscReal :: numerical_jacobian(option%nflowdof,option%nflowdof)
  PetscReal :: analytical_jacobian(option%nflowdof,option%nflowdof)
  PetscReal :: residual(option%nflowdof)
  PetscReal :: residual_pert(option%nflowdof,option%nflowdof)
  PetscReal :: perturbation(option%nflowdof)
  PetscReal :: perturbation_tolerance

  PetscInt :: irow, icol

100 format(2i2,2es13.5,x,a2,es16.8)

  if (len_trim(string) > 1) then
    write(*,'(x,a)') string
  endif
  write(*,'(" Perturbation tolerance: ",es12.4)') perturbation_tolerance
  write(*,'(" r c  numerical    analytical   digits of accuracy")')
  do icol = 1, 3
    do irow = 1, 3
      write(*,100) irow, icol, numerical_jacobian(irow,icol), &
                   analytical_jacobian(irow,icol), &
                   DigitsOfAccuracy(numerical_jacobian(irow,icol), &
                                    analytical_jacobian(irow,icol))
    enddo
  enddo

#if 0
200 format(2es20.12)
300 format(a24,10es20.12)
  do icol = 1, 3
    write(*,'(/," dof = ",i1,"  perturbation = ",es13.5)') icol, perturbation(icol)
!    write(*,300) 'density', general_auxvar(icol)%den(:), general_auxvar(0)%den(:)
!    write(*,300) 'energy', general_auxvar(icol)%U(:), general_auxvar(0)%U(:)
    write(*,'("  residual_pert       residual")')
    do irow = 1, 3
      write(*,200) residual_pert(irow,icol), residual(irow)
    enddo
  enddo
#endif

end subroutine GeneralDiffJacobian

! ************************************************************************** !

subroutine GeneralNonDarcyCorrection(delta_pressure,density_kg_ave,&
                                     perm_ave_over_dist,kr,mobility, &
                                     density_ave,area,v_darcy, &
                                     tot_mole_flux_ddel_pressure,q,tot_mole_flux)

  use Utility_module, only : expm1

  implicit none

  PetscReal, intent(in) :: delta_pressure, density_kg_ave, density_ave
  PetscReal, intent(in) :: mobility, kr, perm_ave_over_dist, area

  PetscReal, intent(out) :: v_darcy,tot_mole_flux_ddel_pressure,q,tot_mole_flux

  PetscReal :: K, I, grad, rhog

  ! based on Liu (2014) "Non-Darcian flow in low-permeability media"
  ! DOI 10.1007/s10040-014-1145-x
  rhog = density_kg_ave*EARTH_GRAVITY
  grad = abs(delta_pressure)/rhog
  ! eqn 11 in Liu (2014), in terms of permeability
  I = non_darcy_A*(perm_ave_over_dist*kr)**(non_darcy_B)
  K = perm_ave_over_dist*mobility*rhog
  ! expm1() to reduce cancelation
  ! v_darcy(iphase) = K*(grad-I*(1.d0-exp(-grad/I)))
  ! eqn 3 in Liu (2014)
  v_darcy = sign(K*(grad + I*expm1(-grad/I)), delta_pressure)

  ! q[m^3 phase/sec] = v_darcy[m/sec] * area[m^2]
  q = v_darcy * area
  ! mole_flux[kmol phase/sec] = q[m^3 phase/sec] *
  !                             density_ave[kmol phase/m^3 phase]
  tot_mole_flux = q*density_ave

  tot_mole_flux_ddel_pressure = area*density_ave*K* &
       grad/rhog*(-expm1(-grad/I))

end subroutine GeneralNonDarcyCorrection

! ************************************************************************** !

subroutine GeneralNonDarcyCorrectionBC(delta_pressure,density_kg_ave, &
                                       perm_ave_over_dist,kr,mobility,v_darcy, &
                                       dv_darcy_ddelta_pressure,dv_darcy_dmobility)

  use Utility_module, only : expm1

  implicit none

  PetscReal, intent(in) :: delta_pressure, density_kg_ave
  PetscReal, intent(in) :: mobility, kr, perm_ave_over_dist

  PetscReal, intent(out) :: v_darcy,dv_darcy_ddelta_pressure,dv_darcy_dmobility

  PetscReal :: K, I, grad, rhog

  ! based on Liu (2014) "Non-Darcian flow in low-permeability media"
  ! DOI 10.1007/s10040-014-1145-x
  rhog = density_kg_ave*EARTH_GRAVITY
  grad = abs(delta_pressure)/rhog
  ! eqn 11 in Liu (2014)
  I = non_darcy_A*(perm_ave_over_dist*kr)**(non_darcy_B)
  K = perm_ave_over_dist*mobility*rhog
  ! expm1() to avoid cancelation
  ! v_darcy(iphase) = K*(grad-I*(1.d0-exp(-(grad)/I)))
  ! eqn 3 in Liu (2014)
  v_darcy = sign(K*(grad + I*expm1(-grad/I)), delta_pressure)

  dv_darcy_ddelta_pressure = K/rhog*grad*(-expm1(-grad/I))
  ! expm1() to avoid cancelation
  ! dv_darcy_dmobility = perm_ave_over_dist*density_kg_ave* &
  !  EARTH_GRAVITY*(grad-I*(1.d0-exp(-grad/I)))
  dv_darcy_dmobility = perm_ave_over_dist*rhog* &
       (grad + I*(expm1(-grad/I)))

end subroutine GeneralNonDarcyCorrectionBC

end module General_Common_module
