module PM_TOWG_class

#include "petsc/finclude/petscsnes.h"
  use petscsnes
  use PM_Base_class
  use PM_Subsurface_Flow_class
  
  use PFLOTRAN_Constants_module

  implicit none

  private

  type, public, extends(pm_subsurface_flow_type) :: pm_towg_type
    !this two vectors could be moved to pm_subsurface_flow_type
    PetscInt, pointer :: max_change_ivar(:)
    PetscInt, pointer :: max_change_isubvar(:)
    ! A) and B) could be moved to pm_subsurface
    !A) used for truncation within CheckUpdatePre
    PetscReal :: trunc_max_pressure_change = 5.5d6
    PetscInt ::  max_it_before_damping = UNINITIALIZED_INTEGER
    PetscReal :: damping_factor = 0.6d0
    !B) used for convergence criteria within CheckUpdatePost
    PetscReal :: itol_rel_update = UNINITIALIZED_DOUBLE
    PetscReal :: itol_scaled_res = 1.d-5
    !check might need different dimension 
    PetscReal :: tgh2_itol_scld_res_e1(3,3) = 1.d-5
    PetscReal :: tgh2_itol_scld_res_e2 = 1.d0
    PetscBool :: tough2_conv_criteria = PETSC_FALSE
    !PetscInt :: miscibility_model = UNINITIALIZED_INTEGER 
    !procedure(TOWGMaxChangeDummy), pointer :: MaxChange => null()
    !procedure(MaxChange), pointer :: MaxChange => null()
  contains
    procedure, public :: ReadSimulationOptionsBlock => PMTOWGReadSimOptionsBlock
    procedure, public :: ReadTSBlock => PMTOWGReadTSSelectCase
    procedure, public :: ReadNewtonBlock => PMTOWGReadNewtonSelectCase
    procedure, public :: InitializeSolver => PMTOWGInitializeSolver
    procedure, public :: InitializeRun => PMTOWGInitializeRun
    procedure, public :: InitializeTimestep => PMTOWGInitializeTimestep
    procedure, public :: Residual => PMTOWGResidual
    procedure, public :: Jacobian => PMTOWGJacobian
    procedure, public :: UpdateTimestep => PMTOWGUpdateTimestep
    procedure, public :: PreSolve => PMTOWGPreSolve
    !procedure, public :: PostSolve => PMTOWGPostSolve
    procedure, public :: CheckUpdatePre => PMTOWGCheckUpdatePre
    !procedure, public :: CheckUpdatePost => PMTOWGCheckUpdatePost
    procedure, public :: TimeCut => PMTOWGTimeCut
    procedure, public :: UpdateSolution => PMTOWGUpdateSolution
    procedure, public :: UpdateAuxVars => PMTOWGUpdateAuxVars
    procedure, public :: MaxChange => PMTOWGMaxChange
    procedure, public :: ComputeMassBalance => PMTOWGComputeMassBalance
    !procedure, public :: InputRecord => PMTOWGInputRecord
    procedure, public :: CheckpointBinary => PMTOWGCheckpointBinary
    procedure, public :: RestartBinary => PMTOWGRestartBinary
    procedure, public :: Destroy => PMTOWGDestroy
  end type pm_towg_type
  
  public :: PMTOWGCreate
  
contains

! ************************************************************************** !
function PMTOWGCreate(input,miscibility_model,option)
  ! 
  ! Creates TOWG process models shell
  ! 
  ! Author: Paolo Orsini
  ! Date: 10/15/16
  ! 
  use Option_module
  use Input_Aux_module
  use PM_TOWG_Aux_module
  use Variables_module, only : OIL_PRESSURE, GAS_PRESSURE, &
                               OIL_SATURATION, GAS_SATURATION, &
                               OIL_MOLE_FRACTION, GAS_MOLE_FRACTION, &  
                               TEMPERATURE, SOLVENT_SATURATION,BUBBLE_POINT
  implicit none
  
  class(pm_towg_type), pointer :: PMTOWGCreate

  type(input_type) :: input
  character(len=MAXWORDLENGTH) :: miscibility_model
  type(option_type), pointer :: option

  class(pm_towg_type), pointer :: towg_pm
  
#ifdef TOWG_DEBUG  
  print *, 'PMTOWGCreate()'
#endif  

  allocate(towg_pm)

  select case(trim(miscibility_model))
    case('TOWG_IMMISCIBLE','TODD_LONGSTAFF','TOWG_MISCIBLE')
      allocate(towg_pm%max_change_ivar(4))
      towg_pm%max_change_ivar = [OIL_PRESSURE, OIL_SATURATION, &
                                 GAS_SATURATION,TEMPERATURE]
      allocate(towg_pm%max_change_isubvar(4))
      towg_pm%max_change_isubvar = [0,0,0,0]
    case('BLACK_OIL')
      allocate(towg_pm%max_change_ivar(5))
      towg_pm%max_change_ivar = [OIL_PRESSURE, OIL_SATURATION, &
                                 GAS_SATURATION,TEMPERATURE,BUBBLE_POINT]
      allocate(towg_pm%max_change_isubvar(5))
      towg_pm%max_change_isubvar = [0,0,0,0,0]
    case('SOLVENT_TL')
      allocate(towg_pm%max_change_ivar(6))
      towg_pm%max_change_ivar = [OIL_PRESSURE, OIL_SATURATION, &
                                 GAS_SATURATION,SOLVENT_SATURATION,TEMPERATURE,BUBBLE_POINT]
      allocate(towg_pm%max_change_isubvar(6))
      towg_pm%max_change_isubvar = [0,0,0,0,0,0]
    case default
      call InputKeywordUnrecognized(input,miscibility_model, &
                         'TOWG MISCIBILITY_MODEL',option)
  end select

  !towg_pm%trunc_max_pressure_change = 5.d4
  towg_pm%trunc_max_pressure_change = 5.d10
  towg_pm%max_it_before_damping = UNINITIALIZED_INTEGER
  towg_pm%damping_factor = 0.6d0
  towg_pm%itol_rel_update = UNINITIALIZED_DOUBLE
  towg_pm%itol_scaled_res = 1.d-5
  towg_pm%tgh2_itol_scld_res_e1(3,3) = 1.d-5
  towg_pm%tgh2_itol_scld_res_e2 = 1.d0
  towg_pm%tough2_conv_criteria = PETSC_FALSE

  select case(trim(miscibility_model)) 
    case('TOWG_IMMISCIBLE')
      towg_miscibility_model = TOWG_IMMISCIBLE
      towg_energy_dof = TOWG_3CMPS_ENERGY_DOF
      towg_energy_eq_idx = TOWG_3CMPS_ENERGY_EQ_IDX
      option%iflow_sub_mode = TOWG_IMMISCIBLE
    case('TODD_LONGSTAFF','TOWG_MISCIBLE')
      towg_miscibility_model = TOWG_TODD_LONGSTAFF
      towg_energy_dof = TOWG_3CMPS_ENERGY_DOF
      towg_energy_eq_idx = TOWG_3CMPS_ENERGY_EQ_IDX
      option%iflow_sub_mode = TOWG_TODD_LONGSTAFF
    case('BLACK_OIL')
      towg_miscibility_model = TOWG_BLACK_OIL
      towg_energy_dof = TOWG_3CMPS_ENERGY_DOF
      towg_energy_eq_idx = TOWG_3CMPS_ENERGY_EQ_IDX
      option%iflow_sub_mode = TOWG_BLACK_OIL
    case('SOLVENT_TL')
      towg_miscibility_model = TOWG_SOLVENT_TL
      towg_energy_dof = TOWG_SOLV_TL_ENERGY_DOF
      towg_energy_eq_idx = TOWG_SOLV_TL_ENERGY_EQ_IDX
      option%iflow_sub_mode = TOWG_SOLVENT_TL
    case default
      call InputKeywordUnrecognized(input,miscibility_model, &
                         'TOWG MISCIBILITY_MODEL',option)
  end select 


  !towg_pm%miscibility_model = UNINITIALIZED_INTEGER

  if (Uninitialized(towg_energy_dof)) then 
    option%io_buffer = 'towg_energy_dof not set up'
    call PrintErrMsg(option)
  end if  

  if (Uninitialized(towg_energy_eq_idx)) then 
    option%io_buffer = 'towg_energy_eq_idx not set up'
    call PrintErrMsg(option)
  end if  


  call PMSubsurfaceFlowInit(towg_pm)
  towg_pm%name = 'TOWG Flow'
  towg_pm%header = 'TOWG FLOW'

  PMTOWGCreate => towg_pm
  
end function PMTOWGCreate

! ************************************************************************** !

subroutine PMTOWGReadSimOptionsBlock(this,input)
  ! 
  ! Read TOWG options and set up miscibility, and functions to be use in 
  ! the Residual and Jacobian 
  ! 
  ! Author: Paolo Orsini (OGS)
  ! Date: 10/15/16
  !
  !use TOWG_module !to set up the functions to be used in R and J
  use PM_TOWG_Aux_module
  use Input_Aux_module
  use String_module
  use Option_module

  implicit none
  
  type(input_type), pointer :: input
  
  character(len=MAXWORDLENGTH) :: keyword 
  class(pm_towg_type) :: this
  type(option_type), pointer :: option
  PetscReal :: tempreal
  character(len=MAXSTRINGLENGTH) :: error_string
  PetscBool :: found

  option => this%option

  error_string = 'TOWG Options'

  !this%miscibility_model = UNINITIALIZED_INTEGER
  
  input%ierr = 0
  call InputPushBlock(input,option)
  do
  
    call InputReadPflotranString(input,option)

    if (InputCheckExit(input,option)) exit  

    call InputReadCard(input,option,keyword)
    call InputErrorMsg(input,option,'keyword',error_string)
    call StringToUpper(keyword)
    
    found = PETSC_FALSE
    call PMSubsurfFlowReadSimOptionsSC(this,input,keyword,found, &
                                       error_string,option)    
    if (found) cycle
    
    select case(trim(keyword))
      case('TL_OMEGA')
        call InputReadDouble(input,option,val_tl_omega)
        call InputDefaultMsg(input,option,'towg tl_omega')
      case('FMIS' )
        call FMISOWGRead(input,option)
      case('ISOTHERMAL')
        towg_isothermal = PETSC_TRUE
      case('NO_OIL')
        towg_no_oil = PETSC_TRUE
      case('NO_GAS')
        towg_no_gas = PETSC_TRUE
      case('DEBUG_CELL')
        call InputReadInt(input,option,towg_debug_cell_id)
        call InputErrorMsg(input,option,'debug cell id',error_string)
      case('TL4P_ALTERNATIVE_DENSITY')
        TL4P_altDensity = PETSC_TRUE
      case('TL4P_NONNEGATIVE_SLVSAT')
        TL4P_slv_sat_truncate = PETSC_TRUE
      case('TL4P_MOBILITY_SAFE')
        TL4P_safemobs = PETSC_TRUE
      case('WINDOW_EPSILON') 
        call InputReadDouble(input,option,towg_window_epsilon)
        call InputErrorMsg(input,option,keyword,error_string)

      !case('DIFFUSE_XMASS')
      !  general_diffuse_xmol = PETSC_FALSE
      !case('HARMONIC_GAS_DIFFUSIVE_DENSITY')
      !  general_harmonic_diff_density = PETSC_TRUE
      !case('ARITHMETIC_GAS_DIFFUSIVE_DENSITY')
      !  general_harmonic_diff_density = PETSC_FALSE
      !case('ANALYTICAL_DERIVATIVES')
      !  general_analytical_derivatives = PETSC_TRUE
      case default
        call InputKeywordUnrecognized(input,keyword,'TOWG Mode',option)
    end select
    
  enddo  
  call InputPopBlock(input,option)

  !if (Uninitialized(towg_miscibility_model)) then 
  !  option%io_buffer = 'TOWG MISCIBILITY_MODEL not set up'
  !  call PrintErrMsg(option)
  !end if  

  !here set up functions in TOWG and pm_TOWG_aux based on miscibility model 
  !select case(towg_miscibility_model)
  !  case(TOWG_IMMISCIBLE) 
  !    !set up towg and pm_towg_aux functions   
  !  case default
  !    option%io_buffer = 'only immiscible TOWG currently implemented' 
  !    call PrintErrMsg(option)
  !end select

end subroutine PMTOWGReadSimOptionsBlock

! ************************************************************************** !

subroutine PMTOWGReadTSSelectCase(this,input,keyword,found, &
                                  error_string,option)
  ! 
  ! Read timestepper settings specific to the TOWG process model
  ! 
  ! Author: Glenn Hammond
  ! Date: 04/06/20
  
  use Input_Aux_module
  use String_module
  use Option_module
  
  implicit none

  class(pm_towg_type) :: this
  type(input_type), pointer :: input
  character(len=MAXWORDLENGTH) :: keyword
  PetscBool :: found
  character(len=MAXSTRINGLENGTH) :: error_string
  type(option_type), pointer :: option

  found = PETSC_TRUE
  call PMSubsurfaceFlowReadTSSelectCase(this,input,keyword,found, &
                                        error_string,option)
  if (found) return

  found = PETSC_TRUE
  select case(trim(keyword))
    case default
      found = PETSC_FALSE
  end select

end subroutine PMTOWGReadTSSelectCase

! ************************************************************************** !

subroutine PMTOWGReadNewtonSelectCase(this,input,keyword,found, &
                                      error_string,option)
  ! 
  ! Reads input file parameters associated with the TOWG process model
  ! Newton solver convergence
  ! 
  ! Author: Glenn Hammond
  ! Date: 03/23/20

  use Input_Aux_module
  use String_module
  use Utility_module
  use Option_module
  use PM_TOWG_Aux_module
 
  implicit none
  
  class(pm_towg_type) :: this
  type(input_type), pointer :: input
  character(len=MAXWORDLENGTH) :: keyword
  character(len=MAXSTRINGLENGTH) :: error_string
  type(option_type), pointer :: option

  PetscBool :: found
  PetscReal :: tempreal

  option => this%option

  error_string = 'TOWG Newton Solver'
  
  found = PETSC_FALSE
  call PMSubsurfaceFlowReadNewtonSelectCase(this,input,keyword,found, &
                                            error_string,option)
  if (found) return
    
  found = PETSC_TRUE
  select case(trim(keyword))
    case('ITOL_SCALED_RESIDUAL')
      call InputReadDouble(input,option,this%itol_scaled_res)
      call InputErrorMsg(input,option,keyword,error_string)
      this%check_post_convergence = PETSC_TRUE
    case('ITOL_RELATIVE_UPDATE')
      call InputReadDouble(input,option,this%itol_rel_update)
      call InputErrorMsg(input,option,keyword,error_string)
      this%check_post_convergence = PETSC_TRUE        
    case('TOUGH2_ITOL_SCALED_RESIDUAL')
      ! tgh2_itol_scld_res_e1 is an array: assign same value to all entries
      tempreal = UNINITIALIZED_DOUBLE
      call InputReadDouble(input,option,tempreal)
      ! tempreal will remain uninitialized if the read fails.
      call InputErrorMsg(input,option,'TOUGH2_ITOL_SCALED_RESIDUAL_E1', &
                         error_string)
      if (Initialized(tempreal)) then
        this%tgh2_itol_scld_res_e1 = tempreal
      endif
      call InputReadDouble(input,option,this%tgh2_itol_scld_res_e2)
      call InputErrorMsg(input,option,'TOUGH2_ITOL_SCALED_RESIDUAL_E2', &
                         error_string)
      this%tough2_conv_criteria = PETSC_TRUE
      this%check_post_convergence = PETSC_TRUE
    case('T2_ITOL_SCALED_RESIDUAL_TEMP')
      call InputReadDouble(input,option,tempreal)
      call InputErrorMsg(input,option,keyword,error_string)
      this%tgh2_itol_scld_res_e1(3,:) = tempreal
    case('MAXIMUM_PRESSURE_CHANGE')
      call InputReadDouble(input,option,this%trunc_max_pressure_change)
      call InputErrorMsg(input,option,keyword,error_string)
    case('MAX_ITERATION_BEFORE_DAMPING')
      call InputReadInt(input,option,this%max_it_before_damping)
      call InputErrorMsg(input,option,keyword,error_string)
    case('DAMPING_FACTOR')
      call InputReadDouble(input,option,this%damping_factor)
      call InputErrorMsg(input,option,keyword,error_string)
    case default
      found = PETSC_FALSE

  end select
  
end subroutine PMTOWGReadNewtonSelectCase

! ************************************************************************** !

subroutine PMTOWGInitializeSolver(this)
  !
  ! Author: Glenn Hammond
  ! Date: 04/06/20

  use Solver_module

  implicit none

  class(pm_towg_type) :: this

  call PMBaseInitializeSolver(this)

  ! helps accommodate rise in residual due to change in state
  this%solver%newton_dtol = 1.d20

end subroutine PMTOWGInitializeSolver

! ************************************************************************** !

recursive subroutine PMTOWGInitializeRun(this)
  ! 
  ! Initializes the time stepping
  ! 
  ! Author: Paolo Orsini
  ! Date: 12/06/16 

  use Realization_Base_class
  
  implicit none
  
  class(pm_towg_type) :: this
  
  PetscInt :: num_var,i
  PetscErrorCode :: ierr

  num_var = size(this%max_change_ivar(:))

  ! need to allocate vectors for max change
  call VecDuplicateVecsF90(this%realization%field%work,num_var, &
                           this%realization%field%max_change_vecs, &
                           ierr);CHKERRQ(ierr)
  ! set initial values
  do i = 1, num_var
    call RealizationGetVariable(this%realization, &
                                this%realization%field%max_change_vecs(i), &
                                this%max_change_ivar(i), &
                                this%max_change_isubvar(i))
  enddo

  ! call parent implementation
  call PMSubsurfaceFlowInitializeRun(this)

end subroutine PMTOWGInitializeRun

! ************************************************************************** !

subroutine PMTOWGInitializeTimestep(this)
  ! 
  ! To be replaced by PreSolve? Which is currently empty...
  ! 
  ! Author: Paolo Orsini
  ! Date: 12/07/16 
  ! 

  use TOWG_module, only : TOWGInitializeTimestep
  use Global_module
  use Variables_module, only : TORTUOSITY
  use Material_module, only : MaterialAuxVarCommunicate
  
  implicit none
  
  class(pm_towg_type) :: this

  call PMSubsurfaceFlowInitializeTimestepA(this)                                 

  !PO: To be removed? And common to all flow modes?
!geh:remove   everywhere                                
  call MaterialAuxVarCommunicate(this%comm1, &
                                 this%realization%patch%aux%Material, &
                                 this%realization%field%work_loc,TORTUOSITY, &
                                 ZERO_INTEGER)
                                 
  call TOWGInitializeTimestep(this%realization)
  call PMSubsurfaceFlowInitializeTimestepB(this)                                 
  
end subroutine PMTOWGInitializeTimestep

! ************************************************************************** !

subroutine PMTOWGUpdateAuxVars(this)
  ! 
  ! Author: Paolo Orsini
  ! Date: 12/06/16 

  use TOWG_module, only : TOWGUpdateAuxVars

  implicit none
  
  class(pm_towg_type) :: this

  call TOWGUpdateAuxVars(this%realization,PETSC_FALSE)

end subroutine PMTOWGUpdateAuxVars   

! ************************************************************************** !

subroutine PMTOWGResidual(this,snes,xx,r,ierr)
  ! 
  ! Author: Paolo Orsini
  ! Date: 12/28/16
  ! 

  use TOWG_module, only : TOWGResidual

  implicit none
  
  class(pm_towg_type) :: this
  SNES :: snes
  Vec :: xx
  Vec :: r
  PetscErrorCode :: ierr
  
  call PMSubsurfaceFlowUpdatePropertiesNI(this)
  call TOWGResidual(snes,xx,r,this%realization,ierr)

end subroutine PMTOWGResidual

! ************************************************************************** !

subroutine PMTOWGJacobian(this,snes,xx,A,B,ierr)
  ! 
  ! Author: Paolo Orsini
  ! Date: 12/28/16
  ! 

  use TOWG_module, only : TOWGJacobian 

  implicit none
  
  class(pm_towg_type) :: this
  SNES :: snes
  Vec :: xx
  Mat :: A, B
  PetscErrorCode :: ierr
  
  call TOWGJacobian(snes,xx,A,B,this%realization,ierr)

end subroutine PMTOWGJacobian

! ************************************************************************** !

subroutine PMTOWGPreSolve(this)
  ! 
  ! Author: Paolo Orsini
  ! Date: 12/28/16

  implicit none

  class(pm_towg_type) :: this

  call PMSubsurfaceFlowPreSolve(this)

end subroutine PMTOWGPreSolve

! ************************************************************************** !

subroutine PMTOWGCheckUpdatePre(this,snes,X,dX,changed,ierr)
  ! 
  ! Author: Paolo Orsini (OGS)
  ! Date: 10/22/15
  ! 

  use TOWG_module, only : TOWGCheckUpdatePre

  implicit none
  
  class(pm_towg_type) :: this
  SNES :: snes
  Vec :: X
  Vec :: dX
  PetscBool :: changed
  PetscErrorCode :: ierr

  call TOWGCheckUpdatePre(snes,X,dX,changed,this%realization, &
                          this%max_it_before_damping,this%damping_factor, &
                          this%trunc_max_pressure_change,ierr)

end subroutine PMTOWGCheckUpdatePre

! ************************************************************************** !


subroutine PMTOWGUpdateSolution(this)
  ! 
  ! Author: Paolo Orsini (OGS)
  ! Date: 12/06/16 
  ! 

  use TOWG_module, only : TOWGUpdateSolution, &
                          TOWGMapBCAuxVarsToGlobal

  implicit none
  
  class(pm_towg_type) :: this
  
  call PMSubsurfaceFlowUpdateSolution(this)
  call TOWGUpdateSolution(this%realization)
  call TOWGMapBCAuxVarsToGlobal(this%realization)

end subroutine PMTOWGUpdateSolution     

! ************************************************************************** !

subroutine PMTOWGTimeCut(this)
  ! 
  ! Author: Glenn Hammond
  ! Date: 03/14/13
  ! 

  use TOWG_module, only : TOWGTimeCut

  implicit none
  
  class(pm_towg_type) :: this
  
  call PMSubsurfaceFlowTimeCut(this)
  call TOWGTimeCut(this%realization)

end subroutine PMTOWGTimeCut

! ************************************************************************** !

subroutine PMTOWGMaxChange(this)
  ! 
  ! Compute primary variable max changes
  ! 
  ! Author: Paolo Orsini
  ! Date: 12/30/16
  ! 
  use TOWG_module, only : TOWGMaxChange

  implicit none
  
  class(pm_towg_type) :: this

  call TOWGMaxChange(this%realization, this%max_change_ivar, &
                     this%max_change_isubvar, this%max_pressure_change, &
                     this%max_xmol_change,this%max_saturation_change, &
                     this%max_temperature_change)

end subroutine PMTOWGMaxChange

! ************************************************************************** !

subroutine PMTOWGUpdateTimestep(this,dt,dt_min,dt_max,iacceleration, &
                                num_newton_iterations,tfac, &
                                time_step_max_growth_factor)
  ! 
  ! Author: Paolo Orsini
  ! Date: 12/30/16
  ! 

  use Realization_Base_class, only : RealizationGetVariable
  use Realization_Subsurface_class, only : RealizationLimitDTByCFL
  use Field_module
  use Variables_module, only : LIQUID_SATURATION, GAS_SATURATION

  implicit none
  
  class(pm_towg_type) :: this
  PetscReal :: dt
  PetscReal :: dt_min,dt_max
  PetscInt :: iacceleration
  PetscInt :: num_newton_iterations
  PetscReal :: tfac(:)
  PetscReal :: time_step_max_growth_factor
  
  PetscReal :: fac
  PetscInt :: ifac
  PetscReal :: up, ut, ux, us, umin
  PetscReal :: dtt
  type(field_type), pointer :: field
  
#ifdef PM_TOWG_DEBUG  
  call PrintMsg(this%option,'PMTOWG%UpdateTimestep()')
#endif
  
  fac = 0.5d0
  if (num_newton_iterations >= iacceleration) then
    fac = 0.33d0
    umin = 0.d0
  else
    up = this%pressure_change_governor/(this%max_pressure_change+0.1)
    ut = this%temperature_change_governor/(this%max_temperature_change+1.d-5)
    ux = this%xmol_change_governor/(this%max_xmol_change+1.d-5)
    us = this%saturation_change_governor/(this%max_saturation_change+1.d-5)
    umin = min(up,ut,ux,us)
  endif
  ifac = max(min(num_newton_iterations,size(tfac)),1)
  dtt = fac * dt * (1.d0 + umin)
  dtt = min(time_step_max_growth_factor*dt,dtt)
  dt = min(dtt,tfac(ifac)*dt,dt_max)
  dt = max(dt,dt_min)

  !if (Initialized(this%cfl_governor)) then
  !  ! Since saturations are not stored in global_auxvar for general mode, we
  !  ! must copy them over for the CFL check
  !  ! liquid saturation
  !  field => this%realization%field
  !  call RealizationGetVariable(this%realization,field%work, &
  !                              LIQUID_SATURATION,ZERO_INTEGER)
  !  call this%realization%comm1%GlobalToLocal(field%work,field%work_loc)
  !  call GlobalSetAuxVarVecLoc(this%realization,field%work_loc, &
  !                             LIQUID_SATURATION,TIME_NULL)
  !  call RealizationGetVariable(this%realization,field%work, &
  !                              GAS_SATURATION,ZERO_INTEGER)
  !  call this%realization%comm1%GlobalToLocal(field%work,field%work_loc)
  !  call GlobalSetAuxVarVecLoc(this%realization,field%work_loc, &
  !                             GAS_SATURATION,TIME_NULL)
  !  call RealizationLimitDTByCFL(this%realization,this%cfl_governor,dt,dt_max)
  !endif

end subroutine PMTOWGUpdateTimestep

! ************************************************************************** !

subroutine PMTOWGComputeMassBalance(this,mass_balance_array)
  ! 
  ! Author: Paolo Orsini (OGS)
  ! Date: 12/30/16
  ! 

  use TOWG_module, only : TOWGComputeMassBalance

  implicit none
  
  class(pm_towg_type) :: this
  PetscReal :: mass_balance_array(:)
  
  call TOWGComputeMassBalance(this%realization,mass_balance_array)

end subroutine PMTOWGComputeMassBalance

! ************************************************************************** !

subroutine PMTOWGCheckpointBinary(this,viewer)
  ! 
  ! Checkpoints data associated with TOWG PM
  ! If saves istate for all TOWG submodels, but required only for black oil
  ! and solvent models
  ! 
  ! Author: Paolo Orsini
  ! Date: 01/19/17

  use Checkpoint_module
  use Global_module
  use Variables_module, only : STATE

  implicit none
#include "petsc/finclude/petscviewer.h"      

  class(pm_towg_type) :: this
  PetscViewer :: viewer
  
  call PMSubsurfaceFlowCheckpointBinary(this,viewer)
  
end subroutine PMTOWGCheckpointBinary

! ************************************************************************** !

subroutine PMTOWGRestartBinary(this,viewer)
  ! 
  ! Restarts data associated with TOWG PM
  ! If loads istate for all TOWG submodels, but required only for black oil
  ! and solvent models
  !
  ! Author: Paolo Orsini
  ! Date: 01/19/17

  use Checkpoint_module
  use Global_module
  use Variables_module, only : STATE

  implicit none
#include "petsc/finclude/petscviewer.h"      

  class(pm_towg_type) :: this
  PetscViewer :: viewer
  
  call PMSubsurfaceFlowRestartBinary(this,viewer)
  
end subroutine PMTOWGRestartBinary

! ************************************************************************** !

subroutine PMTOWGDestroy(this)
  ! 
  ! Destroys TOWG process model
  ! 
  ! Author: Paolo Orsini
  ! Date: 01/19/17
  ! 

  use TOWG_module, only : TOWGDestroy

  implicit none
  
  class(pm_towg_type) :: this
  
  if (associated(this%next)) then
    call this%next%Destroy()
  endif

  deallocate(this%max_change_ivar)
  nullify(this%max_change_ivar)
  deallocate(this%max_change_isubvar)
  nullify(this%max_change_isubvar)

  ! preserve this ordering
  call TOWGDestroy(this%realization)
  call PMSubsurfaceFlowDestroy(this)
  
end subroutine PMTOWGDestroy

subroutine FMISOWGRead(input,option)

!------------------------------------------------------------------------------
! Read in the parameters describing required fs->fm conversion in TL4P
!------------------------------------------------------------------------------
! Author: Dave Ponting
! Date  : May 2018
!------------------------------------------------------------------------------

  use Option_module
  use Input_Aux_module
  use PM_TOWG_Aux_module

  implicit none

  type(input_type), pointer :: input
  type(option_type) :: option

  PetscReal:: fmis_av
  PetscReal,parameter:: eps=0.001

  character(len=MAXSTRINGLENGTH) :: error_string='FMIS'

!--Basic read operations-------------------------------------------------------

  call InputReadDouble(input,option,fmis_sl)
  call InputErrorMsg(input,option,'Lower saturation point',error_string)
  call InputReadDouble(input,option,fmis_su)
  call InputErrorMsg(input,option,'Upper saturation point',error_string)

!--Order, separation and range checks------------------------------------------

  if(      fmis_sl==1.0d0 ) then
    fmis_is_zero =.true.
  else if( fmis_su==0.0d0 ) then
    fmis_is_unity=.true.
  else
    if( fmis_su<=fmis_sl ) then
      option%io_buffer = 'Second FMIS value must exceed first'
      call PrintErrMsg(option)
      fmis_av=0.5*(fmis_sl+fmis_su)
      fmis_sl=fmis_av-0.5*eps
      fmis_su=fmis_av+0.5*eps
    endif
  endif

  fmis_sl=max(fmis_sl,0.0d0)
  fmis_su=min(fmis_su,1.0d0)

end subroutine FMISOWGRead

! ************************************************************************** !

end module PM_TOWG_class
