import sys
import os
from collections import defaultdict
import time
from subprocess import run, PIPE, DEVNULL

if run(f"valgrind -h", shell=True).returncode != 0:
    print("Please install Valgrind. Exiting.")
    sys.exit(1)

def run_time(cmd):
    print(f"~~~~~~~~~~~~~~~~~~~~~ Timing '{cmd}'")
    start = time.time()
    run(cmd, shell=True)
    end = time.time()
    return end - start

def run_mem(cmd):
    print(f"~~~~~~~~~~~~~~~~~~~~~ Massif'ing '{cmd}'")
    run(f"valgrind --tool=massif --massif-out-file=massif.out {cmd}", shell=True)
    peak = None
    for line in open("massif.out", "r"):
        if line.startswith("mem_heap_B="):
            peak = int(line.split('=')[1].strip())
        elif line.strip() == "heap_tree=peak":
            break
    os.remove("massif.out")
    assert peak is not None
    return peak

LUAS = ["unmodified", "syncc", "ett"]
BENCHMARKS = ["binarytrees.lua", "helloworld.lua", "merkletrees.lua",
              "nbody.lua", "spectralnormal.lua", "list.lua", "dbllist.lua"]

timing = defaultdict(dict)
meming = defaultdict(dict)
for benchmark in BENCHMARKS:
    for lua in LUAS:
        times = [run_time(f"../../bins/{lua} {benchmark}") for _ in range(5)]
        timing[benchmark][lua] = sum(times) / len(times)
        meming[benchmark][lua] = run_mem(f"../../bins/{lua} {benchmark}")

lines = []
lines.append(["Benchmark", "Time (s)", "Mem (B)"])
for _ in LUAS[1:]:
    lines[0].append("Time")
    lines[0].append("Mem")

for benchmark in BENCHMARKS:
    line = [benchmark.split('.')[0]]
    ut, um = timing[benchmark]["unmodified"], meming[benchmark]["unmodified"]
    for lua in LUAS:
        t, m = timing[benchmark][lua], meming[benchmark][lua]
        if lua == "unmodified":
            line.append(f"{t:<3.4f}")
            line.append(f"{m:<9.2f}")
        else:
            t = (t / ut)
            m = (m / um)
            line.append(f"${t:<2.4f}\\times$")
            line.append(f"${m:<2.4f}\\times$")
    lines.append(line)

F = open("report_files/table.tex", "w")
F.write("\\documentclass[acmsmall,screen,review,anonymous]{acmart}\n")
F.write("\\usepackage{booktabs}\n")
F.write("\\title{Pathological Cases for Garbage Collection --- Artifact}\n")
F.write("\\begin{document}\n")
F.write("\\begin{tabular}{lrrrrrr}\n")
F.write("\\toprule\n")
F.write("& \\multicolumn{2}{c}{Nonimmediate} & \\multicolumn{2}{c}{SynCC Overhead} & \\multicolumn{2}{c}{Our Overhead} \\\\\n")
F.write("\\cmidrule{2-3} \\cmidrule{4-5} \\cmidrule{6-7}\n")

for row in lines:
    rename = {"binarytrees":    "binarytrees-15",
              "helloworld":     "helloworld",
              "list":           "list-4096",
              "dbllist":        "dbllist-4096",
              "merkletrees":    "merkletrees-15",
              "nbody":          "nbody-100000",
              "spectralnormal": "specnorm-1000"}
    if row[0] not in rename: continue
    row[0] = f"\\texttt{{{rename[row[0]]}}}"

col_widths = [max([len(row[i]) for row in lines]) for i in range(len(lines[0]))]
for i, row in enumerate(lines):
    row = [cell.ljust(3+col_widths[i]) for i, cell in enumerate(row)]
    F.write(' & '.join(row) + " \\\\\n")
    if i == 0: F.write("\\midrule\n")
F.write("\\bottomrule\n")
F.write("\\end{tabular}\n")
F.write("\\end{document}\n")
