/***
 * Implementation of the synchronous cycle collection algorithm.
 * Reference: Figure 2 of
 * https://pages.cs.wisc.edu/~cymen/misc/interests/Bacon01Concurrent.pdf
 ***/
#pragma once

#include <assert.h>
#if 0
#undef assert
#define assert(...)
#endif
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>

extern unsigned long _GC_TOTAL_HEAP_COUNT;
extern unsigned long _GC_N_FREES;
extern unsigned long _GC_HEAP_SIZE;
extern unsigned long _GC_N_REGIONS;

#if 1
#define gcassert(...) if (!(__VA_ARGS__)) __builtin_unreachable()
#else
#include <assert.h>
#define gcassert assert
#endif

// The four colors from their pseudocode
enum GC_COLOR {GC_COLOR_BLACK, GC_COLOR_GRAY, GC_COLOR_WHITE, GC_COLOR_PURPLE};

// Represents an edge (pointer) in the heap graph.
struct _gc_link {
    struct _gc_descriptor *to;  // Destination of the edge
    struct _gc_link *next;      // Next edge from the same source
};

struct _gc_descriptor {
    // Number of incoming edges.
    unsigned refs;
    // The Lua object this refers to (used to call finalizers).
    void *gco;
    // A linked list of the outgoing edges (pointers) from this node.
    struct _gc_link *outgoing_links;
    // The color of this node as used by the SynCC algorithm.
    enum GC_COLOR color : 2;
    // The buffered flag as used by the SynCC algorithm.
    char buffered : 1;
    // Managing the roots list
    char is_root : 1;
    struct _gc_descriptor *next_root;
};

extern void luaC_delete(void *gco);
extern int _GC_IS_FINALIZING;
extern void luaC_finalize(void *);

// Prepends to the linked list of outgoing edges on @from. Also increments the
// count of incoming edges on @link->to.
static inline void _gc_insert_link(struct _gc_descriptor *from, struct _gc_link *link) {
    link->next = from->outgoing_links;
    from->outgoing_links = link;
    link->to->refs++;
}

// Removes an edge and decrements the count of incoming edges on its
// destination. The argument should be a pointer to the pointer to the edge in
// the list of outgoing edges on its source, e.g., something like &(pred->next)
// if @pred is the predecessor of the edge in the list.
static inline int _gc_remove_link(struct _gc_link **plink) {
    struct _gc_link *link = *plink;
    struct _gc_descriptor *to = link->to;
    *plink = link->next;
    free(link);
    return --(to->refs);
}

extern struct _gc_descriptor *_GC_FIRST_ROOT;
// Appends to the linked list of roots.
static inline void _gc_append_to_roots(struct _gc_descriptor *node) {
    if (node->is_root) return;
    node->is_root = 1;
    node->next_root = _GC_FIRST_ROOT;
    _GC_FIRST_ROOT = node;
}

// MarkGray implemented as in their Figure 2.
static inline void _gc_mark_gray(struct _gc_descriptor *node) {
  if (node->color == GC_COLOR_GRAY) return;
  node->color = GC_COLOR_GRAY;
  for (struct _gc_link *link = node->outgoing_links; link; link = link->next) {
    link->to->refs--;
    _gc_mark_gray(link->to);
  }
}

// ScanBlack implemented as in their Figure 2.
static inline void _gc_scan_black(struct _gc_descriptor *node) {
  node->color = GC_COLOR_BLACK;
  for (struct _gc_link *link = node->outgoing_links; link; link = link->next) {
    link->to->refs++;
    if (link->to->color != GC_COLOR_BLACK) {
      _gc_scan_black(link->to);
    }
  }
}

// Scan implemented as in their Figure 2.
static inline void _gc_scan(struct _gc_descriptor *node) {
  if (node->color != GC_COLOR_GRAY) return;
  if (node->refs > 0) {
    _gc_scan_black(node);
  } else {
    node->color = GC_COLOR_WHITE;
    for (struct _gc_link *link = node->outgoing_links; link; link = link->next) {
      _gc_scan(link->to);
    }
  }
}

// Implements PossibleRoot from the SynCC paper, Figure 2.
static inline void _gc_possible_root(struct _gc_descriptor *node) {
  if (node->color != GC_COLOR_PURPLE) {
    node->color = GC_COLOR_PURPLE;
    if (!node->buffered) {
        node->buffered = 1;
        _gc_append_to_roots(node);
    }
  }
}

// Decrement implemented as in their Figure 2.
static inline void _gc_release(struct _gc_descriptor *node);
static inline void _gc_decrement(struct _gc_descriptor *node) {
    node->refs--;
    if (!node->refs)    _gc_release(node);
    else                _gc_possible_root(node);
}

// Decrement, but assuming the caller has already done the refs--.
static inline void _gc_postdecrement(struct _gc_descriptor *node) {
    if (!node->refs)    _gc_release(node);
    else                _gc_possible_root(node);
}

// Implements "Release(S)" from the SynCC paper, Figure 2, modified to support
// finalization.
static inline void _gc_release(struct _gc_descriptor *node) {
  _GC_IS_FINALIZING++;
  if (!node->buffered) {
      luaC_finalize(node->gco);
  }
  while (node->outgoing_links) {
    struct _gc_descriptor *which = node->outgoing_links->to;
    _gc_remove_link(&(node->outgoing_links));
    _gc_postdecrement(which);
  }
  node->color = GC_COLOR_BLACK;
  if (!node->buffered) {
      luaC_delete(node->gco);
      free(node);
  }
  _GC_IS_FINALIZING--;
}

// Implements CollectWhite from the SynCC paper, Figure 2, modified to support
// finalization.
static inline void _gc_collect_white(struct _gc_descriptor *node) {
  if (node->color != GC_COLOR_WHITE)    return;
  if (node->buffered)                   return;
  _GC_IS_FINALIZING++;
  node->color = GC_COLOR_BLACK;
  luaC_finalize(node->gco);
  luaC_delete(node->gco);
  for (struct _gc_link *link = node->outgoing_links; link; link = link->next) {
    _gc_collect_white(link->to);
    // TODO: free the link before moving on.
  }
  free(node);
  _GC_IS_FINALIZING--;
}

// Implements MarkRoots from the SymCC paper, Figure 2.
static inline void _gc_mark_roots(void) {
    for (struct _gc_descriptor **pn = &_GC_FIRST_ROOT; *pn;) {
        struct _gc_descriptor *node = *pn;
        if (node->color == GC_COLOR_PURPLE) {
            _gc_mark_gray(node);
             pn = &((*pn)->next_root);
        } else {
            node->buffered = 0;
            node->is_root = 0;
            *pn = (*pn)->next_root;
            if (node->color == GC_COLOR_BLACK && node->refs == 0) {
                // TODO: finalization interaction with Release...
                luaC_delete(node->gco);
                free(node);
            }
        }
    }
}

// Implements ScanRoots from the SymCC paper, Figure 2.
static inline void _gc_scan_roots(void) {
    for (struct _gc_descriptor *node = _GC_FIRST_ROOT; node; node = node->next_root) {
        _gc_scan(node);
    }
}

// Implements CollectRoots from the SymCC paper, Figure 2.
static inline void _gc_collect_roots(void) {
    while (_GC_FIRST_ROOT) {
        struct _gc_descriptor *node = _GC_FIRST_ROOT;
        node->is_root = 0;
        _GC_FIRST_ROOT = node->next_root;
        node->buffered = 0;
        _gc_collect_white(node);
    }
}

// Implements CollectCycles from the SymCC paper, Figure 2.
static inline void _gc_collect_cycles(void) {
  // MARK ROOT
  _gc_mark_roots();
  // SCAN ROOT
  _gc_scan_roots();
  // COLLECT ROOT
  _gc_collect_roots();
}

//////////////////// GC MODULE
// Helper function that keeps a shadow copy of the heap graph and decides when
// to call the different SynCC methods.  delta=+1 means adding an edge,
// delta=-1 means removing an edge.
static inline void _gc_delta_link(struct _gc_descriptor *from,
                                  struct _gc_descriptor *to,
                                  int delta) {
    struct _gc_link **plink = 0, *link = 0;

    // Self loops and edges to nowhere can be ignored without affecting anything.
    if (from == to || !to) return;

    // We are representing edges from GCRoot implicitly, via a counter "refs,"
    // which gets updated here.
    if (!from) {
        to->refs += delta;
        if (delta < 0) {
            _gc_postdecrement(to);
            _gc_collect_cycles();
        }
        return;
    }

    // If we are removing an edge, find that edge in the outgoing list of
    // @from. (NOTE: this can take asymptotically bad time in a dense graph,
    // but that can be solved by using a hashmap or balanced tree to store the
    // edge lists in addition to a doubly linked list. We did not do so in this
    // implementation to keep it as small as possible.)
    plink = &(from->outgoing_links);
    link = *plink;
    if (delta < 0)
        for (; link && link->to != to;
             plink = &(link->next), link = *plink) ;
    else
        link = 0;

    // If we're adding a new edge, we need to allocate a link for it.
    if (delta > 0 && !link) {
        link = (struct _gc_link *)calloc(1, sizeof(*link));
        link->to = to;
        _gc_insert_link(from, link);
    } else if (!link) { gcassert(0); }

    // And then we're done, if we're adding a new edge, since new edges can
    // never make anything unreachable.
    if (delta > 0) return;

    // Otherwise, we're deleting an edge so call the relevant SynCC methods.
    _gc_remove_link(plink);
    _gc_postdecrement(to);
    _gc_collect_cycles();
}

// Used as the main interface to the GC to indicate writing to a pointer on
// source region @via, pointing to destination region @new_value, and
// overwriting an old pointer that was pointing to @old_value.
static inline void _gc_overwrite_pointer(
            struct _gc_descriptor *via,
            struct _gc_descriptor *old_value,
            struct _gc_descriptor *new_value) {
  if (_GC_IS_FINALIZING) return;
  if (new_value) _gc_delta_link(via, new_value, +1);
  if (old_value) _gc_delta_link(via, old_value, -1);
}

// Helper for _gc_new.
static inline struct _gc_descriptor *_gc_new_inplace(struct _gc_descriptor *d) {
    _GC_TOTAL_HEAP_COUNT++;
    return d;
}

// Equivalent of GCAllocate.
static inline struct _gc_descriptor *_gc_new(void) {
    struct _gc_descriptor *d = (struct _gc_descriptor *)calloc(1, sizeof(*d));
    return _gc_new_inplace(d);
}

// SynCC doesn't use terminality information.
static inline void _gc_set_terminal(struct _gc_descriptor *d) { }
