/*
** $Id: lgc.c $
** Garbage Collector
** See Copyright Notice in lua.h
*/

#define lgc_c
#define LUA_CORE

#include "lprefix.h"

#include <stdio.h>
#include <string.h>
#include <assert.h>


#include "lua.h"

#include "ldebug.h"
#include "ldo.h"
#include "lfunc.h"
#include "lgc.h"
#include "lmem.h"
#include "lobject.h"
#include "lstate.h"
#include "lstring.h"
#include "ltable.h"
#include "ltm.h"


/*
** Maximum number of elements to sweep in each single step.
** (Large enough to dissipate fixed overheads but small enough
** to allow small steps for the collector.)
*/
#define GCSWEEPMAX	100

/*
** Maximum number of finalizers to call in each single step.
*/
#define GCFINMAX	10


/*
** Cost of calling one finalizer.
*/
#define GCFINALIZECOST	50


/*
** The equivalent, in bytes, of one unit of "work" (visiting a slot,
** sweeping an object, etc.)
*/
#define WORK2MEM	sizeof(TValue)


/*
** macro to adjust 'pause': 'pause' is actually used like
** 'pause / PAUSEADJ' (value chosen by tests)
*/
#define PAUSEADJ		100


/* mask to erase all color bits (plus gen. related stuff) */
#define maskcolors	(~(bitmask(BLACKBIT) | WHITEBITS | AGEBITS))


/* macro to erase all color bits then sets only the current white bit */
#define makewhite(g,x)	\
 (x->marked = cast_byte((x->marked & maskcolors) | luaC_white(g)))

#define white2gray(x)	resetbits(x->marked, WHITEBITS)
#define black2gray(x)	resetbit(x->marked, BLACKBIT)


#define valiswhite(x)   (iscollectable(x) && iswhite(gcvalue(x)))

#define keyiswhite(n)   (keyiscollectable(n) && iswhite(gckey(n)))


#define checkconsistency(obj)  \
  lua_longassert(!iscollectable(obj) || righttt(obj))

/*
** Protected access to objects in values
*/
#define gcvalueN(o)     (iscollectable(o) ? gcvalue(o) : NULL)


#define markvalue(g,o) { checkconsistency(o); \
  if (valiswhite(o)) reallymarkobject(g,gcvalue(o)); }

#define markkey(g, n)	{ if keyiswhite(n) reallymarkobject(g,gckey(n)); }

#define markobject(g,t)	{ if (iswhite(t)) reallymarkobject(g, obj2gco(t)); }

/*
** mark an object that can be NULL (either because it is really optional,
** or it was stripped as debug info, or inside an uncompleted structure)
*/
#define markobjectN(g,t)	{ if (t) markobject(g,t); }

// void luaC_barrier_ (lua_State *L, GCObject *o, GCObject *v) { cast_void(L); cast_void(o); cast_void(v); }
// void luaC_barrierback_ (lua_State *L, GCObject *o) { cast_void(L); cast_void(o); }

void luaC_fix (lua_State *L, GCObject *o) {
  luaC_refinc(L, o);
}

/*
** create a new collectable object (with given type and size) and link
** it to 'allgc' list.
*/
static lua_State *LSTATE;
GCObject *luaC_newobj (lua_State *L, int tt, size_t sz) {
  LSTATE = L;
  global_State *g = G(L);
  GCObject *o = cast(GCObject *, luaM_newobject(L, novariant(tt), sz));
  o->marked = luaC_white(g);
  o->tt = tt;
  o->gcd = _gc_new();
  o->gcd->gco = o;
  g->allgc = o;
  return o;
}

static void freeupval (lua_State *L, UpVal *uv) {
  if (upisopen(uv))
    luaF_unlinkupval(uv);
  luaM_free(L, uv);
}

// from old lgc.c
static void dothecall (lua_State *L, void *ud) {
  UNUSED(ud);
  luaD_callnoyield(L, L->top - 2, 0);
}
static void _gc_do_finalizer(TValue *v) {
  const TValue *tm = 0;
  tm = luaT_gettmbyobj(LSTATE, v, TM_GC);
  if (!notm(tm)) {  /* is there a finalizer? */
      StkId old_top = LSTATE->top;
      LSTATE->top = LSTATE->ci->top;

    int status;
    lu_byte oldah = LSTATE->allowhook;
    // int oldgcstp  = g->gcstp;
    // g->gcstp |= GCSTPGC;  /* avoid GC steps */
    LSTATE->allowhook = 0;  /* stop debug hooks during GC metamethod */
    setobj2s(LSTATE, LSTATE->top++, tm);  /* push finalizer... */
    setobj2s(LSTATE, LSTATE->top++, v);  /* ... and its argument */
    // printf("Pushed %p\n", v->value_.gc);
    LSTATE->ci->callstatus |= CIST_FIN;  /* will run a finalizer */
    status = luaD_pcall(LSTATE, dothecall, NULL, savestack(LSTATE, LSTATE->top - 2), 0);
    LSTATE->ci->callstatus &= ~CIST_FIN;  /* not running a finalizer anymore */
    LSTATE->allowhook = oldah;  /* restore hooks */

      LSTATE->top = old_top;

    // g->gcstp = oldgcstp;  /* restore state */
    if (status != LUA_OK) {  /* error while running __gc? */
      luaE_warnerror(LSTATE, "__gc");
      LSTATE->top--;  /* pops error object */
    }
  }
}

void luaC_finalize(void *o_) {
  GCObject *o = (GCObject *)o_;
  TValue value = {0};
  value.tt_ = o->tt;
  if (value.tt_ == LUA_VTABLE) {
    value.tt_ |= BIT_ISCOLLECTABLE;
  }
  // printf("TT: %ld\n", value.tt_);
  value.value_.gc = o;
  // printf("Finalizing: %p\n", o);
  _gc_do_finalizer(&value);
}

// from the old lgc.c
void luaC_delete (void *o_) {
  GCObject *o = (GCObject *)o_;
  switch (o->tt) {
    case LUA_VPROTO:
      luaF_freeproto(LSTATE, gco2p(o));
      break;
    case LUA_VUPVAL:
      freeupval(LSTATE, gco2upv(o));
      break;
    case LUA_VLCL: {
      LClosure *cl = gco2lcl(o);
      luaM_freemem(LSTATE, cl, sizeLclosure(cl->nupvalues));
      break;
    }
    case LUA_VCCL: {
      CClosure *cl = gco2ccl(o);
      luaM_freemem(LSTATE, cl, sizeCclosure(cl->nupvalues));
      break;
    }
    case LUA_VTABLE:
      luaH_free(LSTATE, gco2t(o));
      break;
    case LUA_VTHREAD:
      luaE_freethread(LSTATE, gco2th(o));
      break;
    case LUA_VUSERDATA: {
      Udata *u = gco2u(o);
      luaM_freemem(LSTATE, o, sizeudata(u->nuvalue, u->len));
      break;
    }
    case LUA_VSHRSTR: {
      TString *ts = gco2ts(o);
      luaS_remove(LSTATE, ts);  /* remove it from hash table */
      luaM_freemem(LSTATE, ts, sizelstring(ts->shrlen));
      break;
    }
    case LUA_VLNGSTR: {
      TString *ts = gco2ts(o);
      luaM_freemem(LSTATE, ts, sizelstring(ts->u.lnglen));
      break;
    }
    default: lua_assert(0);
  }
}

/* }====================================================== */


void luaC_checkfinalizer (lua_State *L, GCObject *o, Table *mt) { cast_void(L); cast_void(o); cast_void(mt); }
void luaC_changemode (lua_State *L, int newmode) { cast_void(L); cast_void(newmode); }
void luaC_freeallobjects (lua_State *L) { cast_void(L); }
void luaC_runtilstate (lua_State *L, int statesmask) { cast_void(L); cast_void(statesmask); }
void luaC_step (lua_State *L) { cast_void(L); }
void luaC_fullgc (lua_State *L, int isemergency) { cast_void(L); cast_void(isemergency); }

/* }====================================================== */

unsigned long _GC_N_FREES;
unsigned long _GC_COUNT;
unsigned long _GC_HEAP_SIZE;
unsigned long _GC_TOTAL_HEAP_COUNT;
unsigned long _GC_N_REGIONS;
int _GC_IS_FINALIZING;
unsigned long _GC_D_GENERATION;
#include "gc/gc_globals.h"
