typedef long unsigned size_t;

struct splaynode {
    struct splaynode *left, *right, *parent;
    struct splaynode *next, *prev;
    char is_start : 1;
};

static struct splaynode *SCRATCH;
// Helper to get a pointer to the pointer in the parent that points to @node.
static struct splaynode **splay_pp(struct splaynode *node) {
    if (!(node->parent)) return &SCRATCH;
    gcassert(node->parent->left == node || node->parent->right == node);
    return (node->parent->left == node) ? &(node->parent->left)
                                        : &(node->parent->right);
}

// https://www.cs.cornell.edu/courses/cs3110/2013sp/recitations/rec08-splay/rec08.html
#define SET_CHILD(PAR, WHICH, CHI) \
  do { PAR->WHICH = CHI; if (CHI) CHI->parent = PAR; } while(0)

// Right rotation routine.
static struct splaynode *splay_rotate_r(struct splaynode * a) {
    struct splaynode **ap = splay_pp(a),
          *b = a->left,
          *d = b ? b->right : 0;
    b->parent = a->parent;  *ap = b;
    SET_CHILD(b, right, a);
    SET_CHILD(a, left, d);
    return b;
}

// Left rotation routine.
static struct splaynode *splay_rotate_l(struct splaynode *a) {
    struct splaynode **ap = splay_pp(a),
          *c = a->right,
          *d = c ? c->left : 0;
    c->parent = a->parent;  *ap = c;
    SET_CHILD(c, left, a);
    SET_CHILD(a, right, d);
    return c;
}

// Main splay routine.
static struct splaynode *splay_splay(struct splaynode *node) {
  while (node->parent) {
      if (!(node->parent->parent)) {
        // rotate around the parent first
        if (node->parent->left == node) {
            splay_rotate_r(node->parent);
        } else {
            splay_rotate_l(node->parent);
        }
      } else {
        // 0 -> left, 1 -> right
        int side = (node->parent->right == node);
        int parent_side = (node->parent->parent->right == node->parent);
        if (side == 0 && parent_side == 0) {
          splay_rotate_r(node->parent->parent);
          splay_rotate_r(node->parent);
        } else if (side == 1 && parent_side == 1) {
          splay_rotate_l(node->parent->parent);
          splay_rotate_l(node->parent);
        } else if (side == 0 && parent_side == 1) {
          splay_rotate_r(node->parent);
          splay_rotate_l(node->parent);
        } else if (side == 1 && parent_side == 0) {
          splay_rotate_l(node->parent);
          splay_rotate_r(node->parent);
        }
      }
  }
  return node;
}

// Split a splay tree into everything <@new_head and everything >=@new_head.
static void splay_split(struct splaynode *new_head) {
    // (0) fix the nexts and prevs
    if (new_head->prev) new_head->prev->next = 0;
    new_head->prev = 0;

    splay_splay(new_head);
    if (new_head->left) new_head->left->parent = 0;
    new_head->left = 0;
}

// Join two splay trees, given the largest element in one (@tail) and the
// smallest element in the other (@head), assuming @tail < @head.
static void splay_join(struct splaynode *tail, struct splaynode *head) {
    // (0) fix the nexts and prevs
    gcassert(!(head->prev || tail->next));
    tail->next = head;
    head->prev = tail;

    // (1) splay up the min element in head and make tail its left child
    splay_splay(head);
    gcassert(!(head->left));
    while (tail->parent) tail = tail->parent;
    SET_CHILD(head, left, tail);
}

// Find the smallest element in the tree containing @node by splaying it to the
// root, following left pointers, and then splaying again.
static struct splaynode *splay_head(struct splaynode *node) {
    splay_splay(node);
    while (node->left) node = node->left;
    splay_splay(node);
    return node;
}
