#pragma once

#include <assert.h>
#if 0
#undef assert
#define assert(...)
#endif
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>

extern unsigned long _GC_COUNT;
extern unsigned long _GC_TOTAL_HEAP_COUNT;
extern unsigned long _GC_N_FREES;
extern unsigned long _GC_HEAP_SIZE;
extern unsigned long _GC_N_REGIONS;
extern unsigned long _GC_D_GENERATION;

#if 1
#define gcassert(...) if (!(__VA_ARGS__)) __builtin_unreachable()
#else
#include <assert.h>
#define gcassert assert
#endif

#include "ett.h"

// Represents an edge in the heap graph.
struct _gc_link {
    struct _gc_descriptor *from, *to;
    // following the @from links maintains the @from field, following @to links
    // maintains the @to field.
    struct _gc_link *next_from, *next_to,
                    **pprev_from, **pprev_to;
};

// Represents a node in the heap graph.
struct _gc_descriptor {
    unsigned stack_refs;
    void *gco;

    struct ettnode node;

    struct _gc_link *outgoing_links, // this is an @from field
                    *incoming_links; // this is a @to field

    // if _D_generation or memo_generation = current generation, it's in the D
    // set or the memo set (respectively).
    unsigned long D_generation, memo_generation;
    struct _gc_link **plink_memo;
    struct _gc_descriptor *parent;

    char is_terminal : 1;
    char deleting : 1;
};
#define ETT2GCD(E) (struct _gc_descriptor *)((void*)(E) - offsetof(struct _gc_descriptor, node))
#define GCD2ETT(D) (struct ettnode *)((void*)(D) + offsetof(struct _gc_descriptor, node))

typedef unsigned long size_t;

// Insert the edge described by @edge into the heap graph.
static inline void _gc_insert_link(struct _gc_link *link) {
    link->next_from = link->from->outgoing_links;
    if (link->next_from) link->next_from->pprev_from = &(link->next_from);
    link->pprev_from = &(link->from->outgoing_links);
    link->from->outgoing_links = link;
    if (link->to->is_terminal) {
        link->to->stack_refs++;
    } else {
        link->next_to = link->to->incoming_links;
        if (link->next_to) link->next_to->pprev_to = &(link->next_to);
        link->pprev_to = &(link->to->incoming_links);
        link->to->incoming_links = link;
    }
}

// Remove the edge described by @edge from the heap graph.
static inline void _gc_remove_link(struct _gc_link *link) {
    *(link->pprev_from) = link->next_from;
    if (link->next_from) link->next_from->pprev_from = link->pprev_from;
    if (link->to->is_terminal) {
        link->to->stack_refs--;
    } else {
        *(link->pprev_to) = link->next_to;
        if (link->next_to) link->next_to->pprev_to = link->pprev_to;
    }
    free(link);
}

// True iff @d currently has no parent in the spanning forest (note that we
// don't represent GCRoot explicitly, so if d->stack_refs > 0 this means it
// should be interpreted as a direct child of GCRoot in the spanning tree).
static inline int _gc_is_root(struct _gc_descriptor *d) {
    return !(d->parent);
}

// Cut the node @to and its subtree out of its existing spanning tree.
static inline void _gc_deparent(struct _gc_descriptor *to) {
    to->parent = 0;
    ett_cut(&(to->node));
}

// Treat @plink.from -> @plink.to as the edge in the spanning forest, where
// @plink.to had no parent before (i.e., it was the root of its own spanning
// tree in the forest).
static inline void _gc_parent(struct _gc_link **plink) {
    struct _gc_link *link = *plink;
    struct _gc_descriptor *from = link->from,
                          *to = link->to;

    if (to->parent)
        _gc_deparent(to);
    to->parent = from;

    // update the ETT
    ett_link(&(from->node), &(to->node));
}

extern void luaC_delete(void *gco);
int mallctl(char *, void *, void *, void *, unsigned long long);

extern int _GC_IS_FINALIZING;
extern void luaC_finalize(void *);
// Given a tree in the spanning forest, finalize and free all nodes and their
// associated Lua objects.
static void _gc_free_subtree(struct _gc_descriptor *root) {
    struct ettnode *ett = GCD2ETT(root);
    unsigned n_killing = 0;
    _GC_IS_FINALIZING++;
    ETT_ITER(GCD2ETT(root), node_, {
        struct _gc_descriptor *node = ETT2GCD(node_);
        node->deleting = 1;
    })
    ETT_ITER(GCD2ETT(root), node_, {
        struct _gc_descriptor *node = ETT2GCD(node_);
        luaC_finalize(node->gco);
    })
    ETT_ITER(GCD2ETT(root), node_, {
        struct _gc_descriptor *node = ETT2GCD(node_);
        _GC_N_FREES++;
        // NOTE: incoming links are not our responsibility
        while (node->outgoing_links)
            _gc_remove_link(node->outgoing_links);
        _GC_TOTAL_HEAP_COUNT--;
        luaC_delete(node->gco);
        n_killing++;
    })
    ett_free_all(ett, -(int)offsetof(struct _gc_descriptor, node));
    _GC_IS_FINALIZING--;
}

// This implements the main loop of Algorithm 23.
// @scooby points to the node in the heap that has just been cut out of its
// previous spanning tree and is not directly reachable from the root. The job
// of _gc_rehome is to find it (and all of its children) a new parent (i.e.,
// spanning tree to join) so that it can be rejoined into a tree reachable from
// the root.
static void _gc_rehome(struct _gc_descriptor *scooby) {
    // ETT node corresponding to the node that we need to search for a new
    // parent.
    struct ettnode *scoob_ett = GCD2ETT(scooby),
                   *node_ = 0,
                   *root = 0;
    struct _gc_descriptor *node = 0;
    struct _gc_link **plink = 0,
                     *link = 0;
    // "keep_going" takes the place of 'c'
    for (int keep_going = 1; keep_going--; ) {
      // Has the effect of setting D = emptyset (see the documentation PDF).
      _GC_D_GENERATION++;
      // Loop over all the nodes in Euler tour order.
      for (struct splaynode *_ett_i = &(scoob_ett->start); _ett_i;) {
          // The ETT traversal will visit each node twice, but we only want to
          // visit each node once, so if this is the second visit of that node
          // (i.e., on the way back up the tree) we can just skip it.
          if (!(_ett_i->is_start)) goto next_node;
          node_ = container_of(_ett_i, struct ettnode, start);

          node = ETT2GCD(node_);
          // If there is an (implicit) edge from GCRoot to node, we can use
          // that edge to make the node its own tree in the spanning forest
          // (this is essentially the line 11--15 case but using implicit
          // GCRoot edges).
          if (node->stack_refs) {
              _ett_i = node_->end.next;
              if (node->parent)
                  _gc_deparent(node);
              keep_going = 1;
              continue;
          }

          // Otherwise, we need to search for a new parent @m meeting all of
          // the constraints on line 9 (has an edge to @node, is not in D, and
          // is not a descendant of @node).
          // So we're going to loop over incoming links:
          plink = &(node->incoming_links);
          // This implements the optimization described in Section 7.4;
          // basically, we skip incoming edges if we've already considered and
          // ruled them out earlier.
          if (node->memo_generation == _GC_D_GENERATION)
              plink = node->plink_memo;
          // Now we continue looping over incoming links looking for a new
          // parent @m.
          link = *plink;
          for (; link; plink = &(link->next_to), link = *plink) {
              // The candidate for @m we're considering is @m = @link->from.

              // If it's the old parent, we already know that doesn't work.
              if (link->from == node->parent) continue;

              // This condition means @link->from in D, hence it also doesn't
              // work.
              if (link->from->D_generation == _GC_D_GENERATION)
                  continue;

              // This checks whether @node is an ancestor of @link->from in the
              // ETT forest.
              if (node->parent) ett_cut(&(node->node));
              root = ett_root(GCD2ETT(link->from));
              if (node->parent) ett_link(&(node->parent->node), &(node->node));
              if (root == node_) continue;

              // If we're *not* their ancestor, and they're not in D, reparent
              // this node at that new parent (lines 11--15).
              _ett_i = node_->end.next;
              _gc_parent(plink);

              if (root != scoob_ett) {
                  // If we have reconnected the root of the subtree into the
                  // main spanning tree (forest), then that means all of the
                  // nodes have also been reconnected and we're done.
                  keep_going = 1;
              } else {
                  // Otherwise, set the memo and the memo generation before
                  // continuing so we avoid duplicate work later (again,
                  // Section 7.4 describes this optimization).
                  node->plink_memo = &((*plink)->next_to);
                  node->memo_generation = _GC_D_GENERATION;
              }

              if (node == scooby) return;
              goto immediate_next;
          }

          // If we reach here, we didn't find a new parent. Mark as part of D.
          node->D_generation = _GC_D_GENERATION;

next_node:
          _ett_i = _ett_i->next;
immediate_next:
          continue;
      }
    }
    // This is the equivalent of the line that adds all of D to GCFreeList,
    // except (1) we know at that point that D just contains everything left in
    // the subtree rooted at @scooby, and (2) we can immediately finalize and
    // free them, so we just call _gc_free_subtree to do the work for us.
    _gc_free_subtree(scooby);
    return;
}

// Helper function that decides when to call GCInsert (delta=+1) vs. GCDelete
// (delta=-1), modifies the heap graph accordingly, and calls the other GC
// methods.
static inline void _gc_delta_link(struct _gc_descriptor *from,
                                  struct _gc_descriptor *to,
                                  int delta) {
    struct _gc_link **plink = 0, *link = 0;
    int was_parent = 0;

    // Self loops and edges to nowhere can be ignored without affecting anything.
    if (from == to || !to) return;

    // We are representing edges from GCRoot implicitly, via a counter
    // "stack_refs," which gets updated here.
    if (!from) {
        to->stack_refs += delta;
        // If the stack reference counter drops to zero and this was the root
        // of its own tree in the spanning forest (i.e., it was a child of
        // GCRoot in the spanning forest), we need to find a new parent for it.
        if (!(to->stack_refs) && _gc_is_root(to))
            _gc_rehome(to);
        return;
    }
    // Terminal nodes are promised to have no outgoing edges.
    gcassert(!(from->is_terminal));

    // If we are removing an edge, find that edge in the outgoing list of
    // @from. (NOTE: this can take asymptotically bad time in a dense graph,
    // but that can be solved by using a hashmap or balanced tree to store the
    // edge lists in addition to a doubly linked list. We did not do so in this
    // implementation to keep it as small as possible.)
    plink = &(from->outgoing_links);
    link = *plink;
    if (delta < 0)
        for (; link && link->to != to;
             plink = &(link->next_from), link = *plink)
          ;
    else link = 0;

    // If we're adding a new edge, we need to allocate a link for it.
    if (delta > 0 && !link) {
        link = (struct _gc_link *)calloc(1, sizeof(*link));
        link->from = from;
        link->to = to;
        _gc_insert_link(link);
    } else if (!link) { gcassert(0); }

    // And then we're done, if we're adding a new edge, since new edges can
    // never break an existing spanning tree.
    if (delta > 0) return;

    // Otherwise, we're deleting an edge. First, remember whether this edge was
    // part of the spanning tree.
    was_parent = (from == to->parent);
    // Then remove it.
    _gc_remove_link(*plink);

    // If this was a terminal node, all that matters is the ref count.
    if (to->is_terminal) {
      if (!(to->stack_refs)) _gc_rehome(to);
      return;
    }

    // If we just removed an edge from the spanning tree, we need to try and
    // find a new way to link that newly separated subtree into the spanning
    // forest.
    if (was_parent) {
        _gc_deparent(to);
        if (!(to->stack_refs)) _gc_rehome(to);
        return;
    }
}

// Used as the main interface to the GC to indicate writing to a pointer on
// source region @via, pointing to destination region @new_value, and
// overwriting an old pointer that was pointing to @old_value.
static inline void _gc_overwrite_pointer(
            struct _gc_descriptor *via,
            struct _gc_descriptor *old_value,
            struct _gc_descriptor *new_value) {
  // avoid GC operations while finalizing
  if (via && via->deleting) return;
  // Do the GCInsert.
  if (new_value && !(new_value->deleting))
      _gc_delta_link(via, new_value, +1);
  // Do the GCDelete.
  if (old_value && !(old_value->deleting))
      _gc_delta_link(via, old_value, -1);
}

// Builds the _gc_descriptor in-place.
static inline struct _gc_descriptor *_gc_new_inplace(struct _gc_descriptor *d) {
    ett_singleton_inplace(GCD2ETT(d));
    _GC_TOTAL_HEAP_COUNT++;
    return d;
}

// Builds a _gc_descriptor.
static inline struct _gc_descriptor *_gc_new(void) {
    struct _gc_descriptor *d = (struct _gc_descriptor *)calloc(1, sizeof(*d));
    return _gc_new_inplace(d);
}

// Inform the GCDS that this node can have no outgoing edges (basically only
// used for strings). Because such nodes can never be part of a cycle, we
// simply use reference counting to handle them.
static inline void _gc_set_terminal(struct _gc_descriptor *d) {
    if (d) d->is_terminal = 1;
}
