#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <stdarg.h>
#include "gcds/gc.h"
#include "gcds/gc_globals.h"
unsigned long _GC_N_FREES;
unsigned long _GC_COUNT;
unsigned long _GC_HEAP_SIZE;
unsigned long _GC_TOTAL_HEAP_COUNT;
unsigned long _GC_N_REGIONS;
int _GC_IS_FINALIZING;
unsigned long _GC_D_GENERATION;

// assuming @dst is a location on the stack, gc_assign(dst, expr) will
// overwrite its value with @expr and inform the GCDS of this stack change.
// it's crucial that we pop *after* pushing in this sequence, because otherwise
// gc_assign(x, x) could cause @x to be deleted (if it was the only reference).
#define gc_assign(dst, ...) ({ object *old_##dst = dst; dst = __VA_ARGS__; gc_push(dst); gc_pop(old_##dst); dst; })
// same as gc_assign, except assumes that @expr has already been pushed. this
// should be used when @expr is the return value of a function call, because
// the return value of a function call was already on the stack at the moment
// of return (see documentation PDF).
#define gc_assign2(dst, ...) ({ object *old_##dst = dst; dst = __VA_ARGS__; gc_pop(old_##dst); dst; })

typedef enum { T_CONS, T_ATOM, T_CFUNC, T_LAMBDA } object_tag;

struct object_t;
typedef struct object_t *(*cfunc)(struct object_t *);

typedef struct object_t {
	struct object_t *car, *cdr;
        struct _gc_descriptor *gcd;
	object_tag tag;
} object;

#define TOKEN_MAX 256
#define HASHMAP_SIZE 2048
#define ATOMCHAR(ch) (((ch) >= '!' && (ch) <= '\'') || ((ch) >= '*' && (ch) <= '~'))
#define TEXT(x) (((x) && (x)->tag == T_ATOM) ? ((const char *)((x)->car)) : "")
#define HEAPSIZE 16384
#define MAXROOTS 500
#define MAXFRAMES 50

const char *TQUOTE = NULL, *TLAMBDA = NULL, *TCOND = NULL, *TDEFINE = NULL;
char        token_text[TOKEN_MAX];
int         token_peek = 0;
object     *atom_t = NULL;
object *heap, *tospace, *fromspace, *allocptr, *scanptr;
object ** roots[MAXROOTS];
object fwdmarker = { .tag = T_ATOM, .car = 0, .cdr = 0 };

object *gc_alloc(object_tag tag, object *car, object *cdr);
object *gc_push(object *r);
void    gc_pop(object *r);
// result has extra stackref
object *lisp_read_list(const char *tok, FILE *in);
// result has extra stackref
object *lisp_read_obj(const char *tok, FILE *in);
// result has extra stackref
object *lisp_read(FILE *in);
void    lisp_print(object *obj);
// result has extra stackref
object *lisp_eval(object *obj, object *env);

size_t djbhash(const unsigned char *str) {
    size_t hash = 5381;
    for (int c = *str++; c; c = *str++)
        hash = (hash << 5) + hash + c;
    return hash;
}

const char *intern_string(const char *str) {
	typedef struct node { struct node *next; char data[]; } node_t;
	static node_t* nodes[HASHMAP_SIZE] = {0};
	size_t hash = djbhash((const unsigned char *)str) % HASHMAP_SIZE;
	for (node_t* is = nodes[hash]; is != NULL; is = is->next)
		if (strcmp(is->data, str) == 0)
			return is->data;
	size_t sz = strlen(str) + 1;
	node_t *item = malloc(sizeof(node_t) + sz);
	memcpy(item->data, str, sz);
	item->next = nodes[hash];
	nodes[hash] = item;
	return item->data;
}

int match_number(const char *s) {
	if (*s == '-' || *s == '+') s++;
	do { if (*s < '0' || *s > '9') return 0; } while (*++s != '\0');
	return 1;
}

const char* itos(long n) {
	char buf[TOKEN_MAX], reversed[TOKEN_MAX];
	char *p1 = buf, *p2 = reversed;
	unsigned long u = (unsigned long)n;
	if (n < 0) { *p1++ = '-'; u = ~u + 1; }
	do { *p2++ = (char)(u % 10) + '0'; u /= 10; } while (u > 0);
	do { *p1++ = *--p2; } while (p2 != reversed);
	*p1 = '\0';
	return intern_string(buf);
}

object *new_cfunc(cfunc func) {
	return gc_push(gc_alloc(T_CFUNC, (object *)func, NULL));
}

object *new_atom(const char *str) {
        return gc_push(gc_alloc(T_ATOM, (object *)intern_string(str), NULL));
}

object *new_cons(object *car, object *cdr) {
	return gc_push(gc_alloc(T_CONS, car, cdr));
}

const char *read_token(FILE *in) {
	int n = 0;
	while (isspace(token_peek))
		token_peek = fgetc(in);
	if (token_peek == '(' || token_peek == ')') {
		token_text[n++] = token_peek;
		token_peek = fgetc(in);
	} else while (ATOMCHAR(token_peek)) {
		if (n == TOKEN_MAX)
			abort();
		token_text[n++] = token_peek;
		token_peek = fgetc(in);
	}
	if (token_peek == EOF)
		exit(0);
	token_text[n] = '\0';
	return intern_string(token_text);
}

// stackref +1
object *lisp_read_obj(const char *tok, FILE *in) {
	return (tok[0] != '(') ? new_atom(tok) :
		lisp_read_list(read_token(in), in);
}

// stackref +1
object *lisp_read_list(const char *tok, FILE *in) {
	if (tok[0] == ')')
		return NULL;
	object *obj = NULL, *tmp = NULL, *obj2 = NULL;
        gc_assign2(obj, lisp_read_obj(tok, in));
	tok = read_token(in);
	if (tok[0] == '.' && tok[1] == '\0') {
		tok = read_token(in);
                gc_assign2(tmp, lisp_read_obj(tok, in));
                gc_assign2(obj2, new_cons(obj, tmp));
		tok = read_token(in);
                gc_pop(obj);
                gc_pop(tmp);
		if (tok[0] == ')')
			return obj2;
		fputs("Error: Malformed dotted cons\n", stderr);
                gc_pop(obj2);
		return NULL;
	}
        gc_assign2(tmp, lisp_read_list(tok, in));
        gc_assign2(obj2, new_cons(obj, tmp));
        gc_pop(obj);
        gc_pop(tmp);
	return obj2;
}

// stackref +1
object *lisp_read(FILE *in) {
	const char *tok = read_token(in);
	if (tok == NULL)
		return NULL;
	if (tok[0] != ')')
		return lisp_read_obj(tok, in);
	fputs("Error: Unexpected )\n", stderr);
	return NULL;
}

int lisp_equal(object *a, object *b) {
	if (a == b)
		return 1;
	if (a == NULL || b == NULL || a->tag != b->tag)
		return 0;
	if (a->tag != T_CONS)
		return a->car == b->car;
	return lisp_equal(a->car, b->car) && lisp_equal(a->cdr, b->cdr);
}

object *list_find_pair(object *needle, object *haystack) {
	for (; haystack != NULL; haystack = haystack->cdr)
		if (haystack->car != NULL && lisp_equal(needle, haystack->car->car))
			return haystack->car;
	return NULL;
}

object *env_lookup(object *needle, object *haystack) {
	for (object *pair; haystack != NULL; haystack = haystack->cdr)
		if ((pair = list_find_pair(needle, haystack->car)) != NULL)
			return pair->cdr;
	return NULL;
}

object *env_set(object *env, object *key, object *value) {
	object *pair = NULL, *frame = NULL;
	gc_assign2(pair, new_cons(key, value));
	gc_assign2(frame, new_cons(pair, env->car));

        _gc_overwrite_pointer(env->gcd, env->car ? env->car->gcd : 0,
                              frame ? frame->gcd : 0);
	env->car = frame;

        gc_pop(pair);
        gc_pop(frame);
	return env;
}

// return value is +1 in the rootset
object *list_reverse(object *lst) {
	if (lst == NULL)
		return NULL;
	object *prev = NULL, *curr = lst, *next = lst->cdr;
        gc_push(curr);
        gc_push(next);
	while (curr) {
                _gc_overwrite_pointer(curr->gcd, curr->cdr ? curr->cdr->gcd : 0,
                                      prev ? prev->gcd : 0);
		curr->cdr = prev;
		gc_assign(prev, curr);
		gc_assign(curr, next);
		if (next != NULL)
			gc_assign(next, next->cdr);
	}
        gc_pop(curr);
        gc_pop(next);
	return prev;
}

// result has +1 stackref
object *lisp_eval(object *expr, object *env) {
restart:
	if (expr == NULL)
		return gc_push(expr);
	if (expr->tag == T_ATOM)
		return gc_push(match_number(TEXT(expr)) ? expr : env_lookup(expr, env));
	if (expr->tag != T_CONS)
		return gc_push(expr);
	object *head = expr->car;
        gc_push(head);
        gc_push(expr);
	if (TEXT(head) == TQUOTE) {
                gc_push(expr->cdr->car);
                gc_pop(head);
                gc_pop(expr);
		return expr->cdr->car;
	} else if (TEXT(head) == TCOND) {
		object *item = NULL, *cond = NULL;
		for (gc_assign(item, expr->cdr); item != NULL; gc_assign(item, item->cdr)) {
			gc_assign(cond, item->car);
                        object *e = lisp_eval(cond->car, env);
			if (e != NULL) {
				gc_assign(expr, cond->cdr->car);
                                gc_pop(e);
                                gc_pop(head);
                                gc_pop(expr);
                                gc_pop(item);
                                gc_pop(cond);
				goto restart;
			}
		}
                gc_pop(head);
                gc_pop(expr);
                gc_pop(item);
                gc_pop(cond);
		return NULL; // was abort(), but no match should return nil
	} else if (TEXT(head) == TDEFINE) {
		object *name = NULL, *value = NULL;
                gc_assign(name, expr->cdr->car);
		gc_assign2(value, lisp_eval(expr->cdr->cdr->car, env));
		env_set(env, name, value);
                gc_pop(head);
                gc_pop(expr);
                gc_pop(name);
		return value;
	} else if (TEXT(head) == TLAMBDA) {
		expr->cdr->tag = T_LAMBDA;
                gc_push(expr->cdr);
                gc_pop(head);
                gc_pop(expr);
		return expr->cdr;
	}

	object *fn = NULL, *args = NULL, *params = NULL, *param = NULL;
        gc_push(env);
	gc_assign2(fn, lisp_eval(head, env));
	if (fn->tag == T_CFUNC) {
		for (gc_assign(params, expr->cdr); params != NULL; gc_assign(params, params->cdr)) {
			gc_assign2(param, lisp_eval(params->car, env));
			gc_assign2(args, new_cons(param, args));
		}
                object *rev = 0, *ret = 0;
                gc_assign2(rev, list_reverse(args));
		gc_assign2(ret, ((cfunc)fn->car)(rev));
                gc_pop(rev);
                gc_pop(param);
                gc_pop(args);
                gc_pop(env);
                gc_pop(head);
                gc_pop(expr);
		return ret;
	} else if (fn->tag == T_LAMBDA) {
		object *callenv = 0;
                gc_assign2(callenv, new_cons(NULL, env));
		gc_assign(args, fn->car);
		object *item = NULL;
		for (gc_assign(params, expr->cdr);
                     params != NULL;
                     gc_assign(params, params->cdr), gc_assign(args, args->cdr)) {
			gc_assign2(param, lisp_eval(params->car, env));
			env_set(callenv, args->car, param);
		}
		for (gc_assign(item, fn->cdr); item != NULL; gc_assign(item, item->cdr)) {
			if (item->cdr == NULL) {
				gc_assign(expr, item->car);
				gc_assign(env, callenv);
                                gc_pop(callenv);
                                gc_pop(item);
                                gc_pop(head);
                                gc_pop(expr);
                                gc_pop(args);
				goto restart;
			}
			gc_pop(lisp_eval(item->car, callenv));
		}
                gc_pop(callenv);
                gc_pop(item);
                gc_pop(env);
                gc_pop(head);
                gc_pop(expr);
                gc_pop(args);
	}
	return NULL;
}

void lisp_print(object *obj) {
	if (obj == NULL) {
		fputs("()", stdout);
	} else if (obj->tag == T_ATOM) {
		fputs(TEXT(obj), stdout);
	} else if (obj->tag == T_CFUNC) {
		printf("<C@%p>", (void *)obj);
	} else if (obj->tag == T_LAMBDA) {
		fputs("<lambda ", stdout);
		lisp_print(obj->car);
		fputs(">", stdout);
	} else if (obj->tag == T_CONS) {
		fputs("(", stdout);
		for (;;) {
			lisp_print(obj->car);
			if (obj->cdr == NULL)
				break;
			fputs(" ", stdout);
			if (obj->cdr->tag != T_CONS) {
				fputs(". ", stdout);
				lisp_print(obj->cdr);
				break;
			}
			obj = obj->cdr;
		}
		fputs(")", stdout);
	}
}

object *builtin_car(object *args) {
	return gc_push(args->car->car);
}

object *builtin_cdr(object *args) {
	return gc_push(args->car->cdr);
}

object *builtin_cons(object *args) {
	return new_cons(args->car, args->cdr->car);
}

object *builtin_equal(object *args) {
	object *cmp = args->car;
	for (args = args->cdr; args != NULL; args = args->cdr)
		if (!lisp_equal(cmp, args->car))
			return NULL;
	return gc_push(atom_t);
}

object *builtin_pair(object *args) {
	return gc_push((args->car != NULL && args->car->tag == T_CONS) ? atom_t : NULL);
}

object *builtin_null(object *args) {
	return gc_push((args->car == NULL) ? atom_t : NULL);
}

object *builtin_sum(object *args) {
	long sum = 0;
	for (; args != NULL; args = args->cdr)
		sum += atol(TEXT(args->car));
	return new_atom(itos(sum));
}

object *builtin_sub(object *args) {
	long n;
	if (args->cdr == NULL) {
		n = -atol(TEXT(args->car));
	} else {
		n = atol(TEXT(args->car));
		for (args = args->cdr; args != NULL; args = args->cdr)
			n = n - atol(TEXT(args->car));
	}
	return new_atom(itos(n));
}

object *builtin_mul(object *args) {
	long sum = 1;
	for (; args != NULL; args = args->cdr)
		sum *= atol(TEXT(args->car));
	return new_atom(itos(sum));
}

object *builtin_display(object *args) {
	lisp_print(args->car);
	return NULL;
}

object *builtin_newline(object *args) {
	puts("");
	return NULL;
}

object *builtin_read(object *args) {
	return lisp_read(stdin);
}

void defun(object *env, const char *name, cfunc fn) {
	object *key = NULL, *val = NULL;
        gc_assign2(key, new_atom(name));
	gc_assign2(val, new_cfunc(fn));
	env_set(env, key, val);
        gc_pop(key);
        gc_pop(val);
}

void gc_collect(void) { }

// allocate a new object
object *gc_alloc(object_tag tag, object *car, object *cdr) {
	if (_GC_COUNT++ > HEAPSIZE) {
		fputs("Out of memory\n", stderr);
		abort();
	}

        object *o = calloc(1, sizeof(*o));
        o->gcd = _gc_new();

        if (tag == T_ATOM || tag == T_CFUNC) _gc_set_terminal(o->gcd);
        if (tag == T_CONS) {
            if (car) _gc_overwrite_pointer(o->gcd, 0, car->gcd);
            if (cdr) _gc_overwrite_pointer(o->gcd, 0, cdr->gcd);
        }
        o->gcd->gco = o;
	o->tag = tag;
	o->car = car;
	o->cdr = cdr;
	return o;
}
// add a stack (root set) reference to @r
object *gc_push(object *r) { if (r && r->gcd) _gc_overwrite_pointer(0, 0, r->gcd); return r; }
// remove a stack (root set) reference to @r
void gc_pop(object *r) { if (r && r->gcd) _gc_overwrite_pointer(0, r->gcd, 0); }

extern void luaC_delete(void *gco) { free(gco); _GC_COUNT--; }
extern void luaC_finalize(void *gco) { }

int main(int argc, char* argv[]) {
	TQUOTE = intern_string("quote");
	TLAMBDA = intern_string("lambda");
	TCOND = intern_string("cond");
	TDEFINE = intern_string("define");
	memset(token_text, 0, TOKEN_MAX);
	token_peek = ' ';

	object *env = NULL, *atom_f = NULL, *obj = NULL;
	gc_assign2(env, new_cons(NULL, NULL));
	gc_assign2(atom_t, new_atom("#t"));
	gc_assign2(atom_f, new_atom("#f"));
	env_set(env, atom_t, atom_t);
	env_set(env, atom_f, NULL);
	defun(env, "car", &builtin_car);
	defun(env, "cdr", &builtin_cdr);
	defun(env, "cons", &builtin_cons);
	defun(env, "equal?", &builtin_equal);
	defun(env, "pair?", &builtin_pair);
	defun(env, "null?", &builtin_null);
	defun(env, "+", &builtin_sum);
	defun(env, "-", &builtin_sub);
	defun(env, "*", &builtin_mul);
	defun(env, "display", &builtin_display);
	defun(env, "newline", &builtin_newline);
	defun(env, "read", &builtin_read);
	FILE *in = (argc > 1) ? fopen(argv[1], "r") : stdin;
	for (;;) {
		gc_assign2(obj, lisp_read(in));
	        gc_assign2(obj, lisp_eval(obj, env));
	        if (in == stdin) {
	        	lisp_print(obj);
	        	puts("");
	        }
                gc_assign(obj, 0);
	}
	return 0;
}
